"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateRedirectResponse = exports.validateFixedResponse = exports.ApplicationListenerRule = exports.ContentType = void 0;
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * The content type for a fixed response
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule
 */
class ApplicationListenerRule extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.legacyConditions = {};
        this.conditions = props.conditions || [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (this.conditions.length === 0 && !props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'conditions\', \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * Add a non-standard condition to this rule
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field, values) {
        if (values === undefined) {
            delete this.legacyConditions[field];
            return;
        }
        this.legacyConditions[field] = values;
    }
    /**
     * Add a non-standard condition to this rule
     */
    addCondition(condition) {
        this.conditions.push(condition);
    }
    /**
     * Configure the action to perform for this rule
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            this.node.addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * Add a TargetGroup to load balance to
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * Validate the rule
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        const legacyConditionFields = Object.keys(this.legacyConditions);
        if (legacyConditionFields.length === 0 && this.conditions.length === 0) {
            return ['Listener rule needs at least one condition'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const legacyConditions = Object.entries(this.legacyConditions).map(([field, values]) => {
            return { field, values };
        });
        const conditions = this.conditions.map(condition => condition.renderRawCondition());
        return [
            ...legacyConditions,
            ...conditions,
        ];
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,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