import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import * as ec2 from '@aws-cdk/aws-ec2';
import { Construct, Duration } from '@aws-cdk/core';
import { BaseLoadBalancer, BaseLoadBalancerProps, ILoadBalancerV2 } from '../shared/base-load-balancer';
import { IpAddressType } from '../shared/enums';
import { ApplicationListener, BaseApplicationListenerProps } from './application-listener';
/**
 * Properties for defining an Application Load Balancer
 */
export interface ApplicationLoadBalancerProps extends BaseLoadBalancerProps {
    /**
     * Security group to associate with this load balancer
     *
     * @default A security group is created
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * The type of IP addresses to use
     *
     * Only applies to application load balancers.
     *
     * @default IpAddressType.Ipv4
     */
    readonly ipAddressType?: IpAddressType;
    /**
     * Indicates whether HTTP/2 is enabled.
     *
     * @default true
     */
    readonly http2Enabled?: boolean;
    /**
     * The load balancer idle timeout, in seconds
     *
     * @default 60
     */
    readonly idleTimeout?: Duration;
}
/**
 * Define an Application Load Balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
export declare class ApplicationLoadBalancer extends BaseLoadBalancer implements IApplicationLoadBalancer {
    /**
     * Import an existing Application Load Balancer
     */
    static fromApplicationLoadBalancerAttributes(scope: Construct, id: string, attrs: ApplicationLoadBalancerAttributes): IApplicationLoadBalancer;
    readonly connections: ec2.Connections;
    readonly ipAddressType?: IpAddressType;
    private readonly securityGroup;
    constructor(scope: Construct, id: string, props: ApplicationLoadBalancerProps);
    /**
     * Add a new listener to this load balancer
     */
    addListener(id: string, props: BaseApplicationListenerProps): ApplicationListener;
    /**
     * Return the given named metric for this Application Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of concurrent TCP connections active from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricActiveConnectionCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of TLS connections initiated by the client that did not
     * establish a session with the load balancer. Possible causes include a
     * mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricClientTlsNegotiationErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of fixed-response actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpFixedResponseCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of redirect actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of redirect actions that couldn't be completed because the URL
     * in the response location header is larger than 8K.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectUrlLimitExceededCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of HTTP 3xx/4xx/5xx codes that originate from the load balancer.
     *
     * This does not include any response codes generated by the targets.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeElb(code: HttpCodeElb, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets
     * in the load balancer.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code: HttpCodeTarget, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of bytes processed by the load balancer over IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6ProcessedBytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of IPv6 requests received by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of new TCP connections established from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricNewConnectionCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The total number of bytes processed by the load balancer over IPv4 and IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of connections that were rejected because the load balancer had
     * reached its maximum number of connections.
     *
     * @default Sum over 5 minutes
     */
    metricRejectedConnectionCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of rules processed by the load balancer given a request rate averaged over an hour.
     *
     * @default Sum over 5 minutes
     */
    metricRuleEvaluations(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of user authentications that could not be completed
     *
     * Because an authenticate action was misconfigured, the load balancer
     * couldn't establish a connection with the IdP, or the load balancer
     * couldn't complete the authentication flow due to an internal error.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthError(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of user authentications that could not be completed because the
     * IdP denied access to the user or an authorization code was used more than
     * once.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthFailure(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time elapsed, in milliseconds, to query the IdP for the ID token and user info.
     *
     * If one or more of these operations fail, this is the time to failure.
     *
     * @default Average over 5 minutes
     */
    metricElbAuthLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The number of authenticate actions that were successful.
     *
     * This metric is incremented at the end of the authentication workflow,
     * after the load balancer has retrieved the user claims from the IdP.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthSuccess(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
/**
 * Count of HTTP status originating from the load balancer
 *
 * This count does not include any response codes generated by the targets.
 */
export declare enum HttpCodeElb {
    /**
     * The number of HTTP 3XX redirection codes that originate from the load balancer.
     */
    ELB_3XX_COUNT = "HTTPCode_ELB_3XX_Count",
    /**
     * The number of HTTP 4XX client error codes that originate from the load balancer.
     *
     * Client errors are generated when requests are malformed or incomplete.
     * These requests have not been received by the target. This count does not
     * include any response codes generated by the targets.
     */
    ELB_4XX_COUNT = "HTTPCode_ELB_4XX_Count",
    /**
     * The number of HTTP 5XX server error codes that originate from the load balancer.
     */
    ELB_5XX_COUNT = "HTTPCode_ELB_5XX_Count"
}
/**
 * Count of HTTP status originating from the targets
 */
export declare enum HttpCodeTarget {
    /**
     * The number of 2xx response codes from targets
     */
    TARGET_2XX_COUNT = "HTTPCode_Target_2XX_Count",
    /**
     * The number of 3xx response codes from targets
     */
    TARGET_3XX_COUNT = "HTTPCode_Target_3XX_Count",
    /**
     * The number of 4xx response codes from targets
     */
    TARGET_4XX_COUNT = "HTTPCode_Target_4XX_Count",
    /**
     * The number of 5xx response codes from targets
     */
    TARGET_5XX_COUNT = "HTTPCode_Target_5XX_Count"
}
/**
 * An application load balancer
 */
export interface IApplicationLoadBalancer extends ILoadBalancerV2, ec2.IConnectable {
    /**
     * The ARN of this load balancer
     */
    readonly loadBalancerArn: string;
    /**
     * The VPC this load balancer has been created in (if available).
     * If this interface is the result of an import call to fromApplicationLoadBalancerAttributes,
     * the vpc attribute will be undefined unless specified in the optional properties of that method.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The IP Address Type for this load balancer
     *
     * @default IpAddressType.IPV4
     */
    readonly ipAddressType?: IpAddressType;
    /**
     * Add a new listener to this load balancer
     */
    addListener(id: string, props: BaseApplicationListenerProps): ApplicationListener;
}
/**
 * Properties to reference an existing load balancer
 */
export interface ApplicationLoadBalancerAttributes {
    /**
     * ARN of the load balancer
     */
    readonly loadBalancerArn: string;
    /**
     * ID of the load balancer's security group
     */
    readonly securityGroupId: string;
    /**
     * The canonical hosted zone ID of this load balancer
     *
     * @default - When not provided, LB cannot be used as Route53 Alias target.
     */
    readonly loadBalancerCanonicalHostedZoneId?: string;
    /**
     * The DNS name of this load balancer
     *
     * @default - When not provided, LB cannot be used as Route53 Alias target.
     */
    readonly loadBalancerDnsName?: string;
    /**
     * Whether the security group allows all outbound traffic or not
     *
     * Unless set to `false`, no egress rules will be added to the security group.
     *
     * @default true
     */
    readonly securityGroupAllowsAllOutbound?: boolean;
    /**
     * The VPC this load balancer has been created in, if available
     *
     * @default - If the Load Balancer was imported and a VPC was not specified,
     * the VPC is not available.
     */
    readonly vpc?: ec2.IVpc;
}
