"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationTargetGroup = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define an Application Target Group
 */
class ApplicationTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props = {}) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        super(scope, id, { ...props }, {
            protocol,
            port,
        });
        this.protocol = protocol;
        this.port = port;
        this.connectableMembers = [];
        this.listeners = [];
        if (props) {
            if (props.slowStart !== undefined) {
                this.setAttribute('slow_start.duration_seconds', props.slowStart.toSeconds().toString());
            }
            if (props.stickinessCookieDuration !== undefined) {
                this.enableCookieStickiness(props.stickinessCookieDuration);
            }
            this.addTarget(...(props.targets || []));
        }
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedApplicationTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing target group
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return ApplicationTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Enable sticky routing via a cookie to members of this target group
     */
    enableCookieStickiness(duration) {
        this.setAttribute('stickiness.enabled', 'true');
        this.setAttribute('stickiness.type', 'lb_cookie');
        this.setAttribute('stickiness.lb_cookie.duration_seconds', duration.toSeconds().toString());
    }
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     */
    registerConnectable(connectable, portRange) {
        portRange = portRange || ec2.Port.tcp(this.defaultPort);
        // Notify all listeners that we already know about of this new connectable.
        // Then remember for new listeners that might get added later.
        this.connectableMembers.push({ connectable, portRange });
        for (const listener of this.listeners) {
            listener.registerConnectable(connectable, portRange);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener, associatingConstruct) {
        // Notify this listener of all connectables that we know about.
        // Then remember for new connectables that might get added later.
        for (const member of this.connectableMembers) {
            listener.registerConnectable(member.connectable, member.portRange);
        }
        this.listeners.push(listener);
        this.loadBalancerAttachedDependencies.add(associatingConstruct || listener);
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Return the given named metric for this Application Load Balancer Target Group
     *
     * Returns the metric for this target group from the point of view of the first
     * load balancer load balancing to it. If you have multiple load balancers load
     * sending traffic to the same target group, you will have to override the dimensions
     * on this metric.
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: {
                TargetGroup: this.targetGroupFullName,
                LoadBalancer: this.firstLoadBalancerFullName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of IPv6 requests received by the target group
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of healthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of unhealthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricUnhealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in this target group.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The average number of requests received by each target in a target group.
     *
     * The only valid statistic is Sum. Note that this represents the average not the sum.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCountPerTarget(props) {
        return this.metric('RequestCountPerTarget', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    validate() {
        const ret = super.validate();
        if (this.targetType !== undefined && this.targetType !== enums_1.TargetType.LAMBDA
            && (this.protocol === undefined || this.port === undefined)) {
            ret.push('At least one of \'port\' or \'protocol\' is required for a non-Lambda TargetGroup');
        }
        if (this.healthCheck && this.healthCheck.protocol && !ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
            ret.push([
                `Health check protocol '${this.healthCheck.protocol}' is not supported. `,
                `Must be one of [${ALB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(''));
        }
        return ret;
    }
}
exports.ApplicationTargetGroup = ApplicationTargetGroup;
/**
 * An imported application target group
 */
class ImportedApplicationTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener, _associatingConstruct) {
        // Nothing to do, we know nothing of our members
        this.node.addWarning('Cannot register listener on imported target group -- security groups might need to be updated manually');
    }
    registerConnectable(_connectable, _portRange) {
        this.node.addWarning('Cannot register connectable on imported target group -- security groups might need to be updated manually');
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const ALB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
//# sourceMappingURL=data:application/json;base64,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