"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
const helpers_1 = require("../helpers");
class ResourceWithLBDependency extends cdk.CfnResource {
    constructor(scope, id, targetGroup) {
        super(scope, id, { type: 'Test::Resource' });
        this.node.addDependency(targetGroup.loadBalancerAttached);
    }
}
module.exports = {
    'Listener guesses protocol from port'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            certificateArns: ['bla'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS',
        }));
        test.done();
    },
    'Listener guesses port from protocol'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            protocol: elbv2.ApplicationProtocol.HTTP,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 80,
        }));
        test.done();
    },
    'Listener default to open - IPv4'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const loadBalancer = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        loadBalancer.addListener('MyListener', {
            port: 80,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    Description: 'Allow from anyone on port 80',
                    CidrIp: '0.0.0.0/0',
                    FromPort: 80,
                    IpProtocol: 'tcp',
                    ToPort: 80,
                },
            ],
        }));
        test.done();
    },
    'Listener default to open - IPv4 and IPv6 (dualstack)'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const loadBalancer = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc, ipAddressType: elbv2.IpAddressType.DUAL_STACK });
        // WHEN
        loadBalancer.addListener('MyListener', {
            port: 80,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    Description: 'Allow from anyone on port 80',
                    CidrIp: '0.0.0.0/0',
                    FromPort: 80,
                    IpProtocol: 'tcp',
                    ToPort: 80,
                },
                {
                    Description: 'Allow from anyone on port 80',
                    CidrIpv6: '::/0',
                    FromPort: 80,
                    IpProtocol: 'tcp',
                    ToPort: 80,
                },
            ],
        }));
        test.done();
    },
    'HTTPS listener requires certificate'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        const errors = cdk.ConstructNode.validate(stack.node);
        test.deepEqual(errors.map(e => e.message), ['HTTPS Listener needs at least one certificate (call addCertificates)']);
        test.done();
    },
    'HTTPS listener can add certificate after construction'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        listener.addCertificateArns('Arns', ['cert']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Certificates: [
                { CertificateArn: 'cert' },
            ],
        }));
        test.done();
    },
    'Can configure targetType on TargetGroups'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', {
            vpc,
            port: 80,
            targetType: elbv2.TargetType.IP,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            TargetType: 'ip',
        }));
        test.done();
    },
    'Can configure name on TargetGroups'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', {
            vpc,
            port: 80,
            targetGroupName: 'foo',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            Name: 'foo',
        }));
        test.done();
    },
    'Can add target groups with and without conditions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        // WHEN
        listener.addTargetGroups('Default', {
            targetGroups: [group],
        });
        listener.addTargetGroups('WithPath', {
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [group],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    TargetGroupArn: { Ref: 'TargetGroup3D7CD9B8' },
                    Type: 'forward',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 10,
            Conditions: [
                {
                    Field: 'path-pattern',
                    Values: ['/hello'],
                },
            ],
            Actions: [
                {
                    TargetGroupArn: { Ref: 'TargetGroup3D7CD9B8' },
                    Type: 'forward',
                },
            ],
        }));
        test.done();
    },
    'Can implicitly create target groups with and without conditions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        listener.addTargets('Targets', {
            port: 80,
            targets: [new elbv2.InstanceTarget('i-12345')],
        });
        listener.addTargets('WithPath', {
            priority: 10,
            pathPattern: '/hello',
            port: 80,
            targets: [new elbv2.InstanceTarget('i-5678')],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    TargetGroupArn: { Ref: 'LBListenerTargetsGroup76EF81E8' },
                    Type: 'forward',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            VpcId: { Ref: 'Stack8A423254' },
            Port: 80,
            Protocol: 'HTTP',
            Targets: [
                { Id: 'i-12345' },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Actions: [
                {
                    TargetGroupArn: { Ref: 'LBListenerWithPathGroupE889F9E5' },
                    Type: 'forward',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            VpcId: { Ref: 'Stack8A423254' },
            Port: 80,
            Protocol: 'HTTP',
            Targets: [
                { Id: 'i-5678' },
            ],
        }));
        test.done();
    },
    'Add certificate to constructed listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 443 });
        // WHEN
        listener.addCertificateArns('Arns', ['cert']);
        listener.addTargets('Targets', { port: 8080, targets: [new elbv2.IpTarget('1.2.3.4')] });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Certificates: [
                { CertificateArn: 'cert' },
            ],
        }));
        test.done();
    },
    'Add certificate to imported listener'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        const listener2 = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack2, 'Listener', {
            listenerArn: 'listener-arn',
            defaultPort: 443,
            securityGroupId: 'security-group-id',
        });
        // WHEN
        listener2.addCertificateArns('Arns', ['cert']);
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [
                { CertificateArn: 'cert' },
            ],
        }));
        test.done();
    },
    'Enable stickiness for targets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)],
        });
        group.enableCookieStickiness(cdk.Duration.hours(1));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            TargetGroupAttributes: [
                {
                    Key: 'stickiness.enabled',
                    Value: 'true',
                },
                {
                    Key: 'stickiness.type',
                    Value: 'lb_cookie',
                },
                {
                    Key: 'stickiness.lb_cookie.duration_seconds',
                    Value: '3600',
                },
            ],
        }));
        test.done();
    },
    'Enable health check for targets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)],
        });
        group.configureHealthCheck({
            unhealthyThresholdCount: 3,
            timeout: cdk.Duration.hours(1),
            interval: cdk.Duration.seconds(30),
            path: '/test',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            UnhealthyThresholdCount: 3,
            HealthCheckIntervalSeconds: 30,
            HealthCheckPath: '/test',
            HealthCheckTimeoutSeconds: 3600,
        }));
        test.done();
    },
    'validation error if invalid health check protocol'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        const group = listener.addTargets('Group', {
            port: 80,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc)],
        });
        group.configureHealthCheck({
            unhealthyThresholdCount: 3,
            timeout: cdk.Duration.hours(1),
            interval: cdk.Duration.seconds(30),
            path: '/test',
            protocol: elbv2.Protocol.TCP,
        });
        // THEN
        const validationErrors = group.validate();
        test.deepEqual(validationErrors, ["Health check protocol 'TCP' is not supported. Must be one of [HTTP, HTTPS]"]);
        test.done();
    },
    'Can call addTargetGroups on imported listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const listener = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack, 'Listener', {
            listenerArn: 'ieks',
            securityGroupId: 'sg-12345',
        });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        // WHEN
        listener.addTargetGroups('Gruuup', {
            priority: 30,
            hostHeader: 'example.com',
            targetGroups: [group],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            ListenerArn: 'ieks',
            Priority: 30,
            Actions: [
                {
                    TargetGroupArn: { Ref: 'TargetGroup3D7CD9B8' },
                    Type: 'forward',
                },
            ],
        }));
        test.done();
    },
    'Can depend on eventual listener via TargetGroup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const loadBalancer = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', { vpc });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        // WHEN
        new ResourceWithLBDependency(stack, 'SomeResource', group);
        loadBalancer.addListener('Listener', {
            port: 80,
            defaultTargetGroups: [group],
        });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                SomeResource: {
                    Type: 'Test::Resource',
                    DependsOn: ['LoadBalancerListenerE1A099B9'],
                },
            },
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'Exercise metrics'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        lb.addListener('SomeListener', {
            port: 80,
            defaultTargetGroups: [group],
        });
        // WHEN
        const metrics = new Array();
        metrics.push(group.metricHttpCodeTarget(elbv2.HttpCodeTarget.TARGET_3XX_COUNT));
        metrics.push(group.metricIpv6RequestCount());
        metrics.push(group.metricUnhealthyHostCount());
        metrics.push(group.metricUnhealthyHostCount());
        metrics.push(group.metricRequestCount());
        metrics.push(group.metricTargetConnectionErrorCount());
        metrics.push(group.metricTargetResponseTime());
        metrics.push(group.metricTargetTLSNegotiationErrorCount());
        for (const metric of metrics) {
            test.equal('AWS/ApplicationELB', metric.namespace);
            const loadBalancerArn = { Ref: 'LBSomeListenerCA01F1A0' };
            test.deepEqual(stack.resolve(metric.dimensions), {
                TargetGroup: { 'Fn::GetAtt': ['TargetGroup3D7CD9B8', 'TargetGroupFullName'] },
                LoadBalancer: { 'Fn::Join': ['',
                        [{ 'Fn::Select': [1, { 'Fn::Split': ['/', loadBalancerArn] }] },
                            '/',
                            { 'Fn::Select': [2, { 'Fn::Split': ['/', loadBalancerArn] }] },
                            '/',
                            { 'Fn::Select': [3, { 'Fn::Split': ['/', loadBalancerArn] }] },
                        ],
                    ],
                },
            });
        }
        test.done();
    },
    'Can add dependency on ListenerRule via TargetGroup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const loadBalancer = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', { vpc });
        const group1 = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup1', { vpc, port: 80 });
        const group2 = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup2', { vpc, port: 80 });
        const listener = loadBalancer.addListener('Listener', {
            port: 80,
            defaultTargetGroups: [group1],
        });
        // WHEN
        new ResourceWithLBDependency(stack, 'SomeResource', group2);
        listener.addTargetGroups('SecondGroup', {
            pathPattern: '/bla',
            priority: 10,
            targetGroups: [group2],
        });
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                SomeResource: {
                    Type: 'Test::Resource',
                    DependsOn: ['LoadBalancerListenerSecondGroupRuleF5FDC196'],
                },
            },
        }, assert_1.MatchStyle.SUPERSET);
        test.done();
    },
    'Can add fixed responses'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc,
        });
        const listener = lb.addListener('Listener', {
            port: 80,
        });
        // WHEN
        listener.addFixedResponse('Default', {
            contentType: elbv2.ContentType.TEXT_PLAIN,
            messageBody: 'Not Found',
            statusCode: '404',
        });
        listener.addFixedResponse('Hello', {
            priority: 10,
            pathPattern: '/hello',
            statusCode: '503',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    FixedResponseConfig: {
                        ContentType: 'text/plain',
                        MessageBody: 'Not Found',
                        StatusCode: '404',
                    },
                    Type: 'fixed-response',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Actions: [
                {
                    FixedResponseConfig: {
                        StatusCode: '503',
                    },
                    Type: 'fixed-response',
                },
            ],
        }));
        test.done();
    },
    'Can add redirect responses'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc,
        });
        const listener = lb.addListener('Listener', {
            port: 80,
        });
        // WHEN
        listener.addRedirectResponse('Default', {
            statusCode: 'HTTP_301',
            port: '443',
            protocol: 'HTTPS',
        });
        listener.addRedirectResponse('Hello', {
            priority: 10,
            pathPattern: '/hello',
            path: '/new/#{path}',
            statusCode: 'HTTP_302',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    RedirectConfig: {
                        Port: '443',
                        Protocol: 'HTTPS',
                        StatusCode: 'HTTP_301',
                    },
                    Type: 'redirect',
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Actions: [
                {
                    RedirectConfig: {
                        Path: '/new/#{path}',
                        StatusCode: 'HTTP_302',
                    },
                    Type: 'redirect',
                },
            ],
        }));
        test.done();
    },
    'Can configure deregistration_delay for targets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        // WHEN
        new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', {
            vpc,
            port: 80,
            deregistrationDelay: cdk.Duration.seconds(30),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            TargetGroupAttributes: [
                {
                    Key: 'deregistration_delay.timeout_seconds',
                    Value: '30',
                },
            ],
        }));
        test.done();
    },
    'Throws with bad fixed responses': {
        'status code'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc,
            });
            const listener = lb.addListener('Listener', {
                port: 80,
            });
            // THEN
            test.throws(() => listener.addFixedResponse('Default', {
                statusCode: '301',
            }), /`statusCode`/);
            test.done();
        },
        'message body'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc,
            });
            const listener = lb.addListener('Listener', {
                port: 80,
            });
            // THEN
            test.throws(() => listener.addFixedResponse('Default', {
                messageBody: 'a'.repeat(1025),
                statusCode: '500',
            }), /`messageBody`/);
            test.done();
        },
    },
    'Throws with bad redirect responses': {
        'status code'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc,
            });
            const listener = lb.addListener('Listener', {
                port: 80,
            });
            // THEN
            test.throws(() => listener.addRedirectResponse('Default', {
                statusCode: '301',
            }), /`statusCode`/);
            test.done();
        },
        'protocol'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            const vpc = new ec2.Vpc(stack, 'VPC');
            const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
                vpc,
            });
            const listener = lb.addListener('Listener', {
                port: 80,
            });
            // THEN
            test.throws(() => listener.addRedirectResponse('Default', {
                protocol: 'tcp',
                statusCode: 'HTTP_301',
            }), /`protocol`/);
            test.done();
        },
    },
    'Throws when specifying both target groups and fixed reponse'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc,
        });
        const listener = lb.addListener('Listener', {
            port: 80,
        });
        // THEN
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule', {
            listener,
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 })],
            fixedResponse: {
                statusCode: '500',
            },
        }), /'targetGroups,fixedResponse'.*/);
        test.done();
    },
    'Throws when specifying priority 0'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc,
        });
        const listener = lb.addListener('Listener', {
            port: 80,
        });
        // THEN
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule', {
            listener,
            priority: 0,
            pathPattern: '/hello',
            fixedResponse: {
                statusCode: '500',
            },
        }), Error, 'Priority must have value greater than or equal to 1');
        test.done();
    },
    'Throws when specifying both target groups and redirect reponse'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LoadBalancer', {
            vpc,
        });
        const listener = lb.addListener('Listener', {
            port: 80,
        });
        // THEN
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule', {
            listener,
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 })],
            redirectResponse: {
                statusCode: 'HTTP_301',
            },
        }), /'targetGroups,redirectResponse'.*/);
        test.throws(() => new elbv2.ApplicationListenerRule(stack, 'Rule2', {
            listener,
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [new elbv2.ApplicationTargetGroup(stack, 'TargetGroup2', { vpc, port: 80 })],
            fixedResponse: {
                statusCode: '500',
            },
            redirectResponse: {
                statusCode: 'HTTP_301',
            },
        }), /'targetGroups,fixedResponse,redirectResponse'.*/);
        test.done();
    },
    'Imported listener with imported security group and allowAllOutbound set to false'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const listener = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack, 'Listener', {
            listenerArn: 'listener-arn',
            defaultPort: 443,
            securityGroup: ec2.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'security-group-id', {
                allowAllOutbound: false,
            }),
        });
        // WHEN
        listener.connections.allowToAnyIpv4(ec2.Port.tcp(443));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
            GroupId: 'security-group-id',
        }));
        test.done();
    },
    'Can pass multiple certificate arns to application listener constructor'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert2' }],
        }));
        test.done();
    },
    'Can use certificate wrapper class'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        lb.addListener('Listener', {
            port: 443,
            certificates: [elbv2.ListenerCertificate.fromArn('cert1'), elbv2.ListenerCertificate.fromArn('cert2')],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert2' }],
        }));
        test.done();
    },
    'Can add additional certificates via addCertficateArns to application listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        listener.addCertificateArns('ListenerCertificateX', ['cert3']);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            Protocol: 'HTTPS',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert2' }],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerCertificate', {
            Certificates: [{ CertificateArn: 'cert3' }],
        }));
        test.done();
    },
    'Can add multiple path patterns to listener rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        listener.addTargets('Target1', {
            priority: 10,
            pathPatterns: ['/test/path/1', '/test/path/2'],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 10,
            Conditions: [
                {
                    Field: 'path-pattern',
                    Values: ['/test/path/1', '/test/path/2'],
                },
            ],
        }));
        test.done();
    },
    'Cannot add pathPattern and pathPatterns to listener rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1', 'cert2'],
            defaultTargetGroups: [new elbv2.ApplicationTargetGroup(stack, 'Group', { vpc, port: 80 })],
        });
        // THEN
        test.throws(() => listener.addTargets('Target1', {
            priority: 10,
            pathPatterns: ['/test/path/1', '/test/path/2'],
            pathPattern: '/test/path/3',
        }), Error, 'At least one of \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        test.done();
    },
    'Add additonal condition to listener rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const group1 = new elbv2.ApplicationTargetGroup(stack, 'Group1', { vpc, port: 80 });
        const group2 = new elbv2.ApplicationTargetGroup(stack, 'Group2', { vpc, port: 81, protocol: elbv2.ApplicationProtocol.HTTP });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1'],
            defaultTargetGroups: [group2],
        });
        listener.addTargetGroups('TargetGroup1', {
            priority: 10,
            conditions: [
                elbv2.ListenerCondition.hostHeaders(['app.test']),
                elbv2.ListenerCondition.httpHeader('Accept', ['application/vnd.myapp.v2+json']),
            ],
            targetGroups: [group1],
        });
        listener.addTargetGroups('TargetGroup2', {
            priority: 20,
            conditions: [
                elbv2.ListenerCondition.hostHeaders(['app.test']),
            ],
            targetGroups: [group2],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 10,
            Conditions: [
                {
                    Field: 'host-header',
                    HostHeaderConfig: {
                        Values: ['app.test'],
                    },
                },
                {
                    Field: 'http-header',
                    HttpHeaderConfig: {
                        HttpHeaderName: 'Accept',
                        Values: ['application/vnd.myapp.v2+json'],
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 20,
            Conditions: [
                {
                    Field: 'host-header',
                    HostHeaderConfig: {
                        Values: ['app.test'],
                    },
                },
            ],
        }));
        test.done();
    },
    'Add multiple additonal condition to listener rule'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const group1 = new elbv2.ApplicationTargetGroup(stack, 'Group1', { vpc, port: 80 });
        const group2 = new elbv2.ApplicationTargetGroup(stack, 'Group2', { vpc, port: 81, protocol: elbv2.ApplicationProtocol.HTTP });
        const group3 = new elbv2.ApplicationTargetGroup(stack, 'Group3', { vpc, port: 82, protocol: elbv2.ApplicationProtocol.HTTP });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1'],
            defaultTargetGroups: [group3],
        });
        listener.addTargetGroups('TargetGroup1', {
            priority: 10,
            conditions: [
                elbv2.ListenerCondition.hostHeaders(['app.test']),
                elbv2.ListenerCondition.sourceIps(['192.0.2.0/24']),
                elbv2.ListenerCondition.queryStrings([{ key: 'version', value: '2' }, { value: 'foo*' }]),
            ],
            targetGroups: [group1],
        });
        listener.addTargetGroups('TargetGroup2', {
            priority: 20,
            conditions: [
                elbv2.ListenerCondition.hostHeaders(['app.test']),
                elbv2.ListenerCondition.httpHeader('Accept', ['application/vnd.myapp.v2+json']),
            ],
            targetGroups: [group1],
        });
        listener.addTargetGroups('TargetGroup3', {
            priority: 30,
            conditions: [
                elbv2.ListenerCondition.hostHeaders(['app.test']),
                elbv2.ListenerCondition.httpRequestMethods(['PUT', 'COPY', 'LOCK', 'MKCOL', 'MOVE', 'PROPFIND', 'PROPPATCH', 'UNLOCK']),
            ],
            targetGroups: [group2],
        });
        listener.addTargetGroups('TargetGroup4', {
            priority: 40,
            conditions: [
                elbv2.ListenerCondition.hostHeaders(['app.test']),
            ],
            targetGroups: [group3],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 10,
            Conditions: [
                {
                    Field: 'host-header',
                    HostHeaderConfig: {
                        Values: ['app.test'],
                    },
                },
                {
                    Field: 'source-ip',
                    SourceIpConfig: {
                        Values: ['192.0.2.0/24'],
                    },
                },
                {
                    Field: 'query-string',
                    QueryStringConfig: {
                        Values: [
                            {
                                Key: 'version',
                                Value: '2',
                            },
                            {
                                Value: 'foo*',
                            },
                        ],
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 20,
            Conditions: [
                {
                    Field: 'host-header',
                    HostHeaderConfig: {
                        Values: ['app.test'],
                    },
                },
                {
                    Field: 'http-header',
                    HttpHeaderConfig: {
                        HttpHeaderName: 'Accept',
                        Values: ['application/vnd.myapp.v2+json'],
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 30,
            Conditions: [
                {
                    Field: 'host-header',
                    HostHeaderConfig: {
                        Values: ['app.test'],
                    },
                },
                {
                    Field: 'http-request-method',
                    HttpRequestMethodConfig: {
                        Values: ['PUT', 'COPY', 'LOCK', 'MKCOL', 'MOVE', 'PROPFIND', 'PROPPATCH', 'UNLOCK'],
                    },
                },
            ],
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 40,
            Conditions: [
                {
                    Field: 'host-header',
                    HostHeaderConfig: {
                        Values: ['app.test'],
                    },
                },
            ],
        }));
        test.done();
    },
    'Can exist together legacy style conditions and modan style conditions'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const group1 = new elbv2.ApplicationTargetGroup(stack, 'Group1', { vpc, port: 80 });
        const group2 = new elbv2.ApplicationTargetGroup(stack, 'Group2', { vpc, port: 81, protocol: elbv2.ApplicationProtocol.HTTP });
        // WHEN
        const listener = lb.addListener('Listener', {
            port: 443,
            certificateArns: ['cert1'],
            defaultTargetGroups: [group2],
        });
        listener.addTargetGroups('TargetGroup1', {
            hostHeader: 'app.test',
            pathPattern: '/test',
            conditions: [
                elbv2.ListenerCondition.sourceIps(['192.0.2.0/24']),
            ],
            priority: 10,
            targetGroups: [group1],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 10,
            Conditions: [
                {
                    Field: 'host-header',
                    Values: ['app.test'],
                },
                {
                    Field: 'path-pattern',
                    Values: ['/test'],
                },
                {
                    Field: 'source-ip',
                    SourceIpConfig: {
                        Values: ['192.0.2.0/24'],
                    },
                },
            ],
        }));
        test.done();
    },
    'Add condition to imported application listener'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        const listener = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack, 'Listener', {
            listenerArn: 'listener-arn',
            defaultPort: 443,
            securityGroupId: 'security-group-id',
        });
        // WHEN
        listener.addTargetGroups('OtherTG', {
            targetGroups: [group],
            priority: 1,
            pathPatterns: ['/path1', '/path2'],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Priority: 1,
            Conditions: [
                {
                    Field: 'path-pattern',
                    Values: ['/path1', '/path2'],
                },
            ],
        }));
        test.done();
    },
    'not allowed to combine action specifiers when instantiating a Rule directly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        const listener = lb.addListener('Listener', { port: 80 });
        const baseProps = { listener, priority: 1, pathPatterns: ['/path1', '/path2'] };
        // WHEN
        test.throws(() => {
            new elbv2.ApplicationListenerRule(stack, 'Rule1', {
                ...baseProps,
                fixedResponse: { statusCode: '200' },
                action: elbv2.ListenerAction.fixedResponse(200),
            });
        }, /specify only one/);
        test.throws(() => {
            new elbv2.ApplicationListenerRule(stack, 'Rule2', {
                ...baseProps,
                targetGroups: [group],
                action: elbv2.ListenerAction.fixedResponse(200),
            });
        }, /specify only one/);
        test.done();
    },
    'not allowed to specify defaultTargetGroups and defaultAction together'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        const group = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup', { vpc, port: 80 });
        const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        // WHEN
        test.throws(() => {
            lb.addListener('Listener1', {
                port: 80,
                defaultTargetGroups: [group],
                defaultAction: elbv2.ListenerAction.fixedResponse(200),
            });
        }, /Specify at most one/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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