#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../lib");
/* IPv6 workaround found here: https://github.com/aws/aws-cdk/issues/894 */
const valueOrDie = (value, err) => {
    if (value === undefined) {
        throw err;
    }
    return value;
};
/**
 * Integration test to deployability and use of dualstack ALB. Creates an ALB
 * with dualstack ipAddresType and an ipv6Block to add to VPC subnets. Main
 * test is for the inclusion of default IPv6 ingress rule.
 *
 * Stack Verification steps:
 * VPC is created with subnets that allow for IPv6 connection and then dualstack
 * ALB attaches a listener with dualstack that defaults IPv4/IPv6 ingress rule.
 *
 */
const app = new cdk.App();
const stack = new cdk.Stack(app, 'aws-cdk-elbv2-integ');
const vpc = new ec2.Vpc(stack, 'VPC', {
    maxAzs: 2,
});
const ipv6Block = new ec2.CfnVPCCidrBlock(stack, 'IPv6_Block', {
    vpcId: vpc.vpcId,
    amazonProvidedIpv6CidrBlock: true,
});
// Get the vpc's internet gateway so we can create default routes for the
// public subnets.
const internetGateway = valueOrDie(vpc.node.children.find(c => c instanceof ec2.CfnInternetGateway), new Error('Couldnt find an internet gateway'));
vpc.publicSubnets.forEach((subnet, idx) => {
    // Add a default ipv6 route to the subnet's route table.
    const unboxedSubnet = subnet;
    unboxedSubnet.addRoute('IPv6Default', {
        routerId: internetGateway.ref,
        routerType: ec2.RouterType.GATEWAY,
        destinationIpv6CidrBlock: '::/0',
    });
    // Find a CfnSubnet (raw cloudformation resources) child to the public
    // subnet nodes.
    const cfnSubnet = valueOrDie(subnet.node.children.find(c => c instanceof ec2.CfnSubnet), new Error('Couldnt find a CfnSubnet'));
    // Use the intrinsic Fn::Cidr CloudFormation function on the VPC's
    // first IPv6 block to determine ipv6 /64 cidrs for each subnet as
    // a function of the public subnet's index.
    const vpcCidrBlock = cdk.Fn.select(0, vpc.vpcIpv6CidrBlocks);
    const ipv6Cidrs = cdk.Fn.cidr(vpcCidrBlock, vpc.publicSubnets.length, '64');
    cfnSubnet.ipv6CidrBlock = cdk.Fn.select(idx, ipv6Cidrs);
    // The subnet depends on the ipv6 cidr being allocated.
    cfnSubnet.addDependsOn(ipv6Block);
});
const lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', {
    vpc,
    ipAddressType: elbv2.IpAddressType.DUAL_STACK,
    internetFacing: true,
});
const listener = lb.addListener('Listener', {
    port: 80,
});
const group1 = listener.addTargets('Target', {
    port: 80,
    targets: [new elbv2.IpTarget('10.0.128.4')],
});
const group2 = listener.addTargets('ConditionalTarget', {
    priority: 10,
    hostHeader: 'example.com',
    port: 80,
    targets: [new elbv2.IpTarget('10.0.128.5')],
});
listener.addAction('action1', {
    priority: 1,
    conditions: [
        elbv2.ListenerCondition.hostHeaders(['example.com']),
    ],
    action: elbv2.ListenerAction.fixedResponse(200, { messageBody: 'success' }),
});
group1.metricTargetResponseTime().createAlarm(stack, 'ResponseTimeHigh1', {
    threshold: 5,
    evaluationPeriods: 2,
});
group2.metricTargetResponseTime().createAlarm(stack, 'ResponseTimeHigh2', {
    threshold: 5,
    evaluationPeriods: 2,
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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