"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongAdminService = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const aws_route53_1 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const aws_servicediscovery_1 = require("aws-cdk-lib/aws-servicediscovery");
class KongAdminService extends aws_ecs_1.FargateService {
    constructor(scope, id, props) {
        super(scope, id, props.kongServiceProps);
        this.zone = props.route53_zone;
        // Initialize the port variables or user defined input
        this.adminPort = props.kongFeaturesProps.adminProps.httpPort || 8001;
        this.adminHttpsPort = props.kongFeaturesProps.adminProps.httpsPort || 8444;
        this.managerPort = props.kongFeaturesProps.kongManagerProps.httpPort || 8002;
        this.managerHttpsPort = props.kongFeaturesProps.kongManagerProps.httpsPort || 8445;
        this.telemetryPort = props.kongFeaturesProps.clusterTelemetryProps.port || 8006;
        this.clusterPort = props.kongFeaturesProps.clusterProps.port || 8005;
        this.devPortalPort = props.kongFeaturesProps.devPortalProps.port || 8003;
        // Initialize the endpoints
        this.controlPlaneDnsEp = this.cluster.clusterName;
        this.adminEndpoint = 'admin.' + props.route53_zone.zoneName;
        this.managerEndpoint = 'manager.' + props.route53_zone.zoneName;
        this.devPortalEndpoint = 'devportal.' + props.route53_zone.zoneName;
        this.clusterEndpoint = 'cluster.' + props.route53_zone.zoneName;
        this.telemetryEndpoint = 'telemetry.' + props.route53_zone.zoneName;
        const cloudmap = this.enableCloudMap({
            name: 'kongControlPlaneCloudMapService',
            cloudMapNamespace: new aws_servicediscovery_1.PrivateDnsNamespace(this, 'KongCPNamespace', {
                name: this.controlPlaneDnsEp,
                vpc: props.kongServiceProps.cluster.vpc,
            }),
            dnsRecordType: aws_servicediscovery_1.DnsRecordType.A,
        });
        //Use Cloudmap to expose the https service instead of adding a NLB and map it to a DNS record.
        if (props.kongFeaturesProps.clusterProps.enabled) {
            new aws_route53_1.CnameRecord(this, 'kongAdminCNameRecord', {
                domainName: this.cloudMapService?.serviceName + '.' + this.controlPlaneDnsEp || '',
                zone: props.route53_zone,
                recordName: this.clusterEndpoint,
            }).node.addDependency(cloudmap);
            this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(this.clusterPort));
            new aws_cdk_lib_1.CfnOutput(this, 'KongClusterEndpoint', {
                value: this.clusterEndpoint + ':' + this.clusterPort,
                exportName: 'KongClusterEndpoint',
            });
        }
        if (props.kongFeaturesProps.clusterTelemetryProps.enabled) {
            new aws_route53_1.CnameRecord(this, 'kongTelemetryCNameRecord', {
                domainName: this.cloudMapService?.serviceName + '.' + this.controlPlaneDnsEp || '',
                zone: props.route53_zone,
                recordName: this.telemetryEndpoint,
            }).node.addDependency(cloudmap);
            this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(this.telemetryPort));
            new aws_cdk_lib_1.CfnOutput(this, 'KongTelemetryEndpoint', {
                value: this.telemetryEndpoint + ':' + this.telemetryPort,
                exportName: 'KongTelemetryEndpoint',
            });
        }
        this.registerKongFeatureTargets(443, this.adminHttpsPort, props.adminNlb, 'KongAdminHttps', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', true, this.adminEndpoint);
        if (props.kongFeaturesProps.adminProps.enableHttp) {
            this.registerKongFeatureTargets(80, this.adminPort, props.adminNlb, 'KongAdmin', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', false);
        }
        if (props.kongFeaturesProps.kongManagerProps.enabled && props.managerNlb) {
            this.registerKongFeatureTargets(443, this.managerHttpsPort, props.managerNlb, 'KongManager', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', true, this.managerEndpoint);
            if (props.kongFeaturesProps.kongManagerProps.enableHttp) {
                this.registerKongFeatureTargets(80, this.managerPort, props.managerNlb, 'KongManagerHttps', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', false);
            }
        }
        ;
        if (props.kongFeaturesProps.devPortalProps.enabled && props.devPortalNlb) {
            this.registerKongFeatureTargets(80, this.devPortalPort, props.devPortalNlb, 'KongDevPortal', props.kongServiceProps.taskDefinition.defaultContainer?.containerName ?? '', true, this.devPortalEndpoint);
        }
    }
    registerKongFeatureTargets(listenerPort, containerPort, lb, id, containerName, addARecord, recordName) {
        const listener = new aws_elasticloadbalancingv2_1.NetworkListener(this, 'KongCp' + id + 'Listener', {
            port: listenerPort,
            protocol: aws_elasticloadbalancingv2_1.Protocol.TCP,
            loadBalancer: lb,
        });
        if (addARecord) {
            new aws_route53_1.ARecord(this, 'KongCp' + id + 'ARecord', {
                zone: this.zone,
                recordName: recordName,
                target: aws_route53_1.RecordTarget.fromAlias(new targets.LoadBalancerTarget(lb)),
            });
        }
        this.registerLoadBalancerTargets({
            containerName: containerName,
            containerPort: containerPort,
            newTargetGroupId: id,
            listener: aws_ecs_1.ListenerConfig.networkListener(listener, {
                deregistrationDelay: aws_cdk_lib_1.Duration.seconds(60),
                port: listenerPort,
                healthCheck: {
                    port: containerPort.toString(),
                },
                preserveClientIp: false,
                proxyProtocolV2: false,
                protocol: aws_elasticloadbalancingv2_1.Protocol.TCP,
            }),
        });
        this.connections.allowFrom(aws_ec2_1.Peer.ipv4(this.cluster.vpc.vpcCidrBlock), aws_ec2_1.Port.tcp(containerPort));
        if (recordName) {
            new aws_cdk_lib_1.CfnOutput(this, 'KongCp' + id + 'Endpoint', {
                value: recordName,
                exportName: id,
            });
        }
    }
}
exports.KongAdminService = KongAdminService;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia29uZy1hZG1pbi1zZXJ2aWNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3Jlc291cmNlcy9lY3Mva29uZy1hZG1pbi1zZXJ2aWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDZDQUFrRDtBQUNsRCxpREFBaUQ7QUFDakQsaURBSTZCO0FBQzdCLHVGQUF3RztBQUN4Ryx5REFBZ0c7QUFDaEcsMkRBQTJEO0FBQzNELDJFQUFzRjtBQWF0RixNQUFhLGdCQUFpQixTQUFRLHdCQUFjO0lBbUJsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTRCO1FBRXBFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBRXpDLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUMvQixzREFBc0Q7UUFDdEQsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsVUFBVSxDQUFDLFFBQVEsSUFBSSxJQUFJLENBQUM7UUFDckUsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsVUFBVSxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUM7UUFDM0UsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQztRQUM3RSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDLGdCQUFnQixDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUM7UUFDbkYsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsaUJBQWlCLENBQUMscUJBQXFCLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQztRQUNoRixJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQztRQUNyRSxJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxjQUFjLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQztRQUV6RSwyQkFBMkI7UUFDM0IsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO1FBQ2xELElBQUksQ0FBQyxhQUFhLEdBQUcsUUFBUSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO1FBQzVELElBQUksQ0FBQyxlQUFlLEdBQUcsVUFBVSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDO1FBQ2hFLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUM7UUFDcEUsSUFBSSxDQUFDLGVBQWUsR0FBRyxVQUFVLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUM7UUFDaEUsSUFBSSxDQUFDLGlCQUFpQixHQUFHLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQztRQUVwRSxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDO1lBQ25DLElBQUksRUFBRSxpQ0FBaUM7WUFDdkMsaUJBQWlCLEVBQUUsSUFBSSwwQ0FBbUIsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7Z0JBQ2xFLElBQUksRUFBRSxJQUFJLENBQUMsaUJBQWlCO2dCQUM1QixHQUFHLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxHQUFHO2FBQ3hDLENBQUM7WUFDRixhQUFhLEVBQUUsb0NBQWEsQ0FBQyxDQUFDO1NBQy9CLENBQUMsQ0FBQztRQUdILDhGQUE4RjtRQUM5RixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFO1lBQ2hELElBQUkseUJBQVcsQ0FBQyxJQUFJLEVBQUUsc0JBQXNCLEVBQUU7Z0JBQzVDLFVBQVUsRUFBRSxJQUFJLENBQUMsZUFBZSxFQUFFLFdBQVcsR0FBRyxHQUFHLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixJQUFJLEVBQUU7Z0JBQ2xGLElBQUksRUFBRSxLQUFLLENBQUMsWUFBWTtnQkFDeEIsVUFBVSxFQUFFLElBQUksQ0FBQyxlQUFlO2FBQ2pDLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBRWhDLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLGNBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsY0FBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztZQUVqRyxJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO2dCQUN6QyxLQUFLLEVBQUUsSUFBSSxDQUFDLGVBQWUsR0FBRyxHQUFHLEdBQUcsSUFBSSxDQUFDLFdBQVc7Z0JBQ3BELFVBQVUsRUFBRSxxQkFBcUI7YUFDbEMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxxQkFBcUIsQ0FBQyxPQUFPLEVBQUU7WUFDekQsSUFBSSx5QkFBVyxDQUFDLElBQUksRUFBRSwwQkFBMEIsRUFBRTtnQkFDaEQsVUFBVSxFQUFFLElBQUksQ0FBQyxlQUFlLEVBQUUsV0FBVyxHQUFHLEdBQUcsR0FBRyxJQUFJLENBQUMsaUJBQWlCLElBQUksRUFBRTtnQkFDbEYsSUFBSSxFQUFFLEtBQUssQ0FBQyxZQUFZO2dCQUN4QixVQUFVLEVBQUUsSUFBSSxDQUFDLGlCQUFpQjthQUNuQyxDQUFDLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUVoQyxJQUFJLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxjQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLGNBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUM7WUFFbkcsSUFBSSx1QkFBUyxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtnQkFDM0MsS0FBSyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxHQUFHLEdBQUcsSUFBSSxDQUFDLGFBQWE7Z0JBQ3hELFVBQVUsRUFBRSx1QkFBdUI7YUFDcEMsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLENBQUMsMEJBQTBCLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsS0FBSyxDQUFDLFFBQVEsRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFLGFBQWEsSUFBSSxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNuTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLENBQUMsVUFBVSxFQUFFO1lBQ2pELElBQUksQ0FBQywwQkFBMEIsQ0FBQyxFQUFFLEVBQUUsSUFBSSxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLFdBQVcsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFLGFBQWEsSUFBSSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7U0FDdEs7UUFFRCxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUN4RSxJQUFJLENBQUMsMEJBQTBCLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsVUFBVSxFQUFFLGFBQWEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFLGFBQWEsSUFBRyxFQUFFLEVBQUUsSUFBSSxFQUMzSyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7WUFDeEIsSUFBSSxLQUFLLENBQUMsaUJBQWlCLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxFQUFFO2dCQUN2RCxJQUFJLENBQUMsMEJBQTBCLENBQUMsRUFBRSxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFVBQVUsRUFBRSxrQkFBa0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFLGFBQWEsSUFBRyxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7YUFDaEw7U0FDRjtRQUFBLENBQUM7UUFFRixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxjQUFjLENBQUMsT0FBTyxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDeEUsSUFBSSxDQUFDLDBCQUEwQixDQUFDLEVBQUUsRUFBRSxJQUFJLENBQUMsYUFBYSxFQUFFLEtBQUssQ0FBQyxZQUFZLEVBQUUsZUFBZSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUUsYUFBYSxJQUFHLEVBQUUsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7U0FDeE07SUFDSCxDQUFDO0lBRU8sMEJBQTBCLENBQUMsWUFBb0IsRUFBRSxhQUFxQixFQUFFLEVBQXVCLEVBQUUsRUFBVSxFQUNqSCxhQUFxQixFQUFFLFVBQW1CLEVBQUUsVUFBbUI7UUFFL0QsTUFBTSxRQUFRLEdBQUcsSUFBSSw0Q0FBZSxDQUFDLElBQUksRUFBRSxRQUFRLEdBQUMsRUFBRSxHQUFDLFVBQVUsRUFBRTtZQUNqRSxJQUFJLEVBQUUsWUFBWTtZQUNsQixRQUFRLEVBQUUscUNBQVEsQ0FBQyxHQUFHO1lBQ3RCLFlBQVksRUFBRSxFQUFFO1NBRWpCLENBQUMsQ0FBQztRQUVILElBQUksVUFBVSxFQUFFO1lBQ2QsSUFBSSxxQkFBTyxDQUFDLElBQUksRUFBRSxRQUFRLEdBQUcsRUFBRSxHQUFHLFNBQVMsRUFBRTtnQkFDM0MsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO2dCQUNmLFVBQVUsRUFBRSxVQUFVO2dCQUN0QixNQUFNLEVBQUUsMEJBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxPQUFPLENBQUMsa0JBQWtCLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDbkUsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLENBQUMsMkJBQTJCLENBQzlCO1lBQ0UsYUFBYSxFQUFFLGFBQWE7WUFDNUIsYUFBYSxFQUFFLGFBQWE7WUFDNUIsZ0JBQWdCLEVBQUUsRUFBRTtZQUNwQixRQUFRLEVBQUUsd0JBQWMsQ0FBQyxlQUFlLENBQUMsUUFBUSxFQUFFO2dCQUNqRCxtQkFBbUIsRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7Z0JBQ3pDLElBQUksRUFBRSxZQUFZO2dCQUNsQixXQUFXLEVBQUU7b0JBQ1gsSUFBSSxFQUFFLGFBQWEsQ0FBQyxRQUFRLEVBQUU7aUJBQy9CO2dCQUNELGdCQUFnQixFQUFFLEtBQUs7Z0JBQ3ZCLGVBQWUsRUFBRSxLQUFLO2dCQUN0QixRQUFRLEVBQUUscUNBQVEsQ0FBQyxHQUFHO2FBQ3ZCLENBQUM7U0FDSCxDQUNGLENBQUM7UUFFRixJQUFJLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxjQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLGNBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUM5RixJQUFJLFVBQVUsRUFBRTtZQUNkLElBQUksdUJBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxHQUFHLEVBQUUsR0FBRyxVQUFVLEVBQUU7Z0JBQzlDLEtBQUssRUFBRSxVQUFVO2dCQUNqQixVQUFVLEVBQUUsRUFBRTthQUNmLENBQUMsQ0FBQztTQUNKO0lBQ0gsQ0FBQztDQUNGO0FBaEpELDRDQWdKQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENmbk91dHB1dCwgRHVyYXRpb24gfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBQZWVyLCBQb3J0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWVjMic7XG5pbXBvcnQge1xuICBGYXJnYXRlU2VydmljZSxcbiAgRmFyZ2F0ZVNlcnZpY2VQcm9wcyxcbiAgTGlzdGVuZXJDb25maWcsXG59IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3MnO1xuaW1wb3J0IHsgTmV0d29ya0xpc3RlbmVyLCBQcm90b2NvbCwgTmV0d29ya0xvYWRCYWxhbmNlciB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1lbGFzdGljbG9hZGJhbGFuY2luZ3YyJztcbmltcG9ydCB7IEFSZWNvcmQsIENuYW1lUmVjb3JkLCBQcml2YXRlSG9zdGVkWm9uZSwgUmVjb3JkVGFyZ2V0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXJvdXRlNTMnO1xuaW1wb3J0ICogYXMgdGFyZ2V0cyBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtcm91dGU1My10YXJnZXRzJztcbmltcG9ydCB7IERuc1JlY29yZFR5cGUsIFByaXZhdGVEbnNOYW1lc3BhY2UgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc2VydmljZWRpc2NvdmVyeSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEtvbmdGZWF0dXJlUHJvcHMgfSBmcm9tICcuLi8uLi9rb25nLWVjcyc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgS29uZ0FkbWluU2VydmljZVByb3BzIHtcbiAgcmVhZG9ubHkga29uZ1NlcnZpY2VQcm9wczogRmFyZ2F0ZVNlcnZpY2VQcm9wcztcbiAgcmVhZG9ubHkgYWRtaW5ObGI6IE5ldHdvcmtMb2FkQmFsYW5jZXI7XG4gIHJlYWRvbmx5IG1hbmFnZXJObGI/OiBOZXR3b3JrTG9hZEJhbGFuY2VyO1xuICByZWFkb25seSBkZXZQb3J0YWxObGI/OiBOZXR3b3JrTG9hZEJhbGFuY2VyO1xuICByZWFkb25seSByb3V0ZTUzX3pvbmU6IFByaXZhdGVIb3N0ZWRab25lO1xuICByZWFkb25seSBrb25nRmVhdHVyZXNQcm9wczogS29uZ0ZlYXR1cmVQcm9wcztcbn1cblxuZXhwb3J0IGNsYXNzIEtvbmdBZG1pblNlcnZpY2UgZXh0ZW5kcyBGYXJnYXRlU2VydmljZSB7XG5cbiAgcHJpdmF0ZSB6b25lOiBQcml2YXRlSG9zdGVkWm9uZTtcbiAgcHJpdmF0ZSBhZG1pblBvcnQ6IG51bWJlcjtcbiAgcHJpdmF0ZSBhZG1pbkh0dHBzUG9ydDogbnVtYmVyO1xuICBwcml2YXRlIG1hbmFnZXJQb3J0IDogbnVtYmVyO1xuICBwcml2YXRlIG1hbmFnZXJIdHRwc1BvcnQgOiBudW1iZXI7XG4gIHByaXZhdGUgdGVsZW1ldHJ5UG9ydCA6bnVtYmVyO1xuICBwcml2YXRlIGNsdXN0ZXJQb3J0IDogbnVtYmVyO1xuICBwcml2YXRlIGRldlBvcnRhbFBvcnQgOiBudW1iZXI7XG5cbiAgLy8gSW5pdGlhbGl6ZSB0aGUgZW5kcG9pbnRzXG4gIHByaXZhdGUgY29udHJvbFBsYW5lRG5zRXA6IHN0cmluZztcbiAgcHJpdmF0ZSBhZG1pbkVuZHBvaW50IDogc3RyaW5nO1xuICBwcml2YXRlIG1hbmFnZXJFbmRwb2ludCA6IHN0cmluZztcbiAgcHJpdmF0ZSBkZXZQb3J0YWxFbmRwb2ludCA6IHN0cmluZztcbiAgcHJpdmF0ZSBjbHVzdGVyRW5kcG9pbnQgOiBzdHJpbmc7XG4gIHByaXZhdGUgdGVsZW1ldHJ5RW5kcG9pbnQgOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEtvbmdBZG1pblNlcnZpY2VQcm9wcykge1xuXG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcy5rb25nU2VydmljZVByb3BzKTtcblxuICAgIHRoaXMuem9uZSA9IHByb3BzLnJvdXRlNTNfem9uZTtcbiAgICAvLyBJbml0aWFsaXplIHRoZSBwb3J0IHZhcmlhYmxlcyBvciB1c2VyIGRlZmluZWQgaW5wdXRcbiAgICB0aGlzLmFkbWluUG9ydCA9IHByb3BzLmtvbmdGZWF0dXJlc1Byb3BzLmFkbWluUHJvcHMuaHR0cFBvcnQgfHwgODAwMTtcbiAgICB0aGlzLmFkbWluSHR0cHNQb3J0ID0gcHJvcHMua29uZ0ZlYXR1cmVzUHJvcHMuYWRtaW5Qcm9wcy5odHRwc1BvcnQgfHwgODQ0NDtcbiAgICB0aGlzLm1hbmFnZXJQb3J0ID0gcHJvcHMua29uZ0ZlYXR1cmVzUHJvcHMua29uZ01hbmFnZXJQcm9wcy5odHRwUG9ydCB8fCA4MDAyO1xuICAgIHRoaXMubWFuYWdlckh0dHBzUG9ydCA9IHByb3BzLmtvbmdGZWF0dXJlc1Byb3BzLmtvbmdNYW5hZ2VyUHJvcHMuaHR0cHNQb3J0IHx8IDg0NDU7XG4gICAgdGhpcy50ZWxlbWV0cnlQb3J0ID0gcHJvcHMua29uZ0ZlYXR1cmVzUHJvcHMuY2x1c3RlclRlbGVtZXRyeVByb3BzLnBvcnQgfHwgODAwNjtcbiAgICB0aGlzLmNsdXN0ZXJQb3J0ID0gcHJvcHMua29uZ0ZlYXR1cmVzUHJvcHMuY2x1c3RlclByb3BzLnBvcnQgfHwgODAwNTtcbiAgICB0aGlzLmRldlBvcnRhbFBvcnQgPSBwcm9wcy5rb25nRmVhdHVyZXNQcm9wcy5kZXZQb3J0YWxQcm9wcy5wb3J0IHx8IDgwMDM7XG5cbiAgICAvLyBJbml0aWFsaXplIHRoZSBlbmRwb2ludHNcbiAgICB0aGlzLmNvbnRyb2xQbGFuZURuc0VwID0gdGhpcy5jbHVzdGVyLmNsdXN0ZXJOYW1lO1xuICAgIHRoaXMuYWRtaW5FbmRwb2ludCA9ICdhZG1pbi4nICsgcHJvcHMucm91dGU1M196b25lLnpvbmVOYW1lO1xuICAgIHRoaXMubWFuYWdlckVuZHBvaW50ID0gJ21hbmFnZXIuJyArIHByb3BzLnJvdXRlNTNfem9uZS56b25lTmFtZTtcbiAgICB0aGlzLmRldlBvcnRhbEVuZHBvaW50ID0gJ2RldnBvcnRhbC4nICsgcHJvcHMucm91dGU1M196b25lLnpvbmVOYW1lO1xuICAgIHRoaXMuY2x1c3RlckVuZHBvaW50ID0gJ2NsdXN0ZXIuJyArIHByb3BzLnJvdXRlNTNfem9uZS56b25lTmFtZTtcbiAgICB0aGlzLnRlbGVtZXRyeUVuZHBvaW50ID0gJ3RlbGVtZXRyeS4nICsgcHJvcHMucm91dGU1M196b25lLnpvbmVOYW1lO1xuXG4gICAgY29uc3QgY2xvdWRtYXAgPSB0aGlzLmVuYWJsZUNsb3VkTWFwKHtcbiAgICAgIG5hbWU6ICdrb25nQ29udHJvbFBsYW5lQ2xvdWRNYXBTZXJ2aWNlJyxcbiAgICAgIGNsb3VkTWFwTmFtZXNwYWNlOiBuZXcgUHJpdmF0ZURuc05hbWVzcGFjZSh0aGlzLCAnS29uZ0NQTmFtZXNwYWNlJywge1xuICAgICAgICBuYW1lOiB0aGlzLmNvbnRyb2xQbGFuZURuc0VwLFxuICAgICAgICB2cGM6IHByb3BzLmtvbmdTZXJ2aWNlUHJvcHMuY2x1c3Rlci52cGMsXG4gICAgICB9KSxcbiAgICAgIGRuc1JlY29yZFR5cGU6IERuc1JlY29yZFR5cGUuQSxcbiAgICB9KTtcblxuXG4gICAgLy9Vc2UgQ2xvdWRtYXAgdG8gZXhwb3NlIHRoZSBodHRwcyBzZXJ2aWNlIGluc3RlYWQgb2YgYWRkaW5nIGEgTkxCIGFuZCBtYXAgaXQgdG8gYSBETlMgcmVjb3JkLlxuICAgIGlmIChwcm9wcy5rb25nRmVhdHVyZXNQcm9wcy5jbHVzdGVyUHJvcHMuZW5hYmxlZCkge1xuICAgICAgbmV3IENuYW1lUmVjb3JkKHRoaXMsICdrb25nQWRtaW5DTmFtZVJlY29yZCcsIHtcbiAgICAgICAgZG9tYWluTmFtZTogdGhpcy5jbG91ZE1hcFNlcnZpY2U/LnNlcnZpY2VOYW1lICsgJy4nICsgdGhpcy5jb250cm9sUGxhbmVEbnNFcCB8fCAnJyxcbiAgICAgICAgem9uZTogcHJvcHMucm91dGU1M196b25lLFxuICAgICAgICByZWNvcmROYW1lOiB0aGlzLmNsdXN0ZXJFbmRwb2ludCxcbiAgICAgIH0pLm5vZGUuYWRkRGVwZW5kZW5jeShjbG91ZG1hcCk7XG5cbiAgICAgIHRoaXMuY29ubmVjdGlvbnMuYWxsb3dGcm9tKFBlZXIuaXB2NCh0aGlzLmNsdXN0ZXIudnBjLnZwY0NpZHJCbG9jayksIFBvcnQudGNwKHRoaXMuY2x1c3RlclBvcnQpKTtcblxuICAgICAgbmV3IENmbk91dHB1dCh0aGlzLCAnS29uZ0NsdXN0ZXJFbmRwb2ludCcsIHtcbiAgICAgICAgdmFsdWU6IHRoaXMuY2x1c3RlckVuZHBvaW50ICsgJzonICsgdGhpcy5jbHVzdGVyUG9ydCxcbiAgICAgICAgZXhwb3J0TmFtZTogJ0tvbmdDbHVzdGVyRW5kcG9pbnQnLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmtvbmdGZWF0dXJlc1Byb3BzLmNsdXN0ZXJUZWxlbWV0cnlQcm9wcy5lbmFibGVkKSB7XG4gICAgICBuZXcgQ25hbWVSZWNvcmQodGhpcywgJ2tvbmdUZWxlbWV0cnlDTmFtZVJlY29yZCcsIHtcbiAgICAgICAgZG9tYWluTmFtZTogdGhpcy5jbG91ZE1hcFNlcnZpY2U/LnNlcnZpY2VOYW1lICsgJy4nICsgdGhpcy5jb250cm9sUGxhbmVEbnNFcCB8fCAnJyxcbiAgICAgICAgem9uZTogcHJvcHMucm91dGU1M196b25lLFxuICAgICAgICByZWNvcmROYW1lOiB0aGlzLnRlbGVtZXRyeUVuZHBvaW50LFxuICAgICAgfSkubm9kZS5hZGREZXBlbmRlbmN5KGNsb3VkbWFwKTtcblxuICAgICAgdGhpcy5jb25uZWN0aW9ucy5hbGxvd0Zyb20oUGVlci5pcHY0KHRoaXMuY2x1c3Rlci52cGMudnBjQ2lkckJsb2NrKSwgUG9ydC50Y3AodGhpcy50ZWxlbWV0cnlQb3J0KSk7XG5cbiAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ0tvbmdUZWxlbWV0cnlFbmRwb2ludCcsIHtcbiAgICAgICAgdmFsdWU6IHRoaXMudGVsZW1ldHJ5RW5kcG9pbnQgKyAnOicgKyB0aGlzLnRlbGVtZXRyeVBvcnQsXG4gICAgICAgIGV4cG9ydE5hbWU6ICdLb25nVGVsZW1ldHJ5RW5kcG9pbnQnLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgdGhpcy5yZWdpc3RlcktvbmdGZWF0dXJlVGFyZ2V0cyg0NDMsIHRoaXMuYWRtaW5IdHRwc1BvcnQsIHByb3BzLmFkbWluTmxiLCAnS29uZ0FkbWluSHR0cHMnLCBwcm9wcy5rb25nU2VydmljZVByb3BzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXI/LmNvbnRhaW5lck5hbWUgPz8gJycsIHRydWUsIHRoaXMuYWRtaW5FbmRwb2ludCk7XG4gICAgaWYgKHByb3BzLmtvbmdGZWF0dXJlc1Byb3BzLmFkbWluUHJvcHMuZW5hYmxlSHR0cCkge1xuICAgICAgdGhpcy5yZWdpc3RlcktvbmdGZWF0dXJlVGFyZ2V0cyg4MCwgdGhpcy5hZG1pblBvcnQsIHByb3BzLmFkbWluTmxiLCAnS29uZ0FkbWluJywgcHJvcHMua29uZ1NlcnZpY2VQcm9wcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyPy5jb250YWluZXJOYW1lID8/ICcnLCBmYWxzZSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmtvbmdGZWF0dXJlc1Byb3BzLmtvbmdNYW5hZ2VyUHJvcHMuZW5hYmxlZCAmJiBwcm9wcy5tYW5hZ2VyTmxiKSB7XG4gICAgICB0aGlzLnJlZ2lzdGVyS29uZ0ZlYXR1cmVUYXJnZXRzKDQ0MywgdGhpcy5tYW5hZ2VySHR0cHNQb3J0LCBwcm9wcy5tYW5hZ2VyTmxiLCAnS29uZ01hbmFnZXInLCBwcm9wcy5rb25nU2VydmljZVByb3BzLnRhc2tEZWZpbml0aW9uLmRlZmF1bHRDb250YWluZXI/LmNvbnRhaW5lck5hbWUgPz8nJywgdHJ1ZSxcbiAgICAgICAgdGhpcy5tYW5hZ2VyRW5kcG9pbnQpO1xuICAgICAgaWYgKHByb3BzLmtvbmdGZWF0dXJlc1Byb3BzLmtvbmdNYW5hZ2VyUHJvcHMuZW5hYmxlSHR0cCkge1xuICAgICAgICB0aGlzLnJlZ2lzdGVyS29uZ0ZlYXR1cmVUYXJnZXRzKDgwLCB0aGlzLm1hbmFnZXJQb3J0LCBwcm9wcy5tYW5hZ2VyTmxiLCAnS29uZ01hbmFnZXJIdHRwcycsIHByb3BzLmtvbmdTZXJ2aWNlUHJvcHMudGFza0RlZmluaXRpb24uZGVmYXVsdENvbnRhaW5lcj8uY29udGFpbmVyTmFtZSA/PycnLCBmYWxzZSk7XG4gICAgICB9XG4gICAgfTtcblxuICAgIGlmIChwcm9wcy5rb25nRmVhdHVyZXNQcm9wcy5kZXZQb3J0YWxQcm9wcy5lbmFibGVkICYmIHByb3BzLmRldlBvcnRhbE5sYikge1xuICAgICAgdGhpcy5yZWdpc3RlcktvbmdGZWF0dXJlVGFyZ2V0cyg4MCwgdGhpcy5kZXZQb3J0YWxQb3J0LCBwcm9wcy5kZXZQb3J0YWxObGIsICdLb25nRGV2UG9ydGFsJywgcHJvcHMua29uZ1NlcnZpY2VQcm9wcy50YXNrRGVmaW5pdGlvbi5kZWZhdWx0Q29udGFpbmVyPy5jb250YWluZXJOYW1lID8/JycsIHRydWUsIHRoaXMuZGV2UG9ydGFsRW5kcG9pbnQpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgcmVnaXN0ZXJLb25nRmVhdHVyZVRhcmdldHMobGlzdGVuZXJQb3J0OiBudW1iZXIsIGNvbnRhaW5lclBvcnQ6IG51bWJlciwgbGI6IE5ldHdvcmtMb2FkQmFsYW5jZXIsIGlkOiBzdHJpbmcsXG4gICAgY29udGFpbmVyTmFtZTogc3RyaW5nLCBhZGRBUmVjb3JkOiBib29sZWFuLCByZWNvcmROYW1lPzogc3RyaW5nKSB7XG5cbiAgICBjb25zdCBsaXN0ZW5lciA9IG5ldyBOZXR3b3JrTGlzdGVuZXIodGhpcywgJ0tvbmdDcCcraWQrJ0xpc3RlbmVyJywge1xuICAgICAgcG9ydDogbGlzdGVuZXJQb3J0LFxuICAgICAgcHJvdG9jb2w6IFByb3RvY29sLlRDUCxcbiAgICAgIGxvYWRCYWxhbmNlcjogbGIsXG5cbiAgICB9KTtcblxuICAgIGlmIChhZGRBUmVjb3JkKSB7XG4gICAgICBuZXcgQVJlY29yZCh0aGlzLCAnS29uZ0NwJyArIGlkICsgJ0FSZWNvcmQnLCB7XG4gICAgICAgIHpvbmU6IHRoaXMuem9uZSxcbiAgICAgICAgcmVjb3JkTmFtZTogcmVjb3JkTmFtZSxcbiAgICAgICAgdGFyZ2V0OiBSZWNvcmRUYXJnZXQuZnJvbUFsaWFzKG5ldyB0YXJnZXRzLkxvYWRCYWxhbmNlclRhcmdldChsYikpLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgdGhpcy5yZWdpc3RlckxvYWRCYWxhbmNlclRhcmdldHMoXG4gICAgICB7XG4gICAgICAgIGNvbnRhaW5lck5hbWU6IGNvbnRhaW5lck5hbWUsXG4gICAgICAgIGNvbnRhaW5lclBvcnQ6IGNvbnRhaW5lclBvcnQsXG4gICAgICAgIG5ld1RhcmdldEdyb3VwSWQ6IGlkLFxuICAgICAgICBsaXN0ZW5lcjogTGlzdGVuZXJDb25maWcubmV0d29ya0xpc3RlbmVyKGxpc3RlbmVyLCB7XG4gICAgICAgICAgZGVyZWdpc3RyYXRpb25EZWxheTogRHVyYXRpb24uc2Vjb25kcyg2MCksXG4gICAgICAgICAgcG9ydDogbGlzdGVuZXJQb3J0LFxuICAgICAgICAgIGhlYWx0aENoZWNrOiB7XG4gICAgICAgICAgICBwb3J0OiBjb250YWluZXJQb3J0LnRvU3RyaW5nKCksXG4gICAgICAgICAgfSxcbiAgICAgICAgICBwcmVzZXJ2ZUNsaWVudElwOiBmYWxzZSxcbiAgICAgICAgICBwcm94eVByb3RvY29sVjI6IGZhbHNlLFxuICAgICAgICAgIHByb3RvY29sOiBQcm90b2NvbC5UQ1AsXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgdGhpcy5jb25uZWN0aW9ucy5hbGxvd0Zyb20oUGVlci5pcHY0KHRoaXMuY2x1c3Rlci52cGMudnBjQ2lkckJsb2NrKSwgUG9ydC50Y3AoY29udGFpbmVyUG9ydCkpO1xuICAgIGlmIChyZWNvcmROYW1lKSB7XG4gICAgICBuZXcgQ2ZuT3V0cHV0KHRoaXMsICdLb25nQ3AnICsgaWQgKyAnRW5kcG9pbnQnLCB7XG4gICAgICAgIHZhbHVlOiByZWNvcmROYW1lLFxuICAgICAgICBleHBvcnROYW1lOiBpZCxcbiAgICAgIH0pO1xuICAgIH1cbiAgfVxufVxuXG5cbiJdfQ==