"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityConfiguration = exports.JobBookmarksEncryptionMode = exports.CloudWatchEncryptionMode = exports.S3EncryptionMode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const kms = require("aws-cdk-lib/aws-kms");
const cdk = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
/**
 * Encryption mode for S3.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_S3Encryption.html#Glue-Type-S3Encryption-S3EncryptionMode
 */
var S3EncryptionMode;
(function (S3EncryptionMode) {
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    S3EncryptionMode["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    S3EncryptionMode["KMS"] = "SSE-KMS";
})(S3EncryptionMode = exports.S3EncryptionMode || (exports.S3EncryptionMode = {}));
/**
 * Encryption mode for CloudWatch Logs.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_CloudWatchEncryption.html#Glue-Type-CloudWatchEncryption-CloudWatchEncryptionMode
 */
var CloudWatchEncryptionMode;
(function (CloudWatchEncryptionMode) {
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    CloudWatchEncryptionMode["KMS"] = "SSE-KMS";
})(CloudWatchEncryptionMode = exports.CloudWatchEncryptionMode || (exports.CloudWatchEncryptionMode = {}));
/**
 * Encryption mode for Job Bookmarks.
 * @see https://docs.aws.amazon.com/glue/latest/webapi/API_JobBookmarksEncryption.html#Glue-Type-JobBookmarksEncryption-JobBookmarksEncryptionMode
 */
var JobBookmarksEncryptionMode;
(function (JobBookmarksEncryptionMode) {
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    JobBookmarksEncryptionMode["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(JobBookmarksEncryptionMode = exports.JobBookmarksEncryptionMode || (exports.JobBookmarksEncryptionMode = {}));
/**
 * A security configuration is a set of security properties that can be used by AWS Glue to encrypt data at rest.
 *
 * The following scenarios show some of the ways that you can use a security configuration.
 * - Attach a security configuration to an AWS Glue crawler to write encrypted Amazon CloudWatch Logs.
 * - Attach a security configuration to an extract, transform, and load (ETL) job to write encrypted Amazon Simple Storage Service (Amazon S3) targets and encrypted CloudWatch Logs.
 * - Attach a security configuration to an ETL job to write its jobs bookmarks as encrypted Amazon S3 data.
 * - Attach a security configuration to a development endpoint to write encrypted Amazon S3 targets.
 */
class SecurityConfiguration extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityConfigurationName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_alpha_SecurityConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SecurityConfiguration);
            }
            throw error;
        }
        if (!props.s3Encryption && !props.cloudWatchEncryption && !props.jobBookmarksEncryption) {
            throw new Error('One of cloudWatchEncryption, jobBookmarksEncryption or s3Encryption must be defined');
        }
        const kmsKeyCreationRequired = (props.s3Encryption && props.s3Encryption.mode === S3EncryptionMode.KMS && !props.s3Encryption.kmsKey) ||
            (props.cloudWatchEncryption && !props.cloudWatchEncryption.kmsKey) ||
            (props.jobBookmarksEncryption && !props.jobBookmarksEncryption.kmsKey);
        const autoCreatedKmsKey = kmsKeyCreationRequired ? new kms.Key(this, 'Key') : undefined;
        let cloudWatchEncryption;
        if (props.cloudWatchEncryption) {
            this.cloudWatchEncryptionKey = props.cloudWatchEncryption.kmsKey || autoCreatedKmsKey;
            cloudWatchEncryption = {
                cloudWatchEncryptionMode: props.cloudWatchEncryption.mode,
                kmsKeyArn: this.cloudWatchEncryptionKey?.keyArn,
            };
        }
        let jobBookmarksEncryption;
        if (props.jobBookmarksEncryption) {
            this.jobBookmarksEncryptionKey = props.jobBookmarksEncryption.kmsKey || autoCreatedKmsKey;
            jobBookmarksEncryption = {
                jobBookmarksEncryptionMode: props.jobBookmarksEncryption.mode,
                kmsKeyArn: this.jobBookmarksEncryptionKey?.keyArn,
            };
        }
        let s3Encryptions;
        if (props.s3Encryption) {
            if (props.s3Encryption.mode === S3EncryptionMode.KMS) {
                this.s3EncryptionKey = props.s3Encryption.kmsKey || autoCreatedKmsKey;
            }
            // NOTE: CloudFormations errors out if array is of length > 1. That's why the props don't expose an array
            s3Encryptions = [{
                    s3EncryptionMode: props.s3Encryption.mode,
                    kmsKeyArn: this.s3EncryptionKey?.keyArn,
                }];
        }
        const resource = new aws_glue_1.CfnSecurityConfiguration(this, 'Resource', {
            name: props.securityConfigurationName,
            encryptionConfiguration: {
                cloudWatchEncryption,
                jobBookmarksEncryption,
                s3Encryptions,
            },
        });
        this.securityConfigurationName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Creates a Connection construct that represents an external security configuration.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param securityConfigurationName name of external security configuration.
     */
    static fromSecurityConfigurationName(scope, id, securityConfigurationName) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.securityConfigurationName = securityConfigurationName;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecurityConfiguration = SecurityConfiguration;
_a = JSII_RTTI_SYMBOL_1;
SecurityConfiguration[_a] = { fqn: "@aws-cdk/aws-glue-alpha.SecurityConfiguration", version: "2.37.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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