"""
  scenariogeneration
  https://github.com/pyoscx/scenariogeneration
 
  This Source Code Form is subject to the terms of the Mozilla Public
  License, v. 2.0. If a copy of the MPL was not distributed with this
  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 
  Copyright (c) 2022 The scenariogeneration Authors.

"""
import xml.etree.ElementTree as ET
from .utils import (
    Controller,
    Dimensions,
    Center,
    BoundingBox,
    Properties,
    Parameter,
    convert_float,
)
from .utils import EntityRef
from .utils import ParameterDeclarations
from .enumerations import (
    VehicleCategory,
    PedestrianCategory,
    MiscObjectCategory,
    ObjectType,
    VersionBase,
)
from .utils import DynamicsConstraints, CatalogFile, CatalogReference

# TODO: Add functionality for lists in add_entity_byref & add_entity_bytype
class Entities:
    """The Entities class creates the entities part of OpenScenario

    Attributes
    ----------
        scenario_objects (list of ScenarioObject): ScenarioObject type entities in the scenario

        entities (list of Entity): Entity type of entities in the scenario


    Methods
    -------
        parse(element)
            parses a ElementTree created by the class and returns an instance of the class

        add_scenario_object(entityobject,controller)
            adds a ScenarioObject to the scenario

        add_entity_bytype(name,entity_type)
            adds an Entity by the type of entity

        add_entity_byref(name,reference)
            adds an Entity by a reference name

        get_element()
            Returns the full ElementTree of the class


    """

    def __init__(self):
        """initalizes the Entities class"""
        self.scenario_objects = []
        self.entities = []

    def __eq__(self, other):
        if isinstance(other, Entities):
            if (
                self.scenario_objects == other.scenario_objects
                and self.entities == other.entities
            ):
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element of Entities

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A entities element (same as generated by the class itself)

        Returns
        -------
            entities (Entities): a Entities object

        """
        ent_ret = Entities()
        scenario_objects = []
        entity_selections = []
        if element.find("ScenarioObject") != None:
            for object in element.findall("ScenarioObject"):
                scenario_objects.append(ScenarioObject.parse(object))
        if element.find("EntitySelection") != None:
            for entity in element.findall("EntitySelection"):
                entity_selections.append(Entity.parse(entity))

        ent_ret.entities = entity_selections
        ent_ret.scenario_objects = scenario_objects
        return ent_ret

    def add_scenario_object(self, name, entityobject, controller=None):
        """adds a ScenarioObject to the scenario

        Parameters
        ----------
            name (str): name of the scenario object

            entityobject (CatalogReference, Vehicle, Pedestrian, MiscObject, or ExternalObjectReference (V1.1)): object description

            controller (CatalogReference, or Controller): controller for the object
                Default (None)

        """

        self.scenario_objects.append(ScenarioObject(name, entityobject, controller))
        return self

    def add_entity_bytype(self, name, object_type):
        """adds an Entity to the scenario

        Parameters
        ----------
            name (str): name of the entity

            object_type (ObjectType or list of ObjectType): type of entity

        """
        self.entities.append(Entity(name, object_type=object_type))
        return self

    def add_entity_byref(self, name, entity):
        """adds an Entity to the scenario

        Parameters
        ----------
            name (str): name of the entity

            entity (str): type of entity

        """

        self.entities.append(Entity(name, entityref=entity))
        return self

    def get_element(self):
        """returns the elementTree of the Entities"""
        element = ET.Element("Entities")
        for i in self.scenario_objects:
            element.append(i.get_element())

        for i in self.entities:
            element.append(i.get_element())

        return element


class ScenarioObject(VersionBase):
    """The ScenarioObject creates a scenario object of OpenScenario

    Parameters
    ----------
        name (str): name of the object

        entityobject (CatalogReference, Vehicle, Pedestrian, MiscObject, or ExternalObjectReference): object description

        controller (CatalogReference, or Controller): controller for the object

    Attributes
    ----------
        name (str): name of the object

        entityobject (CatalogReference, Vehicle, Pedestrian, MiscObject, or ExternalObjectReference): object description

        controller (CatalogReference, or Controller): controller for the object

    Methods
    -------
        parse(element)
            parses a ElementTree created by the class and returns an instance of the class

        get_element()
            Returns the full ElementTree of the class

        get_attributes()
            returns the attributes of the class
    """

    def __init__(self, name, entityobject, controller=None):
        """initalizes the ScenarioObject

        Parameters
        ----------
            name (str): name of the object

            entityobject (CatalogReference, Vehicle, Pedestrian, MiscObject, or ExternalObjectReference): object description

            controller (CatalogReference, or Controller): controller for the object
                Default: None

        """
        self.name = name
        if not (
            isinstance(entityobject, CatalogReference)
            or isinstance(entityobject, Vehicle)
            or isinstance(entityobject, Pedestrian)
            or isinstance(entityobject, MiscObject)
            or (
                not self.isVersion(minor=0)
                and isinstance(entityobject, ExternalObjectReference)
            )
        ):
            raise TypeError(
                "entityobject is not of type CatalogReference, Vehicle, Pedestrian, MiscObject, nor ExternalObjectReference (or to old version of openscenario)"
            )

        if controller is not None and not (
            isinstance(controller, CatalogReference)
            or isinstance(controller, Controller)
        ):
            raise TypeError(
                "controller input is not of type CatalogReference or Controller"
            )
        self.entityobject = entityobject
        self.controller = controller

    def __eq__(self, other):
        if isinstance(other, ScenarioObject):
            if (
                self.get_attributes() == other.get_attributes()
                and self.controller == other.controller
                and self.entityobject == other.entityobject
            ):
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element of ScenarioObject

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A scenarioobject element (same as generated by the class itself)

        Returns
        -------
            scenarioobject (ScenarioObject): a ScenarioObject object

        """
        name = element.attrib["name"]
        if element.find("CatalogReference") != None:
            entityobject = CatalogReference.parse(element.find("CatalogReference"))
        elif element.find("Vehicle") != None:
            entityobject = Vehicle.parse(element.find("Vehicle"))
        elif element.find("Pedestrian") != None:
            entityobject = Pedestrian.parse(element.find("Pedestrian"))
        elif element.find("MiscObject") != None:
            entityobject = MiscObject.parse(element.find("MiscObject"))
        elif element.find("ExternalObjectReference") != None:
            entityobject = ExternalObjectReference.parse(
                element.find("ExternalObjectReference")
            )

        controller = None
        if element.find("ObjectController") != None:
            object_controller_element = element.find("ObjectController")
            if object_controller_element.find("Controller") != None:
                controller = Controller.parse(
                    object_controller_element.find("Controller")
                )
            elif object_controller_element.find("CatalogReference") != None:
                controller = CatalogReference.parse(
                    object_controller_element.find("CatalogReference")
                )

        return ScenarioObject(name, entityobject, controller)

    def get_attributes(self):
        """returns the attributes of the Entity as a dict"""
        return {"name": self.name}

    def get_element(self):
        """returns the elementTree of the Entity"""
        element = ET.Element("ScenarioObject", attrib=self.get_attributes())

        element.append(self.entityobject.get_element())
        if self.controller:
            objcont = ET.SubElement(element, "ObjectController")
            objcont.append(self.controller.get_element())

        return element


class Entity:
    """The Entity class creates an Entity of OpenScenario
    Can either use a object_type or entityref (not both)

    Parameters
    ----------
        name (str): name of the Entity

        optionals:
            object_type (ObjectType or list of ObjectType): the object_type to be used

            entityref (str): reference to an entity

    Attributes
    ----------
        name (str): name of the Entity

        object_type (ObjectType): the object_type to be used

        entityref (str): reference to an entity

    Methods
    -------
        parse(element)
            parses a ElementTree created by the class and returns an instance of the class

        get_element()
            Returns the full ElementTree of the class

        get_attributes()
            Returns a dictionary of all attributes of the class

    """

    def __init__(self, name, object_type=None, entityref=None):
        """Initalizes the Entity

        Parameters
        ----------
            name (str): name of the Entity

            optionals (only use one):
                object_type (ObjectType): the object_type to be used

                entityref (str): reference to an entity

        """
        self.name = name
        self.object_type = []
        self.entity = []
        if (object_type != None) and (entityref != None):
            raise KeyError("only one of objecttype or entityref are alowed")
        if (object_type == None) and (entityref == None):
            raise KeyError("either objecttype or entityref is requiered")
        if entityref:
            if isinstance(entityref, list):
                self.entity = [EntityRef(x) for x in entityref]
            else:
                self.entity.append(EntityRef(entityref))
            self.object_type = None
        else:
            if not hasattr(ObjectType, str(object_type)):
                ValueError("object_type input not a valid ObjectType")
            if isinstance(object_type, list):
                self.object_type = object_type
            else:
                self.object_type.append(object_type)
            self.entity = None

    def __eq__(self, other):
        if isinstance(other, Entity):
            if (
                self.get_attributes() == other.get_attributes()
                and self.object_type == other.object_type
                and self.entity == other.entity
            ):
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element of Entity

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A entity element (same as generated by the class itself)

        Returns
        -------
            entity (Entity): a Entity object

        """
        name = element.attrib["name"]
        bytypes = []
        entity_refs = []

        members_element = element.find("Members")
        if members_element.find("ByType") != None:
            types = members_element.findall("ByType")
            for type in types:
                bytypes.append(getattr(ObjectType, type.attrib["value"]))
            entity_refs = None
        elif members_element.find("EntityRef") != None:
            entities = members_element.findall("EntityRef")
            for entity in entities:
                entity_refs.append(entity.attrib["entityRef"])
            bytypes = None

        return Entity(name, object_type=bytypes, entityref=entity_refs)

    def get_attributes(self):
        """returns the attributes of the Entity as a dict"""
        return {"name": self.name}

    def get_element(self):
        """returns the elementTree of the Entity"""
        element = ET.Element("EntitySelection", attrib=self.get_attributes())
        members = ET.SubElement(element, "Members")
        if self.entity:
            for entity in self.entity:
                members.append(entity.get_element())
        if self.object_type:
            for object_type in self.object_type:
                ET.SubElement(
                    members, "ByType", attrib={"value": object_type.get_name()}
                )
        return element


class Pedestrian(VersionBase):
    """the Pedestrian class creates a pedestrian type entity of openscenario

    Parameters
    ----------
        name (str): name of the type (req for catalog)

        mass (float): mass of the pedestrian

        boundingbox (BoundingBox): the bounding box of the pedestrian

        category (PedestrianCategory): type of of pedestrian

        model (str): definition model of the pedestrian
            Default: None

    Attributes
    ----------
        name (str): name of the pedestrian

        model (str): definition model of the pedestrian

        mass (float): mass of the pedestrian

        category (PedestrianCategory): type of pedestrian

        boundingbox (BoundingBox): the bounding box of the pedestrian

        parameters (ParameterDeclaration): Parameter declarations of the pedestrian

        properties (Properties): additional properties of the pedestrian

    Methods
    -------
        parse(element)
            parses a ElementTree created by the class and returns an instance of the class itself

        add_parameter(parameter)
            adds a parameter declaration to the pedestrian

        add_property(name, value)
            adds a single property to the pedestrian

        add_property_file(filename)
            adds a property file to the pedestrian

        append_to_catalog(filename)
            adds the vehicle to an existing pedestrian

        dump_to_catalog(filename,name,description,author)
            crates a new catalog with the pedestrian

        get_element()
            Returns the full ElementTree of the class

        get_attributes()
            Returns a dictionary of all attributes of the class

    """

    def __init__(self, name, mass, category, boundingbox, model=None):
        """initalzie the Pedestrian Class

        Parameters
        ----------
            name (str): name of the type (req for catalog)

            mass (float): mass of the pedestrian

            category (PedestrianCategory): type of of pedestrian

            boundingbox (BoundingBox): the bounding box of the pedestrian

            model (str): definition model of the pedestrian
                Default: None
        """
        self.name = name
        self.model = model
        self.mass = convert_float(mass)
        if not hasattr(PedestrianCategory, str(category)):
            ValueError(str(category) + " is not a valid pedestrian type")
        if not isinstance(boundingbox, BoundingBox):
            raise TypeError("boundingbox input is not of type BoundingBox")

        self.category = category
        self.boundingbox = boundingbox
        self.parameters = ParameterDeclarations()
        self.properties = Properties()

    def __eq__(self, other):
        if isinstance(other, Pedestrian):
            if (
                self.get_attributes() == other.get_attributes()
                and self.boundingbox == other.boundingbox
                and self.properties == other.properties
                and self.parameters == other.parameters
            ):
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element of Pedestrian

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A pedestrian element (same as generated by the class itself)

        Returns
        -------
            pedestrian (Pedestrian): a Pedestrian object

        """
        name = element.attrib["name"]
        mass = convert_float(element.attrib["mass"])
        model = None
        if "model3d" in element.attrib:
            model = element.attrib["model3d"]
        elif "model" in element.attrib:
            model = element.attrib["model"]
        category = getattr(PedestrianCategory, element.attrib["pedestrianCategory"])
        if element.find("ParameterDeclarations") != None:
            parameters = ParameterDeclarations.parse(
                element.find("ParameterDeclarations")
            )
        else:
            parameters = ParameterDeclarations()
        boundingbox = BoundingBox.parse(element.find("BoundingBox"))
        properties = Properties.parse(element.find("Properties"))

        pedestrian = Pedestrian(name, mass, category, boundingbox, model)
        pedestrian.parameters = parameters
        pedestrian.properties = properties

        return pedestrian

    def dump_to_catalog(self, filename, catalogtype, description, author):
        """dump_to_catalog creates a new catalog and adds the Pedestrian to it

        Parameters
        ----------
            filename (str): path of the new catalog file

            catalogtype (str): name of the catalog

            description (str): description of the catalog

            author (str): author of the catalog

        """
        cf = CatalogFile()
        cf.create_catalog(filename, catalogtype, description, author)
        cf.add_to_catalog(self)
        cf.dump()

    def append_to_catalog(self, filename):
        """adds the Pedestrian to an existing catalog

        Parameters
        ----------
            filename (str): path to the catalog file

        """
        cf = CatalogFile()
        cf.open_catalog(filename)
        cf.add_to_catalog(self)
        cf.dump()

    def add_parameter(self, parameter):
        """adds a parameter declaration to the pedestrian

        Parameters
        ----------
            parameter (Parameter): A new parameter declaration for the pedestrian

        """
        if not isinstance(parameter, Parameter):
            raise TypeError("parameter input is not of type Parameter")
        self.parameters.add_parameter(parameter)
        return self

    def add_property(self, name, value):
        """adds a single property to the pedestrian

        Parameters
        ----------
            name (str): name of the property

            value (str): value of the property

        """
        self.properties.add_property(name, value)
        return self

    def add_property_file(self, filename):
        """adds a property file to the pedestrian

        Parameters
        ----------
            filename (str): filename of a property file

        """
        self.properties.add_file(filename)
        return self

    def get_attributes(self):
        """returns the attributes as a dict of the pedestrian"""
        retdict = {}
        retdict["name"] = str(self.name)
        retdict["pedestrianCategory"] = self.category.get_name()
        if self.model is not None:
            if self.isVersion(minor=0):
                retdict["model"] = self.model
            else:
                retdict["model3d"] = self.model
        retdict["mass"] = str(self.mass)
        return retdict

    def get_element(self):
        """returns the elementTree of the pedestrian"""
        element = ET.Element("Pedestrian", attrib=self.get_attributes())
        element.append(self.parameters.get_element())
        element.append(self.boundingbox.get_element())
        element.append(self.properties.get_element())

        return element


class MiscObject(VersionBase):
    """the MiscObject Class creates a MiscObject for openscenario

    Parameters
    ----------
        name (str): name of the MiscObject

        mass (float): mass of the object

        category (MiscObjectCategory): the category of the misc object

        boundingbox (BoundingBox): the bounding box of the MiscObject

        model3d (str): path to model file (valid from V1.1)
            Default: None

    Attributes
    ----------
        name (str): name of the object

        mass (float): mass of the object

        misc_type (MiscObjectCategory): type of misc object

        boundingbox (BoundingBox): the bounding box of the MiscObject

        parameters (ParameterDeclaration): Parameter declarations of the MiscObject

        properties (Properties): additional properties of the MiscObject

        model3d (str): path to model file (valid from V1.1)
            Default: None

    Methods
    -------
        add_parameter(parameter)
            adds a parameter declaration to the MiscObject

        add_property(name, value)
            adds a single property to the MiscObject

        add_property_file(filename)
            adds a property file to the MiscObject

        append_to_catalog(filename)
            adds the MiscObject to an existing catalog

        dump_to_catalog(filename,name,description,author)
            crates a new catalog with the MiscObject

        get_element()
            Returns the full ElementTree of the class

        get_attributes()
            Returns a dictionary of all attributes of the class

        parse(element)
            parses a ElementTree created by the class and returns an instance of the class

    """

    def __init__(self, name, mass, category, boundingbox, model3d=None):
        """initalzie the MiscObject Class

        Parameters
        ----------
            name (str): name of the MiscObject

            mass (float): mass of the object

            category (MiscObjectCategory): the category of the misc object

            boundingbox (BoundingBox): the bounding box of the MiscObject

            model3d (str): path to model file (valid from V1.1)
                Default: None
        """
        self.name = name
        self.mass = convert_float(mass)
        if not hasattr(MiscObjectCategory, str(category)):
            raise TypeError(str(category) + " is not a valid MiscObject type")
        self.category = category
        if not isinstance(boundingbox, BoundingBox):
            raise TypeError("boundingbox input is not of type BoundingBox")
        self.boundingbox = boundingbox
        self.parameters = ParameterDeclarations()
        self.properties = Properties()
        self.model3d = model3d

    def __eq__(self, other):
        if isinstance(other, MiscObject):
            if (
                self.get_attributes() == other.get_attributes()
                and self.boundingbox == other.boundingbox
                and self.properties == other.properties
                and self.parameters == other.parameters
            ):
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element to MiscObject

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A MiscObject element (same as generated by the class itself)

        Returns
        ------
            object (MiscObject): a MiscObject object

        """
        model3d = None
        if "model3d" in element.attrib:
            model3d = element.attrib["model3d"]
        mass = convert_float(element.attrib["mass"])
        name = element.attrib["name"]
        properties = Properties.parse(element.find("Properties"))
        boundingbox = BoundingBox.parse(element.find("BoundingBox"))
        category = getattr(MiscObjectCategory, element.attrib["miscObjectCategory"])

        if element.find("ParameterDeclarations") != None:
            parameters = ParameterDeclarations.parse(
                element.find("ParameterDeclarations")
            )
        else:
            parameters = ParameterDeclarations()

        object = MiscObject(name, mass, category, boundingbox, model3d)
        object.parameters = parameters
        object.properties = properties
        return object

    def dump_to_catalog(self, filename, catalogtype, description, author):
        """dump_to_catalog creates a new catalog and adds the MiscObject to it

        Parameters
        ----------
            filename (str): path of the new catalog file

            catalogtype (str): name of the catalog

            description (str): description of the catalog

            author (str): author of the catalog

        """
        cf = CatalogFile()
        cf.create_catalog(filename, catalogtype, description, author)
        cf.add_to_catalog(self)
        cf.dump()

    def append_to_catalog(self, filename):
        """adds the MiscObject to an existing catalog

        Parameters
        ----------
            filename (str): path to the catalog file

        """
        cf = CatalogFile()
        cf.open_catalog(filename)
        cf.add_to_catalog(self)
        cf.dump()

    def add_parameter(self, parameter):
        """adds a parameter declaration to the MiscObject

        Parameters
        ----------
            parameter (Parameter): A new parameter declaration for the MiscObject

        """
        if not isinstance(parameter, Parameter):
            raise TypeError("parameter input is not of type Parameter")
        self.parameters.add_parameter(parameter)
        return self

    def add_property(self, name, value):
        """adds a single property to the MiscObject

        Parameters
        ----------
            name (str): name of the property

            value (str): value of the property

        """
        self.properties.add_property(name, value)
        return self

    def add_property_file(self, filename):
        """adds a property file to the MiscObject

        Parameters
        ----------
            filename (str): filename of a property file

        """
        self.properties.add_file(filename)
        return self

    def get_attributes(self):
        """returns the attributes as a dict of the MiscObject"""
        retdict = {}
        retdict["name"] = str(self.name)
        retdict["miscObjectCategory"] = self.category.get_name()
        retdict["mass"] = str(self.mass)
        if not self.isVersion(minor=0) and self.model3d:
            retdict["model3d"] = self.model3d
        return retdict

    def get_element(self):
        """returns the elementTree of the MiscObject"""
        element = ET.Element("MiscObject", attrib=self.get_attributes())
        element.append(self.parameters.get_element())
        element.append(self.boundingbox.get_element())
        element.append(self.properties.get_element())

        return element


class Vehicle(VersionBase):
    """the Vehicle Class creates a Vehicle for openscenario

    Parameters
    ----------
        name (str): name of the vehicle

        vehicle_type (VehicleCategory): type of vehicle

        boundingbox (BoundingBox): the bounding box of the vehicle

        frontaxle (Axle): the front axle properties of the vehicle

        rearaxle (Axle): the back axle properties of the vehicle

        max_speed (float): the maximum speed of the vehicle

        max_acceleration (float): the maximum acceleration of the vehicle

        max_deceleration (float): the maximum deceleration of the vehicle

        mass (float): the mass of the vehicle (valid from OpenSCENARIO V1.1)
            Default: None

        model3d (str): path to model file (valid from V1.1)
            Default: None

    Attributes
    ----------
        name (str): name of the vehicle

        vehicle_type (VehicleCategory): type of vehicle

        boundingbox (BoundingBox): the bounding box of the vehicle

        axles (Axles): an Axles object

        dynamics (DynamicsConstraints): the allowed dynamics of the vehicle

        parameters (ParameterDeclaration): Parameter declarations of the vehicle

        properties (Properties): additional properties of the vehicle

        mass (float): the mass of the vehicle

        model3d (str): path to model file (valid from V1.1)
            Default: None

    Methods
    -------
        parse(element)
            parses a ElementTree created by the class and returns an instance of the class itself

        add_axle(axle)
            adds an additional axle to the vehicle

        add_parameter(parameter)
            adds a parameter declaration to the vehicle

        add_property(name, value)
            adds a single property to the vehicle

        add_property_file(filename)
            adds a property file to the vehicle

        append_to_catalog(filename)
            adds the vehicle to an existing catalog

        dump_to_catalog(filename,name,description,author)
            crates a new catalog with the vehicle

        get_element()
            Returns the full ElementTree of the class

        get_attributes()
            Returns a dictionary of all attributes of the class

    """

    def __init__(
        self,
        name,
        vehicle_type,
        boundingbox,
        frontaxle,
        rearaxle,
        max_speed,
        max_acceleration,
        max_deceleration,
        mass=None,
        model3d=None,
    ):
        """initalzie the Vehicle Class

        Parameters
        ----------
            name (str): name of the vehicle

            vehicle_type (VehicleCategory): type of vehicle

            boundingbox (BoundingBox): the bounding box of the vehicle

            frontaxle (Axle): the front axle properties of the vehicle

            rearaxle (Axle): the back axle properties of the vehicle

            max_speed (float): the maximum speed of the vehicle

            max_acceleration (float): the maximum acceleration of the vehicle

            max_deceleration (float): the maximum deceleration of the vehicle

            mass (float): the mass of the vehicle (valid from OpenSCENARIO V1.1)
                Default: None

            model3d (str): path to model file (valid from V1.1)
                Default: None
        """
        self.name = name
        if not hasattr(VehicleCategory, str(vehicle_type)):
            raise TypeError(vehicle_type + " is not a valid vehicle type.")
        if not isinstance(boundingbox, BoundingBox):
            raise TypeError("boundingbox input is not of type BoundingBox")

        self.vehicle_type = vehicle_type
        self.boundingbox = boundingbox

        self.axles = Axles(frontaxle, rearaxle)
        self.dynamics = DynamicsConstraints(
            max_acceleration, max_deceleration, max_speed
        )
        self.parameters = ParameterDeclarations()
        self.properties = Properties()
        self.mass = mass
        self.model3d = model3d

    def __eq__(self, other):
        if isinstance(other, Vehicle):
            if (
                self.get_attributes() == other.get_attributes()
                and self.boundingbox == other.boundingbox
                and self.properties == other.properties
                and self.axles == other.axles
                and self.dynamics == other.dynamics
                and self.parameters == other.parameters
                and self.mass == other.mass
            ):
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element of Vehicle

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A vehicle element (same as generated by the class itself)

        Returns
        -------
            vehicle (Vehicle): a Vehicle object

        """
        name = element.attrib["name"]
        mass = None
        if "mass" in element.attrib:
            mass = element.attrib["mass"]
        vehicle_type = getattr(VehicleCategory, element.attrib["vehicleCategory"])
        model3d = None
        if "model3d" in element.attrib:
            model3d = element.attrib["model3d"]
        # if element.find('ParameterDeclarations'):
        if element.find("ParameterDeclarations") != None:
            parameters = ParameterDeclarations.parse(
                element.find("ParameterDeclarations")
            )
        else:
            parameters = ParameterDeclarations()
        boundingbox = BoundingBox.parse(element.find("BoundingBox"))
        properties = Properties.parse(element.find("Properties"))

        performance = DynamicsConstraints.parse(element.find("Performance"))
        max_speed = performance.get_attributes()["maxSpeed"]
        max_acc = performance.get_attributes()["maxAcceleration"]
        max_dec = performance.get_attributes()["maxDeceleration"]

        axles_element = element.find("Axles")
        frontaxle = Axle.parse(axles_element.find("FrontAxle"))
        rearaxle = Axle.parse(axles_element.find("RearAxle"))

        vehicle = Vehicle(
            name,
            vehicle_type,
            boundingbox,
            frontaxle,
            rearaxle,
            max_speed,
            max_acc,
            max_dec,
            mass,
            model3d,
        )
        vehicle.properties = properties
        vehicle.parameters = parameters

        additional_axles = axles_element.findall("AdditionalAxle")
        if additional_axles != None:
            for axle in additional_axles:
                vehicle.axles.add_axle(Axle.parse(axle))

        return vehicle

    def dump_to_catalog(self, filename, catalogtype, description, author):
        """dump_to_catalog creates a new catalog and adds the vehicle to it

        Parameters
        ----------
            filename (str): path of the new catalog file

            catalogtype (str): name of the catalog

            description (str): description of the catalog

            author (str): author of the catalog

        """
        cf = CatalogFile()
        cf.create_catalog(filename, catalogtype, description, author)
        cf.add_to_catalog(self)
        cf.dump()

    def append_to_catalog(self, filename):
        """adds the vehicle to an existing catalog

        Parameters
        ----------
            filename (str): path to the catalog file

        """
        cf = CatalogFile()
        cf.open_catalog(filename)
        cf.add_to_catalog(self)
        cf.dump()

    def add_axle(self, axle):
        """adds an additional axle to the vehicle

        Parameters
        ----------
            axle (Axle): an additional Axle

        """

        self.axles.add_axle(axle)
        return self

    def add_parameter(self, parameter):
        """adds a parameter declaration to the vehicle

        Parameters
        ----------
            parameter (Parameter): A new parameter declaration for the vehicle

        """
        if not isinstance(parameter, Parameter):
            raise TypeError("parameter input is not of type Parameter")
        self.parameters.add_parameter(parameter)
        return self

    def add_property(self, name, value):
        """adds a single property to the vehicle

        Parameters
        ----------
            name (str): name of the property

            value (str): value of the property

        """
        self.properties.add_property(name, value)
        return self

    def add_property_file(self, filename):
        """adds a property file to the vehicle

        Parameters
        ----------
            filename (str): filename of a property file

        """
        self.properties.add_file(filename)
        return self

    def get_attributes(self):
        """returns the attributes as a dict of the Center"""
        retdict = {}
        retdict["name"] = str(self.name)
        retdict["vehicleCategory"] = self.vehicle_type.get_name()
        if self.mass and not self.isVersion(minor=0):
            retdict["mass"] = str(self.mass)
        if not self.isVersion(minor=0) and self.model3d:
            retdict["model3d"] = self.model3d

        return retdict

    def get_element(self):
        """returns the elementTree of the Center"""
        element = ET.Element("Vehicle", attrib=self.get_attributes())
        element.append(self.parameters.get_element())
        element.append(self.boundingbox.get_element())
        element.append(self.dynamics.get_element("Performance"))
        element.append(self.axles.get_element())
        element.append(self.properties.get_element())

        return element


class Axle:
    """the Axle describes the axle properties of a vehicle

    Parameters
    ----------
        maxsteer (float): max steering angle

        wheeldia (float): diameter of wheel

        track_width (float): distance between wheelcenter

        xpos (float): x position of axle relative to car reference

        zpos (float): z position of axle relative to car reference

    Attributes
    ----------
        maxsteer (float): max steering angle

        wheeldia (float): diameter of wheel

        track_width (float): distance between wheelcenter

        xpos (float): x position of axle relative to car reference

        zpos (float): z position of axle relative to car reference

    Methods
    -------
        get_element()
            Returns the full ElementTree of the class

        get_attributes()
            Returns the attributes of the class

        parse(element)
            parses a ElementTree created by the class and returns an instance of the class


    """

    def __init__(self, maxsteer, wheeldia, track_width, xpos, zpos):
        """initalzie the Axle

        Parameters
        ----------
            maxsteer (float): max steering angle

            wheeldia (float): diameter of wheel

            track_width (float): distance between wheelcenter

            xpos (float): x position of axle relative to car reference

            zpos (float): z position of axle relative to car reference
        """
        self.maxsteer = convert_float(maxsteer)
        self.wheeldia = convert_float(wheeldia)
        self.track_width = convert_float(track_width)
        self.xpos = convert_float(xpos)
        self.zpos = convert_float(zpos)

    @staticmethod
    def parse(element):
        """Parses the xml element of Axle

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A axle element (same as generated by the class itself

        Returns
        -------
            axle (Axle): a Axle object

        """
        maxsteer = convert_float(element.attrib["maxSteering"])
        wheeldia = convert_float(element.attrib["wheelDiameter"])
        track_width = convert_float(element.attrib["trackWidth"])
        xpos = convert_float(element.attrib["positionX"])
        zpos = convert_float(element.attrib["positionZ"])
        return Axle(maxsteer, wheeldia, track_width, xpos, zpos)

    def __eq__(self, other):
        if isinstance(other, Axle):
            if self.get_attributes() == other.get_attributes():
                return True
        return False

    def get_attributes(self):
        """returns the attributes of the Axle as a dict"""
        return {
            "maxSteering": str(self.maxsteer),
            "wheelDiameter": str(self.wheeldia),
            "trackWidth": str(self.track_width),
            "positionX": str(self.xpos),
            "positionZ": str(self.zpos),
        }

    def get_element(self, elementname="AdditionalAxle"):
        """returns the elementTree of the Axle"""
        return ET.Element(elementname, attrib=self.get_attributes())


class Axles:
    """the Axles combines the different Axles to one Element

    Parameters
    ----------
        frontaxle (Axle): Axle properties of the front axle

        rearaxle (Axle): Axle properties of the rear axle

    Attributes
    ----------
        frontaxle (Axle): Axle properties of the front axle

        rearaxle (Axle): Axle properties of the rear axle

        additionals (Axle): additional axles if requiered


    Methods
    -------
        add_axle(Axle)
            adds an additional axle to the Axles

        get_element()
            Returns the full ElementTree of the class

        parse(element)
            parses a ElementTree created by the class and returns an instance of the class


    """

    def __init__(self, frontaxle, rearaxle):
        """initalzie the Axle

        Parameters
        ----------
            frontaxle (Axle): Axle properties of the front axle

            rearaxle (Axle): Axle properties of the rear axle

        """
        if not isinstance(frontaxle, Axle):
            raise TypeError("frontaxle input is not of type Axle")
        if not isinstance(rearaxle, Axle):
            raise TypeError("rearaxle input is not of type Axle")
        self.frontaxle = frontaxle
        self.rearaxle = rearaxle
        self.additionals = []

    def __eq__(self, other):
        if isinstance(other, Axles):
            if (
                self.frontaxle == other.frontaxle
                and self.rearaxle == other.rearaxle
                and self.additionals == other.additionals
            ):
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element of Axles

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A axles element (same as generated by the class itself

        Returns
        -------
            axles (Axles): axles object

        """
        frontaxle = Axle.parse(element.find("FrontAxle"))
        rearaxle = Axle.parse(element.find("RearAxle"))
        axles = Axles(frontaxle, rearaxle)
        additionals = element.findall("AdditionalAxle")
        for additional in additionals:
            axle = Axle.parse(additional)
            axles.add_axle(axle)
        return axles

    def add_axle(self, axle):
        """adds an additional axle to the Axles

        Parameters
        ----------
            frontaxle (Axle): Axle properties of the front axle

        """
        if not isinstance(axle, Axle):
            raise TypeError("axle input is not of type Axle")
        self.additionals.append(axle)
        return self

    def get_element(self):
        """returns the elementTree of the Axle"""
        element = ET.Element("Axles")
        element.append(self.frontaxle.get_element(elementname="FrontAxle"))
        element.append(self.rearaxle.get_element(elementname="RearAxle"))
        for ax in self.additionals:
            element.append(ax.get_element())

        return element


class ExternalObjectReference:
    """the ExternalObjectReference describes the EntityObject ExternalObjectReference (valid from V1.1)

    Parameters
    ----------
        name (str): identifier of the external object

    Attributes
    ----------
        name (str): identifier of the external object

    Methods
    -------
        parse(element)
            parses a ElementTree created by the class and returns an instance of the class

        get_element()
            Returns the full ElementTree of the class

        get_attributes()
            Returns the attributes of the class


    """

    def __init__(self, name):
        """initalzie the ExternalObjectReference

        Parameters
        ----------
            name (str): identifier of the external object
        """
        self.name = name

    def __eq__(self, other):
        if isinstance(other, ExternalObjectReference):
            if self.get_attributes() == other.get_attributes():
                return True
        return False

    @staticmethod
    def parse(element):
        """Parses the xml element of ExternalObjectReference

        Parameters
        ----------
            element (xml.etree.ElementTree.Element): A ExternalObjectReference element (same as generated by the class itself)

        Returns
        -------
            reference (ExternalObjectReference): a ExternalObjectReference object

        """
        name = element.attrib["name"]
        return ExternalObjectReference(name)

    def get_attributes(self):
        """returns the attributes of the Axle as a dict"""
        return {"name": self.name}

    def get_element(self):
        """returns the elementTree of the Axle"""
        return ET.Element("ExternalObjectReference", attrib=self.get_attributes())
