"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("@aws-cdk/core");
const cfn_parse = require("@aws-cdk/core/lib/cfn-parse");
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_1 = require("@aws-cdk/core");
/**
 * (experimental) Construct to import an existing CloudFormation template file into a CDK application.
 *
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 *
 * @experimental
 */
class CfnInclude extends core.CfnElement {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_a = props.preserveLogicalIds) !== null && _a !== void 0 ? _a : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core_1.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core_1.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core_1.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core_1.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core_1.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * (experimental) Returns the low-level CfnResource from the template with the given logical ID.
     *
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *      // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *      const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *      // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file.
     * @experimental
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnCondition object from the 'Conditions' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file.
     * @experimental
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnParameter object from the 'Parameters' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve.
     * @experimental
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnMapping object from the 'Mappings' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve.
     * @experimental
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnOutput object from the 'Outputs' section of the included template.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve.
     * @experimental
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnRule object from the 'Rules' section of the CloudFormation template with the given name.
     *
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template.
     * @experimental
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns the CfnHook object from the 'Hooks' section of the included CloudFormation template with the given logical ID.
     *
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section.
     * @experimental
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * (experimental) Returns a loaded NestedStack with name logicalId.
     *
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @experimental
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * (experimental) Includes a template for a child stack inside of this parent template.
     *
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template.
     * @param nestedStackProps the properties of the included child Stack.
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     * @experimental
     */
    loadNestedStack(logicalId, nestedStackProps) {
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _a;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_a = cfnParser.parseValue(hookAttributes.Properties)) !== null && _a !== void 0 ? _a : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Condition', 'DependsOn', 'Description', 'Metadata', 'Properties', 'Type', 'Version',
            'CreationPolicy', 'DeletionPolicy', 'UpdatePolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _a;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_a = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _a !== void 0 ? _a : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLWluY2x1ZGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4taW5jbHVkZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBc0M7QUFFdEMseURBQXlEO0FBQ3pELG1FQUFtRTtBQUNuRSx1Q0FBdUM7QUFFdkMsZ0hBQWdIO0FBQ2hILDJCQUEyQjtBQUMzQix3Q0FBMkQ7Ozs7Ozs7OztBQTRFM0QsTUFBYSxVQUFXLFNBQVEsSUFBSSxDQUFDLFVBQVU7Ozs7SUFrQjdDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7O1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFsQkYsZUFBVSxHQUFtRCxFQUFFLENBQUM7UUFFaEUsY0FBUyxHQUE4QyxFQUFFLENBQUM7UUFDMUQsZUFBVSxHQUErQyxFQUFFLENBQUM7UUFHNUQsYUFBUSxHQUErQyxFQUFFLENBQUM7UUFDMUQsVUFBSyxHQUF5QyxFQUFFLENBQUM7UUFFakQsVUFBSyxHQUF5QyxFQUFFLENBQUM7UUFFakQsWUFBTyxHQUE0QyxFQUFFLENBQUM7UUFDdEQsaUJBQVksR0FBaUQsRUFBRSxDQUFDO1FBUS9FLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQztRQUVsRCxxQ0FBcUM7UUFDckMsSUFBSSxDQUFDLFFBQVEsR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUV4RCxJQUFJLENBQUMsa0JBQWtCLFNBQUcsS0FBSyxDQUFDLGtCQUFrQixtQ0FBSSxJQUFJLENBQUM7UUFFM0QscUVBQXFFO1FBQ3JFLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRTtZQUM3RCxJQUFJLENBQUMsQ0FBQyxTQUFTLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFO2dCQUNwRCxNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixTQUFTLGlDQUFpQyxDQUFDLENBQUM7YUFDM0Y7U0FDRjtRQUVELDJCQUEyQjtRQUMzQixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksZ0JBQWEsQ0FBQyxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDMUQsS0FBSyxNQUFNLFdBQVcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ25FLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDakM7UUFFRCw2QkFBNkI7UUFDN0IsS0FBSyxNQUFNLFNBQVMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ25FLElBQUksQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDakM7UUFFRCw2QkFBNkI7UUFDN0IsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLGdCQUFhLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQzlELEtBQUssTUFBTSxhQUFhLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUN2RSxJQUFJLENBQUMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDMUM7UUFFRCx3QkFBd0I7UUFDeEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLGdCQUFhLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3BELEtBQUssTUFBTSxRQUFRLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUMsRUFBRTtZQUM3RCxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzNCO1FBRUQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxFQUFFLENBQUM7UUFDMUQsOENBQThDO1FBQzlDLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUNsRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDckM7UUFDRCxzREFBc0Q7UUFDdEQsS0FBSyxNQUFNLGFBQWEsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUNyRSxJQUFJLENBQUMsQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFO2dCQUN0QyxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxhQUFhLGlDQUFpQyxDQUFDLENBQUM7YUFDbEc7U0FDRjtRQUVELHdCQUF3QjtRQUN4QixJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksZ0JBQWEsQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDcEQsS0FBSyxNQUFNLFFBQVEsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQzdELElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDM0I7UUFFRCxNQUFNLFdBQVcsR0FBRyxJQUFJLGdCQUFhLENBQUMsSUFBSSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3ZELEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUMsRUFBRTtZQUNoRSxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRSxXQUFXLENBQUMsQ0FBQztTQUMzQztJQUNILENBQUM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7SUFrQk0sV0FBVyxDQUFDLFNBQWlCO1FBQ2xDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsNkJBQTZCLFNBQVMsaUNBQWlDLENBQUMsQ0FBQztTQUMxRjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7Ozs7Ozs7Ozs7O0lBWU0sWUFBWSxDQUFDLGFBQXFCO1FBQ3ZDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsd0JBQXdCLGFBQWEsaUNBQWlDLENBQUMsQ0FBQztTQUN6RjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7Ozs7Ozs7Ozs7O0lBWU0sWUFBWSxDQUFDLGFBQXFCO1FBQ3ZDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsd0JBQXdCLGFBQWEsaUNBQWlDLENBQUMsQ0FBQztTQUN6RjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7Ozs7Ozs7Ozs7O0lBV00sVUFBVSxDQUFDLFdBQW1CO1FBQ25DLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsc0JBQXNCLFdBQVcsaUNBQWlDLENBQUMsQ0FBQztTQUNyRjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7Ozs7Ozs7Ozs7O0lBWU0sU0FBUyxDQUFDLFNBQWlCO1FBQ2hDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDcEMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLFNBQVMsaUNBQWlDLENBQUMsQ0FBQztTQUN4RjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7Ozs7Ozs7Ozs7O0lBWU0sT0FBTyxDQUFDLFFBQWdCO1FBQzdCLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDakMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLFFBQVEsaUNBQWlDLENBQUMsQ0FBQztTQUMvRTtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7Ozs7Ozs7Ozs7O0lBWU0sT0FBTyxDQUFDLGFBQXFCO1FBQ2xDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLGFBQWEsaUNBQWlDLENBQUMsQ0FBQztTQUMxRjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQzs7Ozs7Ozs7Ozs7SUFVTSxjQUFjLENBQUMsU0FBaUI7UUFDckMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDakMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxFQUFFO2dCQUN2QyxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxTQUFTLGlDQUFpQyxDQUFDLENBQUM7YUFDOUY7aUJBQU0sSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLEtBQUssNEJBQTRCLEVBQUU7Z0JBQ25GLE1BQU0sSUFBSSxLQUFLLENBQUMsNkJBQTZCLFNBQVMsaUNBQWlDLENBQUMsQ0FBQzthQUMxRjtpQkFBTTtnQkFDTCxNQUFNLElBQUksS0FBSyxDQUFDLGlCQUFpQixTQUFTLDZDQUE2QztvQkFDckYsK0lBQStJLENBQUMsQ0FBQzthQUNwSjtTQUNGO1FBQ0QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7Ozs7Ozs7Ozs7Ozs7O0lBYU0sZUFBZSxDQUFDLFNBQWlCLEVBQUUsZ0JBQWlDO1FBQ3pFLElBQUksU0FBUyxJQUFJLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDbEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsU0FBUywrQ0FBK0MsQ0FBQyxDQUFDO1NBQzVGO1FBQ0QsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsU0FBUyxpQ0FBaUMsQ0FBQyxDQUFDO1NBQzlGO1FBQ0QsSUFBSSxRQUFRLFlBQVksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNyQyxnRkFBZ0Y7WUFDaEYsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDcEMseUVBQXlFO1lBQ3pFLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUNqQyxrRkFBa0Y7WUFDbEYsSUFBSSxDQUFDLHFCQUFxQixDQUFDLFNBQVMsQ0FBQyxHQUFHLGdCQUFnQixDQUFDO1lBRXpELElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUNwQyxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDckM7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLFNBQVMsaURBQWlELENBQUMsQ0FBQztTQUM5RztJQUNILENBQUM7SUFFRCxnQkFBZ0I7SUFDVCxpQkFBaUI7UUFDdEIsTUFBTSxHQUFHLEdBQStCLEVBQUUsQ0FBQztRQUUzQyxLQUFLLE1BQU0sT0FBTyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ2hELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztZQUNsQixNQUFNLE1BQU0sR0FBeUI7Z0JBQ25DLFlBQVksQ0FBQyxHQUFHO29CQUNkLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDN0IsQ0FBQztnQkFDRCxhQUFhLENBQUMsV0FBbUI7O29CQUMvQixhQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLG1DQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3JFLENBQUM7Z0JBQ0QsYUFBYSxDQUFDLGFBQXFCO29CQUNqQyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7Z0JBQ3hDLENBQUM7Z0JBQ0QsV0FBVyxDQUFDLFdBQVc7b0JBQ3JCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDcEMsQ0FBQzthQUNGLENBQUM7WUFDRixNQUFNLFNBQVMsR0FBRyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUM7Z0JBQ3hDLE1BQU07Z0JBQ04sVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7YUFDckMsQ0FBQyxDQUFDO1lBRUgsUUFBUSxPQUFPLEVBQUU7Z0JBQ2YsS0FBSyxZQUFZLENBQUM7Z0JBQ2xCLEtBQUssVUFBVSxDQUFDO2dCQUNoQixLQUFLLFdBQVcsQ0FBQztnQkFDakIsS0FBSyxZQUFZLENBQUM7Z0JBQ2xCLEtBQUssT0FBTyxDQUFDO2dCQUNiLEtBQUssT0FBTyxDQUFDO2dCQUNiLEtBQUssU0FBUztvQkFDWiwrREFBK0Q7b0JBQy9ELE1BQU07Z0JBQ1I7b0JBQ0UsR0FBRyxDQUFDLE9BQU8sQ0FBQyxHQUFHLFNBQVMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2FBQy9EO1NBQ0Y7UUFFRCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFTyxhQUFhLENBQUMsV0FBbUI7UUFDdkMsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3hDLE1BQU0sRUFBRTtnQkFDTixhQUFhLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDckcsV0FBVyxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsbUVBQW1FLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3ZHLGFBQWEsS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNuRyxZQUFZLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUN0RztZQUNELFVBQVUsRUFBRSxFQUFFO1NBQ2YsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxVQUFVLEdBQUcsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsV0FBVyxFQUFFO1lBQ3RFLE9BQU8sRUFBRSxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ25FLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLEdBQUcsVUFBVSxDQUFDO1FBQ3hDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxVQUFVLEVBQUUsV0FBVyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVPLGVBQWUsQ0FBQyxTQUFpQjtRQUN2QyxJQUFJLFNBQVMsSUFBSSxJQUFJLENBQUMsbUJBQW1CLEVBQUU7WUFDekMsT0FBTztTQUNSO1FBRUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3pDLE1BQU0sRUFBRTtnQkFDTixZQUFZLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDdkcsYUFBYSxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3JHLGFBQWEsS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLGlFQUFpRSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN2RyxXQUFXLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUNwRztZQUNELFVBQVUsRUFBRSxFQUFFO1NBQ2YsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ25ELE1BQU0sWUFBWSxHQUFHLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzFELElBQUksRUFBRSxVQUFVLENBQUMsSUFBSTtZQUNyQixPQUFPLEVBQUUsVUFBVSxDQUFDLE9BQU87WUFDM0IsY0FBYyxFQUFFLFVBQVUsQ0FBQyxjQUFjO1lBQ3pDLGFBQWEsRUFBRSxVQUFVLENBQUMsYUFBYTtZQUN2QyxxQkFBcUIsRUFBRSxVQUFVLENBQUMscUJBQXFCO1lBQ3ZELFdBQVcsRUFBRSxVQUFVLENBQUMsV0FBVztZQUNuQyxTQUFTLEVBQUUsVUFBVSxDQUFDLFNBQVM7WUFDL0IsUUFBUSxFQUFFLFVBQVUsQ0FBQyxRQUFRO1lBQzdCLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUztZQUMvQixRQUFRLEVBQUUsVUFBVSxDQUFDLFFBQVE7WUFDN0IsTUFBTSxFQUFFLFVBQVUsQ0FBQyxNQUFNO1NBQzFCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxZQUFZLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDeEQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsR0FBRyxZQUFZLENBQUM7SUFDNUMsQ0FBQztJQUVPLFVBQVUsQ0FBQyxRQUFnQjtRQUNqQyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3hDLE1BQU0sRUFBRTtnQkFDTixhQUFhLENBQUMsU0FBaUI7b0JBQzdCLDZDQUE2QztvQkFDN0MsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUNwQyxDQUFDO2dCQUNELFlBQVksS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNsRyxhQUFhLENBQUMsYUFBcUI7b0JBQ2pDLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQztnQkFDeEMsQ0FBQztnQkFDRCxXQUFXLENBQUMsV0FBbUI7b0JBQzdCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDcEMsQ0FBQzthQUNGO1lBQ0QsVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7WUFDcEMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLO1NBQzNDLENBQUMsQ0FBQztRQUNILE1BQU0sY0FBYyxHQUFHLFNBQVMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztRQUMzRSxNQUFNLElBQUksR0FBRyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxRQUFRLEVBQUU7WUFDdkQsYUFBYSxFQUFFLGNBQWMsQ0FBQyxhQUFhO1lBQzNDLFVBQVUsRUFBRSxjQUFjLENBQUMsVUFBVTtTQUN0QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxHQUFHLElBQUksQ0FBQztRQUM1QixJQUFJLENBQUMseUJBQXlCLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFTyxVQUFVLENBQUMsUUFBZ0I7O1FBQ2pDLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixNQUFNLFNBQVMsR0FBRyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUM7WUFDeEMsTUFBTSxFQUFFO2dCQUNOLFlBQVksQ0FBQyxHQUFHO29CQUNkLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDN0IsQ0FBQztnQkFDRCxhQUFhLENBQUMsV0FBbUI7O29CQUMvQixhQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLG1DQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3JFLENBQUM7Z0JBQ0QsYUFBYSxDQUFDLGFBQXFCO29CQUNqQyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7Z0JBQ3hDLENBQUM7Z0JBQ0QsV0FBVyxDQUFDLFdBQVc7b0JBQ3JCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDcEMsQ0FBQzthQUNGO1lBQ0QsVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7U0FDckMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFckQsSUFBSSxJQUFrQixDQUFDO1FBQ3ZCLFFBQVEsY0FBYyxDQUFDLElBQUksRUFBRTtZQUMzQixLQUFLLDRCQUE0QjtnQkFDL0IsSUFBSSxHQUFJLElBQUksQ0FBQywwQkFBa0MsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLFFBQVEsRUFBRSxjQUFjLEVBQUU7b0JBQzdHLE1BQU0sRUFBRSxTQUFTO2lCQUNsQixDQUFDLENBQUM7Z0JBQ0gsTUFBTTtZQUNSLE9BQU8sQ0FBQyxDQUFDO2dCQUNQLE1BQU0sY0FBYyxTQUFHLFNBQVMsQ0FBQyxVQUFVLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxtQ0FBSSxFQUFFLENBQUM7Z0JBQzdFLElBQUksR0FBRyxJQUFJLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxRQUFRLEVBQUU7b0JBQ2pELElBQUksRUFBRSxjQUFjLENBQUMsSUFBSTtvQkFDekIsVUFBVSxFQUFFLGNBQWM7aUJBQzNCLENBQUMsQ0FBQzthQUNKO1NBQ0Y7UUFDRCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxHQUFHLElBQUksQ0FBQztRQUM1QixJQUFJLENBQUMseUJBQXlCLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFTyxZQUFZLENBQUMsU0FBaUIsRUFBRSxLQUFnQjtRQUN0RCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUM7WUFDL0MsTUFBTSxFQUFFO2dCQUNOLFlBQVksQ0FBQyxHQUFHO29CQUNkLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDN0IsQ0FBQztnQkFDRCxhQUFhLENBQUMsV0FBbUI7O29CQUMvQixhQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLG1DQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3JFLENBQUM7Z0JBQ0QsYUFBYSxDQUFDLGFBQXFCO29CQUNqQyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7Z0JBQ3hDLENBQUM7Z0JBQ0QsV0FBVyxDQUFDLFdBQVc7b0JBQ3JCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDcEMsQ0FBQzthQUNGO1lBQ0QsVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7U0FDckMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sU0FBUyxHQUFHLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ3JELEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxLQUFLO1lBQzdCLFdBQVcsRUFBRSxnQkFBZ0IsQ0FBQyxXQUFXO1lBQ3pDLFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDOUUsU0FBUyxFQUFFLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUU7b0JBQy9CLE9BQU8sU0FBUyxDQUFDO2lCQUNsQjtxQkFBTSxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLEVBQUU7b0JBQ3RELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztpQkFDdEQ7Z0JBRUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsU0FBUyxvQ0FBb0M7b0JBQ2hGLElBQUksZ0JBQWdCLENBQUMsU0FBUyx3Q0FBd0MsQ0FBQyxDQUFDO1lBQzVFLENBQUMsQ0FBQyxFQUFFO1NBQ0wsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHlCQUF5QixDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNyRCxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsQ0FBQztJQUN0QyxDQUFDO0lBRU8sb0JBQW9CLENBQUMsYUFBcUI7UUFDaEQsSUFBSSxhQUFhLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNwQyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDdkM7UUFFRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3hDLE1BQU0sRUFBRTtnQkFDTixZQUFZLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDM0YsYUFBYSxDQUFDLFdBQW1CO29CQUMvQixnRUFBZ0U7b0JBQ2hFLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDdEMsQ0FBQztnQkFDRCxhQUFhLENBQUMsS0FBYTtvQkFDekIsT0FBTyxLQUFLLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUM7d0JBQzlDLENBQUMsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsS0FBSyxDQUFDO3dCQUNsQyxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUNoQixDQUFDO2dCQUNELFdBQVcsQ0FBQyxXQUFtQjtvQkFDN0IsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUNwQyxDQUFDO2FBQ0Y7WUFDRCxPQUFPLEVBQUUsU0FBUyxDQUFDLGlCQUFpQixDQUFDLFVBQVU7WUFDL0MsVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7U0FDckMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsYUFBYSxFQUFFO1lBQzlFLFVBQVUsRUFBRSxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzFFLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxZQUFZLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDNUQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsR0FBRyxZQUFZLENBQUM7UUFDOUMsT0FBTyxZQUFZLENBQUM7SUFDdEIsQ0FBQztJQUVPLG1CQUFtQixDQUFDLFNBQWlCO1FBQzNDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdEMsSUFBSSxHQUFHLEVBQUU7WUFDUCxPQUFPLEdBQUcsQ0FBQztTQUNaO1FBRUQsTUFBTSxrQkFBa0IsR0FBUSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUVuRSwwREFBMEQ7UUFDMUQsTUFBTSxlQUFlLEdBQUc7WUFDdEIsV0FBVyxFQUFFLFdBQVcsRUFBRSxhQUFhLEVBQUUsVUFBVSxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUUsU0FBUztZQUNwRixnQkFBZ0IsRUFBRSxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUscUJBQXFCO1NBQzFFLENBQUM7UUFDRixLQUFLLE1BQU0sU0FBUyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsRUFBRTtZQUN2RCxJQUFJLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsRUFBRTtnQkFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxRQUFRLFNBQVMsdUVBQXVFO29CQUN0RyxnR0FBZ0csQ0FBQyxDQUFDO2FBQ3JHO1NBQ0Y7UUFFRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxNQUFNLEdBQXlCO1lBQ25DLGFBQWEsQ0FBQyxhQUFxQjtnQkFDakMsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBQ3hDLENBQUM7WUFFRCxXQUFXLENBQUMsV0FBVztnQkFDckIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQ3BDLENBQUM7WUFFRCxZQUFZLENBQUMsR0FBVztnQkFDdEIsSUFBSSxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLENBQUMsRUFBRTtvQkFDN0MsT0FBTyxTQUFTLENBQUM7aUJBQ2xCO2dCQUNELE9BQU8sSUFBSSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ3ZDLENBQUM7WUFFRCxhQUFhLENBQUMsV0FBbUI7Z0JBQy9CLElBQUksV0FBVyxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7b0JBQ2xDLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztpQkFDckM7Z0JBRUQsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQ3hDLENBQUM7U0FDRixDQUFDO1FBQ0YsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3hDLE1BQU07WUFDTixVQUFVLEVBQUUsSUFBSSxDQUFDLG1CQUFtQjtTQUNyQyxDQUFDLENBQUM7UUFFSCxJQUFJLFVBQTRCLENBQUM7UUFDakMsSUFBSSxJQUFJLENBQUMscUJBQXFCLENBQUMsU0FBUyxDQUFDLEVBQUU7WUFDekMsVUFBVSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDLENBQUM7U0FDM0Q7YUFBTTtZQUNMLE1BQU0sVUFBVSxHQUFHLHNCQUFzQixDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUMxRSwyRkFBMkY7WUFDM0Ysa0ZBQWtGO1lBQ2xGLDRFQUE0RTtZQUM1RSwwREFBMEQ7WUFDMUQsSUFBSSxVQUFVLElBQUksa0JBQWtCLENBQUMsSUFBSSxLQUFLLHFDQUFxQyxFQUFFO2dCQUNuRixNQUFNLE9BQU8sR0FBd0M7b0JBQ25ELE1BQU0sRUFBRSxTQUFTO2lCQUNsQixDQUFDO2dCQUNGLE1BQU0sQ0FBQyxVQUFVLEVBQUUsR0FBRyxTQUFTLENBQUMsR0FBRyxVQUFVLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO2dCQUN6RCxNQUFNLE1BQU0sR0FBRyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyw0REFBNEQ7Z0JBQ2hHLE1BQU0saUJBQWlCLEdBQUcsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDdEQsVUFBVSxHQUFHLGlCQUFpQixDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUUsa0JBQWtCLEVBQUUsT0FBTyxDQUFDLENBQUM7YUFDbEc7aUJBQU07Z0JBQ0wsVUFBVSxHQUFHLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO29CQUNqRCxJQUFJLEVBQUUsa0JBQWtCLENBQUMsSUFBSTtvQkFDN0IsVUFBVSxFQUFFLFNBQVMsQ0FBQyxVQUFVLENBQUMsa0JBQWtCLENBQUMsVUFBVSxDQUFDO2lCQUNoRSxDQUFDLENBQUM7Z0JBQ0gsU0FBUyxDQUFDLGdCQUFnQixDQUFDLFVBQVUsRUFBRSxrQkFBa0IsRUFBRSxTQUFTLENBQUMsQ0FBQzthQUN2RTtTQUNGO1FBRUQsSUFBSSxDQUFDLHlCQUF5QixDQUFDLFVBQVUsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxHQUFHLFVBQVUsQ0FBQztRQUN2QyxPQUFPLFVBQVUsQ0FBQztJQUNwQixDQUFDO0lBRU8saUJBQWlCLENBQUMsYUFBcUIsRUFBRSxTQUE4QjtRQUM3RSxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxJQUFJLEVBQUUsQ0FBQztRQUN4RCxNQUFNLHFCQUFxQixHQUFHLGlCQUFpQixDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUVyRSxJQUFJLHFCQUFxQixDQUFDLElBQUksS0FBSyw0QkFBNEIsRUFBRTtZQUMvRCxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxhQUFhLGlEQUFpRCxDQUFDLENBQUM7U0FDbEg7UUFDRCxJQUFJLHFCQUFxQixDQUFDLGNBQWMsRUFBRTtZQUN4QyxNQUFNLElBQUksS0FBSyxDQUFDLDRFQUE0RSxDQUFDLENBQUM7U0FDL0Y7UUFDRCxJQUFJLHFCQUFxQixDQUFDLFlBQVksRUFBRTtZQUN0QyxNQUFNLElBQUksS0FBSyxDQUFDLDBFQUEwRSxDQUFDLENBQUM7U0FDN0Y7UUFFRCxNQUFNLGdCQUFnQixHQUFHLFNBQVMsQ0FBQyxVQUFVLENBQUMscUJBQXFCLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDaEYsTUFBTSxXQUFXLEdBQUcsSUFBSSxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDNUQsVUFBVSxFQUFFLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLEVBQUUsYUFBYSxDQUFDO1lBQ3JGLGdCQUFnQixFQUFFLGdCQUFnQixDQUFDLGdCQUFnQjtZQUNuRCxPQUFPLEVBQUUsZ0JBQWdCLENBQUMsT0FBTztTQUNsQyxDQUFDLENBQUM7UUFDSCxNQUFNLFFBQVEsR0FBRyxJQUFJLFVBQVUsQ0FBQyxXQUFXLEVBQUUsYUFBYSxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDO1FBQ3ZHLElBQUksQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsV0FBVyxFQUFFLGdCQUFnQixFQUFFLFFBQVEsRUFBRSxDQUFDO1FBRXRGLG9EQUFvRDtRQUNwRCxNQUFNLG1CQUFtQixHQUFxQixXQUFXLENBQUMsbUJBQW9CLENBQUM7UUFDL0UsU0FBUyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixFQUFFLHFCQUFxQixFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQ3RGLE9BQU8sbUJBQW1CLENBQUM7SUFDN0IsQ0FBQztJQUVPLHdCQUF3QixDQUFDLFVBQWUsRUFBRSxhQUFxQjs7UUFDckUsSUFBSSxVQUFVLElBQUksSUFBSSxFQUFFO1lBQ3RCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsTUFBTSxtQkFBbUIsU0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUMsYUFBYSxDQUFDLENBQUMsVUFBVSxtQ0FBSSxFQUFFLENBQUM7UUFDdkYsTUFBTSxHQUFHLEdBQThCLEVBQUUsQ0FBQztRQUMxQyxLQUFLLE1BQU0sU0FBUyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDL0MsSUFBSSxDQUFDLENBQUMsU0FBUyxJQUFJLG1CQUFtQixDQUFDLEVBQUU7Z0JBQ3ZDLEdBQUcsQ0FBQyxTQUFTLENBQUMsR0FBRyxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUM7YUFDeEM7U0FDRjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztJQUVPLHlCQUF5QixDQUFDLE9BQXdCLEVBQUUsRUFBVTtRQUNwRSxJQUFJLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtZQUMzQixPQUFPLENBQUMsaUJBQWlCLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDL0I7SUFDSCxDQUFDO0NBQ0Y7QUF4bkJELGdDQXduQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjb3JlIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBjZm5fcGFyc2UgZnJvbSAnQGF3cy1jZGsvY29yZS9saWIvY2ZuLXBhcnNlJztcbmltcG9ydCAqIGFzIGNmbl90eXBlX3RvX2wxX21hcHBpbmcgZnJvbSAnLi9jZm4tdHlwZS10by1sMS1tYXBwaW5nJztcbmltcG9ydCAqIGFzIGZ1dGlscyBmcm9tICcuL2ZpbGUtdXRpbHMnO1xuXG4vLyB2MiAtIGtlZXAgdGhpcyBpbXBvcnQgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHRvIHJlZHVjZSBtZXJnZSBjb25mbGljdCB3aGVuIGZvcndhcmQgbWVyZ2luZyB3aXRoIHRoZSB2MiBicmFuY2guXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBDZm5JbmNsdWRlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdGVtcGxhdGVGaWxlOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByZXNlcnZlTG9naWNhbElkcz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbG9hZE5lc3RlZFN0YWNrcz86IHsgW3N0YWNrTmFtZTogc3RyaW5nXTogQ2ZuSW5jbHVkZVByb3BzIH07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHsgW3BhcmFtZXRlck5hbWU6IHN0cmluZ106IGFueSB9O1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSW5jbHVkZWROZXN0ZWRTdGFjayB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdGFjazogY29yZS5OZXN0ZWRTdGFjaztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbmNsdWRlZFRlbXBsYXRlOiBDZm5JbmNsdWRlO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBDZm5JbmNsdWRlIGV4dGVuZHMgY29yZS5DZm5FbGVtZW50IHtcbiAgcHJpdmF0ZSByZWFkb25seSBjb25kaXRpb25zOiB7IFtjb25kaXRpb25OYW1lOiBzdHJpbmddOiBjb3JlLkNmbkNvbmRpdGlvbiB9ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgY29uZGl0aW9uc1Njb3BlOiBDb25zdHJ1Y3Q7XG4gIHByaXZhdGUgcmVhZG9ubHkgcmVzb3VyY2VzOiB7IFtsb2dpY2FsSWQ6IHN0cmluZ106IGNvcmUuQ2ZuUmVzb3VyY2UgfSA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IHBhcmFtZXRlcnM6IHsgW2xvZ2ljYWxJZDogc3RyaW5nXTogY29yZS5DZm5QYXJhbWV0ZXIgfSA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IHBhcmFtZXRlcnNUb1JlcGxhY2U6IHsgW3BhcmFtZXRlck5hbWU6IHN0cmluZ106IGFueSB9O1xuICBwcml2YXRlIHJlYWRvbmx5IG1hcHBpbmdzU2NvcGU6IENvbnN0cnVjdDtcbiAgcHJpdmF0ZSByZWFkb25seSBtYXBwaW5nczogeyBbbWFwcGluZ05hbWU6IHN0cmluZ106IGNvcmUuQ2ZuTWFwcGluZyB9ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgcnVsZXM6IHsgW3J1bGVOYW1lOiBzdHJpbmddOiBjb3JlLkNmblJ1bGUgfSA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IHJ1bGVzU2NvcGU6IENvbnN0cnVjdDtcbiAgcHJpdmF0ZSByZWFkb25seSBob29rczogeyBbaG9va05hbWU6IHN0cmluZ106IGNvcmUuQ2ZuSG9vayB9ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgaG9va3NTY29wZTogQ29uc3RydWN0O1xuICBwcml2YXRlIHJlYWRvbmx5IG91dHB1dHM6IHsgW2xvZ2ljYWxJZDogc3RyaW5nXTogY29yZS5DZm5PdXRwdXQgfSA9IHt9O1xuICBwcml2YXRlIHJlYWRvbmx5IG5lc3RlZFN0YWNrczogeyBbbG9naWNhbElkOiBzdHJpbmddOiBJbmNsdWRlZE5lc3RlZFN0YWNrIH0gPSB7fTtcbiAgcHJpdmF0ZSByZWFkb25seSBuZXN0ZWRTdGFja3NUb0luY2x1ZGU6IHsgW25hbWU6IHN0cmluZ106IENmbkluY2x1ZGVQcm9wcyB9O1xuICBwcml2YXRlIHJlYWRvbmx5IHRlbXBsYXRlOiBhbnk7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJlc2VydmVMb2dpY2FsSWRzOiBib29sZWFuO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZm5JbmNsdWRlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlID0gcHJvcHMucGFyYW1ldGVycyB8fCB7fTtcblxuICAgIC8vIHJlYWQgdGhlIHRlbXBsYXRlIGludG8gYSBKUyBvYmplY3RcbiAgICB0aGlzLnRlbXBsYXRlID0gZnV0aWxzLnJlYWRZYW1sU3luYyhwcm9wcy50ZW1wbGF0ZUZpbGUpO1xuXG4gICAgdGhpcy5wcmVzZXJ2ZUxvZ2ljYWxJZHMgPSBwcm9wcy5wcmVzZXJ2ZUxvZ2ljYWxJZHMgPz8gdHJ1ZTtcblxuICAgIC8vIGNoZWNrIGlmIGFsbCB1c2VyIHNwZWNpZmllZCBwYXJhbWV0ZXIgdmFsdWVzIGV4aXN0IGluIHRoZSB0ZW1wbGF0ZVxuICAgIGZvciAoY29uc3QgbG9naWNhbElkIG9mIE9iamVjdC5rZXlzKHRoaXMucGFyYW1ldGVyc1RvUmVwbGFjZSkpIHtcbiAgICAgIGlmICghKGxvZ2ljYWxJZCBpbiAodGhpcy50ZW1wbGF0ZS5QYXJhbWV0ZXJzIHx8IHt9KSkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBQYXJhbWV0ZXIgd2l0aCBsb2dpY2FsIElEICcke2xvZ2ljYWxJZH0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gaW5zdGFudGlhdGUgdGhlIE1hcHBpbmdzXG4gICAgdGhpcy5tYXBwaW5nc1Njb3BlID0gbmV3IENvcmVDb25zdHJ1Y3QodGhpcywgJyRNYXBwaW5ncycpO1xuICAgIGZvciAoY29uc3QgbWFwcGluZ05hbWUgb2YgT2JqZWN0LmtleXModGhpcy50ZW1wbGF0ZS5NYXBwaW5ncyB8fCB7fSkpIHtcbiAgICAgIHRoaXMuY3JlYXRlTWFwcGluZyhtYXBwaW5nTmFtZSk7XG4gICAgfVxuXG4gICAgLy8gaW5zdGFudGlhdGUgYWxsIHBhcmFtZXRlcnNcbiAgICBmb3IgKGNvbnN0IGxvZ2ljYWxJZCBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLlBhcmFtZXRlcnMgfHwge30pKSB7XG4gICAgICB0aGlzLmNyZWF0ZVBhcmFtZXRlcihsb2dpY2FsSWQpO1xuICAgIH1cblxuICAgIC8vIGluc3RhbnRpYXRlIHRoZSBjb25kaXRpb25zXG4gICAgdGhpcy5jb25kaXRpb25zU2NvcGUgPSBuZXcgQ29yZUNvbnN0cnVjdCh0aGlzLCAnJENvbmRpdGlvbnMnKTtcbiAgICBmb3IgKGNvbnN0IGNvbmRpdGlvbk5hbWUgb2YgT2JqZWN0LmtleXModGhpcy50ZW1wbGF0ZS5Db25kaXRpb25zIHx8IHt9KSkge1xuICAgICAgdGhpcy5nZXRPckNyZWF0ZUNvbmRpdGlvbihjb25kaXRpb25OYW1lKTtcbiAgICB9XG5cbiAgICAvLyBpbnN0YW50aWF0ZSB0aGUgcnVsZXNcbiAgICB0aGlzLnJ1bGVzU2NvcGUgPSBuZXcgQ29yZUNvbnN0cnVjdCh0aGlzLCAnJFJ1bGVzJyk7XG4gICAgZm9yIChjb25zdCBydWxlTmFtZSBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLlJ1bGVzIHx8IHt9KSkge1xuICAgICAgdGhpcy5jcmVhdGVSdWxlKHJ1bGVOYW1lKTtcbiAgICB9XG5cbiAgICB0aGlzLm5lc3RlZFN0YWNrc1RvSW5jbHVkZSA9IHByb3BzLmxvYWROZXN0ZWRTdGFja3MgfHwge307XG4gICAgLy8gaW5zdGFudGlhdGUgYWxsIHJlc291cmNlcyBhcyBDREsgTDEgb2JqZWN0c1xuICAgIGZvciAoY29uc3QgbG9naWNhbElkIG9mIE9iamVjdC5rZXlzKHRoaXMudGVtcGxhdGUuUmVzb3VyY2VzIHx8IHt9KSkge1xuICAgICAgdGhpcy5nZXRPckNyZWF0ZVJlc291cmNlKGxvZ2ljYWxJZCk7XG4gICAgfVxuICAgIC8vIHZlcmlmeSB0aGF0IGFsbCBuZXN0ZWRTdGFja3MgaGF2ZSBiZWVuIGluc3RhbnRpYXRlZFxuICAgIGZvciAoY29uc3QgbmVzdGVkU3RhY2tJZCBvZiBPYmplY3Qua2V5cyhwcm9wcy5sb2FkTmVzdGVkU3RhY2tzIHx8IHt9KSkge1xuICAgICAgaWYgKCEobmVzdGVkU3RhY2tJZCBpbiB0aGlzLnJlc291cmNlcykpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBOZXN0ZWQgU3RhY2sgd2l0aCBsb2dpY2FsIElEICcke25lc3RlZFN0YWNrSWR9JyB3YXMgbm90IGZvdW5kIGluIHRoZSB0ZW1wbGF0ZWApO1xuICAgICAgfVxuICAgIH1cblxuICAgIC8vIGluc3RhbnRpYXRlIHRoZSBIb29rc1xuICAgIHRoaXMuaG9va3NTY29wZSA9IG5ldyBDb3JlQ29uc3RydWN0KHRoaXMsICckSG9va3MnKTtcbiAgICBmb3IgKGNvbnN0IGhvb2tOYW1lIG9mIE9iamVjdC5rZXlzKHRoaXMudGVtcGxhdGUuSG9va3MgfHwge30pKSB7XG4gICAgICB0aGlzLmNyZWF0ZUhvb2soaG9va05hbWUpO1xuICAgIH1cblxuICAgIGNvbnN0IG91dHB1dFNjb3BlID0gbmV3IENvcmVDb25zdHJ1Y3QodGhpcywgJyRPdXB1dHMnKTtcbiAgICBmb3IgKGNvbnN0IGxvZ2ljYWxJZCBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLk91dHB1dHMgfHwge30pKSB7XG4gICAgICB0aGlzLmNyZWF0ZU91dHB1dChsb2dpY2FsSWQsIG91dHB1dFNjb3BlKTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldFJlc291cmNlKGxvZ2ljYWxJZDogc3RyaW5nKTogY29yZS5DZm5SZXNvdXJjZSB7XG4gICAgY29uc3QgcmV0ID0gdGhpcy5yZXNvdXJjZXNbbG9naWNhbElkXTtcbiAgICBpZiAoIXJldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBSZXNvdXJjZSB3aXRoIGxvZ2ljYWwgSUQgJyR7bG9naWNhbElkfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ2V0Q29uZGl0aW9uKGNvbmRpdGlvbk5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuQ29uZGl0aW9uIHtcbiAgICBjb25zdCByZXQgPSB0aGlzLmNvbmRpdGlvbnNbY29uZGl0aW9uTmFtZV07XG4gICAgaWYgKCFyZXQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ29uZGl0aW9uIHdpdGggbmFtZSAnJHtjb25kaXRpb25OYW1lfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldFBhcmFtZXRlcihwYXJhbWV0ZXJOYW1lOiBzdHJpbmcpOiBjb3JlLkNmblBhcmFtZXRlciB7XG4gICAgY29uc3QgcmV0ID0gdGhpcy5wYXJhbWV0ZXJzW3BhcmFtZXRlck5hbWVdO1xuICAgIGlmICghcmV0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFBhcmFtZXRlciB3aXRoIG5hbWUgJyR7cGFyYW1ldGVyTmFtZX0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBnZXRNYXBwaW5nKG1hcHBpbmdOYW1lOiBzdHJpbmcpOiBjb3JlLkNmbk1hcHBpbmcge1xuICAgIGNvbnN0IHJldCA9IHRoaXMubWFwcGluZ3NbbWFwcGluZ05hbWVdO1xuICAgIGlmICghcmV0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYE1hcHBpbmcgd2l0aCBuYW1lICcke21hcHBpbmdOYW1lfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldE91dHB1dChsb2dpY2FsSWQ6IHN0cmluZyk6IGNvcmUuQ2ZuT3V0cHV0IHtcbiAgICBjb25zdCByZXQgPSB0aGlzLm91dHB1dHNbbG9naWNhbElkXTtcbiAgICBpZiAoIXJldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBPdXRwdXQgd2l0aCBsb2dpY2FsIElEICcke2xvZ2ljYWxJZH0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ2V0UnVsZShydWxlTmFtZTogc3RyaW5nKTogY29yZS5DZm5SdWxlIHtcbiAgICBjb25zdCByZXQgPSB0aGlzLnJ1bGVzW3J1bGVOYW1lXTtcbiAgICBpZiAoIXJldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBSdWxlIHdpdGggbmFtZSAnJHtydWxlTmFtZX0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGdldEhvb2soaG9va0xvZ2ljYWxJZDogc3RyaW5nKTogY29yZS5DZm5Ib29rIHtcbiAgICBjb25zdCByZXQgPSB0aGlzLmhvb2tzW2hvb2tMb2dpY2FsSWRdO1xuICAgIGlmICghcmV0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEhvb2sgd2l0aCBsb2dpY2FsIElEICcke2hvb2tMb2dpY2FsSWR9JyB3YXMgbm90IGZvdW5kIGluIHRoZSB0ZW1wbGF0ZWApO1xuICAgIH1cbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ2V0TmVzdGVkU3RhY2sobG9naWNhbElkOiBzdHJpbmcpOiBJbmNsdWRlZE5lc3RlZFN0YWNrIHtcbiAgICBpZiAoIXRoaXMubmVzdGVkU3RhY2tzW2xvZ2ljYWxJZF0pIHtcbiAgICAgIGlmICghdGhpcy50ZW1wbGF0ZS5SZXNvdXJjZXNbbG9naWNhbElkXSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYE5lc3RlZCBTdGFjayB3aXRoIGxvZ2ljYWwgSUQgJyR7bG9naWNhbElkfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICAgIH0gZWxzZSBpZiAodGhpcy50ZW1wbGF0ZS5SZXNvdXJjZXNbbG9naWNhbElkXS5UeXBlICE9PSAnQVdTOjpDbG91ZEZvcm1hdGlvbjo6U3RhY2snKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgUmVzb3VyY2Ugd2l0aCBsb2dpY2FsIElEICcke2xvZ2ljYWxJZH0nIGlzIG5vdCBhIENsb3VkRm9ybWF0aW9uIFN0YWNrYCk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYE5lc3RlZCBTdGFjayAnJHtsb2dpY2FsSWR9JyB3YXMgbm90IGluY2x1ZGVkIGluIHRoZSBwYXJlbnQgdGVtcGxhdGUuIGAgK1xuICAgICAgICAgICdUbyByZXRyaWV2ZSBhbiBpbmNsdWRlZCBuZXN0ZWQgc3RhY2ssIGl0IG11c3QgYmUgc3BlY2lmaWVkIGVpdGhlciBpbiB0aGUgYGxvYWROZXN0ZWRTdGFja3NgIHByb3BlcnR5LCBvciB0aHJvdWdoIHRoZSBgbG9hZE5lc3RlZFN0YWNrYCBtZXRob2QnKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHRoaXMubmVzdGVkU3RhY2tzW2xvZ2ljYWxJZF07XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBsb2FkTmVzdGVkU3RhY2sobG9naWNhbElkOiBzdHJpbmcsIG5lc3RlZFN0YWNrUHJvcHM6IENmbkluY2x1ZGVQcm9wcyk6IEluY2x1ZGVkTmVzdGVkU3RhY2sge1xuICAgIGlmIChsb2dpY2FsSWQgaW4gdGhpcy5uZXN0ZWRTdGFja3MpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTmVzdGVkIFN0YWNrICcke2xvZ2ljYWxJZH0nIHdhcyBhbHJlYWR5IGluY2x1ZGVkIGluIGl0cyBwYXJlbnQgdGVtcGxhdGVgKTtcbiAgICB9XG4gICAgY29uc3QgY2ZuU3RhY2sgPSB0aGlzLnJlc291cmNlc1tsb2dpY2FsSWRdO1xuICAgIGlmICghY2ZuU3RhY2spIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTmVzdGVkIFN0YWNrIHdpdGggbG9naWNhbCBJRCAnJHtsb2dpY2FsSWR9JyB3YXMgbm90IGZvdW5kIGluIHRoZSB0ZW1wbGF0ZWApO1xuICAgIH1cbiAgICBpZiAoY2ZuU3RhY2sgaW5zdGFuY2VvZiBjb3JlLkNmblN0YWNrKSB7XG4gICAgICAvLyBkZWxldGUgdGhlIG9sZCBDZm5TdGFjayBjaGlsZCAtIG9uZSB3aWxsIGJlIGNyZWF0ZWQgYnkgdGhlIE5lc3RlZFN0YWNrIG9iamVjdFxuICAgICAgdGhpcy5ub2RlLnRyeVJlbW92ZUNoaWxkKGxvZ2ljYWxJZCk7XG4gICAgICAvLyByZW1vdmUgdGhlIHByZXZpb3VzbHkgY3JlYXRlZCBDZm5TdGFjayByZXNvdXJjZSBmcm9tIHRoZSByZXNvdXJjZXMgbWFwXG4gICAgICBkZWxldGUgdGhpcy5yZXNvdXJjZXNbbG9naWNhbElkXTtcbiAgICAgIC8vIGNyZWF0ZU5lc3RlZFN0YWNrKCkgKGNhbGxlZCBieSBnZXRPckNyZWF0ZVJlc291cmNlKCkpIGV4cGVjdHMgdGhpcyB0byBiZSBmaWxsZWRcbiAgICAgIHRoaXMubmVzdGVkU3RhY2tzVG9JbmNsdWRlW2xvZ2ljYWxJZF0gPSBuZXN0ZWRTdGFja1Byb3BzO1xuXG4gICAgICB0aGlzLmdldE9yQ3JlYXRlUmVzb3VyY2UobG9naWNhbElkKTtcbiAgICAgIHJldHVybiB0aGlzLm5lc3RlZFN0YWNrc1tsb2dpY2FsSWRdO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYE5lc3RlZCBTdGFjayB3aXRoIGxvZ2ljYWwgSUQgJyR7bG9naWNhbElkfScgaXMgbm90IGFuIEFXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrIHJlc291cmNlYCk7XG4gICAgfVxuICB9XG5cbiAgLyoqIEBpbnRlcm5hbCAqL1xuICBwdWJsaWMgX3RvQ2xvdWRGb3JtYXRpb24oKTogb2JqZWN0IHtcbiAgICBjb25zdCByZXQ6IHsgW3NlY3Rpb246IHN0cmluZ106IGFueSB9ID0ge307XG5cbiAgICBmb3IgKGNvbnN0IHNlY3Rpb24gb2YgT2JqZWN0LmtleXModGhpcy50ZW1wbGF0ZSkpIHtcbiAgICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuICAgICAgY29uc3QgZmluZGVyOiBjZm5fcGFyc2UuSUNmbkZpbmRlciA9IHtcbiAgICAgICAgZmluZFJlc291cmNlKGxJZCk6IGNvcmUuQ2ZuUmVzb3VyY2UgfCB1bmRlZmluZWQge1xuICAgICAgICAgIHJldHVybiBzZWxmLnJlc291cmNlc1tsSWRdO1xuICAgICAgICB9LFxuICAgICAgICBmaW5kUmVmVGFyZ2V0KGVsZW1lbnROYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkVsZW1lbnQgfCB1bmRlZmluZWQge1xuICAgICAgICAgIHJldHVybiBzZWxmLnJlc291cmNlc1tlbGVtZW50TmFtZV0gPz8gc2VsZi5wYXJhbWV0ZXJzW2VsZW1lbnROYW1lXTtcbiAgICAgICAgfSxcbiAgICAgICAgZmluZENvbmRpdGlvbihjb25kaXRpb25OYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgcmV0dXJuIHNlbGYuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXTtcbiAgICAgICAgfSxcbiAgICAgICAgZmluZE1hcHBpbmcobWFwcGluZ05hbWUpOiBjb3JlLkNmbk1hcHBpbmcgfCB1bmRlZmluZWQge1xuICAgICAgICAgIHJldHVybiBzZWxmLm1hcHBpbmdzW21hcHBpbmdOYW1lXTtcbiAgICAgICAgfSxcbiAgICAgIH07XG4gICAgICBjb25zdCBjZm5QYXJzZXIgPSBuZXcgY2ZuX3BhcnNlLkNmblBhcnNlcih7XG4gICAgICAgIGZpbmRlcixcbiAgICAgICAgcGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlLFxuICAgICAgfSk7XG5cbiAgICAgIHN3aXRjaCAoc2VjdGlvbikge1xuICAgICAgICBjYXNlICdDb25kaXRpb25zJzpcbiAgICAgICAgY2FzZSAnTWFwcGluZ3MnOlxuICAgICAgICBjYXNlICdSZXNvdXJjZXMnOlxuICAgICAgICBjYXNlICdQYXJhbWV0ZXJzJzpcbiAgICAgICAgY2FzZSAnUnVsZXMnOlxuICAgICAgICBjYXNlICdIb29rcyc6XG4gICAgICAgIGNhc2UgJ091dHB1dHMnOlxuICAgICAgICAgIC8vIHRoZXNlIGFyZSByZW5kZXJlZCBhcyBhIHNpZGUgZWZmZWN0IG9mIGluc3RhbnRpYXRpbmcgdGhlIEwxc1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgIHJldFtzZWN0aW9uXSA9IGNmblBhcnNlci5wYXJzZVZhbHVlKHRoaXMudGVtcGxhdGVbc2VjdGlvbl0pO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZU1hcHBpbmcobWFwcGluZ05hbWU6IHN0cmluZyk6IHZvaWQge1xuICAgIGNvbnN0IGNmblBhcnNlciA9IG5ldyBjZm5fcGFyc2UuQ2ZuUGFyc2VyKHtcbiAgICAgIGZpbmRlcjoge1xuICAgICAgICBmaW5kQ29uZGl0aW9uKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1JlZmVycmluZyB0byBDb25kaXRpb25zIGluIE1hcHBpbmcgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgZmluZE1hcHBpbmcoKSB7IHRocm93IG5ldyBFcnJvcignUmVmZXJyaW5nIHRvIG90aGVyIE1hcHBpbmdzIGluIE1hcHBpbmcgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgZmluZFJlZlRhcmdldCgpIHsgdGhyb3cgbmV3IEVycm9yKCdVc2luZyBSZWYgZXhwcmVzc2lvbnMgaW4gTWFwcGluZyBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgICBmaW5kUmVzb3VyY2UoKSB7IHRocm93IG5ldyBFcnJvcignVXNpbmcgR2V0QXR0IGV4cHJlc3Npb25zIGluIE1hcHBpbmcgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgIH0sXG4gICAgICBwYXJhbWV0ZXJzOiB7fSxcbiAgICB9KTtcbiAgICBjb25zdCBjZm5NYXBwaW5nID0gbmV3IGNvcmUuQ2ZuTWFwcGluZyh0aGlzLm1hcHBpbmdzU2NvcGUsIG1hcHBpbmdOYW1lLCB7XG4gICAgICBtYXBwaW5nOiBjZm5QYXJzZXIucGFyc2VWYWx1ZSh0aGlzLnRlbXBsYXRlLk1hcHBpbmdzW21hcHBpbmdOYW1lXSksXG4gICAgfSk7XG4gICAgdGhpcy5tYXBwaW5nc1ttYXBwaW5nTmFtZV0gPSBjZm5NYXBwaW5nO1xuICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChjZm5NYXBwaW5nLCBtYXBwaW5nTmFtZSk7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVBhcmFtZXRlcihsb2dpY2FsSWQ6IHN0cmluZyk6IHZvaWQge1xuICAgIGlmIChsb2dpY2FsSWQgaW4gdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgZXhwcmVzc2lvbiA9IG5ldyBjZm5fcGFyc2UuQ2ZuUGFyc2VyKHtcbiAgICAgIGZpbmRlcjoge1xuICAgICAgICBmaW5kUmVzb3VyY2UoKSB7IHRocm93IG5ldyBFcnJvcignVXNpbmcgR2V0QXR0IGV4cHJlc3Npb25zIGluIFBhcmFtZXRlciBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgICBmaW5kUmVmVGFyZ2V0KCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1VzaW5nIFJlZiBleHByZXNzaW9ucyBpbiBQYXJhbWV0ZXIgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgZmluZENvbmRpdGlvbigpIHsgdGhyb3cgbmV3IEVycm9yKCdSZWZlcnJpbmcgdG8gQ29uZGl0aW9ucyBpbiBQYXJhbWV0ZXIgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgZmluZE1hcHBpbmcoKSB7IHRocm93IG5ldyBFcnJvcignUmVmZXJyaW5nIHRvIE1hcHBpbmdzIGluIFBhcmFtZXRlciBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgfSxcbiAgICAgIHBhcmFtZXRlcnM6IHt9LFxuICAgIH0pLnBhcnNlVmFsdWUodGhpcy50ZW1wbGF0ZS5QYXJhbWV0ZXJzW2xvZ2ljYWxJZF0pO1xuICAgIGNvbnN0IGNmblBhcmFtZXRlciA9IG5ldyBjb3JlLkNmblBhcmFtZXRlcih0aGlzLCBsb2dpY2FsSWQsIHtcbiAgICAgIHR5cGU6IGV4cHJlc3Npb24uVHlwZSxcbiAgICAgIGRlZmF1bHQ6IGV4cHJlc3Npb24uRGVmYXVsdCxcbiAgICAgIGFsbG93ZWRQYXR0ZXJuOiBleHByZXNzaW9uLkFsbG93ZWRQYXR0ZXJuLFxuICAgICAgYWxsb3dlZFZhbHVlczogZXhwcmVzc2lvbi5BbGxvd2VkVmFsdWVzLFxuICAgICAgY29uc3RyYWludERlc2NyaXB0aW9uOiBleHByZXNzaW9uLkNvbnN0cmFpbnREZXNjcmlwdGlvbixcbiAgICAgIGRlc2NyaXB0aW9uOiBleHByZXNzaW9uLkRlc2NyaXB0aW9uLFxuICAgICAgbWF4TGVuZ3RoOiBleHByZXNzaW9uLk1heExlbmd0aCxcbiAgICAgIG1heFZhbHVlOiBleHByZXNzaW9uLk1heFZhbHVlLFxuICAgICAgbWluTGVuZ3RoOiBleHByZXNzaW9uLk1pbkxlbmd0aCxcbiAgICAgIG1pblZhbHVlOiBleHByZXNzaW9uLk1pblZhbHVlLFxuICAgICAgbm9FY2hvOiBleHByZXNzaW9uLk5vRWNobyxcbiAgICB9KTtcblxuICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChjZm5QYXJhbWV0ZXIsIGxvZ2ljYWxJZCk7XG4gICAgdGhpcy5wYXJhbWV0ZXJzW2xvZ2ljYWxJZF0gPSBjZm5QYXJhbWV0ZXI7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVJ1bGUocnVsZU5hbWU6IHN0cmluZyk6IHZvaWQge1xuICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuICAgIGNvbnN0IGNmblBhcnNlciA9IG5ldyBjZm5fcGFyc2UuQ2ZuUGFyc2VyKHtcbiAgICAgIGZpbmRlcjoge1xuICAgICAgICBmaW5kUmVmVGFyZ2V0KHJlZlRhcmdldDogc3RyaW5nKTogY29yZS5DZm5FbGVtZW50IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAvLyBvbmx5IHBhcmFtZXRlcnMgY2FuIGJlIHJlZmVyZW5jZWQgaW4gUnVsZXNcbiAgICAgICAgICByZXR1cm4gc2VsZi5wYXJhbWV0ZXJzW3JlZlRhcmdldF07XG4gICAgICAgIH0sXG4gICAgICAgIGZpbmRSZXNvdXJjZSgpIHsgdGhyb3cgbmV3IEVycm9yKCdVc2luZyBHZXRBdHQgZXhwcmVzc2lvbnMgaW4gUnVsZSBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgICBmaW5kQ29uZGl0aW9uKGNvbmRpdGlvbk5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuQ29uZGl0aW9uIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICByZXR1cm4gc2VsZi5jb25kaXRpb25zW2NvbmRpdGlvbk5hbWVdO1xuICAgICAgICB9LFxuICAgICAgICBmaW5kTWFwcGluZyhtYXBwaW5nTmFtZTogc3RyaW5nKTogY29yZS5DZm5NYXBwaW5nIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICByZXR1cm4gc2VsZi5tYXBwaW5nc1ttYXBwaW5nTmFtZV07XG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgcGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlLFxuICAgICAgY29udGV4dDogY2ZuX3BhcnNlLkNmblBhcnNpbmdDb250ZXh0LlJVTEVTLFxuICAgIH0pO1xuICAgIGNvbnN0IHJ1bGVQcm9wZXJ0aWVzID0gY2ZuUGFyc2VyLnBhcnNlVmFsdWUodGhpcy50ZW1wbGF0ZS5SdWxlc1tydWxlTmFtZV0pO1xuICAgIGNvbnN0IHJ1bGUgPSBuZXcgY29yZS5DZm5SdWxlKHRoaXMucnVsZXNTY29wZSwgcnVsZU5hbWUsIHtcbiAgICAgIHJ1bGVDb25kaXRpb246IHJ1bGVQcm9wZXJ0aWVzLlJ1bGVDb25kaXRpb24sXG4gICAgICBhc3NlcnRpb25zOiBydWxlUHJvcGVydGllcy5Bc3NlcnRpb25zLFxuICAgIH0pO1xuICAgIHRoaXMucnVsZXNbcnVsZU5hbWVdID0gcnVsZTtcbiAgICB0aGlzLm92ZXJyaWRlTG9naWNhbElkSWZOZWVkZWQocnVsZSwgcnVsZU5hbWUpO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVIb29rKGhvb2tOYW1lOiBzdHJpbmcpOiB2b2lkIHtcbiAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICBjb25zdCBjZm5QYXJzZXIgPSBuZXcgY2ZuX3BhcnNlLkNmblBhcnNlcih7XG4gICAgICBmaW5kZXI6IHtcbiAgICAgICAgZmluZFJlc291cmNlKGxJZCk6IGNvcmUuQ2ZuUmVzb3VyY2UgfCB1bmRlZmluZWQge1xuICAgICAgICAgIHJldHVybiBzZWxmLnJlc291cmNlc1tsSWRdO1xuICAgICAgICB9LFxuICAgICAgICBmaW5kUmVmVGFyZ2V0KGVsZW1lbnROYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkVsZW1lbnQgfCB1bmRlZmluZWQge1xuICAgICAgICAgIHJldHVybiBzZWxmLnJlc291cmNlc1tlbGVtZW50TmFtZV0gPz8gc2VsZi5wYXJhbWV0ZXJzW2VsZW1lbnROYW1lXTtcbiAgICAgICAgfSxcbiAgICAgICAgZmluZENvbmRpdGlvbihjb25kaXRpb25OYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgcmV0dXJuIHNlbGYuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXTtcbiAgICAgICAgfSxcbiAgICAgICAgZmluZE1hcHBpbmcobWFwcGluZ05hbWUpOiBjb3JlLkNmbk1hcHBpbmcgfCB1bmRlZmluZWQge1xuICAgICAgICAgIHJldHVybiBzZWxmLm1hcHBpbmdzW21hcHBpbmdOYW1lXTtcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBwYXJhbWV0ZXJzOiB0aGlzLnBhcmFtZXRlcnNUb1JlcGxhY2UsXG4gICAgfSk7XG4gICAgY29uc3QgaG9va0F0dHJpYnV0ZXMgPSB0aGlzLnRlbXBsYXRlLkhvb2tzW2hvb2tOYW1lXTtcblxuICAgIGxldCBob29rOiBjb3JlLkNmbkhvb2s7XG4gICAgc3dpdGNoIChob29rQXR0cmlidXRlcy5UeXBlKSB7XG4gICAgICBjYXNlICdBV1M6OkNvZGVEZXBsb3k6OkJsdWVHcmVlbic6XG4gICAgICAgIGhvb2sgPSAoY29yZS5DZm5Db2RlRGVwbG95Qmx1ZUdyZWVuSG9vayBhcyBhbnkpLl9mcm9tQ2xvdWRGb3JtYXRpb24odGhpcy5ob29rc1Njb3BlLCBob29rTmFtZSwgaG9va0F0dHJpYnV0ZXMsIHtcbiAgICAgICAgICBwYXJzZXI6IGNmblBhcnNlcixcbiAgICAgICAgfSk7XG4gICAgICAgIGJyZWFrO1xuICAgICAgZGVmYXVsdDoge1xuICAgICAgICBjb25zdCBob29rUHJvcGVydGllcyA9IGNmblBhcnNlci5wYXJzZVZhbHVlKGhvb2tBdHRyaWJ1dGVzLlByb3BlcnRpZXMpID8/IHt9O1xuICAgICAgICBob29rID0gbmV3IGNvcmUuQ2ZuSG9vayh0aGlzLmhvb2tzU2NvcGUsIGhvb2tOYW1lLCB7XG4gICAgICAgICAgdHlwZTogaG9va0F0dHJpYnV0ZXMuVHlwZSxcbiAgICAgICAgICBwcm9wZXJ0aWVzOiBob29rUHJvcGVydGllcyxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICAgIHRoaXMuaG9va3NbaG9va05hbWVdID0gaG9vaztcbiAgICB0aGlzLm92ZXJyaWRlTG9naWNhbElkSWZOZWVkZWQoaG9vaywgaG9va05hbWUpO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVPdXRwdXQobG9naWNhbElkOiBzdHJpbmcsIHNjb3BlOiBDb25zdHJ1Y3QpOiB2b2lkIHtcbiAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICBjb25zdCBvdXRwdXRBdHRyaWJ1dGVzID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgZmluZGVyOiB7XG4gICAgICAgIGZpbmRSZXNvdXJjZShsSWQpOiBjb3JlLkNmblJlc291cmNlIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICByZXR1cm4gc2VsZi5yZXNvdXJjZXNbbElkXTtcbiAgICAgICAgfSxcbiAgICAgICAgZmluZFJlZlRhcmdldChlbGVtZW50TmFtZTogc3RyaW5nKTogY29yZS5DZm5FbGVtZW50IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICByZXR1cm4gc2VsZi5yZXNvdXJjZXNbZWxlbWVudE5hbWVdID8/IHNlbGYucGFyYW1ldGVyc1tlbGVtZW50TmFtZV07XG4gICAgICAgIH0sXG4gICAgICAgIGZpbmRDb25kaXRpb24oY29uZGl0aW9uTmFtZTogc3RyaW5nKTogY29yZS5DZm5Db25kaXRpb24gfCB1bmRlZmluZWQge1xuICAgICAgICAgIHJldHVybiBzZWxmLmNvbmRpdGlvbnNbY29uZGl0aW9uTmFtZV07XG4gICAgICAgIH0sXG4gICAgICAgIGZpbmRNYXBwaW5nKG1hcHBpbmdOYW1lKTogY29yZS5DZm5NYXBwaW5nIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICByZXR1cm4gc2VsZi5tYXBwaW5nc1ttYXBwaW5nTmFtZV07XG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgcGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlLFxuICAgIH0pLnBhcnNlVmFsdWUodGhpcy50ZW1wbGF0ZS5PdXRwdXRzW2xvZ2ljYWxJZF0pO1xuICAgIGNvbnN0IGNmbk91dHB1dCA9IG5ldyBjb3JlLkNmbk91dHB1dChzY29wZSwgbG9naWNhbElkLCB7XG4gICAgICB2YWx1ZTogb3V0cHV0QXR0cmlidXRlcy5WYWx1ZSxcbiAgICAgIGRlc2NyaXB0aW9uOiBvdXRwdXRBdHRyaWJ1dGVzLkRlc2NyaXB0aW9uLFxuICAgICAgZXhwb3J0TmFtZTogb3V0cHV0QXR0cmlidXRlcy5FeHBvcnQgPyBvdXRwdXRBdHRyaWJ1dGVzLkV4cG9ydC5OYW1lIDogdW5kZWZpbmVkLFxuICAgICAgY29uZGl0aW9uOiAoKCkgPT4ge1xuICAgICAgICBpZiAoIW91dHB1dEF0dHJpYnV0ZXMuQ29uZGl0aW9uKSB7XG4gICAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgICAgfSBlbHNlIGlmICh0aGlzLmNvbmRpdGlvbnNbb3V0cHV0QXR0cmlidXRlcy5Db25kaXRpb25dKSB7XG4gICAgICAgICAgcmV0dXJuIHNlbGYuZ2V0Q29uZGl0aW9uKG91dHB1dEF0dHJpYnV0ZXMuQ29uZGl0aW9uKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgT3V0cHV0IHdpdGggbmFtZSAnJHtsb2dpY2FsSWR9JyByZWZlcnMgdG8gYSBDb25kaXRpb24gd2l0aCBuYW1lIGAgK1xuICAgICAgICAgIGAnJHtvdXRwdXRBdHRyaWJ1dGVzLkNvbmRpdGlvbn0nIHdoaWNoIHdhcyBub3QgZm91bmQgaW4gdGhpcyB0ZW1wbGF0ZWApO1xuICAgICAgfSkoKSxcbiAgICB9KTtcblxuICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChjZm5PdXRwdXQsIGxvZ2ljYWxJZCk7XG4gICAgdGhpcy5vdXRwdXRzW2xvZ2ljYWxJZF0gPSBjZm5PdXRwdXQ7XG4gIH1cblxuICBwcml2YXRlIGdldE9yQ3JlYXRlQ29uZGl0aW9uKGNvbmRpdGlvbk5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuQ29uZGl0aW9uIHtcbiAgICBpZiAoY29uZGl0aW9uTmFtZSBpbiB0aGlzLmNvbmRpdGlvbnMpIHtcbiAgICAgIHJldHVybiB0aGlzLmNvbmRpdGlvbnNbY29uZGl0aW9uTmFtZV07XG4gICAgfVxuXG4gICAgY29uc3Qgc2VsZiA9IHRoaXM7XG4gICAgY29uc3QgY2ZuUGFyc2VyID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgZmluZGVyOiB7XG4gICAgICAgIGZpbmRSZXNvdXJjZSgpIHsgdGhyb3cgbmV3IEVycm9yKCdVc2luZyBHZXRBdHQgaW4gQ29uZGl0aW9uIGRlZmluaXRpb25zIGlzIG5vdCBhbGxvd2VkJyk7IH0sXG4gICAgICAgIGZpbmRSZWZUYXJnZXQoZWxlbWVudE5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuRWxlbWVudCB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgLy8gb25seSBQYXJhbWV0ZXJzIGNhbiBiZSByZWZlcmVuY2VkIGluIHRoZSAnQ29uZGl0aW9ucycgc2VjdGlvblxuICAgICAgICAgIHJldHVybiBzZWxmLnBhcmFtZXRlcnNbZWxlbWVudE5hbWVdO1xuICAgICAgICB9LFxuICAgICAgICBmaW5kQ29uZGl0aW9uKGNOYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgcmV0dXJuIGNOYW1lIGluIChzZWxmLnRlbXBsYXRlLkNvbmRpdGlvbnMgfHwge30pXG4gICAgICAgICAgICA/IHNlbGYuZ2V0T3JDcmVhdGVDb25kaXRpb24oY05hbWUpXG4gICAgICAgICAgICA6IHVuZGVmaW5lZDtcbiAgICAgICAgfSxcbiAgICAgICAgZmluZE1hcHBpbmcobWFwcGluZ05hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuTWFwcGluZyB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgcmV0dXJuIHNlbGYubWFwcGluZ3NbbWFwcGluZ05hbWVdO1xuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIGNvbnRleHQ6IGNmbl9wYXJzZS5DZm5QYXJzaW5nQ29udGV4dC5DT05ESVRJT05TLFxuICAgICAgcGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlLFxuICAgIH0pO1xuICAgIGNvbnN0IGNmbkNvbmRpdGlvbiA9IG5ldyBjb3JlLkNmbkNvbmRpdGlvbih0aGlzLmNvbmRpdGlvbnNTY29wZSwgY29uZGl0aW9uTmFtZSwge1xuICAgICAgZXhwcmVzc2lvbjogY2ZuUGFyc2VyLnBhcnNlVmFsdWUodGhpcy50ZW1wbGF0ZS5Db25kaXRpb25zW2NvbmRpdGlvbk5hbWVdKSxcbiAgICB9KTtcblxuICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChjZm5Db25kaXRpb24sIGNvbmRpdGlvbk5hbWUpO1xuICAgIHRoaXMuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXSA9IGNmbkNvbmRpdGlvbjtcbiAgICByZXR1cm4gY2ZuQ29uZGl0aW9uO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRPckNyZWF0ZVJlc291cmNlKGxvZ2ljYWxJZDogc3RyaW5nKTogY29yZS5DZm5SZXNvdXJjZSB7XG4gICAgY29uc3QgcmV0ID0gdGhpcy5yZXNvdXJjZXNbbG9naWNhbElkXTtcbiAgICBpZiAocmV0KSB7XG4gICAgICByZXR1cm4gcmV0O1xuICAgIH1cblxuICAgIGNvbnN0IHJlc291cmNlQXR0cmlidXRlczogYW55ID0gdGhpcy50ZW1wbGF0ZS5SZXNvdXJjZXNbbG9naWNhbElkXTtcblxuICAgIC8vIGZhaWwgZWFybHkgZm9yIHJlc291cmNlIGF0dHJpYnV0ZXMgd2UgZG9uJ3Qgc3VwcG9ydCB5ZXRcbiAgICBjb25zdCBrbm93bkF0dHJpYnV0ZXMgPSBbXG4gICAgICAnQ29uZGl0aW9uJywgJ0RlcGVuZHNPbicsICdEZXNjcmlwdGlvbicsICdNZXRhZGF0YScsICdQcm9wZXJ0aWVzJywgJ1R5cGUnLCAnVmVyc2lvbicsXG4gICAgICAnQ3JlYXRpb25Qb2xpY3knLCAnRGVsZXRpb25Qb2xpY3knLCAnVXBkYXRlUG9saWN5JywgJ1VwZGF0ZVJlcGxhY2VQb2xpY3knLFxuICAgIF07XG4gICAgZm9yIChjb25zdCBhdHRyaWJ1dGUgb2YgT2JqZWN0LmtleXMocmVzb3VyY2VBdHRyaWJ1dGVzKSkge1xuICAgICAgaWYgKCFrbm93bkF0dHJpYnV0ZXMuaW5jbHVkZXMoYXR0cmlidXRlKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSAnJHthdHRyaWJ1dGV9JyByZXNvdXJjZSBhdHRyaWJ1dGUgaXMgbm90IHN1cHBvcnRlZCBieSBjbG91ZGZvcm1hdGlvbi1pbmNsdWRlIHlldC4gYCArXG4gICAgICAgICAgJ0VpdGhlciByZW1vdmUgaXQgZnJvbSB0aGUgdGVtcGxhdGUsIG9yIHVzZSB0aGUgQ2RrSW5jbHVkZSBjbGFzcyBmcm9tIHRoZSBjb3JlIHBhY2thZ2UgaW5zdGVhZC4nKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICBjb25zdCBmaW5kZXI6IGNmbl9wYXJzZS5JQ2ZuRmluZGVyID0ge1xuICAgICAgZmluZENvbmRpdGlvbihjb25kaXRpb25OYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgIHJldHVybiBzZWxmLmNvbmRpdGlvbnNbY29uZGl0aW9uTmFtZV07XG4gICAgICB9LFxuXG4gICAgICBmaW5kTWFwcGluZyhtYXBwaW5nTmFtZSk6IGNvcmUuQ2ZuTWFwcGluZyB8IHVuZGVmaW5lZCB7XG4gICAgICAgIHJldHVybiBzZWxmLm1hcHBpbmdzW21hcHBpbmdOYW1lXTtcbiAgICAgIH0sXG5cbiAgICAgIGZpbmRSZXNvdXJjZShsSWQ6IHN0cmluZyk6IGNvcmUuQ2ZuUmVzb3VyY2UgfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAoIShsSWQgaW4gKHNlbGYudGVtcGxhdGUuUmVzb3VyY2VzIHx8IHt9KSkpIHtcbiAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBzZWxmLmdldE9yQ3JlYXRlUmVzb3VyY2UobElkKTtcbiAgICAgIH0sXG5cbiAgICAgIGZpbmRSZWZUYXJnZXQoZWxlbWVudE5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuRWxlbWVudCB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmIChlbGVtZW50TmFtZSBpbiBzZWxmLnBhcmFtZXRlcnMpIHtcbiAgICAgICAgICByZXR1cm4gc2VsZi5wYXJhbWV0ZXJzW2VsZW1lbnROYW1lXTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiB0aGlzLmZpbmRSZXNvdXJjZShlbGVtZW50TmFtZSk7XG4gICAgICB9LFxuICAgIH07XG4gICAgY29uc3QgY2ZuUGFyc2VyID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgZmluZGVyLFxuICAgICAgcGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlLFxuICAgIH0pO1xuXG4gICAgbGV0IGwxSW5zdGFuY2U6IGNvcmUuQ2ZuUmVzb3VyY2U7XG4gICAgaWYgKHRoaXMubmVzdGVkU3RhY2tzVG9JbmNsdWRlW2xvZ2ljYWxJZF0pIHtcbiAgICAgIGwxSW5zdGFuY2UgPSB0aGlzLmNyZWF0ZU5lc3RlZFN0YWNrKGxvZ2ljYWxJZCwgY2ZuUGFyc2VyKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgbDFDbGFzc0ZxbiA9IGNmbl90eXBlX3RvX2wxX21hcHBpbmcubG9va3VwKHJlc291cmNlQXR0cmlidXRlcy5UeXBlKTtcbiAgICAgIC8vIFRoZSBBV1M6OkNsb3VkRm9ybWF0aW9uOjpDdXN0b21SZXNvdXJjZSB0eXBlIGNvcnJlc3BvbmRzIHRvIHRoZSBDZm5DdXN0b21SZXNvdXJjZSBjbGFzcy5cbiAgICAgIC8vIFVuZm9ydHVuYXRlbHksIGl0J3MgcXVpdGUgdXNlbGVzczsgaXQgb25seSBoYXMgYSBzaW5nbGUgcHJvcGVydHksIFNlcnZpY2VUb2tlbi5cbiAgICAgIC8vIEZvciB0aGF0IHJlYXNvbiwgZXZlbiB0aGUgQ3VzdG9tUmVzb3VyY2UgY2xhc3MgZnJvbSBAY29yZSBkb2Vzbid0IHVzZSBpdCFcbiAgICAgIC8vIFNvLCBzcGVjaWFsLWNhc2UgdGhlIGhhbmRsaW5nIG9mIHRoaXMgb25lIHJlc291cmNlIHR5cGVcbiAgICAgIGlmIChsMUNsYXNzRnFuICYmIHJlc291cmNlQXR0cmlidXRlcy5UeXBlICE9PSAnQVdTOjpDbG91ZEZvcm1hdGlvbjo6Q3VzdG9tUmVzb3VyY2UnKSB7XG4gICAgICAgIGNvbnN0IG9wdGlvbnM6IGNmbl9wYXJzZS5Gcm9tQ2xvdWRGb3JtYXRpb25PcHRpb25zID0ge1xuICAgICAgICAgIHBhcnNlcjogY2ZuUGFyc2VyLFxuICAgICAgICB9O1xuICAgICAgICBjb25zdCBbbW9kdWxlTmFtZSwgLi4uY2xhc3NOYW1lXSA9IGwxQ2xhc3NGcW4uc3BsaXQoJy4nKTtcbiAgICAgICAgY29uc3QgbW9kdWxlID0gcmVxdWlyZShtb2R1bGVOYW1lKTsgLy8gZXNsaW50LWRpc2FibGUtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tcmVxdWlyZS1pbXBvcnRzXG4gICAgICAgIGNvbnN0IGpzQ2xhc3NGcm9tTW9kdWxlID0gbW9kdWxlW2NsYXNzTmFtZS5qb2luKCcuJyldO1xuICAgICAgICBsMUluc3RhbmNlID0ganNDbGFzc0Zyb21Nb2R1bGUuX2Zyb21DbG91ZEZvcm1hdGlvbih0aGlzLCBsb2dpY2FsSWQsIHJlc291cmNlQXR0cmlidXRlcywgb3B0aW9ucyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBsMUluc3RhbmNlID0gbmV3IGNvcmUuQ2ZuUmVzb3VyY2UodGhpcywgbG9naWNhbElkLCB7XG4gICAgICAgICAgdHlwZTogcmVzb3VyY2VBdHRyaWJ1dGVzLlR5cGUsXG4gICAgICAgICAgcHJvcGVydGllczogY2ZuUGFyc2VyLnBhcnNlVmFsdWUocmVzb3VyY2VBdHRyaWJ1dGVzLlByb3BlcnRpZXMpLFxuICAgICAgICB9KTtcbiAgICAgICAgY2ZuUGFyc2VyLmhhbmRsZUF0dHJpYnV0ZXMobDFJbnN0YW5jZSwgcmVzb3VyY2VBdHRyaWJ1dGVzLCBsb2dpY2FsSWQpO1xuICAgICAgfVxuICAgIH1cblxuICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChsMUluc3RhbmNlLCBsb2dpY2FsSWQpO1xuICAgIHRoaXMucmVzb3VyY2VzW2xvZ2ljYWxJZF0gPSBsMUluc3RhbmNlO1xuICAgIHJldHVybiBsMUluc3RhbmNlO1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVOZXN0ZWRTdGFjayhuZXN0ZWRTdGFja0lkOiBzdHJpbmcsIGNmblBhcnNlcjogY2ZuX3BhcnNlLkNmblBhcnNlcik6IGNvcmUuQ2ZuUmVzb3VyY2Uge1xuICAgIGNvbnN0IHRlbXBsYXRlUmVzb3VyY2VzID0gdGhpcy50ZW1wbGF0ZS5SZXNvdXJjZXMgfHwge307XG4gICAgY29uc3QgbmVzdGVkU3RhY2tBdHRyaWJ1dGVzID0gdGVtcGxhdGVSZXNvdXJjZXNbbmVzdGVkU3RhY2tJZF0gfHwge307XG5cbiAgICBpZiAobmVzdGVkU3RhY2tBdHRyaWJ1dGVzLlR5cGUgIT09ICdBV1M6OkNsb3VkRm9ybWF0aW9uOjpTdGFjaycpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTmVzdGVkIFN0YWNrIHdpdGggbG9naWNhbCBJRCAnJHtuZXN0ZWRTdGFja0lkfScgaXMgbm90IGFuIEFXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrIHJlc291cmNlYCk7XG4gICAgfVxuICAgIGlmIChuZXN0ZWRTdGFja0F0dHJpYnV0ZXMuQ3JlYXRpb25Qb2xpY3kpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ3JlYXRpb25Qb2xpY3kgaXMgbm90IHN1cHBvcnRlZCBieSB0aGUgQVdTOjpDbG91ZEZvcm1hdGlvbjo6U3RhY2sgcmVzb3VyY2UnKTtcbiAgICB9XG4gICAgaWYgKG5lc3RlZFN0YWNrQXR0cmlidXRlcy5VcGRhdGVQb2xpY3kpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVXBkYXRlUG9saWN5IGlzIG5vdCBzdXBwb3J0ZWQgYnkgdGhlIEFXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrIHJlc291cmNlJyk7XG4gICAgfVxuXG4gICAgY29uc3QgbmVzdGVkU3RhY2tQcm9wcyA9IGNmblBhcnNlci5wYXJzZVZhbHVlKG5lc3RlZFN0YWNrQXR0cmlidXRlcy5Qcm9wZXJ0aWVzKTtcbiAgICBjb25zdCBuZXN0ZWRTdGFjayA9IG5ldyBjb3JlLk5lc3RlZFN0YWNrKHRoaXMsIG5lc3RlZFN0YWNrSWQsIHtcbiAgICAgIHBhcmFtZXRlcnM6IHRoaXMucGFyYW1ldGVyc0Zvck5lc3RlZFN0YWNrKG5lc3RlZFN0YWNrUHJvcHMuUGFyYW1ldGVycywgbmVzdGVkU3RhY2tJZCksXG4gICAgICBub3RpZmljYXRpb25Bcm5zOiBuZXN0ZWRTdGFja1Byb3BzLk5vdGlmaWNhdGlvbkFybnMsXG4gICAgICB0aW1lb3V0OiBuZXN0ZWRTdGFja1Byb3BzLlRpbWVvdXQsXG4gICAgfSk7XG4gICAgY29uc3QgdGVtcGxhdGUgPSBuZXcgQ2ZuSW5jbHVkZShuZXN0ZWRTdGFjaywgbmVzdGVkU3RhY2tJZCwgdGhpcy5uZXN0ZWRTdGFja3NUb0luY2x1ZGVbbmVzdGVkU3RhY2tJZF0pO1xuICAgIHRoaXMubmVzdGVkU3RhY2tzW25lc3RlZFN0YWNrSWRdID0geyBzdGFjazogbmVzdGVkU3RhY2ssIGluY2x1ZGVkVGVtcGxhdGU6IHRlbXBsYXRlIH07XG5cbiAgICAvLyB3ZSBrbm93IHRoaXMgaXMgbmV2ZXIgdW5kZWZpbmVkIGZvciBuZXN0ZWQgc3RhY2tzXG4gICAgY29uc3QgbmVzdGVkU3RhY2tSZXNvdXJjZTogY29yZS5DZm5SZXNvdXJjZSA9IG5lc3RlZFN0YWNrLm5lc3RlZFN0YWNrUmVzb3VyY2UhO1xuICAgIGNmblBhcnNlci5oYW5kbGVBdHRyaWJ1dGVzKG5lc3RlZFN0YWNrUmVzb3VyY2UsIG5lc3RlZFN0YWNrQXR0cmlidXRlcywgbmVzdGVkU3RhY2tJZCk7XG4gICAgcmV0dXJuIG5lc3RlZFN0YWNrUmVzb3VyY2U7XG4gIH1cblxuICBwcml2YXRlIHBhcmFtZXRlcnNGb3JOZXN0ZWRTdGFjayhwYXJhbWV0ZXJzOiBhbnksIG5lc3RlZFN0YWNrSWQ6IHN0cmluZyk6IHsgW2tleTogc3RyaW5nXTogYW55IH0gfCB1bmRlZmluZWQge1xuICAgIGlmIChwYXJhbWV0ZXJzID09IG51bGwpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgY29uc3QgcGFyYW1ldGVyc1RvUmVwbGFjZSA9IHRoaXMubmVzdGVkU3RhY2tzVG9JbmNsdWRlW25lc3RlZFN0YWNrSWRdLnBhcmFtZXRlcnMgPz8ge307XG4gICAgY29uc3QgcmV0OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9ID0ge307XG4gICAgZm9yIChjb25zdCBwYXJhbU5hbWUgb2YgT2JqZWN0LmtleXMocGFyYW1ldGVycykpIHtcbiAgICAgIGlmICghKHBhcmFtTmFtZSBpbiBwYXJhbWV0ZXJzVG9SZXBsYWNlKSkge1xuICAgICAgICByZXRbcGFyYW1OYW1lXSA9IHBhcmFtZXRlcnNbcGFyYW1OYW1lXTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuXG4gIHByaXZhdGUgb3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChlbGVtZW50OiBjb3JlLkNmbkVsZW1lbnQsIGlkOiBzdHJpbmcpOiB2b2lkIHtcbiAgICBpZiAodGhpcy5wcmVzZXJ2ZUxvZ2ljYWxJZHMpIHtcbiAgICAgIGVsZW1lbnQub3ZlcnJpZGVMb2dpY2FsSWQoaWQpO1xuICAgIH1cbiAgfVxufVxuIl19