"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPartitionKeyNameFromTable = exports.buildDynamoDBTableWithStream = exports.checkTableProps = exports.buildDynamoDBTable = void 0;
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const dynamodb_table_defaults_1 = require("./dynamodb-table-defaults");
const utils_1 = require("./utils");
function buildDynamoDBTable(scope, props) {
    checkTableProps(props);
    // Conditional DynamoDB Table creation
    if (props.existingTableObj) {
        return [props.existingTableObj, props.existingTableObj];
    }
    else if (props.existingTableInterface) {
        return [props.existingTableInterface, undefined];
    }
    else {
        const consolidatedTableProps = utils_1.consolidateProps(dynamodb_table_defaults_1.DefaultTableProps, props.dynamoTableProps);
        const newTable = new dynamodb.Table(scope, 'DynamoTable', consolidatedTableProps);
        return [newTable, newTable];
    }
}
exports.buildDynamoDBTable = buildDynamoDBTable;
function checkTableProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if (props.dynamoTableProps && props.existingTableObj) {
        errorMessages += 'Error - Either provide existingTableObj or dynamoTableProps, but not both.\n';
        errorFound = true;
    }
    if (props.dynamoTableProps && props.existingTableInterface) {
        errorMessages += 'Error - Either provide existingTableInterface or dynamoTableProps, but not both.\n';
        errorFound = true;
    }
    if (props.existingTableObj && props.existingTableInterface) {
        errorMessages += 'Error - Either provide existingTableInterface or existingTableObj, but not both.\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.checkTableProps = checkTableProps;
function buildDynamoDBTableWithStream(scope, props) {
    // Conditional DynamoDB Table creation
    if (!props.existingTableInterface) {
        // Set the default props for DynamoDB table
        const dynamoTableProps = utils_1.consolidateProps(dynamodb_table_defaults_1.DefaultTableWithStreamProps, props.dynamoTableProps);
        const dynamoTable = new dynamodb.Table(scope, 'DynamoTable', dynamoTableProps);
        return [dynamoTable, dynamoTable];
    }
    else {
        return [props.existingTableInterface, undefined];
    }
}
exports.buildDynamoDBTableWithStream = buildDynamoDBTableWithStream;
function getPartitionKeyNameFromTable(table) {
    const cfnTable = table.node.findChild('Resource');
    const keySchema = cfnTable.keySchema;
    const partitionKey = keySchema.find((keyPart) => keyPart.keyType === 'HASH');
    if (!partitionKey) {
        throw new Error('Partition key for table not defined');
    }
    return partitionKey.attributeName;
}
exports.getPartitionKeyNameFromTable = getPartitionKeyNameFromTable;
//# sourceMappingURL=data:application/json;base64,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