"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildKinesisStreamCWAlarms = exports.buildKinesisStream = void 0;
// Imports
const kinesis = require("aws-cdk-lib/aws-kinesis");
const kinesis_streams_defaults_1 = require("./kinesis-streams-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
function buildKinesisStream(scope, props) {
    if (props.existingStreamObj) {
        return props.existingStreamObj;
    }
    // Setup the stream properties
    let kinesisStreamProps;
    // If property overrides have been provided, incorporate them and deploy
    kinesisStreamProps = utils_1.consolidateProps(kinesis_streams_defaults_1.DefaultStreamProps, props.kinesisStreamProps);
    // Create the stream and return
    return new kinesis.Stream(scope, 'KinesisStream', kinesisStreamProps);
}
exports.buildKinesisStream = buildKinesisStream;
function buildKinesisStreamCWAlarms(scope) {
    // Setup CW Alarms for KinesisStream
    const alarms = new Array();
    // Alarm if Max (GetRecords.IteratorAgeMilliseconds): >= 12 hours (50% of 24 hours default retention period)
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamGetRecordsIteratorAgeAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'GetRecords.IteratorAgeMilliseconds',
            statistic: 'Maximum',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 43200,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Consumer Record Processing Falling Behind, there is risk for data loss due to record expiration.'
    }));
    // Alarm if Avg (ReadProvisionedThroughputExceeded): > 0
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamReadProvisionedThroughputExceededAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'ReadProvisionedThroughputExceeded',
            statistic: 'Average',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 0,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
        alarmDescription: 'Consumer Application is Reading at a Slower Rate Than Expected.'
    }));
    return alarms;
}
exports.buildKinesisStreamCWAlarms = buildKinesisStreamCWAlarms;
//# sourceMappingURL=data:application/json;base64,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