"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const ec2 = require("aws-cdk-lib/aws-ec2");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test("Test minimal deployment with no properties", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [], []);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "0.0.0.0/0",
                Description: "Allow all outbound traffic by default",
                IpProtocol: "-1",
            },
        ],
    });
});
test("Test deployment with ingress rules", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) }], []);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupIngress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
        ],
    });
});
test("Test deployment with egress rule", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: false,
    }, [], [
        { peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) },
        { peer: ec2.Peer.ipv4("2.2.2.2/24"), connection: ec2.Port.tcp(200) },
    ]);
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
            {
                CidrIp: "2.2.2.2/24",
                Description: "from 2.2.2.2/24:200",
                FromPort: 200,
                IpProtocol: "tcp",
                ToPort: 200,
            },
        ],
    });
});
test("Test self referencing security group", () => {
    const testPort = 33333;
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.CreateSelfReferencingSecurityGroup(stack, "testsg", vpc, testPort);
    expect(stack).toHaveResourceLike("AWS::EC2::SecurityGroupIngress", {
        IpProtocol: "TCP",
        FromPort: testPort,
        ToPort: testPort,
        GroupId: {
            "Fn::GetAtt": [
                "testsgcachesg72A723EA",
                "GroupId"
            ]
        },
        SourceSecurityGroupId: {
            "Fn::GetAtt": [
                "testsgcachesg72A723EA",
                "GroupId"
            ]
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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