from typing import List, Tuple
import requests
import urllib.parse
import json

from agricore_sp_models.import_models import SyntheticPopulationJsonDTO, FarmJsonDTO

class ABMInterface:
    baseUrl: str = ""
    currentPopulationId = None
    
    populationPath : str = "population/"
    populationAddPath : str= "population/add"
    farmsAddPath : str= "farms/add"
    farmsPath : str= "farms/"
    dumpDataPath : str= "dumpData"
    filterDataPath : str= "filterData"
    yearAddPath : str= "years/add"
    productGroupAddPath : str= "productGroup/add"
    fadnProductsGetPath : str= "FADNProducts/get"
    fadnProductRelationAddPath : str= "FADNProductRelation/add"
    agriculturalProductionAddPath : str= "agriculturalProduction/add"
    livestockProductionAddPath : str= "livestockProduction/add"
    closingValueAddPath : str= "closingValFarmValues/add"
    policyAddPath : str= "policies/add"
    policyGetPath: str = "policies/get"
    policyProductGroupRelationAddPath:str  = "policyProductGroupRelation/add"
    holderFarmYearDataAddPath:str  = "/HolderData/add"
    farmYearSubsidiesAddPath:str = "/farmYearSubsidy/add"
    syntheticPopulationAddPath:str = "/synthetic/add"
    syntheticPopulationImportPath:str = "synthetic/import"
    
    debug = False
    
    fadnProducts = []

    # Helper dicts to transform between values in the original data set and the ones generated by the ABM
    farmsOrigToABMdict = {}
    farmsABMToOrigdict = {}
    
    productGroupsOrigToABMDict = {}
    productGroupsABMToOrigDict = {}
    
    # Extra helper to save the groupings created by the SpecificDatasetConfiguration : ProductOrig -> CustomGroup
    productOrigToProductGroupOrig = {}
        
    def setBaseURL(self, url: str) -> None:
        self.baseUrl = url
        
    def cleanDicts(self) -> None:
        self.farmsOrigToABMDict = {}
        self.farmsABMToOrigDict = {}
        self.productGroupsOrigToABMDict = {}
        self.productGroupsABMToOrigDict = {}
        self.productOrigToProductGroupOrig = {}
        self.fadnProducts = []
        self.fadnCodeToIdDict = {}
        self.fadnIdToCodeDict = {}
        
    def loadFADNProducts(self) -> None:
        self.fadnProducts = self.getFADNProducts()
        self.fadnCodeToIdDict = {}
        self.fadnIdToCodeDict = {}
        for product in self.fadnProducts:
            abmId = product["id"]
            fadnId = product["fadnIdentifier"]
            if self.debug:
                print("saving relation {0}-{1}".format(fadnId, abmId))
            self.fadnCodeToIdDict[fadnId]=abmId
            self.fadnIdToCodeDict[abmId]=fadnId
            
    def transformOrigFarmIdToAbm(self, origFarmId: str | int):
        if type(origFarmId) is int:
            return self.farmsOrigToABMdict[str(origFarmId)]
        elif type(origFarmId) is str:
            return self.farmsOrigToABMdict[origFarmId]
        else:
            return None
    
    def transformABMFarmIdToOrig(self, farmId: str | int):
        if type(farmId) is int:
            return self.farmsABMToOrigdict[str(farmId)]
        elif type(farmId) is str:
            return self.farmsABMToOrigdict[farmId]
        else:
            return None
        
    def transformABMProductGroupToOrig(self, productGroupId: str | int):
        if type(productGroupId) is int:
            return self.productGroupsABMToOrig[str(productGroupId)]
        elif type(productGroupId) is str:
            return self.productGroupsABMToOrig[productGroupId]
        else:
            return None
        
    def transformOrigProductGroupToABM(self, productGroupId: str | int):
        if type(productGroupId) is int:
            return self.productGroupsOrigToABMDict[str(productGroupId)]
        elif type(productGroupId) is str:
            return self.productGroupsOrigToABMDict[productGroupId]
        else:
            return None
    
    def transformFADNIdToCode(self, fadnProductId: str | int):
        if type(fadnProductId) is int:
            return self.fadnIdToCodeDict[str(fadnProductId)]
        elif type(fadnProductId) is str:
            return self.fadnIdToCodeDict[fadnProductId]
        else:
            return None
        
    def transformFADNCodeIntoId(self, fadnProductId: str | int):
        if type(fadnProductId) is int:
            return self.fadnCodeToIdDict[str(fadnProductId)]
        elif type(fadnProductId) is str:
            return self.fadnCodeToIdDict[fadnProductId]
        else:
            return None
    
    def populationAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.populationAddPath)

    def farmAddURL(self, populationId) -> str:
        return urllib.parse.urljoin(self.baseUrl, '{0}/{1}/{2}'.format(self.populationPath, str(populationId), self.farmsAddPath))

    def farmYearSubsidiesAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.farmYearSubsidiesAddPath)
    
    def dumpDataURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.dumpDataPath)
    
    def filterDataURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.filterDataPath)
    
    def yearAddURL(self, populationId) -> str:
        return urllib.parse.urljoin(self.baseUrl, '{0}/{1}/{2}'.format(self.populationPath, str(populationId), self.yearAddPath))
        
    def productGroupAddURL(self, populationId) -> str:
        return urllib.parse.urljoin(self.baseUrl, '{0}/{1}/{2}'.format(self.populationPath, str(populationId), self.productGroupAddPath))
    
    def fadnProductGetURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.fadnProductsGetPath)
    
    def fadnProductRelationAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.fadnProductRelationAddPath)
    
    def agriculturalProductionAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.agriculturalProductionAddPath)
    
    def livestockProductionAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.livestockProductionAddPath)
    
    def closingValueAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.closingValueAddPath)
    
    def policyAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.policyAddPath)

    def policyGetURL(self, policyIdentifier) -> str:
        return urllib.parse.urljoin(self.baseUrl, "{0}/{1}".format(self.policyGetPath, policyIdentifier))

    def policyProductGroupRelationAddURL(self, populationId) -> str:
        return urllib.parse.urljoin(self.baseUrl, '{0}/{1}/{2}'.format(self.populationPath, str(populationId), self.policyProductGroupRelationAddPath))

    def holderFarmYearDataAddUrl(self, populationId, farmCode) -> str:
        return urllib.parse.urljoin(self.baseUrl, '{0}/{1}/{2}/{3}/{4}'.format(self.populationPath, 
                                                                               str(populationId),
                                                                               self.farmsPath,
                                                                               farmCode,
                                                                               self.holderFarmYearDataAddPath))
    
    def syntheticPopulationAddURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.syntheticPopulationAddPath)
    
    def importSyntheticPopulationURL(self) -> str:
        return urllib.parse.urljoin(self.baseUrl, self.syntheticPopulationImportPath)
    
    def addPartialDataURL(self, populationId: int) -> str:
        return urllib.parse.urljoin(self.baseUrl, f'population/{populationId}/addPartialData')
        
    def createPopulation(self, population):        
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.populationAddURL(), json.dumps(population.dict(exclude={'id'})), headers=headers)
        if response.status_code == 201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            population.id = id
            self.currentPopulationId = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None

    def createSyntheticPopulation(self, synthetic):        
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.syntheticPopulationAddURL(), json.dumps(synthetic.dict(exclude={'id'})), headers=headers)
        if response.status_code == 201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            synthetic.id = id
            self.currentSyntheticPopulationId = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
    
    # Returns the ID of the created farm or None
    def createFarm(self, farm):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.farmAddURL(farm.dict().get('populationId')), json.dumps(farm.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id= response.json()["id"]
            farm.id = id
            self.farmsOrigToABMdict[farm.farmCode]=id
            self.farmsABMToOrigdict[id]=farm.farmCode
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
    
    def createProductGroup(self, productGroup):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.productGroupAddURL(productGroup.populationId), json.dumps(productGroup.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            self.productGroupsOrigToABMDict[productGroup.name]=id
            self.productGroupsABMToOrigDict[id]=productGroup.name
            originalProductList = productGroup.productsIncludedInOriginalDataset.strip().split(';')
            for product in originalProductList:
                self.productOrigToProductGroupOrig[(product, productGroup.organic)] = productGroup.name
            productGroup.id = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
        
    def createFADNProductRelation(self, fadnProductRelation):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        #print(message)
        response = requests.post(self.fadnProductRelationAddURL(), json.dumps(fadnProductRelation.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            fadnProductRelation.id = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
        
    def createAgriculturalProduction(self, agriculturalProduction):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        #print(message)
        response = requests.post(self.agriculturalProductionAddURL(), json.dumps(agriculturalProduction.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            agriculturalProduction.id = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
        
    def createLivestockProduction(self, livestockProduction):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        #print(message)
        response = requests.post(self.livestockProductionAddURL(), json.dumps(livestockProduction.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            livestockProduction.id = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None

    def createFarmYearSubsidy(self, subsidy):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        #print(message)
        response = requests.post(self.farmYearSubsidiesAddURL(), json.dumps(subsidy.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            subsidy.id = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None

        
    def createPolicy(self, policy):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.policyAddURL(), json.dumps(policy.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            policy.id = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None

    def getPolicy(self, policy):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.get(self.policyGetURL(policy.policyIdentifier), headers=headers)
        if response.status_code == 200:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            policy.id = id
            return id
        else:
            if self.debug:
                print("Request Error")
                print("{0}-{1}-{2}".format(response.status_code, response.content, response.url))
            return None
        
    def createPolicyProductGroupRelation (self, policyProductGroupRelation):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.policyProductGroupRelationAddURL(policyProductGroupRelation.dict().get('populationId')), json.dumps(policyProductGroupRelation.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            policyProductGroupRelation.id = id
            return id 
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
        
    def createYear(self, year: int, population: int):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.yearAddURL(population), json.dumps({"yearNumber": year}), headers=headers)
        if response.status_code ==201:
            id = response.json()["id"]
            if self.debug:
                print("Request successful")
            return id
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
        
    def createClosingValue(self, closingvalue):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.closingValueAddURL(), json.dumps(closingvalue.dict(exclude={'id'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            closingvalue.id = id
            return id 
        else:
            if self.debug:
                print ("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
        
    def dumpAll(self):
        headers = {'Accept': '*/*'}
        response = requests.delete(self.dumpDataURL(), headers=headers)
        if response.status_code ==200:
            if self.debug:
                print("The database was cleaned")
            return True 
        else:
            if self.debug:
                print ("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return False
        
    def filterData(self):
        headers = {'Accept' : '*/*'}
        response = requests.put(self.filterDataURL(), headers=headers)
        if response.status_code==200:
            if self.debug:
                print("The database was filtered")
            return True
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return False
        
        
    def getFADNProducts(self):
        headers = {'Accept': 'text/plain'}
        response = requests.get(self.fadnProductGetURL(), headers=headers)
        if response.status_code ==200:
            if self.debug:
                print("Request successful")
            return json.loads(response.content)
        else:
            if self.debug:
                print ("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None

    def createHolderFarmYearData(self, farmCode, holderData):
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.holderFarmYearDataAddUrl(self.currentPopulationId, farmCode),
                                 json.dumps(holderData.dict(exclude={'id', 'farmId'})), headers=headers)
        if response.status_code ==201:
            if self.debug:
                print("Request successful")
            id = response.json()["id"]
            holderData.id = id
            return id
        else:
            if self.debug:
                print("Request Error")
                print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return None
        
    def importSyntheticPopulation(self,  population: SyntheticPopulationJsonDTO) -> Tuple[int, int]:        
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.importSyntheticPopulationURL(), json.dumps(population.dict()), headers=headers)
            
        if response.status_code == 201:
            print("Request successful")
            id = response.json()["id"]
            populationId = response.json()["populationId"]
            return id, populationId
        else:
            print("Request Error")
            print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return 0,0
    
    def sendPartialFarmData(self,  population_id: int, farm_list: List[FarmJsonDTO]) -> bool:        
        headers = {'Accept': 'text/plain', 'Content-Type': 'application/json'}
        response = requests.post(self.addPartialDataURL(population_id), json.dumps([x.dict() for x in farm_list]), headers=headers)
        if response.status_code == 201:
            print("Request successful")
            return True
        else:
            print("Request Error")
            print ("{0}-{1}-{2}".format(response.status_code,response.content, response.url))
            return False

    @staticmethod
    def splitPopulationJsonDto(population: SyntheticPopulationJsonDTO, chunkSize: int) -> Tuple[SyntheticPopulationJsonDTO, List[FarmJsonDTO]]:
        chunks = []
        farm_list = population.population.farms
        population.population.farms = []
        for i in range(0, len(farm_list), chunkSize):
            if i == 0:
                population.farms = farm_list[i:i + chunkSize]
            else:
                chunk = farm_list[i:i + chunkSize]
            chunks.append(chunk)
        return population, chunks
