import logging
from importlib import import_module


class ServiceException(Exception):
    """
    We raise this exception when we can't connect to a third-party, online service.
    We expect this to happen whenever there's no internet connection.

    This exception is useful to reattempt operations when they are being processed
    by the Redis queue.
    """

    pass


class Module(object):
    def __init__(
        self,
        pipeline_id: str = None,
        configuration_directory: str = None,
        module_id=None,
        **kwargs,
    ):
        self.pipeline_id = pipeline_id
        self.configuration_directory = configuration_directory

        if module_id:
            self.module_id = module_id

    def _load_function(self, function_name: str):
        if function_name is None:
            return None

        try:
            module_path, _, fn_name = function_name.rpartition(".")
            function = getattr(import_module(module_path), fn_name)
            logging.info(f"Loaded function {function_name}")

            return function
        except Exception as e:
            logging.exception(f"Unable to load function {function_name}")
            return None


class Retriable(Module):
    def __init__(
        self,
        module_id=None,
        schedule=None,
        **kwargs,
    ):
        super().__init__(module_id=module_id, **kwargs)
        self.schedule = schedule or [180, 60]
