"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Athena = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [athena](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonathena.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Athena extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [athena](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonathena.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'athena';
        this.actionList = {
            "BatchGetNamedQuery": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetNamedQuery.html",
                "description": "Grants permissions to get information about one or more named queries.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "BatchGetQueryExecution": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetQueryExecution.html",
                "description": "Grants permissions to get information about one or more query executions.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "CreateDataCatalog": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateDataCatalog.html",
                "description": "Grants permissions to create a datacatalog.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateNamedQuery": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateNamedQuery.html",
                "description": "Grants permissions to create a named query.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "CreateWorkGroup": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateWorkGroup.html",
                "description": "Grants permissions to create a workgroup.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteDataCatalog": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteDataCatalog.html",
                "description": "Grants permissions to delete a datacatalog.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                }
            },
            "DeleteNamedQuery": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteNamedQuery.html",
                "description": "Grants permissions to delete a named query specified.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "DeleteWorkGroup": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteWorkGroup.html",
                "description": "Grants permissions to delete a workgroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "GetDataCatalog": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDataCatalog.html",
                "description": "Grants permissions to get a datacatalog.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                }
            },
            "GetDatabase": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDatabase.html",
                "description": "Grants permissions to get a database for a given datacatalog.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                }
            },
            "GetNamedQuery": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_GetNamedQuery.html",
                "description": "Grants permissions to get information about the specified named query.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "GetQueryExecution": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html",
                "description": "Grants permissions to get information about the specified query execution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "GetQueryResults": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html",
                "description": "Grants permissions to get the query results.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "GetQueryResultsStream": {
                "url": "https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies",
                "description": "Grants permissions to get the query results stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "GetTableMetadata": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ GetTableMetadata.html",
                "description": "Grants permissions to get a metadata about a table for a given datacatalog.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                }
            },
            "GetWorkGroup": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_GetWorkGroup.html",
                "description": "Grants permissions to get a workgroup.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "ListDataCatalogs": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDataCatalogs.html",
                "description": "Grants permissions to return a list of datacatalogs for the specified AWS account.",
                "accessLevel": "List"
            },
            "ListDatabases": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDatabases.html",
                "description": "Grants permissions to return a list of databases for a given datacatalog.",
                "accessLevel": "List",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                }
            },
            "ListNamedQueries": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ListNamedQueries.html",
                "description": "Grants permissions to return a list of named queries in Amazon Athena for the specified AWS account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "ListQueryExecutions": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ListQueryExecutions.html",
                "description": "Grants permissions to return a list of query executions for the specified AWS account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "ListTableMetadata": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTableMetadata.html",
                "description": "Grants permissions to return a list of table metadata in a database for a given datacatalog.",
                "accessLevel": "List",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTagsForResource.html",
                "description": "Grants permissions to return a list of tags for a resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    },
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "ListWorkGroups": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_ListWorkGroups.html",
                "description": "Grants permissions to return a list of workgroups for the specified AWS account.",
                "accessLevel": "List"
            },
            "StartQueryExecution": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html",
                "description": "Grants permissions to start a query execution using an SQL query provided as a string.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "StopQueryExecution": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html",
                "description": "Grants permissions to stop the specified query execution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_TagResource.html",
                "description": "Grants permissions to add a tag to a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    },
                    "workgroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_UntagResource.html",
                "description": "Grants permissions to remove a tag from a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    },
                    "workgroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDataCatalog": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateDataCatalog.html",
                "description": "Grants permissions to update a datacatalog.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "datacatalog": {
                        "required": true
                    }
                }
            },
            "UpdateWorkGroup": {
                "url": "https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateWorkGroup.html",
                "description": "Grants permissions to update a workgroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workgroup": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "datacatalog": {
                "name": "datacatalog",
                "url": "https://docs.aws.amazon.com/athena/latest/ug/datacatalogs-example-policies.html",
                "arn": "arn:${Partition}:athena:${Region}:${Account}:datacatalog/${DataCatalogName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "workgroup": {
                "name": "workgroup",
                "url": "https://docs.aws.amazon.com/athena/latest/ug/example-policies-workgroup.html",
                "arn": "arn:${Partition}:athena:${Region}:${Account}:workgroup/${WorkGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permissions to get information about one or more named queries.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetNamedQuery.html
     */
    batchGetNamedQuery() {
        this.add('athena:BatchGetNamedQuery');
        return this;
    }
    /**
     * Grants permissions to get information about one or more query executions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_BatchGetQueryExecution.html
     */
    batchGetQueryExecution() {
        this.add('athena:BatchGetQueryExecution');
        return this;
    }
    /**
     * Grants permissions to create a datacatalog.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateDataCatalog.html
     */
    createDataCatalog() {
        this.add('athena:CreateDataCatalog');
        return this;
    }
    /**
     * Grants permissions to create a named query.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateNamedQuery.html
     */
    createNamedQuery() {
        this.add('athena:CreateNamedQuery');
        return this;
    }
    /**
     * Grants permissions to create a workgroup.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_CreateWorkGroup.html
     */
    createWorkGroup() {
        this.add('athena:CreateWorkGroup');
        return this;
    }
    /**
     * Grants permissions to delete a datacatalog.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteDataCatalog.html
     */
    deleteDataCatalog() {
        this.add('athena:DeleteDataCatalog');
        return this;
    }
    /**
     * Grants permissions to delete a named query specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteNamedQuery.html
     */
    deleteNamedQuery() {
        this.add('athena:DeleteNamedQuery');
        return this;
    }
    /**
     * Grants permissions to delete a workgroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_DeleteWorkGroup.html
     */
    deleteWorkGroup() {
        this.add('athena:DeleteWorkGroup');
        return this;
    }
    /**
     * Grants permissions to get a datacatalog.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDataCatalog.html
     */
    getDataCatalog() {
        this.add('athena:GetDataCatalog');
        return this;
    }
    /**
     * Grants permissions to get a database for a given datacatalog.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetDatabase.html
     */
    getDatabase() {
        this.add('athena:GetDatabase');
        return this;
    }
    /**
     * Grants permissions to get information about the specified named query.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetNamedQuery.html
     */
    getNamedQuery() {
        this.add('athena:GetNamedQuery');
        return this;
    }
    /**
     * Grants permissions to get information about the specified query execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html
     */
    getQueryExecution() {
        this.add('athena:GetQueryExecution');
        return this;
    }
    /**
     * Grants permissions to get the query results.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html
     */
    getQueryResults() {
        this.add('athena:GetQueryResults');
        return this;
    }
    /**
     * Grants permissions to get the query results stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/ug/connect-with-previous-jdbc.html#jdbc-prev-version-policies
     */
    getQueryResultsStream() {
        this.add('athena:GetQueryResultsStream');
        return this;
    }
    /**
     * Grants permissions to get a metadata about a table for a given datacatalog.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ GetTableMetadata.html
     */
    getTableMetadata() {
        this.add('athena:GetTableMetadata');
        return this;
    }
    /**
     * Grants permissions to get a workgroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_GetWorkGroup.html
     */
    getWorkGroup() {
        this.add('athena:GetWorkGroup');
        return this;
    }
    /**
     * Grants permissions to return a list of datacatalogs for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDataCatalogs.html
     */
    listDataCatalogs() {
        this.add('athena:ListDataCatalogs');
        return this;
    }
    /**
     * Grants permissions to return a list of databases for a given datacatalog.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListDatabases.html
     */
    listDatabases() {
        this.add('athena:ListDatabases');
        return this;
    }
    /**
     * Grants permissions to return a list of named queries in Amazon Athena for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListNamedQueries.html
     */
    listNamedQueries() {
        this.add('athena:ListNamedQueries');
        return this;
    }
    /**
     * Grants permissions to return a list of query executions for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListQueryExecutions.html
     */
    listQueryExecutions() {
        this.add('athena:ListQueryExecutions');
        return this;
    }
    /**
     * Grants permissions to return a list of table metadata in a database for a given datacatalog.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTableMetadata.html
     */
    listTableMetadata() {
        this.add('athena:ListTableMetadata');
        return this;
    }
    /**
     * Grants permissions to return a list of tags for a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('athena:ListTagsForResource');
        return this;
    }
    /**
     * Grants permissions to return a list of workgroups for the specified AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_ListWorkGroups.html
     */
    listWorkGroups() {
        this.add('athena:ListWorkGroups');
        return this;
    }
    /**
     * Grants permissions to start a query execution using an SQL query provided as a string.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html
     */
    startQueryExecution() {
        this.add('athena:StartQueryExecution');
        return this;
    }
    /**
     * Grants permissions to stop the specified query execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html
     */
    stopQueryExecution() {
        this.add('athena:StopQueryExecution');
        return this;
    }
    /**
     * Grants permissions to add a tag to a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('athena:TagResource');
        return this;
    }
    /**
     * Grants permissions to remove a tag from a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('athena:UntagResource');
        return this;
    }
    /**
     * Grants permissions to update a datacatalog.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateDataCatalog.html
     */
    updateDataCatalog() {
        this.add('athena:UpdateDataCatalog');
        return this;
    }
    /**
     * Grants permissions to update a workgroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/athena/latest/APIReference/API_UpdateWorkGroup.html
     */
    updateWorkGroup() {
        this.add('athena:UpdateWorkGroup');
        return this;
    }
    /**
     * Adds a resource of type datacatalog to the statement
     *
     * https://docs.aws.amazon.com/athena/latest/ug/datacatalogs-example-policies.html
     *
     * @param dataCatalogName - Identifier for the dataCatalogName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDatacatalog(dataCatalogName, account, region, partition) {
        var arn = 'arn:${Partition}:athena:${Region}:${Account}:datacatalog/${DataCatalogName}';
        arn = arn.replace('${DataCatalogName}', dataCatalogName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workgroup to the statement
     *
     * https://docs.aws.amazon.com/athena/latest/ug/example-policies-workgroup.html
     *
     * @param workGroupName - Identifier for the workGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onWorkgroup(workGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:athena:${Region}:${Account}:workgroup/${WorkGroupName}';
        arn = arn.replace('${WorkGroupName}', workGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Athena = Athena;
//# sourceMappingURL=data:application/json;base64,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