"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoIdentity = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cognito-identity](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitoidentity.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoIdentity extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cognito-identity](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitoidentity.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cognito-identity';
        this.actionList = {
            "CreateIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_CreateIdentityPool.html",
                "description": "Creates a new identity pool.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DeleteIdentities": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentities.html",
                "description": "Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.",
                "accessLevel": "Write"
            },
            "DeleteIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentityPool.html",
                "description": "Deletes a user pool. Once a pool is deleted, users will not be able to authenticate with the pool.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "DescribeIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentity.html",
                "description": "Returns metadata related to the given identity, including when the identity was created and any associated linked logins.",
                "accessLevel": "Read"
            },
            "DescribeIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentityPool.html",
                "description": "Gets details about a particular identity pool, including the pool name, ID description, creation date, and current number of users.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetCredentialsForIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetCredentialsForIdentity.html",
                "description": "Returns credentials for the provided identity ID.",
                "accessLevel": "Read"
            },
            "GetId": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetId.html",
                "description": "Generates (or retrieves) a Cognito ID. Supplying multiple logins will create an implicit linked account.",
                "accessLevel": "Write"
            },
            "GetIdentityPoolRoles": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolRoles.html",
                "description": "Gets the roles for an identity pool.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "GetOpenIdToken": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdToken.html",
                "description": "Gets an OpenID token, using a known Cognito ID.",
                "accessLevel": "Read"
            },
            "GetOpenIdTokenForDeveloperIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdTokenForDeveloperIdentity.html",
                "description": "Registers (or retrieves) a Cognito IdentityId and an OpenID Connect token for a user authenticated by your backend authentication process.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListIdentities": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentities.html",
                "description": "Lists the identities in a pool.",
                "accessLevel": "List",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "ListIdentityPools": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentityPools.html",
                "description": "Lists all of the Cognito identity pools registered for your account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListTagsForResource.html",
                "description": "Lists the tags that are assigned to an Amazon Cognito identity pool.",
                "accessLevel": "List",
                "resourceTypes": {
                    "identitypool": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "LookupDeveloperIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_LookupDeveloperIdentity.html",
                "description": "Retrieves the IdentityID associated with a DeveloperUserIdentifier or the list of DeveloperUserIdentifiers associated with an IdentityId for an existing identity.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "MergeDeveloperIdentities": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_MergeDeveloperIdentities.html",
                "description": "Merges two users having different IdentityIds, existing in the same identity pool, and identified by the same developer provider.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "SetIdentityPoolRoles": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_SetIdentityPoolRoles.html",
                "description": "Sets the roles for an identity pool. These roles are used when making calls to GetCredentialsForIdentity action.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_TagResource.html",
                "description": "Assigns a set of tags to an Amazon Cognito identity pool.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "identitypool": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UnlinkDeveloperIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkDeveloperIdentity.html",
                "description": "Unlinks a DeveloperUserIdentifier from an existing identity.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            },
            "UnlinkIdentity": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkIdentity.html",
                "description": "Unlinks a federated identity from an existing account.",
                "accessLevel": "Write"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UntagResource.html",
                "description": "Removes the specified tags from an Amazon Cognito identity pool.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "identitypool": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateIdentityPool": {
                "url": "https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UpdateIdentityPool.html",
                "description": "Updates a user pool.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identitypool": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "identitypool": {
                "name": "identitypool",
                "url": "https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html",
                "arn": "arn:${Partition}:cognito-identity:${Region}:${Account}:identitypool/${IdentityPoolId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a new identity pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_CreateIdentityPool.html
     */
    createIdentityPool() {
        this.add('cognito-identity:CreateIdentityPool');
        return this;
    }
    /**
     * Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentities.html
     */
    deleteIdentities() {
        this.add('cognito-identity:DeleteIdentities');
        return this;
    }
    /**
     * Deletes a user pool. Once a pool is deleted, users will not be able to authenticate with the pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentityPool.html
     */
    deleteIdentityPool() {
        this.add('cognito-identity:DeleteIdentityPool');
        return this;
    }
    /**
     * Returns metadata related to the given identity, including when the identity was created and any associated linked logins.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentity.html
     */
    describeIdentity() {
        this.add('cognito-identity:DescribeIdentity');
        return this;
    }
    /**
     * Gets details about a particular identity pool, including the pool name, ID description, creation date, and current number of users.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentityPool.html
     */
    describeIdentityPool() {
        this.add('cognito-identity:DescribeIdentityPool');
        return this;
    }
    /**
     * Returns credentials for the provided identity ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetCredentialsForIdentity.html
     */
    getCredentialsForIdentity() {
        this.add('cognito-identity:GetCredentialsForIdentity');
        return this;
    }
    /**
     * Generates (or retrieves) a Cognito ID. Supplying multiple logins will create an implicit linked account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetId.html
     */
    getId() {
        this.add('cognito-identity:GetId');
        return this;
    }
    /**
     * Gets the roles for an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolRoles.html
     */
    getIdentityPoolRoles() {
        this.add('cognito-identity:GetIdentityPoolRoles');
        return this;
    }
    /**
     * Gets an OpenID token, using a known Cognito ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdToken.html
     */
    getOpenIdToken() {
        this.add('cognito-identity:GetOpenIdToken');
        return this;
    }
    /**
     * Registers (or retrieves) a Cognito IdentityId and an OpenID Connect token for a user authenticated by your backend authentication process.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdTokenForDeveloperIdentity.html
     */
    getOpenIdTokenForDeveloperIdentity() {
        this.add('cognito-identity:GetOpenIdTokenForDeveloperIdentity');
        return this;
    }
    /**
     * Lists the identities in a pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentities.html
     */
    listIdentities() {
        this.add('cognito-identity:ListIdentities');
        return this;
    }
    /**
     * Lists all of the Cognito identity pools registered for your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentityPools.html
     */
    listIdentityPools() {
        this.add('cognito-identity:ListIdentityPools');
        return this;
    }
    /**
     * Lists the tags that are assigned to an Amazon Cognito identity pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('cognito-identity:ListTagsForResource');
        return this;
    }
    /**
     * Retrieves the IdentityID associated with a DeveloperUserIdentifier or the list of DeveloperUserIdentifiers associated with an IdentityId for an existing identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_LookupDeveloperIdentity.html
     */
    lookupDeveloperIdentity() {
        this.add('cognito-identity:LookupDeveloperIdentity');
        return this;
    }
    /**
     * Merges two users having different IdentityIds, existing in the same identity pool, and identified by the same developer provider.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_MergeDeveloperIdentities.html
     */
    mergeDeveloperIdentities() {
        this.add('cognito-identity:MergeDeveloperIdentities');
        return this;
    }
    /**
     * Sets the roles for an identity pool. These roles are used when making calls to GetCredentialsForIdentity action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_SetIdentityPoolRoles.html
     */
    setIdentityPoolRoles() {
        this.add('cognito-identity:SetIdentityPoolRoles');
        return this;
    }
    /**
     * Assigns a set of tags to an Amazon Cognito identity pool.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('cognito-identity:TagResource');
        return this;
    }
    /**
     * Unlinks a DeveloperUserIdentifier from an existing identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkDeveloperIdentity.html
     */
    unlinkDeveloperIdentity() {
        this.add('cognito-identity:UnlinkDeveloperIdentity');
        return this;
    }
    /**
     * Unlinks a federated identity from an existing account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkIdentity.html
     */
    unlinkIdentity() {
        this.add('cognito-identity:UnlinkIdentity');
        return this;
    }
    /**
     * Removes the specified tags from an Amazon Cognito identity pool.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('cognito-identity:UntagResource');
        return this;
    }
    /**
     * Updates a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UpdateIdentityPool.html
     */
    updateIdentityPool() {
        this.add('cognito-identity:UpdateIdentityPool');
        return this;
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-identity:${Region}:${Account}:identitypool/${IdentityPoolId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoIdentity = CognitoIdentity;
//# sourceMappingURL=data:application/json;base64,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