"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ec2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ec2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ec2';
        this.actionList = {
            "AcceptReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html",
                "description": "Grants permission to accept a Convertible Reserved Instance exchange quote",
                "accessLevel": "Write"
            },
            "AcceptTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to accept a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html",
                "description": "Grants permission to accept a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html",
                "description": "Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AcceptVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html",
                "description": "Grants permission to accept a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AdvertiseByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html",
                "description": "Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            },
            "AllocateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html",
                "description": "Grants permission to allocate an Elastic IP address (EIP) to your account",
                "accessLevel": "Write"
            },
            "AllocateHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html",
                "description": "Grants permission to allocate a Dedicated Host to your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dedicated-host": {
                        "required": true
                    }
                }
            },
            "ApplySecurityGroupsToClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html",
                "description": "Grants permission to apply a security group to the association between a Client VPN endpoint and a target network",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html",
                "description": "Grants permission to assign one or more IPv6 addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html",
                "description": "Grants permission to assign one or more secondary private IP addresses to a network interface",
                "accessLevel": "Write"
            },
            "AssociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html",
                "description": "Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface",
                "accessLevel": "Write"
            },
            "AssociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html",
                "description": "Grants permission to associate a target network with a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html",
                "description": "Grants permission to associate or disassociate a set of DHCP options with a VPC",
                "accessLevel": "Write"
            },
            "AssociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html",
                "description": "Grants permission to associate an IAM instance profile with a running or stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AssociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html",
                "description": "Grants permission to associate a subnet or gateway with a route table",
                "accessLevel": "Write"
            },
            "AssociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a subnet",
                "accessLevel": "Write"
            },
            "AssociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html",
                "description": "Grants permission to associate an attachment with a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AssociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html",
                "description": "Grants permission to associate a CIDR block with a VPC",
                "accessLevel": "Write"
            },
            "AttachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html",
                "description": "Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "AttachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html",
                "description": "Grants permission to attach an internet gateway to a VPC",
                "accessLevel": "Write"
            },
            "AttachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html",
                "description": "Grants permission to attach a network interface to an instance",
                "accessLevel": "Write"
            },
            "AttachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html",
                "description": "Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "AttachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html",
                "description": "Grants permission to attach a virtual private gateway to a VPC",
                "accessLevel": "Write"
            },
            "AuthorizeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html",
                "description": "Grants permission to add an inbound authorization rule to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html",
                "description": "Grants permission to add one or more outbound rules to a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "AuthorizeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html",
                "description": "Grants permission to add one or more inbound rules to a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "BundleInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html",
                "description": "Grants permission to bundle an instance store-backed Windows instance",
                "accessLevel": "Write"
            },
            "CancelBundleTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html",
                "description": "Grants permission to cancel a bundling operation",
                "accessLevel": "Write"
            },
            "CancelCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html",
                "description": "Grants permission to cancel a Capacity Reservation and release the reserved capacity",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CancelConversionTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html",
                "description": "Grants permission to cancel an active conversion task",
                "accessLevel": "Write"
            },
            "CancelExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html",
                "description": "Grants permission to cancel an active export task",
                "accessLevel": "Write"
            },
            "CancelImportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html",
                "description": "Grants permission to cancel an in-process import virtual machine or import snapshot task",
                "accessLevel": "Write"
            },
            "CancelReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html",
                "description": "Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CancelSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html",
                "description": "Grants permission to cancel one or more Spot Fleet requests",
                "accessLevel": "Write"
            },
            "CancelSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html",
                "description": "Grants permission to cancel one or more Spot Instance requests",
                "accessLevel": "Write"
            },
            "ConfirmProductInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html",
                "description": "Grants permission to determine whether an owned product code is associated with an instance",
                "accessLevel": "Write"
            },
            "CopyFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html",
                "description": "Grants permission to copy a source Amazon FPGA image (AFI) to the current Region",
                "accessLevel": "Write"
            },
            "CopyImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html",
                "description": "Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region",
                "accessLevel": "Write"
            },
            "CopySnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html",
                "description": "Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html",
                "description": "Grants permission to create a Capacity Reservation",
                "accessLevel": "Write"
            },
            "CreateClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html",
                "description": "Grants permission to create a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html",
                "description": "Grants permission to add a network route to a Client VPN endpoint's route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html",
                "description": "Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device",
                "accessLevel": "Write"
            },
            "CreateDefaultSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html",
                "description": "Grants permission to create a default subnet in a specified Availability Zone in a default VPC",
                "accessLevel": "Write"
            },
            "CreateDefaultVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html",
                "description": "Grants permission to create a default VPC with a default subnet in each Availability Zone",
                "accessLevel": "Write"
            },
            "CreateDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html",
                "description": "Grants permission to create a set of DHCP options for a VPC",
                "accessLevel": "Write"
            },
            "CreateEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html",
                "description": "Grants permission to create an egress-only internet gateway for a VPC",
                "accessLevel": "Write"
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html",
                "description": "Grants permission to launch an EC2 Fleet",
                "accessLevel": "Write"
            },
            "CreateFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html",
                "description": "Grants permission to create one or more flow logs to capture IP traffic for a network interface",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-flow-log": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:Subnet",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:Vpc"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html",
                "description": "Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)",
                "accessLevel": "Write"
            },
            "CreateImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html",
                "description": "Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance",
                "accessLevel": "Write"
            },
            "CreateInstanceExportTask": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html",
                "description": "Grants permission to export a running or stopped instance to an Amazon S3 bucket",
                "accessLevel": "Write"
            },
            "CreateInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html",
                "description": "Grants permission to create an internet gateway for a VPC",
                "accessLevel": "Write"
            },
            "CreateKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html",
                "description": "Grants permission to create a 2048-bit RSA key pair",
                "accessLevel": "Write"
            },
            "CreateLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html",
                "description": "Grants permission to create a launch template",
                "accessLevel": "Write"
            },
            "CreateLaunchTemplateVersion": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html",
                "description": "Grants permission to create a new version of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html",
                "description": "Grants permission to create a static route for a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to associate a VPC with a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "CreateNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html",
                "description": "Grants permission to create a NAT gateway in a subnet",
                "accessLevel": "Write"
            },
            "CreateNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html",
                "description": "Grants permission to create a network ACL in a VPC",
                "accessLevel": "Write"
            },
            "CreateNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html",
                "description": "Grants permission to create a numbered entry (a rule) in a network ACL",
                "accessLevel": "Write"
            },
            "CreateNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html",
                "description": "Grants permission to create a network interface in a subnet",
                "accessLevel": "Write"
            },
            "CreateNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html",
                "description": "Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "ec2:AuthorizedUser",
                            "ec2:AvailabilityZone",
                            "ec2:Permission",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AuthorizedService"
                        ]
                    }
                }
            },
            "CreatePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html",
                "description": "Grants permission to create a placement group",
                "accessLevel": "Write"
            },
            "CreateReservedInstancesListing": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html",
                "description": "Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace",
                "accessLevel": "Write"
            },
            "CreateRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html",
                "description": "Grants permission to create a route in a VPC route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "CreateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html",
                "description": "Grants permission to create a route table for a VPC",
                "accessLevel": "Write"
            },
            "CreateSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html",
                "description": "Grants permission to create a security group",
                "accessLevel": "Write"
            },
            "CreateSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html",
                "description": "Grants permission to create a snapshot of an EBS volume and store it in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html",
                "description": "Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "aws:TagKeys",
                            "aws:RequestTag/${TagKey}",
                            "ec2:ParentVolume",
                            "ec2:Region"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "ec2:Encrypted",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html",
                "description": "Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs",
                "accessLevel": "Write"
            },
            "CreateSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html",
                "description": "Grants permission to create a subnet in a VPC",
                "accessLevel": "Write"
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html",
                "description": "Grants permission to add or overwrite one or more tags for Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:ImageType",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:InstanceType",
                            "ec2:Region",
                            "ec2:ReservedInstancesOfferingType",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                },
                "conditions": [
                    "ec2:CreateAction"
                ]
            },
            "CreateTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html",
                "description": "Grants permission to create a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html",
                "description": "Grants permission to create a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html",
                "description": "Grants permission to create a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html",
                "description": "Grants permission to create a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html",
                "description": "Grants permission to create a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to create a multicast domain for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html",
                "description": "Grants permission to create a static route for a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html",
                "description": "Grants permission to create a route table for a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    }
                }
            },
            "CreateTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html",
                "description": "Grants permission to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html",
                "description": "Grants permission to create an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "CreateVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html",
                "description": "Grants permission to create a VPC with a specified CIDR block",
                "accessLevel": "Write"
            },
            "CreateVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html",
                "description": "Grants permission to create a VPC endpoint for an AWS service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServiceName",
                            "ec2:VpceServiceOwner"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "CreateVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html",
                "description": "Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "CreateVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName"
                        ]
                    }
                }
            },
            "CreateVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html",
                "description": "Grants permission to request a VPC peering connection between two VPCs",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc"
                        ]
                    }
                }
            },
            "CreateVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html",
                "description": "Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:GatewayType",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "CreateVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html",
                "description": "Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "CreateVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html",
                "description": "Grants permission to create a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeleteClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html",
                "description": "Grants permission to delete a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteClientVpnRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html",
                "description": "Grants permission to delete a route from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteCustomerGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html",
                "description": "Grants permission to delete a customer gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "customer-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html",
                "description": "Grants permission to delete a set of DHCP options",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dhcp-options": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteEgressOnlyInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html",
                "description": "Grants permission to delete an egress-only internet gateway",
                "accessLevel": "Write"
            },
            "DeleteFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html",
                "description": "Grants permission to delete one or more EC2 Fleets",
                "accessLevel": "Write"
            },
            "DeleteFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html",
                "description": "Grants permission to delete one or more flow logs",
                "accessLevel": "Write"
            },
            "DeleteFpgaImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html",
                "description": "Grants permission to delete an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "DeleteInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html",
                "description": "Grants permission to delete an internet gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "internet-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html",
                "description": "Grants permission to delete a key pair by removing the public key from Amazon EC2",
                "accessLevel": "Write"
            },
            "DeleteLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html",
                "description": "Grants permission to delete a launch template and its associated versions",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html",
                "description": "Grants permission to delete one or more versions of a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html",
                "description": "Grants permission to delete a route from a local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteLocalGatewayRouteTableVpcAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html",
                "description": "Grants permission to delete an association between a VPC and local gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "local-gateway-route-table-vpc-association": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteNatGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html",
                "description": "Grants permission to delete a NAT gateway",
                "accessLevel": "Write"
            },
            "DeleteNetworkAcl": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html",
                "description": "Grants permission to delete a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html",
                "description": "Grants permission to delete an inbound or outbound entry (rule) from a network ACL",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-acl": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html",
                "description": "Grants permission to delete a detached network interface",
                "accessLevel": "Write"
            },
            "DeleteNetworkInterfacePermission": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html",
                "description": "Grants permission to delete a permission that is associated with a network interface",
                "accessLevel": "Permissions management"
            },
            "DeletePlacementGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html",
                "description": "Grants permission to delete a placement group",
                "accessLevel": "Write"
            },
            "DeleteRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html",
                "description": "Grants permission to delete a route from a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html",
                "description": "Grants permission to delete a route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSecurityGroup": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html",
                "description": "Grants permission to delete a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "DeleteSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html",
                "description": "Grants permission to delete a snapshot of an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "DeleteSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html",
                "description": "Grants permission to delete a data feed for Spot Instances",
                "accessLevel": "Write"
            },
            "DeleteSubnet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html",
                "description": "Grants permission to delete a subnet",
                "accessLevel": "Write"
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html",
                "description": "Grants permission to delete one or more tags from Amazon EC2 resources",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "client-vpn-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "dedicated-host": {
                        "required": false
                    },
                    "dhcp-options": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "fpga-image": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "image": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "instance": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "internet-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "local-gateway": {
                        "required": false
                    },
                    "local-gateway-route-table": {
                        "required": false
                    },
                    "local-gateway-route-table-virtual-interface-group-association": {
                        "required": false
                    },
                    "local-gateway-route-table-vpc-association": {
                        "required": false
                    },
                    "local-gateway-virtual-interface": {
                        "required": false
                    },
                    "local-gateway-virtual-interface-group": {
                        "required": false
                    },
                    "network-acl": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "network-interface": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "reserved-instances": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "spot-instance-request": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "volume": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-endpoint-service": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpc-flow-log": {
                        "required": false
                    },
                    "vpn-connection": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "vpn-gateway": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilter": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html",
                "description": "Grants permission to delete a traffic mirror filter",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html",
                "description": "Grants permission to delete a traffic mirror filter rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html",
                "description": "Grants permission to delete a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTrafficMirrorTarget": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html",
                "description": "Grants permission to delete a traffic mirror target",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-target": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html",
                "description": "Grants permission to delete a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html",
                "description": "Grants permissions to delete a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to delete a peering attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html",
                "description": "Grants permission to delete a route from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html",
                "description": "Grants permission to delete a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html",
                "description": "Grants permission to delete a VPC attachment from a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html",
                "description": "Grants permission to delete an EBS volume",
                "accessLevel": "Write",
                "resourceTypes": {
                    "volume": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DeleteVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html",
                "description": "Grants permission to delete a VPC",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to delete one or more VPC endpoint connection notifications",
                "accessLevel": "Write"
            },
            "DeleteVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to delete one or more VPC endpoint service configurations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html",
                "description": "Grants permission to delete one or more VPC endpoints",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html",
                "description": "Grants permission to delete a VPC peering connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeleteVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html",
                "description": "Grants permission to delete a VPN connection",
                "accessLevel": "Write"
            },
            "DeleteVpnConnectionRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html",
                "description": "Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway",
                "accessLevel": "Write"
            },
            "DeleteVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html",
                "description": "Grants permission to delete a virtual private gateway",
                "accessLevel": "Write"
            },
            "DeprovisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html",
                "description": "Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool",
                "accessLevel": "Write"
            },
            "DeregisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html",
                "description": "Grants permission to deregister an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "DeregisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "DeregisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DeregisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Grants permission to describe the attributes of the AWS account",
                "accessLevel": "List"
            },
            "DescribeAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html",
                "description": "Grants permission to describe one or more Elastic IP addresses",
                "accessLevel": "List"
            },
            "DescribeAggregateIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html",
                "description": "Grants permission to describe the longer ID format settings for all resource types",
                "accessLevel": "List"
            },
            "DescribeAvailabilityZones": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html",
                "description": "Grants permission to describe one or more of the Availability Zones that are available to you",
                "accessLevel": "List"
            },
            "DescribeBundleTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html",
                "description": "Grants permission to describe one or more bundling tasks",
                "accessLevel": "List"
            },
            "DescribeByoipCidrs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html",
                "description": "Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)",
                "accessLevel": "List"
            },
            "DescribeCapacityReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html",
                "description": "Grants permission to describe one or more Capacity Reservations",
                "accessLevel": "List"
            },
            "DescribeClassicLinkInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html",
                "description": "Grants permission to describe one or more linked EC2-Classic instances",
                "accessLevel": "List"
            },
            "DescribeClientVpnAuthorizationRules": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html",
                "description": "Grants permission to describe the authorization rules for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html",
                "description": "Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html",
                "description": "Grants permission to describe one or more Client VPN endpoints",
                "accessLevel": "List"
            },
            "DescribeClientVpnRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html",
                "description": "Grants permission to describe the routes for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeClientVpnTargetNetworks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html",
                "description": "Grants permission to describe the target networks that are associated with a Client VPN endpoint",
                "accessLevel": "List"
            },
            "DescribeCoipPools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html",
                "description": "Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools",
                "accessLevel": "List"
            },
            "DescribeConversionTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html",
                "description": "Grants permission to describe one or more conversion tasks",
                "accessLevel": "List"
            },
            "DescribeCustomerGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html",
                "description": "Grants permission to describe one or more customer gateways",
                "accessLevel": "List"
            },
            "DescribeDhcpOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html",
                "description": "Grants permission to describe one or more DHCP options sets",
                "accessLevel": "List"
            },
            "DescribeEgressOnlyInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html",
                "description": "Grants permission to describe one or more egress-only internet gateways",
                "accessLevel": "List"
            },
            "DescribeElasticGpus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html",
                "description": "Grants permission to describe an Elastic Graphics accelerator that is associated with an instance",
                "accessLevel": "Read"
            },
            "DescribeExportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html",
                "description": "Grants permission to describe one or more export image tasks",
                "accessLevel": "List"
            },
            "DescribeExportTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html",
                "description": "Grants permission to describe one or more export instance tasks",
                "accessLevel": "List"
            },
            "DescribeFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html",
                "description": "Grants permission to describe the state of fast snapshot restores for snapshots",
                "accessLevel": "Read"
            },
            "DescribeFleetHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html",
                "description": "Grants permission to describe the events for an EC2 Fleet during a specified time",
                "accessLevel": "List"
            },
            "DescribeFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html",
                "description": "Grants permission to describe the running instances for an EC2 Fleet",
                "accessLevel": "List"
            },
            "DescribeFleets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html",
                "description": "Grants permission to describe one or more EC2 Fleets",
                "accessLevel": "List"
            },
            "DescribeFlowLogs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html",
                "description": "Grants permission to describe one or more flow logs",
                "accessLevel": "List"
            },
            "DescribeFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html",
                "description": "Grants permission to describe the attributes of an Amazon FPGA Image (AFI)",
                "accessLevel": "List"
            },
            "DescribeFpgaImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html",
                "description": "Grants permission to describe one or more Amazon FPGA Images (AFIs)",
                "accessLevel": "List"
            },
            "DescribeHostReservationOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are available to purchase",
                "accessLevel": "List"
            },
            "DescribeHostReservations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html",
                "description": "Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account",
                "accessLevel": "List"
            },
            "DescribeHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html",
                "description": "Grants permission to describe one or more Dedicated Hosts",
                "accessLevel": "List"
            },
            "DescribeIamInstanceProfileAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html",
                "description": "Grants permission to describe the IAM instance profile associations",
                "accessLevel": "List"
            },
            "DescribeIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources",
                "accessLevel": "List"
            },
            "DescribeIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html",
                "description": "Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user",
                "accessLevel": "List"
            },
            "DescribeImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html",
                "description": "Grants permission to describe an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "List"
            },
            "DescribeImages": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html",
                "description": "Grants permission to describe one or more images (AMIs, AKIs, and ARIs)",
                "accessLevel": "List"
            },
            "DescribeImportImageTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html",
                "description": "Grants permission to describe import virtual machine or import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeImportSnapshotTasks": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html",
                "description": "Grants permission to describe import snapshot tasks",
                "accessLevel": "List"
            },
            "DescribeInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html",
                "description": "Grants permission to describe the attributes of an instance",
                "accessLevel": "List"
            },
            "DescribeInstanceCreditSpecifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html",
                "description": "Grants permission to describe the credit option for CPU usage of one or more burstable performance instances",
                "accessLevel": "List"
            },
            "DescribeInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html",
                "description": "Grants permission to describe the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "List"
            },
            "DescribeInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html",
                "description": "Grants permission to describe the status of one or more instances",
                "accessLevel": "List"
            },
            "DescribeInstanceTypeOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html",
                "description": "Grants permission to describe the set of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstanceTypes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html",
                "description": "Grants permission to describe the details of instance types that are offered in a location",
                "accessLevel": "List"
            },
            "DescribeInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html",
                "description": "Grants permission to describe one or more instances",
                "accessLevel": "List"
            },
            "DescribeInternetGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html",
                "description": "Grants permission to describe one or more internet gateways",
                "accessLevel": "List"
            },
            "DescribeKeyPairs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html",
                "description": "Grants permission to describe one or more key pairs",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplateVersions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html",
                "description": "Grants permission to describe one or more launch template versions",
                "accessLevel": "List"
            },
            "DescribeLaunchTemplates": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html",
                "description": "Grants permission to describe one or more launch templates",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html",
                "description": "Grants permission to describe the associations between virtual interface groups and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTableVpcAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html",
                "description": "Grants permission to describe an association between VPCs and local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html",
                "description": "Grants permission to describe one or more local gateway route tables",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaceGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html",
                "description": "Grants permission to describe local gateway virtual interface groups",
                "accessLevel": "List"
            },
            "DescribeLocalGatewayVirtualInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html",
                "description": "Grants permission to describe local gateway virtual interfaces",
                "accessLevel": "List"
            },
            "DescribeLocalGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html",
                "description": "Grants permission to describe one or more local gateways",
                "accessLevel": "List"
            },
            "DescribeMovingAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html",
                "description": "Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform",
                "accessLevel": "List"
            },
            "DescribeNatGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html",
                "description": "Grants permission to describe one or more NAT gateways",
                "accessLevel": "List"
            },
            "DescribeNetworkAcls": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html",
                "description": "Grants permission to describe one or more network ACLs",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html",
                "description": "Grants permission to describe a network interface attribute",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfacePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html",
                "description": "Grants permission to describe the permissions that are associated with a network interface",
                "accessLevel": "List"
            },
            "DescribeNetworkInterfaces": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html",
                "description": "Grants permission to describe one or more network interfaces",
                "accessLevel": "List"
            },
            "DescribePlacementGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html",
                "description": "Grants permission to describe one or more placement groups",
                "accessLevel": "List"
            },
            "DescribePrefixLists": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html",
                "description": "Grants permission to describe available AWS services in a prefix list format",
                "accessLevel": "List"
            },
            "DescribePrincipalIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html",
                "description": "Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference",
                "accessLevel": "List"
            },
            "DescribePublicIpv4Pools": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html",
                "description": "Grants permission to describe one or more IPv4 address pools",
                "accessLevel": "List"
            },
            "DescribeRegions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html",
                "description": "Grants permission to describe one or more AWS Regions that are currently available in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html",
                "description": "Grants permission to describe one or more purchased Reserved Instances in your account",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesListings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html",
                "description": "Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html",
                "description": "Grants permission to describe the modifications made to one or more Reserved Instances",
                "accessLevel": "List"
            },
            "DescribeReservedInstancesOfferings": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html",
                "description": "Grants permission to describe the Reserved Instance offerings that are available for purchase",
                "accessLevel": "List"
            },
            "DescribeRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html",
                "description": "Grants permission to describe one or more route tables",
                "accessLevel": "List"
            },
            "DescribeScheduledInstanceAvailability": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html",
                "description": "Grants permission to find available schedules for Scheduled Instances",
                "accessLevel": "Read"
            },
            "DescribeScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html",
                "description": "Grants permission to describe one or more Scheduled Instances in your account",
                "accessLevel": "Read"
            },
            "DescribeSecurityGroupReferences": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html",
                "description": "Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups",
                "accessLevel": "List"
            },
            "DescribeSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html",
                "description": "Grants permission to describe one or more security groups",
                "accessLevel": "List"
            },
            "DescribeSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html",
                "description": "Grants permission to describe an attribute of a snapshot",
                "accessLevel": "List"
            },
            "DescribeSnapshots": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html",
                "description": "Grants permission to describe one or more EBS snapshots",
                "accessLevel": "List"
            },
            "DescribeSpotDatafeedSubscription": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html",
                "description": "Grants permission to describe the data feed for Spot Instances",
                "accessLevel": "List"
            },
            "DescribeSpotFleetInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html",
                "description": "Grants permission to describe the running instances for a Spot Fleet",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequestHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html",
                "description": "Grants permission to describe the events for a Spot Fleet request during a specified time",
                "accessLevel": "List"
            },
            "DescribeSpotFleetRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html",
                "description": "Grants permission to describe one or more Spot Fleet requests",
                "accessLevel": "List"
            },
            "DescribeSpotInstanceRequests": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html",
                "description": "Grants permission to describe one or more Spot Instance requests",
                "accessLevel": "List"
            },
            "DescribeSpotPriceHistory": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html",
                "description": "Grants permission to describe the Spot Instance price history",
                "accessLevel": "List"
            },
            "DescribeStaleSecurityGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html",
                "description": "Grants permission to describe the stale security group rules for security groups in a specified VPC",
                "accessLevel": "List"
            },
            "DescribeSubnets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html",
                "description": "Grants permission to describe one or more subnets",
                "accessLevel": "List"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html",
                "description": "Grants permission to describe one or more tags for an Amazon EC2 resource",
                "accessLevel": "Read"
            },
            "DescribeTrafficMirrorFilters": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html",
                "description": "Grants permission to describe one or more traffic mirror filters",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorSessions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html",
                "description": "Grants permission to describe one or more traffic mirror sessions",
                "accessLevel": "List"
            },
            "DescribeTrafficMirrorTargets": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html",
                "description": "Grants permission to describe one or more traffic mirror targets",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html",
                "description": "Grants permission to describe one or more attachments between resources and transit gateways",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayMulticastDomains": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html",
                "description": "Grants permission to describe one or more transit gateway multicast domains",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayPeeringAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html",
                "description": "Grants permission to describe one or more transit gateway peering attachments",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayRouteTables": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html",
                "description": "Grants permission to describe one or more transit gateway route tables",
                "accessLevel": "List"
            },
            "DescribeTransitGatewayVpcAttachments": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html",
                "description": "Grants permission to describe one or more VPC attachments on a transit gateway",
                "accessLevel": "List"
            },
            "DescribeTransitGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html",
                "description": "Grants permission to describe one or more transit gateways",
                "accessLevel": "List"
            },
            "DescribeVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html",
                "description": "Grants permission to describe an attribute of an EBS volume",
                "accessLevel": "List"
            },
            "DescribeVolumeStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html",
                "description": "Grants permission to describe the status of one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html",
                "description": "Grants permission to describe one or more EBS volumes",
                "accessLevel": "List"
            },
            "DescribeVolumesModifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html",
                "description": "Grants permission to describe the current modification status of one or more EBS volumes",
                "accessLevel": "Read"
            },
            "DescribeVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html",
                "description": "Grants permission to describe an attribute of a VPC",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html",
                "description": "Grants permission to describe the ClassicLink status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to describe the ClassicLink DNS support status of one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnectionNotifications": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html",
                "description": "Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html",
                "description": "Grants permission to describe the VPC endpoint connections to your VPC endpoint services",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServiceConfigurations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html",
                "description": "Grants permission to describe VPC endpoint service configurations (your services)",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html",
                "description": "Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service",
                "accessLevel": "List"
            },
            "DescribeVpcEndpointServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html",
                "description": "Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint",
                "accessLevel": "List"
            },
            "DescribeVpcEndpoints": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html",
                "description": "Grants permission to describe one or more VPC endpoints",
                "accessLevel": "List"
            },
            "DescribeVpcPeeringConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html",
                "description": "Grants permission to describe one or more VPC peering connections",
                "accessLevel": "List"
            },
            "DescribeVpcs": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html",
                "description": "Grants permission to describe one or more VPCs",
                "accessLevel": "List"
            },
            "DescribeVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html",
                "description": "Grants permission to describe one or more VPN connections",
                "accessLevel": "Read"
            },
            "DescribeVpnGateways": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html",
                "description": "Grants permission to describe one or more virtual private gateways",
                "accessLevel": "List"
            },
            "DetachClassicLinkVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html",
                "description": "Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DetachInternetGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html",
                "description": "Grants permission to detach an internet gateway from a VPC",
                "accessLevel": "Write"
            },
            "DetachNetworkInterface": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html",
                "description": "Grants permission to detach a network interface from an instance",
                "accessLevel": "Write"
            },
            "DetachVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html",
                "description": "Grants permission to detach an EBS volume from an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    }
                }
            },
            "DetachVpnGateway": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html",
                "description": "Grants permission to detach a virtual private gateway from a VPC",
                "accessLevel": "Write"
            },
            "DisableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html",
                "description": "Grants permission to disable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "DisableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html",
                "description": "Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to disable a resource attachment from propagating routes to the specified propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html",
                "description": "Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC",
                "accessLevel": "Write"
            },
            "DisableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html",
                "description": "Grants permission to disable ClassicLink for a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to disable ClassicLink DNS support for a VPC",
                "accessLevel": "Write"
            },
            "DisassociateAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html",
                "description": "Grants permission to disassociate an Elastic IP address from an instance or network interface",
                "accessLevel": "Write"
            },
            "DisassociateClientVpnTargetNetwork": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html",
                "description": "Grants permission to disassociate a target network from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateIamInstanceProfile": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html",
                "description": "Grants permission to disassociate an IAM instance profile from a running or stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "DisassociateRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html",
                "description": "Grants permission to disassociate a subnet from a route table",
                "accessLevel": "Write"
            },
            "DisassociateSubnetCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a subnet",
                "accessLevel": "Write"
            },
            "DisassociateTransitGatewayMulticastDomain": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html",
                "description": "Grants permission to disassociate one or more subnets from a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateTransitGatewayRouteTable": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html",
                "description": "Grants permission to disassociate a resource attachment from a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "DisassociateVpcCidrBlock": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html",
                "description": "Grants permission to disassociate a CIDR block from a VPC",
                "accessLevel": "Write"
            },
            "EnableEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html",
                "description": "Grants permission to enable EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "EnableFastSnapshotRestores": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html",
                "description": "Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:AvailabilityZone",
                            "ec2:SnapshotTime",
                            "ec2:Encrypted",
                            "ec2:VolumeSize",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableTransitGatewayRouteTablePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html",
                "description": "Grants permission to enable an attachment to propagate routes to a propagation route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "EnableVgwRoutePropagation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html",
                "description": "Grants permission to enable a virtual private gateway to propagate routes to a VPC route table",
                "accessLevel": "Write"
            },
            "EnableVolumeIO": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html",
                "description": "Grants permission to enable I/O operations for a volume that had I/O operations disabled",
                "accessLevel": "Write"
            },
            "EnableVpcClassicLink": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html",
                "description": "Grants permission to enable a VPC for ClassicLink",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "EnableVpcClassicLinkDnsSupport": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html",
                "description": "Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink",
                "accessLevel": "Write"
            },
            "ExportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to download the client certificate revocation list for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportClientVpnClientConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html",
                "description": "Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint",
                "accessLevel": "List"
            },
            "ExportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html",
                "description": "Grants permission to export an Amazon Machine Image (AMI) to a VM file",
                "accessLevel": "Write"
            },
            "ExportTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html",
                "description": "Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket",
                "accessLevel": "Write"
            },
            "GetCapacityReservationUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html",
                "description": "Grants permission to get usage information about a Capacity Reservation",
                "accessLevel": "Read"
            },
            "GetCoipPoolUsage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html",
                "description": "Grants permission to describe the allocations from the specified customer-owned address pool",
                "accessLevel": "Read"
            },
            "GetConsoleOutput": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html",
                "description": "Grants permission to get the console output for an instance",
                "accessLevel": "Read"
            },
            "GetConsoleScreenshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html",
                "description": "Grants permission to retrieve a JPG-format screenshot of a running instance",
                "accessLevel": "Read",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "GetDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html",
                "description": "Grants permission to get the default credit option for CPU usage of a burstable performance instance family",
                "accessLevel": "Read"
            },
            "GetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default",
                "accessLevel": "Read"
            },
            "GetEbsEncryptionByDefault": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html",
                "description": "Grants permission to describe whether EBS encryption by default is enabled for your account",
                "accessLevel": "Read"
            },
            "GetHostReservationPurchasePreview": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html",
                "description": "Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host",
                "accessLevel": "Read"
            },
            "GetLaunchTemplateData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html",
                "description": "Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version",
                "accessLevel": "Read"
            },
            "GetPasswordData": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html",
                "description": "Grants permission to retrieve the encrypted administrator password for a running Windows instance",
                "accessLevel": "Read"
            },
            "GetReservedInstancesExchangeQuote": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html",
                "description": "Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance",
                "accessLevel": "Read"
            },
            "GetTransitGatewayAttachmentPropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html",
                "description": "Grants permission to list the route tables to which a resource attachment propagates routes",
                "accessLevel": "List"
            },
            "GetTransitGatewayMulticastDomainAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html",
                "description": "Grants permission to get information about the associations for a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTableAssociations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html",
                "description": "Grants permission to get information about associations for a transit gateway route table",
                "accessLevel": "List"
            },
            "GetTransitGatewayRouteTablePropagations": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html",
                "description": "Grants permission to get information about the route table propagations for a transit gateway route table",
                "accessLevel": "List"
            },
            "ImportClientVpnClientCertificateRevocationList": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html",
                "description": "Grants permission to upload a client certificate revocation list to a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ImportImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html",
                "description": "Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ImportInstance": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html",
                "description": "Grants permission to create an import instance task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ImportKeyPair": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html",
                "description": "Grants permission to import a public key from an RSA key pair that was created with a third-party tool",
                "accessLevel": "Write"
            },
            "ImportSnapshot": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html",
                "description": "Grants permission to import a disk into an EBS snapshot",
                "accessLevel": "Write"
            },
            "ImportVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html",
                "description": "Grants permission to create an import volume task using metadata from a disk image",
                "accessLevel": "Write"
            },
            "ModifyCapacityReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html",
                "description": "Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released",
                "accessLevel": "Write",
                "resourceTypes": {
                    "capacity-reservation": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyClientVpnEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html",
                "description": "Grants permission to modify a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyDefaultCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html",
                "description": "Grants permission to change the account level default credit option for CPU usage of burstable performance instances",
                "accessLevel": "Write"
            },
            "ModifyEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html",
                "description": "Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account",
                "accessLevel": "Write"
            },
            "ModifyFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html",
                "description": "Grants permission to modify an EC2 Fleet",
                "accessLevel": "Write"
            },
            "ModifyFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon FPGA Image (AFI)",
                "accessLevel": "Write"
            },
            "ModifyHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html",
                "description": "Grants permission to modify a Dedicated Host",
                "accessLevel": "Write"
            },
            "ModifyIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html",
                "description": "Grants permission to modify the ID format for a resource",
                "accessLevel": "Write"
            },
            "ModifyIdentityIdFormat": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html",
                "description": "Grants permission to modify the ID format of a resource for a specific principal in your account",
                "accessLevel": "Write"
            },
            "ModifyImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html",
                "description": "Grants permission to modify an attribute of an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "ModifyInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html",
                "description": "Grants permission to modify an attribute of an instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCapacityReservationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html",
                "description": "Grants permission to modify the Capacity Reservation settings for a stopped instance",
                "accessLevel": "Write"
            },
            "ModifyInstanceCreditSpecification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html",
                "description": "Grants permission to modify the credit option for CPU usage on an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true
                    }
                }
            },
            "ModifyInstanceEventStartTime": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html",
                "description": "Grants permission to modify the start time for a scheduled EC2 instance event",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyInstanceMetadataOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html",
                "description": "Grants permission to modify the metadata options for an instance",
                "accessLevel": "Write"
            },
            "ModifyInstancePlacement": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html",
                "description": "Grants permission to modify the placement attributes for an instance",
                "accessLevel": "Write"
            },
            "ModifyLaunchTemplate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html",
                "description": "Grants permission to modify a launch template",
                "accessLevel": "Write",
                "resourceTypes": {
                    "launch-template": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html",
                "description": "Grants permission to modify an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ModifyReservedInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html",
                "description": "Grants permission to modify attributes of one or more Reserved Instances",
                "accessLevel": "Write"
            },
            "ModifySnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html",
                "description": "Grants permission to add or remove permission settings for a snapshot",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "snapshot": {
                        "required": true,
                        "conditions": [
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "ModifySpotFleetRequest": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html",
                "description": "Grants permission to modify a Spot Fleet request",
                "accessLevel": "Write"
            },
            "ModifySubnetAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html",
                "description": "Grants permission to modify an attribute of a subnet",
                "accessLevel": "Write"
            },
            "ModifyTrafficMirrorFilterNetworkServices": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html",
                "description": "Grants permission to allow or restrict mirroring network services",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorFilterRule": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html",
                "description": "Grants permission to modify a traffic mirror rule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-filter": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter-rule": {
                        "required": true,
                        "conditions": [
                            "ec2:Region"
                        ]
                    }
                }
            },
            "ModifyTrafficMirrorSession": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html",
                "description": "Grants permission to modify a traffic mirror session",
                "accessLevel": "Write",
                "resourceTypes": {
                    "traffic-mirror-session": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-filter": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "traffic-mirror-target": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html",
                "description": "Grants permission to modify a VPC attachment on a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVolume": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html",
                "description": "Grants permission to modify the parameters of an EBS volume",
                "accessLevel": "Write"
            },
            "ModifyVolumeAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html",
                "description": "Grants permission to modify an attribute of a volume",
                "accessLevel": "Write"
            },
            "ModifyVpcAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html",
                "description": "Grants permission to modify an attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpoint": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html",
                "description": "Grants permission to modify an attribute of a VPC endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "route-table": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "security-group": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "subnet": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointConnectionNotification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html",
                "description": "Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service",
                "accessLevel": "Write"
            },
            "ModifyVpcEndpointServiceConfiguration": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html",
                "description": "Grants permission to modify the attributes of a VPC endpoint service configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:VpceServicePrivateDnsName",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcEndpointServicePermissions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html",
                "description": "Grants permission to modify the permissions for a VPC endpoint service",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ModifyVpcPeeringConnectionOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html",
                "description": "Grants permission to modify the VPC peering connection options on one side of a VPC peering connection",
                "accessLevel": "Write"
            },
            "ModifyVpcTenancy": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html",
                "description": "Grants permission to modify the instance tenancy attribute of a VPC",
                "accessLevel": "Write"
            },
            "ModifyVpnConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html",
                "description": "Grants permission to modify the target gateway of a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:GatewayType"
                        ]
                    }
                }
            },
            "ModifyVpnTunnelCertificate": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate",
                "description": "Grants permission to modify the certificate for a Site-to-Site VPN connection",
                "accessLevel": "Write"
            },
            "ModifyVpnTunnelOptions": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html",
                "description": "Grants permission to modify the options for a Site-to-Site VPN connection",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpn-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:AuthenticationType",
                            "ec2:DPDTimeoutSeconds",
                            "ec2:IKEVersions",
                            "ec2:InsideTunnelCidr",
                            "ec2:Phase1DHGroupNumbers",
                            "ec2:Phase2DHGroupNumbers",
                            "ec2:Phase1EncryptionAlgorithms",
                            "ec2:Phase2EncryptionAlgorithms",
                            "ec2:Phase1IntegrityAlgorithms",
                            "ec2:Phase2IntegrityAlgorithms",
                            "ec2:Phase1LifetimeSeconds",
                            "ec2:Phase2LifetimeSeconds",
                            "ec2:PresharedKeys",
                            "ec2:RekeyFuzzPercentage",
                            "ec2:RekeyMarginTimeSeconds",
                            "ec2:RoutingType"
                        ]
                    }
                }
            },
            "MonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html",
                "description": "Grants permission to enable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "MoveAddressToVpc": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html",
                "description": "Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform",
                "accessLevel": "Write"
            },
            "ProvisionByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html",
                "description": "Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool",
                "accessLevel": "Write"
            },
            "PurchaseHostReservation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html",
                "description": "Grants permission to purchase a reservation with configurations that match those of a Dedicated Host",
                "accessLevel": "Write"
            },
            "PurchaseReservedInstancesOffering": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html",
                "description": "Grants permission to purchase a Reserved Instance offering",
                "accessLevel": "Write"
            },
            "PurchaseScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html",
                "description": "Grants permission to purchase one or more Scheduled Instances with a specified schedule",
                "accessLevel": "Write"
            },
            "RebootInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html",
                "description": "Grants permission to request a reboot of one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "RegisterImage": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html",
                "description": "Grants permission to register an Amazon Machine Image (AMI)",
                "accessLevel": "Write"
            },
            "RegisterInstanceEventNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html",
                "description": "Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances",
                "accessLevel": "Write"
            },
            "RegisterTransitGatewayMulticastGroupMembers": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html",
                "description": "Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RegisterTransitGatewayMulticastGroupSources": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html",
                "description": "Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-multicast-domain": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayPeeringAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html",
                "description": "Grants permission to reject a transit gateway peering attachment request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectTransitGatewayVpcAttachment": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html",
                "description": "Grants permission to reject a request to attach a VPC to a transit gateway",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-attachment": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcEndpointConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html",
                "description": "Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RejectVpcPeeringConnection": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html",
                "description": "Grants permission to reject a VPC peering connection request",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-peering-connection": {
                        "required": true,
                        "conditions": [
                            "ec2:AccepterVpc",
                            "ec2:Region",
                            "ec2:RequesterVpc",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReleaseAddress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html",
                "description": "Grants permission to release an Elastic IP address",
                "accessLevel": "Write"
            },
            "ReleaseHosts": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html",
                "description": "Grants permission to release one or more On-Demand Dedicated Hosts",
                "accessLevel": "Write"
            },
            "ReplaceIamInstanceProfileAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html",
                "description": "Grants permission to replace an IAM instance profile for an instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "ReplaceNetworkAclAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html",
                "description": "Grants permission to change which network ACL a subnet is associated with",
                "accessLevel": "Write"
            },
            "ReplaceNetworkAclEntry": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html",
                "description": "Grants permission to replace an entry (rule) in a network ACL",
                "accessLevel": "Write"
            },
            "ReplaceRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html",
                "description": "Grants permission to replace a route within a route table in a VPC",
                "accessLevel": "Write",
                "resourceTypes": {
                    "route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "ReplaceRouteTableAssociation": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html",
                "description": "Grants permission to change the route table that is associated with a subnet",
                "accessLevel": "Write"
            },
            "ReplaceTransitGatewayRoute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html",
                "description": "Grants permission to replace a route in a transit gateway route table",
                "accessLevel": "Write",
                "resourceTypes": {
                    "transit-gateway-route-table": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "transit-gateway-attachment": {
                        "required": false,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "ReportInstanceStatus": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html",
                "description": "Grants permission to submit feedback about the status of an instance",
                "accessLevel": "Write"
            },
            "RequestSpotFleet": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html",
                "description": "Grants permission to create a Spot Fleet request",
                "accessLevel": "Write"
            },
            "RequestSpotInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html",
                "description": "Grants permission to create a Spot Instance request",
                "accessLevel": "Write"
            },
            "ResetEbsDefaultKmsKeyId": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html",
                "description": "Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS",
                "accessLevel": "Write"
            },
            "ResetFpgaImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value",
                "accessLevel": "Write"
            },
            "ResetImageAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html",
                "description": "Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value",
                "accessLevel": "Write"
            },
            "ResetInstanceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html",
                "description": "Grants permission to reset an attribute of an instance to its default value",
                "accessLevel": "Write"
            },
            "ResetNetworkInterfaceAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html",
                "description": "Grants permission to reset an attribute of a network interface",
                "accessLevel": "Write"
            },
            "ResetSnapshotAttribute": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html",
                "description": "Grants permission to reset permission settings for a snapshot",
                "accessLevel": "Permissions management"
            },
            "RestoreAddressToClassic": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html",
                "description": "Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform",
                "accessLevel": "Write"
            },
            "RevokeClientVpnIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html",
                "description": "Grants permission to remove an inbound authorization rule from a Client VPN endpoint",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html",
                "description": "Grants permission to remove one or more outbound rules from a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RevokeSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html",
                "description": "Grants permission to remove one or more inbound rules from a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "RunInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html",
                "description": "Grants permission to launch one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "image": {
                        "required": true,
                        "conditions": [
                            "ec2:ImageType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:Public",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType"
                        ]
                    },
                    "instance": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy",
                            "ec2:MetadataHttpEndpoint",
                            "ec2:MetadataHttpTokens",
                            "ec2:MetadataHttpPutResponseHopLimit"
                        ]
                    },
                    "network-interface": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/",
                            "ec2:Subnet",
                            "ec2:Vpc",
                            "ec2:AssociatePublicIpAddress"
                        ]
                    },
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "subnet": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    },
                    "volume": {
                        "required": true,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:AvailabilityZone",
                            "ec2:Encrypted",
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:ParentSnapshot",
                            "ec2:Region",
                            "ec2:VolumeIops",
                            "ec2:VolumeSize",
                            "ec2:VolumeType"
                        ]
                    },
                    "capacity-reservation": {
                        "required": false,
                        "conditions": [
                            "aws:RequestTag/${TagKey}",
                            "aws:TagKeys",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    },
                    "elastic-gpu": {
                        "required": false,
                        "conditions": [
                            "ec2:ElasticGpuType"
                        ]
                    },
                    "elastic-inference": {
                        "required": false
                    },
                    "key-pair": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "launch-template": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Region"
                        ]
                    },
                    "placement-group": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:PlacementGroupStrategy",
                            "ec2:Region"
                        ]
                    },
                    "snapshot": {
                        "required": false,
                        "conditions": [
                            "ec2:IsLaunchTemplateResource",
                            "ec2:LaunchTemplate",
                            "ec2:Owner",
                            "ec2:ParentVolume",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:SnapshotTime",
                            "ec2:VolumeSize"
                        ]
                    }
                }
            },
            "RunScheduledInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html",
                "description": "Grants permission to launch one or more Scheduled Instances",
                "accessLevel": "Write"
            },
            "SearchLocalGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html",
                "description": "Grants permission to search for routes in a local gateway route table",
                "accessLevel": "List"
            },
            "SearchTransitGatewayMulticastGroups": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html",
                "description": "Grants permission to search for groups, sources, and members in a transit gateway multicast domain",
                "accessLevel": "List"
            },
            "SearchTransitGatewayRoutes": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html",
                "description": "Grants permission to search for routes in a transit gateway route table",
                "accessLevel": "List"
            },
            "SendDiagnosticInterrupt": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html",
                "description": "Grants permission to send a diagnostic interrupt to an Amazon EC2 instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html",
                "description": "Grants permission to start a stopped instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "StartVpcEndpointServicePrivateDnsVerification": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html",
                "description": "Grants permission to start the private DNS verification process for a VPC endpoint service",
                "accessLevel": "Write",
                "resourceTypes": {
                    "vpc-endpoint-service": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "StopInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html",
                "description": "Grants permission to stop an Amazon EBS-backed instance",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "TerminateClientVpnConnections": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html",
                "description": "Grants permission to terminate active Client VPN endpoint connections",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client-vpn-endpoint": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}"
                        ]
                    }
                }
            },
            "TerminateInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html",
                "description": "Grants permission to shut down one or more instances",
                "accessLevel": "Write",
                "resourceTypes": {
                    "instance": {
                        "required": true,
                        "conditions": [
                            "ec2:AvailabilityZone",
                            "ec2:EbsOptimized",
                            "ec2:InstanceProfile",
                            "ec2:InstanceType",
                            "ec2:PlacementGroup",
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:RootDeviceType",
                            "ec2:Tenancy"
                        ]
                    }
                }
            },
            "UnassignIpv6Addresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html",
                "description": "Grants permission to unassign one or more IPv6 addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnassignPrivateIpAddresses": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html",
                "description": "Grants permission to unassign one or more secondary private IP addresses from a network interface",
                "accessLevel": "Write"
            },
            "UnmonitorInstances": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html",
                "description": "Grants permission to disable detailed monitoring for a running instance",
                "accessLevel": "Write"
            },
            "UpdateSecurityGroupRuleDescriptionsEgress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html",
                "description": "Grants permission to update descriptions for one or more outbound rules in a VPC security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "UpdateSecurityGroupRuleDescriptionsIngress": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html",
                "description": "Grants permission to update descriptions for one or more inbound rules in a security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "security-group": {
                        "required": true,
                        "conditions": [
                            "ec2:Region",
                            "ec2:ResourceTag/${TagKey}",
                            "ec2:Vpc"
                        ]
                    }
                }
            },
            "WithdrawByoipCidr": {
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html",
                "description": "Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "capacity-reservation": {
                "name": "capacity-reservation",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "client-vpn-endpoint": {
                "name": "client-vpn-endpoint",
                "url": "https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "customer-gateway": {
                "name": "customer-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}",
                "conditionKeys": [
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dedicated-host": {
                "name": "dedicated-host",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AutoPlacement",
                    "ec2:AvailabilityZone",
                    "ec2:HostRecovery",
                    "ec2:InstanceType",
                    "ec2:Quantity",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "dhcp-options": {
                "name": "dhcp-options",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "elastic-gpu": {
                "name": "elastic-gpu",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}",
                "conditionKeys": [
                    "ec2:ElasticGpuType"
                ]
            },
            "elastic-inference": {
                "name": "elastic-inference",
                "url": "https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html",
                "arn": "arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}",
                "conditionKeys": []
            },
            "fpga-image": {
                "name": "fpga-image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format",
                "arn": "arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "image": {
                "name": "image",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html",
                "arn": "arn:${Partition}:ec2:${Region}::image/${ImageId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:ImageType",
                    "ec2:Owner",
                    "ec2:Public",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType"
                ]
            },
            "instance": {
                "name": "instance",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:EbsOptimized",
                    "ec2:InstanceProfile",
                    "ec2:InstanceType",
                    "ec2:PlacementGroup",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RootDeviceType",
                    "ec2:Tenancy"
                ]
            },
            "internet-gateway": {
                "name": "internet-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "key-pair": {
                "name": "key-pair",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}",
                "conditionKeys": [
                    "ec2:Region"
                ]
            },
            "launch-template": {
                "name": "launch-template",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}",
                "conditionKeys": [
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway": {
                "name": "local-gateway",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table": {
                "name": "local-gateway-route-table",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-virtual-interface-group-association": {
                "name": "local-gateway-route-table-virtual-interface-group-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-route-table-vpc-association": {
                "name": "local-gateway-route-table-vpc-association",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface": {
                "name": "local-gateway-virtual-interface",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "local-gateway-virtual-interface-group": {
                "name": "local-gateway-virtual-interface-group",
                "url": "https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "network-acl": {
                "name": "network-acl",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "network-interface": {
                "name": "network-interface",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AssociatePublicIpAddress",
                    "ec2:AuthorizedService",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Subnet",
                    "ec2:Vpc"
                ]
            },
            "placement-group": {
                "name": "placement-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}",
                "conditionKeys": [
                    "ec2:PlacementGroupStrategy",
                    "ec2:Region"
                ]
            },
            "reserved-instances": {
                "name": "reserved-instances",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:InstanceType",
                    "ec2:Region",
                    "ec2:ReservedInstancesOfferingType",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "route-table": {
                "name": "route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "security-group": {
                "name": "security-group",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html",
                "arn": "arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Owner",
                    "ec2:ParentVolume",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:SnapshotTime",
                    "ec2:VolumeSize"
                ]
            },
            "spot-instance-request": {
                "name": "spot-instance-request",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "subnet": {
                "name": "subnet",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Vpc"
                ]
            },
            "traffic-mirror-session": {
                "name": "traffic-mirror-session",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-target": {
                "name": "traffic-mirror-target",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter": {
                "name": "traffic-mirror-filter",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "traffic-mirror-filter-rule": {
                "name": "traffic-mirror-filter-rule",
                "url": "https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}",
                "conditionKeys": [
                    "ec2:Region"
                ]
            },
            "transit-gateway-attachment": {
                "name": "transit-gateway-attachment",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-multicast-domain": {
                "name": "transit-gateway-multicast-domain",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway-route-table": {
                "name": "transit-gateway-route-table",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "transit-gateway": {
                "name": "transit-gateway",
                "url": "https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "volume": {
                "name": "volume",
                "url": "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AvailabilityZone",
                    "ec2:Encrypted",
                    "ec2:ParentSnapshot",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VolumeIops",
                    "ec2:VolumeSize",
                    "ec2:VolumeType"
                ]
            },
            "vpc": {
                "name": "vpc",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:Tenancy"
                ]
            },
            "vpc-endpoint": {
                "name": "vpc-endpoint",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServiceName",
                    "ec2:VpceServiceOwner"
                ]
            },
            "vpc-endpoint-service": {
                "name": "vpc-endpoint-service",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:VpceServicePrivateDnsName"
                ]
            },
            "vpc-flow-log": {
                "name": "vpc-flow-log",
                "url": "https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:Region",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpc-peering-connection": {
                "name": "vpc-peering-connection",
                "url": "https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}",
                "conditionKeys": [
                    "ec2:AccepterVpc",
                    "ec2:Region",
                    "ec2:RequesterVpc",
                    "ec2:ResourceTag/${TagKey}"
                ]
            },
            "vpn-connection": {
                "name": "vpn-connection",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}",
                "conditionKeys": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "ec2:AuthenticationType",
                    "ec2:DPDTimeoutSeconds",
                    "ec2:GatewayType",
                    "ec2:IKEVersions",
                    "ec2:InsideTunnelCidr",
                    "ec2:Phase1DHGroupNumbers",
                    "ec2:Phase1EncryptionAlgorithms",
                    "ec2:Phase1IntegrityAlgorithms",
                    "ec2:Phase1LifetimeSeconds",
                    "ec2:Phase2DHGroupNumbers",
                    "ec2:Phase2EncryptionAlgorithms",
                    "ec2:Phase2IntegrityAlgorithms",
                    "ec2:Phase2LifetimeSeconds",
                    "ec2:PresharedKeys",
                    "ec2:Region",
                    "ec2:RekeyFuzzPercentage",
                    "ec2:RekeyMarginTimeSeconds",
                    "ec2:ResourceTag/${TagKey}",
                    "ec2:RoutingType"
                ]
            },
            "vpn-gateway": {
                "name": "vpn-gateway",
                "url": "https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html",
                "arn": "arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to accept a Convertible Reserved Instance exchange quote
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptReservedInstancesExchangeQuote.html
     */
    acceptReservedInstancesExchangeQuote() {
        this.add('ec2:AcceptReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to accept a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayPeeringAttachment.html
     */
    acceptTransitGatewayPeeringAttachment() {
        this.add('ec2:AcceptTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to accept a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptTransitGatewayVpcAttachment.html
     */
    acceptTransitGatewayVpcAttachment() {
        this.add('ec2:AcceptTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to accept one or more interface VPC endpoint connections to your VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcEndpointConnections.html
     */
    acceptVpcEndpointConnections() {
        this.add('ec2:AcceptVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to accept a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AcceptVpcPeeringConnection.html
     */
    acceptVpcPeeringConnection() {
        this.add('ec2:AcceptVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to advertise an IP address range that is provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AdvertiseByoipCidr.html
     */
    advertiseByoipCidr() {
        this.add('ec2:AdvertiseByoipCidr');
        return this;
    }
    /**
     * Grants permission to allocate an Elastic IP address (EIP) to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateAddress.html
     */
    allocateAddress() {
        this.add('ec2:AllocateAddress');
        return this;
    }
    /**
     * Grants permission to allocate a Dedicated Host to your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AllocateHosts.html
     */
    allocateHosts() {
        this.add('ec2:AllocateHosts');
        return this;
    }
    /**
     * Grants permission to apply a security group to the association between a Client VPN endpoint and a target network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ApplySecurityGroupsToClientVpnTargetNetwork.html
     */
    applySecurityGroupsToClientVpnTargetNetwork() {
        this.add('ec2:ApplySecurityGroupsToClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to assign one or more IPv6 addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignIpv6Addresses.html
     */
    assignIpv6Addresses() {
        this.add('ec2:AssignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to assign one or more secondary private IP addresses to a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssignPrivateIpAddresses.html
     */
    assignPrivateIpAddresses() {
        this.add('ec2:AssignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to associate an Elastic IP address (EIP) with an instance or a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateAddress.html
     */
    associateAddress() {
        this.add('ec2:AssociateAddress');
        return this;
    }
    /**
     * Grants permission to associate a target network with a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateClientVpnTargetNetwork.html
     */
    associateClientVpnTargetNetwork() {
        this.add('ec2:AssociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to associate or disassociate a set of DHCP options with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateDhcpOptions.html
     */
    associateDhcpOptions() {
        this.add('ec2:AssociateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to associate an IAM instance profile with a running or stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateIamInstanceProfile.html
     */
    associateIamInstanceProfile() {
        this.add('ec2:AssociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to associate a subnet or gateway with a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateRouteTable.html
     */
    associateRouteTable() {
        this.add('ec2:AssociateRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateSubnetCidrBlock.html
     */
    associateSubnetCidrBlock() {
        this.add('ec2:AssociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to associate an attachment and list of subnets with a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayMulticastDomain.html
     */
    associateTransitGatewayMulticastDomain() {
        this.add('ec2:AssociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to associate an attachment with a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateTransitGatewayRouteTable.html
     */
    associateTransitGatewayRouteTable() {
        this.add('ec2:AssociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to associate a CIDR block with a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AssociateVpcCidrBlock.html
     */
    associateVpcCidrBlock() {
        this.add('ec2:AssociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to link an EC2-Classic instance to a ClassicLink-enabled VPC through one or more of the VPC's security groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachClassicLinkVpc.html
     */
    attachClassicLinkVpc() {
        this.add('ec2:AttachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to attach an internet gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachInternetGateway.html
     */
    attachInternetGateway() {
        this.add('ec2:AttachInternetGateway');
        return this;
    }
    /**
     * Grants permission to attach a network interface to an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachNetworkInterface.html
     */
    attachNetworkInterface() {
        this.add('ec2:AttachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to attach an EBS volume to a running or stopped instance and expose it to the instance with the specified device name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVolume.html
     */
    attachVolume() {
        this.add('ec2:AttachVolume');
        return this;
    }
    /**
     * Grants permission to attach a virtual private gateway to a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AttachVpnGateway.html
     */
    attachVpnGateway() {
        this.add('ec2:AttachVpnGateway');
        return this;
    }
    /**
     * Grants permission to add an inbound authorization rule to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeClientVpnIngress.html
     */
    authorizeClientVpnIngress() {
        this.add('ec2:AuthorizeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to add one or more outbound rules to a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupEgress.html
     */
    authorizeSecurityGroupEgress() {
        this.add('ec2:AuthorizeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to add one or more inbound rules to a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_AuthorizeSecurityGroupIngress.html
     */
    authorizeSecurityGroupIngress() {
        this.add('ec2:AuthorizeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to bundle an instance store-backed Windows instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_BundleInstance.html
     */
    bundleInstance() {
        this.add('ec2:BundleInstance');
        return this;
    }
    /**
     * Grants permission to cancel a bundling operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelBundleTask.html
     */
    cancelBundleTask() {
        this.add('ec2:CancelBundleTask');
        return this;
    }
    /**
     * Grants permission to cancel a Capacity Reservation and release the reserved capacity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelCapacityReservation.html
     */
    cancelCapacityReservation() {
        this.add('ec2:CancelCapacityReservation');
        return this;
    }
    /**
     * Grants permission to cancel an active conversion task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelConversionTask.html
     */
    cancelConversionTask() {
        this.add('ec2:CancelConversionTask');
        return this;
    }
    /**
     * Grants permission to cancel an active export task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelExportTask.html
     */
    cancelExportTask() {
        this.add('ec2:CancelExportTask');
        return this;
    }
    /**
     * Grants permission to cancel an in-process import virtual machine or import snapshot task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelImportTask.html
     */
    cancelImportTask() {
        this.add('ec2:CancelImportTask');
        return this;
    }
    /**
     * Grants permission to cancel a Reserved Instance listing on the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelReservedInstancesListing.html
     */
    cancelReservedInstancesListing() {
        this.add('ec2:CancelReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Fleet requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotFleetRequests.html
     */
    cancelSpotFleetRequests() {
        this.add('ec2:CancelSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to cancel one or more Spot Instance requests
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CancelSpotInstanceRequests.html
     */
    cancelSpotInstanceRequests() {
        this.add('ec2:CancelSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to determine whether an owned product code is associated with an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ConfirmProductInstance.html
     */
    confirmProductInstance() {
        this.add('ec2:ConfirmProductInstance');
        return this;
    }
    /**
     * Grants permission to copy a source Amazon FPGA image (AFI) to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyFpgaImage.html
     */
    copyFpgaImage() {
        this.add('ec2:CopyFpgaImage');
        return this;
    }
    /**
     * Grants permission to copy an Amazon Machine Image (AMI) from a source Region to the current Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopyImage.html
     */
    copyImage() {
        this.add('ec2:CopyImage');
        return this;
    }
    /**
     * Grants permission to copy a point-in-time snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CopySnapshot.html
     */
    copySnapshot() {
        this.add('ec2:CopySnapshot');
        return this;
    }
    /**
     * Grants permission to create a Capacity Reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCapacityReservation.html
     */
    createCapacityReservation() {
        this.add('ec2:CreateCapacityReservation');
        return this;
    }
    /**
     * Grants permission to create a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnEndpoint.html
     */
    createClientVpnEndpoint() {
        this.add('ec2:CreateClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to add a network route to a Client VPN endpoint's route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateClientVpnRoute.html
     */
    createClientVpnRoute() {
        this.add('ec2:CreateClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to create a customer gateway, which provides information to AWS about your customer gateway device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateCustomerGateway.html
     */
    createCustomerGateway() {
        this.add('ec2:CreateCustomerGateway');
        return this;
    }
    /**
     * Grants permission to create a default subnet in a specified Availability Zone in a default VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultSubnet.html
     */
    createDefaultSubnet() {
        this.add('ec2:CreateDefaultSubnet');
        return this;
    }
    /**
     * Grants permission to create a default VPC with a default subnet in each Availability Zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDefaultVpc.html
     */
    createDefaultVpc() {
        this.add('ec2:CreateDefaultVpc');
        return this;
    }
    /**
     * Grants permission to create a set of DHCP options for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateDhcpOptions.html
     */
    createDhcpOptions() {
        this.add('ec2:CreateDhcpOptions');
        return this;
    }
    /**
     * Grants permission to create an egress-only internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateEgressOnlyInternetGateway.html
     */
    createEgressOnlyInternetGateway() {
        this.add('ec2:CreateEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to launch an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFleet.html
     */
    createFleet() {
        this.add('ec2:CreateFleet');
        return this;
    }
    /**
     * Grants permission to create one or more flow logs to capture IP traffic for a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFlowLogs.html
     */
    createFlowLogs() {
        this.add('ec2:CreateFlowLogs');
        return this;
    }
    /**
     * Grants permission to create an Amazon FPGA Image (AFI) from a design checkpoint (DCP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateFpgaImage.html
     */
    createFpgaImage() {
        this.add('ec2:CreateFpgaImage');
        return this;
    }
    /**
     * Grants permission to create an Amazon EBS-backed AMI from a stopped or running Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateImage.html
     */
    createImage() {
        this.add('ec2:CreateImage');
        return this;
    }
    /**
     * Grants permission to export a running or stopped instance to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInstanceExportTask.html
     */
    createInstanceExportTask() {
        this.add('ec2:CreateInstanceExportTask');
        return this;
    }
    /**
     * Grants permission to create an internet gateway for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateInternetGateway.html
     */
    createInternetGateway() {
        this.add('ec2:CreateInternetGateway');
        return this;
    }
    /**
     * Grants permission to create a 2048-bit RSA key pair
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateKeyPair.html
     */
    createKeyPair() {
        this.add('ec2:CreateKeyPair');
        return this;
    }
    /**
     * Grants permission to create a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplate.html
     */
    createLaunchTemplate() {
        this.add('ec2:CreateLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to create a new version of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLaunchTemplateVersion.html
     */
    createLaunchTemplateVersion() {
        this.add('ec2:CreateLaunchTemplateVersion');
        return this;
    }
    /**
     * Grants permission to create a static route for a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRoute.html
     */
    createLocalGatewayRoute() {
        this.add('ec2:CreateLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to associate a VPC with a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateLocalGatewayRouteTableVpcAssociation.html
     */
    createLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:CreateLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to create a NAT gateway in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNatGateway.html
     */
    createNatGateway() {
        this.add('ec2:CreateNatGateway');
        return this;
    }
    /**
     * Grants permission to create a network ACL in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAcl.html
     */
    createNetworkAcl() {
        this.add('ec2:CreateNetworkAcl');
        return this;
    }
    /**
     * Grants permission to create a numbered entry (a rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkAclEntry.html
     */
    createNetworkAclEntry() {
        this.add('ec2:CreateNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to create a network interface in a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterface.html
     */
    createNetworkInterface() {
        this.add('ec2:CreateNetworkInterface');
        return this;
    }
    /**
     * Grants permission to create a permission for an AWS-authorized user to perform certain operations on a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateNetworkInterfacePermission.html
     */
    createNetworkInterfacePermission() {
        this.add('ec2:CreateNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to create a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreatePlacementGroup.html
     */
    createPlacementGroup() {
        this.add('ec2:CreatePlacementGroup');
        return this;
    }
    /**
     * Grants permission to create a listing for Standard Reserved Instances to be sold in the Reserved Instance Marketplace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateReservedInstancesListing.html
     */
    createReservedInstancesListing() {
        this.add('ec2:CreateReservedInstancesListing');
        return this;
    }
    /**
     * Grants permission to create a route in a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRoute.html
     */
    createRoute() {
        this.add('ec2:CreateRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateRouteTable.html
     */
    createRouteTable() {
        this.add('ec2:CreateRouteTable');
        return this;
    }
    /**
     * Grants permission to create a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSecurityGroup.html
     */
    createSecurityGroup() {
        this.add('ec2:CreateSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a snapshot of an EBS volume and store it in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshot.html
     */
    createSnapshot() {
        this.add('ec2:CreateSnapshot');
        return this;
    }
    /**
     * Grants permission to create crash-consistent snapshots of multiple EBS volumes and store them in Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSnapshots.html
     */
    createSnapshots() {
        this.add('ec2:CreateSnapshots');
        return this;
    }
    /**
     * Grants permission to create a data feed for Spot Instances to view Spot Instance usage logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSpotDatafeedSubscription.html
     */
    createSpotDatafeedSubscription() {
        this.add('ec2:CreateSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to create a subnet in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateSubnet.html
     */
    createSubnet() {
        this.add('ec2:CreateSubnet');
        return this;
    }
    /**
     * Grants permission to add or overwrite one or more tags for Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html
     */
    createTags() {
        this.add('ec2:CreateTags');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilter.html
     */
    createTrafficMirrorFilter() {
        this.add('ec2:CreateTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorFilterRule.html
     */
    createTrafficMirrorFilterRule() {
        this.add('ec2:CreateTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorSession.html
     */
    createTrafficMirrorSession() {
        this.add('ec2:CreateTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to create a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTrafficMirrorTarget.html
     */
    createTrafficMirrorTarget() {
        this.add('ec2:CreateTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to create a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGateway.html
     */
    createTransitGateway() {
        this.add('ec2:CreateTransitGateway');
        return this;
    }
    /**
     * Grants permission to create a multicast domain for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayMulticastDomain.html
     */
    createTransitGatewayMulticastDomain() {
        this.add('ec2:CreateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to request a transit gateway peering attachment between a requester and accepter transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayPeeringAttachment.html
     */
    createTransitGatewayPeeringAttachment() {
        this.add('ec2:CreateTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to create a static route for a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRoute.html
     */
    createTransitGatewayRoute() {
        this.add('ec2:CreateTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to create a route table for a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayRouteTable.html
     */
    createTransitGatewayRouteTable() {
        this.add('ec2:CreateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTransitGatewayVpcAttachment.html
     */
    createTransitGatewayVpcAttachment() {
        this.add('ec2:CreateTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to create an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVolume.html
     */
    createVolume() {
        this.add('ec2:CreateVolume');
        return this;
    }
    /**
     * Grants permission to create a VPC with a specified CIDR block
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpc.html
     */
    createVpc() {
        this.add('ec2:CreateVpc');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint for an AWS service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpoint.html
     */
    createVpcEndpoint() {
        this.add('ec2:CreateVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to create a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointConnectionNotification.html
     */
    createVpcEndpointConnectionNotification() {
        this.add('ec2:CreateVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to create a VPC endpoint service configuration to which service consumers (AWS accounts, IAM users, and IAM roles) can connect
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcEndpointServiceConfiguration.html
     */
    createVpcEndpointServiceConfiguration() {
        this.add('ec2:CreateVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to request a VPC peering connection between two VPCs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpcPeeringConnection.html
     */
    createVpcPeeringConnection() {
        this.add('ec2:CreateVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to create a VPN connection between a virtual private gateway or transit gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnection.html
     */
    createVpnConnection() {
        this.add('ec2:CreateVpnConnection');
        return this;
    }
    /**
     * Grants permission to create a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnConnectionRoute.html
     */
    createVpnConnectionRoute() {
        this.add('ec2:CreateVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to create a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateVpnGateway.html
     */
    createVpnGateway() {
        this.add('ec2:CreateVpnGateway');
        return this;
    }
    /**
     * Grants permission to delete a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnEndpoint.html
     */
    deleteClientVpnEndpoint() {
        this.add('ec2:DeleteClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a route from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteClientVpnRoute.html
     */
    deleteClientVpnRoute() {
        this.add('ec2:DeleteClientVpnRoute');
        return this;
    }
    /**
     * Grants permission to delete a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteCustomerGateway.html
     */
    deleteCustomerGateway() {
        this.add('ec2:DeleteCustomerGateway');
        return this;
    }
    /**
     * Grants permission to delete a set of DHCP options
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteDhcpOptions.html
     */
    deleteDhcpOptions() {
        this.add('ec2:DeleteDhcpOptions');
        return this;
    }
    /**
     * Grants permission to delete an egress-only internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteEgressOnlyInternetGateway.html
     */
    deleteEgressOnlyInternetGateway() {
        this.add('ec2:DeleteEgressOnlyInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete one or more EC2 Fleets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFleets.html
     */
    deleteFleets() {
        this.add('ec2:DeleteFleets');
        return this;
    }
    /**
     * Grants permission to delete one or more flow logs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFlowLogs.html
     */
    deleteFlowLogs() {
        this.add('ec2:DeleteFlowLogs');
        return this;
    }
    /**
     * Grants permission to delete an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteFpgaImage.html
     */
    deleteFpgaImage() {
        this.add('ec2:DeleteFpgaImage');
        return this;
    }
    /**
     * Grants permission to delete an internet gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteInternetGateway.html
     */
    deleteInternetGateway() {
        this.add('ec2:DeleteInternetGateway');
        return this;
    }
    /**
     * Grants permission to delete a key pair by removing the public key from Amazon EC2
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteKeyPair.html
     */
    deleteKeyPair() {
        this.add('ec2:DeleteKeyPair');
        return this;
    }
    /**
     * Grants permission to delete a launch template and its associated versions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplate.html
     */
    deleteLaunchTemplate() {
        this.add('ec2:DeleteLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLaunchTemplateVersions.html
     */
    deleteLaunchTemplateVersions() {
        this.add('ec2:DeleteLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to delete a route from a local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRoute.html
     */
    deleteLocalGatewayRoute() {
        this.add('ec2:DeleteLocalGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete an association between a VPC and local gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteLocalGatewayRouteTableVpcAssociation.html
     */
    deleteLocalGatewayRouteTableVpcAssociation() {
        this.add('ec2:DeleteLocalGatewayRouteTableVpcAssociation');
        return this;
    }
    /**
     * Grants permission to delete a NAT gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNatGateway.html
     */
    deleteNatGateway() {
        this.add('ec2:DeleteNatGateway');
        return this;
    }
    /**
     * Grants permission to delete a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAcl.html
     */
    deleteNetworkAcl() {
        this.add('ec2:DeleteNetworkAcl');
        return this;
    }
    /**
     * Grants permission to delete an inbound or outbound entry (rule) from a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkAclEntry.html
     */
    deleteNetworkAclEntry() {
        this.add('ec2:DeleteNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to delete a detached network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterface.html
     */
    deleteNetworkInterface() {
        this.add('ec2:DeleteNetworkInterface');
        return this;
    }
    /**
     * Grants permission to delete a permission that is associated with a network interface
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteNetworkInterfacePermission.html
     */
    deleteNetworkInterfacePermission() {
        this.add('ec2:DeleteNetworkInterfacePermission');
        return this;
    }
    /**
     * Grants permission to delete a placement group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeletePlacementGroup.html
     */
    deletePlacementGroup() {
        this.add('ec2:DeletePlacementGroup');
        return this;
    }
    /**
     * Grants permission to delete a route from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRoute.html
     */
    deleteRoute() {
        this.add('ec2:DeleteRoute');
        return this;
    }
    /**
     * Grants permission to delete a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteRouteTable.html
     */
    deleteRouteTable() {
        this.add('ec2:DeleteRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSecurityGroup.html
     */
    deleteSecurityGroup() {
        this.add('ec2:DeleteSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a snapshot of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSnapshot.html
     */
    deleteSnapshot() {
        this.add('ec2:DeleteSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a data feed for Spot Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSpotDatafeedSubscription.html
     */
    deleteSpotDatafeedSubscription() {
        this.add('ec2:DeleteSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to delete a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteSubnet.html
     */
    deleteSubnet() {
        this.add('ec2:DeleteSubnet');
        return this;
    }
    /**
     * Grants permission to delete one or more tags from Amazon EC2 resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTags.html
     */
    deleteTags() {
        this.add('ec2:DeleteTags');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilter.html
     */
    deleteTrafficMirrorFilter() {
        this.add('ec2:DeleteTrafficMirrorFilter');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror filter rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorFilterRule.html
     */
    deleteTrafficMirrorFilterRule() {
        this.add('ec2:DeleteTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorSession.html
     */
    deleteTrafficMirrorSession() {
        this.add('ec2:DeleteTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to delete a traffic mirror target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTrafficMirrorTarget.html
     */
    deleteTrafficMirrorTarget() {
        this.add('ec2:DeleteTrafficMirrorTarget');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGateway.html
     */
    deleteTransitGateway() {
        this.add('ec2:DeleteTransitGateway');
        return this;
    }
    /**
     * Grants permissions to delete a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayMulticastDomain.html
     */
    deleteTransitGatewayMulticastDomain() {
        this.add('ec2:DeleteTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to delete a peering attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayPeeringAttachment.html
     */
    deleteTransitGatewayPeeringAttachment() {
        this.add('ec2:DeleteTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to delete a route from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRoute.html
     */
    deleteTransitGatewayRoute() {
        this.add('ec2:DeleteTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to delete a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayRouteTable.html
     */
    deleteTransitGatewayRouteTable() {
        this.add('ec2:DeleteTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to delete a VPC attachment from a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteTransitGatewayVpcAttachment.html
     */
    deleteTransitGatewayVpcAttachment() {
        this.add('ec2:DeleteTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to delete an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVolume.html
     */
    deleteVolume() {
        this.add('ec2:DeleteVolume');
        return this;
    }
    /**
     * Grants permission to delete a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpc.html
     */
    deleteVpc() {
        this.add('ec2:DeleteVpc');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint connection notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointConnectionNotifications.html
     */
    deleteVpcEndpointConnectionNotifications() {
        this.add('ec2:DeleteVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoint service configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpointServiceConfigurations.html
     */
    deleteVpcEndpointServiceConfigurations() {
        this.add('ec2:DeleteVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to delete one or more VPC endpoints
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcEndpoints.html
     */
    deleteVpcEndpoints() {
        this.add('ec2:DeleteVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to delete a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpcPeeringConnection.html
     */
    deleteVpcPeeringConnection() {
        this.add('ec2:DeleteVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to delete a VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnection.html
     */
    deleteVpnConnection() {
        this.add('ec2:DeleteVpnConnection');
        return this;
    }
    /**
     * Grants permission to delete a static route for a VPN connection between a virtual private gateway and a customer gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnConnectionRoute.html
     */
    deleteVpnConnectionRoute() {
        this.add('ec2:DeleteVpnConnectionRoute');
        return this;
    }
    /**
     * Grants permission to delete a virtual private gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeleteVpnGateway.html
     */
    deleteVpnGateway() {
        this.add('ec2:DeleteVpnGateway');
        return this;
    }
    /**
     * Grants permission to release an IP address range that was provisioned through bring your own IP addresses (BYOIP), and to delete the corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeprovisionByoipCidr.html
     */
    deprovisionByoipCidr() {
        this.add('ec2:DeprovisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to deregister an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterImage.html
     */
    deregisterImage() {
        this.add('ec2:DeregisterImage');
        return this;
    }
    /**
     * Grants permission to remove tags from the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterInstanceEventNotificationAttributes.html
     */
    deregisterInstanceEventNotificationAttributes() {
        this.add('ec2:DeregisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface members from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupMembers.html
     */
    deregisterTransitGatewayMulticastGroupMembers() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to deregister one or more network interface sources from a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DeregisterTransitGatewayMulticastGroupSources.html
     */
    deregisterTransitGatewayMulticastGroupSources() {
        this.add('ec2:DeregisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to describe the attributes of the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAccountAttributes.html
     */
    describeAccountAttributes() {
        this.add('ec2:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to describe one or more Elastic IP addresses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAddresses.html
     */
    describeAddresses() {
        this.add('ec2:DescribeAddresses');
        return this;
    }
    /**
     * Grants permission to describe the longer ID format settings for all resource types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAggregateIdFormat.html
     */
    describeAggregateIdFormat() {
        this.add('ec2:DescribeAggregateIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more of the Availability Zones that are available to you
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeAvailabilityZones.html
     */
    describeAvailabilityZones() {
        this.add('ec2:DescribeAvailabilityZones');
        return this;
    }
    /**
     * Grants permission to describe one or more bundling tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeBundleTasks.html
     */
    describeBundleTasks() {
        this.add('ec2:DescribeBundleTasks');
        return this;
    }
    /**
     * Grants permission to describe the IP address ranges that were provisioned through bring your own IP addresses (BYOIP)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeByoipCidrs.html
     */
    describeByoipCidrs() {
        this.add('ec2:DescribeByoipCidrs');
        return this;
    }
    /**
     * Grants permission to describe one or more Capacity Reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCapacityReservations.html
     */
    describeCapacityReservations() {
        this.add('ec2:DescribeCapacityReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more linked EC2-Classic instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClassicLinkInstances.html
     */
    describeClassicLinkInstances() {
        this.add('ec2:DescribeClassicLinkInstances');
        return this;
    }
    /**
     * Grants permission to describe the authorization rules for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnAuthorizationRules.html
     */
    describeClientVpnAuthorizationRules() {
        this.add('ec2:DescribeClientVpnAuthorizationRules');
        return this;
    }
    /**
     * Grants permission to describe active client connections and connections that have been terminated within the last 60 minutes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnConnections.html
     */
    describeClientVpnConnections() {
        this.add('ec2:DescribeClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more Client VPN endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnEndpoints.html
     */
    describeClientVpnEndpoints() {
        this.add('ec2:DescribeClientVpnEndpoints');
        return this;
    }
    /**
     * Grants permission to describe the routes for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnRoutes.html
     */
    describeClientVpnRoutes() {
        this.add('ec2:DescribeClientVpnRoutes');
        return this;
    }
    /**
     * Grants permission to describe the target networks that are associated with a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeClientVpnTargetNetworks.html
     */
    describeClientVpnTargetNetworks() {
        this.add('ec2:DescribeClientVpnTargetNetworks');
        return this;
    }
    /**
     * Grants permission to describe the specified customer-owned address pools or all of your customer-owned address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCoipPools.html
     */
    describeCoipPools() {
        this.add('ec2:DescribeCoipPools');
        return this;
    }
    /**
     * Grants permission to describe one or more conversion tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeConversionTasks.html
     */
    describeConversionTasks() {
        this.add('ec2:DescribeConversionTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more customer gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeCustomerGateways.html
     */
    describeCustomerGateways() {
        this.add('ec2:DescribeCustomerGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more DHCP options sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeDhcpOptions.html
     */
    describeDhcpOptions() {
        this.add('ec2:DescribeDhcpOptions');
        return this;
    }
    /**
     * Grants permission to describe one or more egress-only internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeEgressOnlyInternetGateways.html
     */
    describeEgressOnlyInternetGateways() {
        this.add('ec2:DescribeEgressOnlyInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe an Elastic Graphics accelerator that is associated with an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeElasticGpus.html
     */
    describeElasticGpus() {
        this.add('ec2:DescribeElasticGpus');
        return this;
    }
    /**
     * Grants permission to describe one or more export image tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportImageTasks.html
     */
    describeExportImageTasks() {
        this.add('ec2:DescribeExportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe one or more export instance tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeExportTasks.html
     */
    describeExportTasks() {
        this.add('ec2:DescribeExportTasks');
        return this;
    }
    /**
     * Grants permission to describe the state of fast snapshot restores for snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFastSnapshotRestores.html
     */
    describeFastSnapshotRestores() {
        this.add('ec2:DescribeFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to describe the events for an EC2 Fleet during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetHistory.html
     */
    describeFleetHistory() {
        this.add('ec2:DescribeFleetHistory');
        return this;
    }
    /**
     * Grants permission to describe the running instances for an EC2 Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleetInstances.html
     */
    describeFleetInstances() {
        this.add('ec2:DescribeFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more EC2 Fleets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFleets.html
     */
    describeFleets() {
        this.add('ec2:DescribeFleets');
        return this;
    }
    /**
     * Grants permission to describe one or more flow logs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFlowLogs.html
     */
    describeFlowLogs() {
        this.add('ec2:DescribeFlowLogs');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an Amazon FPGA Image (AFI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImageAttribute.html
     */
    describeFpgaImageAttribute() {
        this.add('ec2:DescribeFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more Amazon FPGA Images (AFIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeFpgaImages.html
     */
    describeFpgaImages() {
        this.add('ec2:DescribeFpgaImages');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are available to purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservationOfferings.html
     */
    describeHostReservationOfferings() {
        this.add('ec2:DescribeHostReservationOfferings');
        return this;
    }
    /**
     * Grants permission to describe the Dedicated Host Reservations that are associated with Dedicated Hosts in the AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHostReservations.html
     */
    describeHostReservations() {
        this.add('ec2:DescribeHostReservations');
        return this;
    }
    /**
     * Grants permission to describe one or more Dedicated Hosts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeHosts.html
     */
    describeHosts() {
        this.add('ec2:DescribeHosts');
        return this;
    }
    /**
     * Grants permission to describe the IAM instance profile associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIamInstanceProfileAssociations.html
     */
    describeIamInstanceProfileAssociations() {
        this.add('ec2:DescribeIamInstanceProfileAssociations');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdFormat.html
     */
    describeIdFormat() {
        this.add('ec2:DescribeIdFormat');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for resources for an IAM user, IAM role, or root user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeIdentityIdFormat.html
     */
    describeIdentityIdFormat() {
        this.add('ec2:DescribeIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImageAttribute.html
     */
    describeImageAttribute() {
        this.add('ec2:DescribeImageAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more images (AMIs, AKIs, and ARIs)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImages.html
     */
    describeImages() {
        this.add('ec2:DescribeImages');
        return this;
    }
    /**
     * Grants permission to describe import virtual machine or import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportImageTasks.html
     */
    describeImportImageTasks() {
        this.add('ec2:DescribeImportImageTasks');
        return this;
    }
    /**
     * Grants permission to describe import snapshot tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeImportSnapshotTasks.html
     */
    describeImportSnapshotTasks() {
        this.add('ec2:DescribeImportSnapshotTasks');
        return this;
    }
    /**
     * Grants permission to describe the attributes of an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceAttribute.html
     */
    describeInstanceAttribute() {
        this.add('ec2:DescribeInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the credit option for CPU usage of one or more burstable performance instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceCreditSpecifications.html
     */
    describeInstanceCreditSpecifications() {
        this.add('ec2:DescribeInstanceCreditSpecifications');
        return this;
    }
    /**
     * Grants permission to describe the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceEventNotificationAttributes.html
     */
    describeInstanceEventNotificationAttributes() {
        this.add('ec2:DescribeInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceStatus.html
     */
    describeInstanceStatus() {
        this.add('ec2:DescribeInstanceStatus');
        return this;
    }
    /**
     * Grants permission to describe the set of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypeOfferings.html
     */
    describeInstanceTypeOfferings() {
        this.add('ec2:DescribeInstanceTypeOfferings');
        return this;
    }
    /**
     * Grants permission to describe the details of instance types that are offered in a location
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstanceTypes.html
     */
    describeInstanceTypes() {
        this.add('ec2:DescribeInstanceTypes');
        return this;
    }
    /**
     * Grants permission to describe one or more instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInstances.html
     */
    describeInstances() {
        this.add('ec2:DescribeInstances');
        return this;
    }
    /**
     * Grants permission to describe one or more internet gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeInternetGateways.html
     */
    describeInternetGateways() {
        this.add('ec2:DescribeInternetGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more key pairs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeKeyPairs.html
     */
    describeKeyPairs() {
        this.add('ec2:DescribeKeyPairs');
        return this;
    }
    /**
     * Grants permission to describe one or more launch template versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplateVersions.html
     */
    describeLaunchTemplateVersions() {
        this.add('ec2:DescribeLaunchTemplateVersions');
        return this;
    }
    /**
     * Grants permission to describe one or more launch templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLaunchTemplates.html
     */
    describeLaunchTemplates() {
        this.add('ec2:DescribeLaunchTemplates');
        return this;
    }
    /**
     * Grants permission to describe the associations between virtual interface groups and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations.html
     */
    describeLocalGatewayRouteTableVirtualInterfaceGroupAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVirtualInterfaceGroupAssociations');
        return this;
    }
    /**
     * Grants permission to describe an association between VPCs and local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTableVpcAssociations.html
     */
    describeLocalGatewayRouteTableVpcAssociations() {
        this.add('ec2:DescribeLocalGatewayRouteTableVpcAssociations');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayRouteTables.html
     */
    describeLocalGatewayRouteTables() {
        this.add('ec2:DescribeLocalGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interface groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaceGroups.html
     */
    describeLocalGatewayVirtualInterfaceGroups() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaceGroups');
        return this;
    }
    /**
     * Grants permission to describe local gateway virtual interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGatewayVirtualInterfaces.html
     */
    describeLocalGatewayVirtualInterfaces() {
        this.add('ec2:DescribeLocalGatewayVirtualInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more local gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeLocalGateways.html
     */
    describeLocalGateways() {
        this.add('ec2:DescribeLocalGateways');
        return this;
    }
    /**
     * Grants permission to describe Elastic IP addresses that are being moved to the EC2-VPC platform
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeMovingAddresses.html
     */
    describeMovingAddresses() {
        this.add('ec2:DescribeMovingAddresses');
        return this;
    }
    /**
     * Grants permission to describe one or more NAT gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNatGateways.html
     */
    describeNatGateways() {
        this.add('ec2:DescribeNatGateways');
        return this;
    }
    /**
     * Grants permission to describe one or more network ACLs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkAcls.html
     */
    describeNetworkAcls() {
        this.add('ec2:DescribeNetworkAcls');
        return this;
    }
    /**
     * Grants permission to describe a network interface attribute
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaceAttribute.html
     */
    describeNetworkInterfaceAttribute() {
        this.add('ec2:DescribeNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to describe the permissions that are associated with a network interface
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfacePermissions.html
     */
    describeNetworkInterfacePermissions() {
        this.add('ec2:DescribeNetworkInterfacePermissions');
        return this;
    }
    /**
     * Grants permission to describe one or more network interfaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeNetworkInterfaces.html
     */
    describeNetworkInterfaces() {
        this.add('ec2:DescribeNetworkInterfaces');
        return this;
    }
    /**
     * Grants permission to describe one or more placement groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePlacementGroups.html
     */
    describePlacementGroups() {
        this.add('ec2:DescribePlacementGroups');
        return this;
    }
    /**
     * Grants permission to describe available AWS services in a prefix list format
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrefixLists.html
     */
    describePrefixLists() {
        this.add('ec2:DescribePrefixLists');
        return this;
    }
    /**
     * Grants permission to describe the ID format settings for the root user and all IAM roles and IAM users that have explicitly specified a longer ID (17-character ID) preference
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePrincipalIdFormat.html
     */
    describePrincipalIdFormat() {
        this.add('ec2:DescribePrincipalIdFormat');
        return this;
    }
    /**
     * Grants permission to describe one or more IPv4 address pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribePublicIpv4Pools.html
     */
    describePublicIpv4Pools() {
        this.add('ec2:DescribePublicIpv4Pools');
        return this;
    }
    /**
     * Grants permission to describe one or more AWS Regions that are currently available in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRegions.html
     */
    describeRegions() {
        this.add('ec2:DescribeRegions');
        return this;
    }
    /**
     * Grants permission to describe one or more purchased Reserved Instances in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstances.html
     */
    describeReservedInstances() {
        this.add('ec2:DescribeReservedInstances');
        return this;
    }
    /**
     * Grants permission to describe your account's Reserved Instance listings in the Reserved Instance Marketplace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesListings.html
     */
    describeReservedInstancesListings() {
        this.add('ec2:DescribeReservedInstancesListings');
        return this;
    }
    /**
     * Grants permission to describe the modifications made to one or more Reserved Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesModifications.html
     */
    describeReservedInstancesModifications() {
        this.add('ec2:DescribeReservedInstancesModifications');
        return this;
    }
    /**
     * Grants permission to describe the Reserved Instance offerings that are available for purchase
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeReservedInstancesOfferings.html
     */
    describeReservedInstancesOfferings() {
        this.add('ec2:DescribeReservedInstancesOfferings');
        return this;
    }
    /**
     * Grants permission to describe one or more route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeRouteTables.html
     */
    describeRouteTables() {
        this.add('ec2:DescribeRouteTables');
        return this;
    }
    /**
     * Grants permission to find available schedules for Scheduled Instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstanceAvailability.html
     */
    describeScheduledInstanceAvailability() {
        this.add('ec2:DescribeScheduledInstanceAvailability');
        return this;
    }
    /**
     * Grants permission to describe one or more Scheduled Instances in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeScheduledInstances.html
     */
    describeScheduledInstances() {
        this.add('ec2:DescribeScheduledInstances');
        return this;
    }
    /**
     * Grants permission to describe the VPCs on the other side of a VPC peering connection that are referencing specified VPC security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroupReferences.html
     */
    describeSecurityGroupReferences() {
        this.add('ec2:DescribeSecurityGroupReferences');
        return this;
    }
    /**
     * Grants permission to describe one or more security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSecurityGroups.html
     */
    describeSecurityGroups() {
        this.add('ec2:DescribeSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a snapshot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshotAttribute.html
     */
    describeSnapshotAttribute() {
        this.add('ec2:DescribeSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSnapshots.html
     */
    describeSnapshots() {
        this.add('ec2:DescribeSnapshots');
        return this;
    }
    /**
     * Grants permission to describe the data feed for Spot Instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotDatafeedSubscription.html
     */
    describeSpotDatafeedSubscription() {
        this.add('ec2:DescribeSpotDatafeedSubscription');
        return this;
    }
    /**
     * Grants permission to describe the running instances for a Spot Fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetInstances.html
     */
    describeSpotFleetInstances() {
        this.add('ec2:DescribeSpotFleetInstances');
        return this;
    }
    /**
     * Grants permission to describe the events for a Spot Fleet request during a specified time
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequestHistory.html
     */
    describeSpotFleetRequestHistory() {
        this.add('ec2:DescribeSpotFleetRequestHistory');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Fleet requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotFleetRequests.html
     */
    describeSpotFleetRequests() {
        this.add('ec2:DescribeSpotFleetRequests');
        return this;
    }
    /**
     * Grants permission to describe one or more Spot Instance requests
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotInstanceRequests.html
     */
    describeSpotInstanceRequests() {
        this.add('ec2:DescribeSpotInstanceRequests');
        return this;
    }
    /**
     * Grants permission to describe the Spot Instance price history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSpotPriceHistory.html
     */
    describeSpotPriceHistory() {
        this.add('ec2:DescribeSpotPriceHistory');
        return this;
    }
    /**
     * Grants permission to describe the stale security group rules for security groups in a specified VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeStaleSecurityGroups.html
     */
    describeStaleSecurityGroups() {
        this.add('ec2:DescribeStaleSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe one or more subnets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeSubnets.html
     */
    describeSubnets() {
        this.add('ec2:DescribeSubnets');
        return this;
    }
    /**
     * Grants permission to describe one or more tags for an Amazon EC2 resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTags.html
     */
    describeTags() {
        this.add('ec2:DescribeTags');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorFilters.html
     */
    describeTrafficMirrorFilters() {
        this.add('ec2:DescribeTrafficMirrorFilters');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror sessions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorSessions.html
     */
    describeTrafficMirrorSessions() {
        this.add('ec2:DescribeTrafficMirrorSessions');
        return this;
    }
    /**
     * Grants permission to describe one or more traffic mirror targets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTrafficMirrorTargets.html
     */
    describeTrafficMirrorTargets() {
        this.add('ec2:DescribeTrafficMirrorTargets');
        return this;
    }
    /**
     * Grants permission to describe one or more attachments between resources and transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayAttachments.html
     */
    describeTransitGatewayAttachments() {
        this.add('ec2:DescribeTransitGatewayAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway multicast domains
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayMulticastDomains.html
     */
    describeTransitGatewayMulticastDomains() {
        this.add('ec2:DescribeTransitGatewayMulticastDomains');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway peering attachments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayPeeringAttachments.html
     */
    describeTransitGatewayPeeringAttachments() {
        this.add('ec2:DescribeTransitGatewayPeeringAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateway route tables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayRouteTables.html
     */
    describeTransitGatewayRouteTables() {
        this.add('ec2:DescribeTransitGatewayRouteTables');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC attachments on a transit gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGatewayVpcAttachments.html
     */
    describeTransitGatewayVpcAttachments() {
        this.add('ec2:DescribeTransitGatewayVpcAttachments');
        return this;
    }
    /**
     * Grants permission to describe one or more transit gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeTransitGateways.html
     */
    describeTransitGateways() {
        this.add('ec2:DescribeTransitGateways');
        return this;
    }
    /**
     * Grants permission to describe an attribute of an EBS volume
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeAttribute.html
     */
    describeVolumeAttribute() {
        this.add('ec2:DescribeVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to describe the status of one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumeStatus.html
     */
    describeVolumeStatus() {
        this.add('ec2:DescribeVolumeStatus');
        return this;
    }
    /**
     * Grants permission to describe one or more EBS volumes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumes.html
     */
    describeVolumes() {
        this.add('ec2:DescribeVolumes');
        return this;
    }
    /**
     * Grants permission to describe the current modification status of one or more EBS volumes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVolumesModifications.html
     */
    describeVolumesModifications() {
        this.add('ec2:DescribeVolumesModifications');
        return this;
    }
    /**
     * Grants permission to describe an attribute of a VPC
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcAttribute.html
     */
    describeVpcAttribute() {
        this.add('ec2:DescribeVpcAttribute');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLink.html
     */
    describeVpcClassicLink() {
        this.add('ec2:DescribeVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to describe the ClassicLink DNS support status of one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcClassicLinkDnsSupport.html
     */
    describeVpcClassicLinkDnsSupport() {
        this.add('ec2:DescribeVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to describe the connection notifications for VPC endpoints and VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnectionNotifications.html
     */
    describeVpcEndpointConnectionNotifications() {
        this.add('ec2:DescribeVpcEndpointConnectionNotifications');
        return this;
    }
    /**
     * Grants permission to describe the VPC endpoint connections to your VPC endpoint services
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointConnections.html
     */
    describeVpcEndpointConnections() {
        this.add('ec2:DescribeVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to describe VPC endpoint service configurations (your services)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServiceConfigurations.html
     */
    describeVpcEndpointServiceConfigurations() {
        this.add('ec2:DescribeVpcEndpointServiceConfigurations');
        return this;
    }
    /**
     * Grants permission to describe the principals (service consumers) that are permitted to discover your VPC endpoint service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServicePermissions.html
     */
    describeVpcEndpointServicePermissions() {
        this.add('ec2:DescribeVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to describe all supported AWS services that can be specified when creating a VPC endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpointServices.html
     */
    describeVpcEndpointServices() {
        this.add('ec2:DescribeVpcEndpointServices');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcEndpoints.html
     */
    describeVpcEndpoints() {
        this.add('ec2:DescribeVpcEndpoints');
        return this;
    }
    /**
     * Grants permission to describe one or more VPC peering connections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcPeeringConnections.html
     */
    describeVpcPeeringConnections() {
        this.add('ec2:DescribeVpcPeeringConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more VPCs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpcs.html
     */
    describeVpcs() {
        this.add('ec2:DescribeVpcs');
        return this;
    }
    /**
     * Grants permission to describe one or more VPN connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnConnections.html
     */
    describeVpnConnections() {
        this.add('ec2:DescribeVpnConnections');
        return this;
    }
    /**
     * Grants permission to describe one or more virtual private gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DescribeVpnGateways.html
     */
    describeVpnGateways() {
        this.add('ec2:DescribeVpnGateways');
        return this;
    }
    /**
     * Grants permission to unlink (detach) a linked EC2-Classic instance from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachClassicLinkVpc.html
     */
    detachClassicLinkVpc() {
        this.add('ec2:DetachClassicLinkVpc');
        return this;
    }
    /**
     * Grants permission to detach an internet gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachInternetGateway.html
     */
    detachInternetGateway() {
        this.add('ec2:DetachInternetGateway');
        return this;
    }
    /**
     * Grants permission to detach a network interface from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachNetworkInterface.html
     */
    detachNetworkInterface() {
        this.add('ec2:DetachNetworkInterface');
        return this;
    }
    /**
     * Grants permission to detach an EBS volume from an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVolume.html
     */
    detachVolume() {
        this.add('ec2:DetachVolume');
        return this;
    }
    /**
     * Grants permission to detach a virtual private gateway from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DetachVpnGateway.html
     */
    detachVpnGateway() {
        this.add('ec2:DetachVpnGateway');
        return this;
    }
    /**
     * Grants permission to disable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableEbsEncryptionByDefault.html
     */
    disableEbsEncryptionByDefault() {
        this.add('ec2:DisableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to disable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableFastSnapshotRestores.html
     */
    disableFastSnapshotRestores() {
        this.add('ec2:DisableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to disable a resource attachment from propagating routes to the specified propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableTransitGatewayRouteTablePropagation.html
     */
    disableTransitGatewayRouteTablePropagation() {
        this.add('ec2:DisableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to disable a virtual private gateway from propagating routes to a specified route table of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVgwRoutePropagation.html
     */
    disableVgwRoutePropagation() {
        this.add('ec2:DisableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLink.html
     */
    disableVpcClassicLink() {
        this.add('ec2:DisableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to disable ClassicLink DNS support for a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisableVpcClassicLinkDnsSupport.html
     */
    disableVpcClassicLinkDnsSupport() {
        this.add('ec2:DisableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to disassociate an Elastic IP address from an instance or network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateAddress.html
     */
    disassociateAddress() {
        this.add('ec2:DisassociateAddress');
        return this;
    }
    /**
     * Grants permission to disassociate a target network from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateClientVpnTargetNetwork.html
     */
    disassociateClientVpnTargetNetwork() {
        this.add('ec2:DisassociateClientVpnTargetNetwork');
        return this;
    }
    /**
     * Grants permission to disassociate an IAM instance profile from a running or stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateIamInstanceProfile.html
     */
    disassociateIamInstanceProfile() {
        this.add('ec2:DisassociateIamInstanceProfile');
        return this;
    }
    /**
     * Grants permission to disassociate a subnet from a route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateRouteTable.html
     */
    disassociateRouteTable() {
        this.add('ec2:DisassociateRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateSubnetCidrBlock.html
     */
    disassociateSubnetCidrBlock() {
        this.add('ec2:DisassociateSubnetCidrBlock');
        return this;
    }
    /**
     * Grants permission to disassociate one or more subnets from a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayMulticastDomain.html
     */
    disassociateTransitGatewayMulticastDomain() {
        this.add('ec2:DisassociateTransitGatewayMulticastDomain');
        return this;
    }
    /**
     * Grants permission to disassociate a resource attachment from a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateTransitGatewayRouteTable.html
     */
    disassociateTransitGatewayRouteTable() {
        this.add('ec2:DisassociateTransitGatewayRouteTable');
        return this;
    }
    /**
     * Grants permission to disassociate a CIDR block from a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_DisassociateVpcCidrBlock.html
     */
    disassociateVpcCidrBlock() {
        this.add('ec2:DisassociateVpcCidrBlock');
        return this;
    }
    /**
     * Grants permission to enable EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableEbsEncryptionByDefault.html
     */
    enableEbsEncryptionByDefault() {
        this.add('ec2:EnableEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to enable fast snapshot restores for one or more snapshots in specified Availability Zones
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableFastSnapshotRestores.html
     */
    enableFastSnapshotRestores() {
        this.add('ec2:EnableFastSnapshotRestores');
        return this;
    }
    /**
     * Grants permission to enable an attachment to propagate routes to a propagation route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableTransitGatewayRouteTablePropagation.html
     */
    enableTransitGatewayRouteTablePropagation() {
        this.add('ec2:EnableTransitGatewayRouteTablePropagation');
        return this;
    }
    /**
     * Grants permission to enable a virtual private gateway to propagate routes to a VPC route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVgwRoutePropagation.html
     */
    enableVgwRoutePropagation() {
        this.add('ec2:EnableVgwRoutePropagation');
        return this;
    }
    /**
     * Grants permission to enable I/O operations for a volume that had I/O operations disabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVolumeIO.html
     */
    enableVolumeIO() {
        this.add('ec2:EnableVolumeIO');
        return this;
    }
    /**
     * Grants permission to enable a VPC for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLink.html
     */
    enableVpcClassicLink() {
        this.add('ec2:EnableVpcClassicLink');
        return this;
    }
    /**
     * Grants permission to enable a VPC to support DNS hostname resolution for ClassicLink
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_EnableVpcClassicLinkDnsSupport.html
     */
    enableVpcClassicLinkDnsSupport() {
        this.add('ec2:EnableVpcClassicLinkDnsSupport');
        return this;
    }
    /**
     * Grants permission to download the client certificate revocation list for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientCertificateRevocationList.html
     */
    exportClientVpnClientCertificateRevocationList() {
        this.add('ec2:ExportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to download the contents of the Client VPN endpoint configuration file for a Client VPN endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportClientVpnClientConfiguration.html
     */
    exportClientVpnClientConfiguration() {
        this.add('ec2:ExportClientVpnClientConfiguration');
        return this;
    }
    /**
     * Grants permission to export an Amazon Machine Image (AMI) to a VM file
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportImage.html
     */
    exportImage() {
        this.add('ec2:ExportImage');
        return this;
    }
    /**
     * Grants permission to export routes from a transit gateway route table to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ExportTransitGatewayRoutes.html
     */
    exportTransitGatewayRoutes() {
        this.add('ec2:ExportTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to get usage information about a Capacity Reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCapacityReservationUsage.html
     */
    getCapacityReservationUsage() {
        this.add('ec2:GetCapacityReservationUsage');
        return this;
    }
    /**
     * Grants permission to describe the allocations from the specified customer-owned address pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetCoipPoolUsage.html
     */
    getCoipPoolUsage() {
        this.add('ec2:GetCoipPoolUsage');
        return this;
    }
    /**
     * Grants permission to get the console output for an instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleOutput.html
     */
    getConsoleOutput() {
        this.add('ec2:GetConsoleOutput');
        return this;
    }
    /**
     * Grants permission to retrieve a JPG-format screenshot of a running instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetConsoleScreenshot.html
     */
    getConsoleScreenshot() {
        this.add('ec2:GetConsoleScreenshot');
        return this;
    }
    /**
     * Grants permission to get the default credit option for CPU usage of a burstable performance instance family
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetDefaultCreditSpecification.html
     */
    getDefaultCreditSpecification() {
        this.add('ec2:GetDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to get the ID of the default customer master key (CMK) for EBS encryption by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsDefaultKmsKeyId.html
     */
    getEbsDefaultKmsKeyId() {
        this.add('ec2:GetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to describe whether EBS encryption by default is enabled for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetEbsEncryptionByDefault.html
     */
    getEbsEncryptionByDefault() {
        this.add('ec2:GetEbsEncryptionByDefault');
        return this;
    }
    /**
     * Grants permission to preview a reservation purchase with configurations that match those of a Dedicated Host
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetHostReservationPurchasePreview.html
     */
    getHostReservationPurchasePreview() {
        this.add('ec2:GetHostReservationPurchasePreview');
        return this;
    }
    /**
     * Grants permission to get the configuration data of the specified instance for use with a new launch template or launch template version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetLaunchTemplateData.html
     */
    getLaunchTemplateData() {
        this.add('ec2:GetLaunchTemplateData');
        return this;
    }
    /**
     * Grants permission to retrieve the encrypted administrator password for a running Windows instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetPasswordData.html
     */
    getPasswordData() {
        this.add('ec2:GetPasswordData');
        return this;
    }
    /**
     * Grants permission to return a quote and exchange information for exchanging one or more Convertible Reserved Instances for a new Convertible Reserved Instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetReservedInstancesExchangeQuote.html
     */
    getReservedInstancesExchangeQuote() {
        this.add('ec2:GetReservedInstancesExchangeQuote');
        return this;
    }
    /**
     * Grants permission to list the route tables to which a resource attachment propagates routes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayAttachmentPropagations.html
     */
    getTransitGatewayAttachmentPropagations() {
        this.add('ec2:GetTransitGatewayAttachmentPropagations');
        return this;
    }
    /**
     * Grants permission to get information about the associations for a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayMulticastDomainAssociations.html
     */
    getTransitGatewayMulticastDomainAssociations() {
        this.add('ec2:GetTransitGatewayMulticastDomainAssociations');
        return this;
    }
    /**
     * Grants permission to get information about associations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTableAssociations.html
     */
    getTransitGatewayRouteTableAssociations() {
        this.add('ec2:GetTransitGatewayRouteTableAssociations');
        return this;
    }
    /**
     * Grants permission to get information about the route table propagations for a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetTransitGatewayRouteTablePropagations.html
     */
    getTransitGatewayRouteTablePropagations() {
        this.add('ec2:GetTransitGatewayRouteTablePropagations');
        return this;
    }
    /**
     * Grants permission to upload a client certificate revocation list to a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportClientVpnClientCertificateRevocationList.html
     */
    importClientVpnClientCertificateRevocationList() {
        this.add('ec2:ImportClientVpnClientCertificateRevocationList');
        return this;
    }
    /**
     * Grants permission to import single or multi-volume disk images or EBS snapshots into an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportImage.html
     */
    importImage() {
        this.add('ec2:ImportImage');
        return this;
    }
    /**
     * Grants permission to create an import instance task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportInstance.html
     */
    importInstance() {
        this.add('ec2:ImportInstance');
        return this;
    }
    /**
     * Grants permission to import a public key from an RSA key pair that was created with a third-party tool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportKeyPair.html
     */
    importKeyPair() {
        this.add('ec2:ImportKeyPair');
        return this;
    }
    /**
     * Grants permission to import a disk into an EBS snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportSnapshot.html
     */
    importSnapshot() {
        this.add('ec2:ImportSnapshot');
        return this;
    }
    /**
     * Grants permission to create an import volume task using metadata from a disk image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ImportVolume.html
     */
    importVolume() {
        this.add('ec2:ImportVolume');
        return this;
    }
    /**
     * Grants permission to modify a Capacity Reservation's capacity and the conditions under which it is to be released
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyCapacityReservation.html
     */
    modifyCapacityReservation() {
        this.add('ec2:ModifyCapacityReservation');
        return this;
    }
    /**
     * Grants permission to modify a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyClientVpnEndpoint.html
     */
    modifyClientVpnEndpoint() {
        this.add('ec2:ModifyClientVpnEndpoint');
        return this;
    }
    /**
     * Grants permission to change the account level default credit option for CPU usage of burstable performance instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyDefaultCreditSpecification.html
     */
    modifyDefaultCreditSpecification() {
        this.add('ec2:ModifyDefaultCreditSpecification');
        return this;
    }
    /**
     * Grants permission to change the default customer master key (CMK) for EBS encryption by default for your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyEbsDefaultKmsKeyId.html
     */
    modifyEbsDefaultKmsKeyId() {
        this.add('ec2:ModifyEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to modify an EC2 Fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFleet.html
     */
    modifyFleet() {
        this.add('ec2:ModifyFleet');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon FPGA Image (AFI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyFpgaImageAttribute.html
     */
    modifyFpgaImageAttribute() {
        this.add('ec2:ModifyFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyHosts.html
     */
    modifyHosts() {
        this.add('ec2:ModifyHosts');
        return this;
    }
    /**
     * Grants permission to modify the ID format for a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdFormat.html
     */
    modifyIdFormat() {
        this.add('ec2:ModifyIdFormat');
        return this;
    }
    /**
     * Grants permission to modify the ID format of a resource for a specific principal in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyIdentityIdFormat.html
     */
    modifyIdentityIdFormat() {
        this.add('ec2:ModifyIdentityIdFormat');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyImageAttribute.html
     */
    modifyImageAttribute() {
        this.add('ec2:ModifyImageAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceAttribute.html
     */
    modifyInstanceAttribute() {
        this.add('ec2:ModifyInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to modify the Capacity Reservation settings for a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCapacityReservationAttributes.html
     */
    modifyInstanceCapacityReservationAttributes() {
        this.add('ec2:ModifyInstanceCapacityReservationAttributes');
        return this;
    }
    /**
     * Grants permission to modify the credit option for CPU usage on an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceCreditSpecification.html
     */
    modifyInstanceCreditSpecification() {
        this.add('ec2:ModifyInstanceCreditSpecification');
        return this;
    }
    /**
     * Grants permission to modify the start time for a scheduled EC2 instance event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceEventStartTime.html
     */
    modifyInstanceEventStartTime() {
        this.add('ec2:ModifyInstanceEventStartTime');
        return this;
    }
    /**
     * Grants permission to modify the metadata options for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstanceMetadataOptions.html
     */
    modifyInstanceMetadataOptions() {
        this.add('ec2:ModifyInstanceMetadataOptions');
        return this;
    }
    /**
     * Grants permission to modify the placement attributes for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyInstancePlacement.html
     */
    modifyInstancePlacement() {
        this.add('ec2:ModifyInstancePlacement');
        return this;
    }
    /**
     * Grants permission to modify a launch template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyLaunchTemplate.html
     */
    modifyLaunchTemplate() {
        this.add('ec2:ModifyLaunchTemplate');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyNetworkInterfaceAttribute.html
     */
    modifyNetworkInterfaceAttribute() {
        this.add('ec2:ModifyNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to modify attributes of one or more Reserved Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyReservedInstances.html
     */
    modifyReservedInstances() {
        this.add('ec2:ModifyReservedInstances');
        return this;
    }
    /**
     * Grants permission to add or remove permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySnapshotAttribute.html
     */
    modifySnapshotAttribute() {
        this.add('ec2:ModifySnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to modify a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySpotFleetRequest.html
     */
    modifySpotFleetRequest() {
        this.add('ec2:ModifySpotFleetRequest');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifySubnetAttribute.html
     */
    modifySubnetAttribute() {
        this.add('ec2:ModifySubnetAttribute');
        return this;
    }
    /**
     * Grants permission to allow or restrict mirroring network services
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterNetworkServices.html
     */
    modifyTrafficMirrorFilterNetworkServices() {
        this.add('ec2:ModifyTrafficMirrorFilterNetworkServices');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorFilterRule.html
     */
    modifyTrafficMirrorFilterRule() {
        this.add('ec2:ModifyTrafficMirrorFilterRule');
        return this;
    }
    /**
     * Grants permission to modify a traffic mirror session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTrafficMirrorSession.html
     */
    modifyTrafficMirrorSession() {
        this.add('ec2:ModifyTrafficMirrorSession');
        return this;
    }
    /**
     * Grants permission to modify a VPC attachment on a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyTransitGatewayVpcAttachment.html
     */
    modifyTransitGatewayVpcAttachment() {
        this.add('ec2:ModifyTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to modify the parameters of an EBS volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolume.html
     */
    modifyVolume() {
        this.add('ec2:ModifyVolume');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVolumeAttribute.html
     */
    modifyVolumeAttribute() {
        this.add('ec2:ModifyVolumeAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcAttribute.html
     */
    modifyVpcAttribute() {
        this.add('ec2:ModifyVpcAttribute');
        return this;
    }
    /**
     * Grants permission to modify an attribute of a VPC endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpoint.html
     */
    modifyVpcEndpoint() {
        this.add('ec2:ModifyVpcEndpoint');
        return this;
    }
    /**
     * Grants permission to modify a connection notification for a VPC endpoint or VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointConnectionNotification.html
     */
    modifyVpcEndpointConnectionNotification() {
        this.add('ec2:ModifyVpcEndpointConnectionNotification');
        return this;
    }
    /**
     * Grants permission to modify the attributes of a VPC endpoint service configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServiceConfiguration.html
     */
    modifyVpcEndpointServiceConfiguration() {
        this.add('ec2:ModifyVpcEndpointServiceConfiguration');
        return this;
    }
    /**
     * Grants permission to modify the permissions for a VPC endpoint service
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcEndpointServicePermissions.html
     */
    modifyVpcEndpointServicePermissions() {
        this.add('ec2:ModifyVpcEndpointServicePermissions');
        return this;
    }
    /**
     * Grants permission to modify the VPC peering connection options on one side of a VPC peering connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcPeeringConnectionOptions.html
     */
    modifyVpcPeeringConnectionOptions() {
        this.add('ec2:ModifyVpcPeeringConnectionOptions');
        return this;
    }
    /**
     * Grants permission to modify the instance tenancy attribute of a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpcTenancy.html
     */
    modifyVpcTenancy() {
        this.add('ec2:ModifyVpcTenancy');
        return this;
    }
    /**
     * Grants permission to modify the target gateway of a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnConnection.html
     */
    modifyVpnConnection() {
        this.add('ec2:ModifyVpnConnection');
        return this;
    }
    /**
     * Grants permission to modify the certificate for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelCertificate
     */
    modifyVpnTunnelCertificate() {
        this.add('ec2:ModifyVpnTunnelCertificate');
        return this;
    }
    /**
     * Grants permission to modify the options for a Site-to-Site VPN connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ModifyVpnTunnelOptions.html
     */
    modifyVpnTunnelOptions() {
        this.add('ec2:ModifyVpnTunnelOptions');
        return this;
    }
    /**
     * Grants permission to enable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MonitorInstances.html
     */
    monitorInstances() {
        this.add('ec2:MonitorInstances');
        return this;
    }
    /**
     * Grants permission to move an Elastic IP address from the EC2-Classic platform to the EC2-VPC platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_MoveAddressToVpc.html
     */
    moveAddressToVpc() {
        this.add('ec2:MoveAddressToVpc');
        return this;
    }
    /**
     * Grants permission to provision an address range for use in AWS through bring your own IP addresses (BYOIP), and to create a corresponding address pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ProvisionByoipCidr.html
     */
    provisionByoipCidr() {
        this.add('ec2:ProvisionByoipCidr');
        return this;
    }
    /**
     * Grants permission to purchase a reservation with configurations that match those of a Dedicated Host
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseHostReservation.html
     */
    purchaseHostReservation() {
        this.add('ec2:PurchaseHostReservation');
        return this;
    }
    /**
     * Grants permission to purchase a Reserved Instance offering
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseReservedInstancesOffering.html
     */
    purchaseReservedInstancesOffering() {
        this.add('ec2:PurchaseReservedInstancesOffering');
        return this;
    }
    /**
     * Grants permission to purchase one or more Scheduled Instances with a specified schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_PurchaseScheduledInstances.html
     */
    purchaseScheduledInstances() {
        this.add('ec2:PurchaseScheduledInstances');
        return this;
    }
    /**
     * Grants permission to request a reboot of one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RebootInstances.html
     */
    rebootInstances() {
        this.add('ec2:RebootInstances');
        return this;
    }
    /**
     * Grants permission to register an Amazon Machine Image (AMI)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterImage.html
     */
    registerImage() {
        this.add('ec2:RegisterImage');
        return this;
    }
    /**
     * Grants permission to add tags to the set of tags to include in notifications about scheduled events for your instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterInstanceEventNotificationAttributes.html
     */
    registerInstanceEventNotificationAttributes() {
        this.add('ec2:RegisterInstanceEventNotificationAttributes');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a member of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupMembers.html
     */
    registerTransitGatewayMulticastGroupMembers() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupMembers');
        return this;
    }
    /**
     * Grants permission to register one or more network interfaces as a source of a group IP address in a transit gateway multicast domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RegisterTransitGatewayMulticastGroupSources.html
     */
    registerTransitGatewayMulticastGroupSources() {
        this.add('ec2:RegisterTransitGatewayMulticastGroupSources');
        return this;
    }
    /**
     * Grants permission to reject a transit gateway peering attachment request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayPeeringAttachment.html
     */
    rejectTransitGatewayPeeringAttachment() {
        this.add('ec2:RejectTransitGatewayPeeringAttachment');
        return this;
    }
    /**
     * Grants permission to reject a request to attach a VPC to a transit gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectTransitGatewayVpcAttachment.html
     */
    rejectTransitGatewayVpcAttachment() {
        this.add('ec2:RejectTransitGatewayVpcAttachment');
        return this;
    }
    /**
     * Grants permission to reject one or more VPC endpoint connection requests to a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcEndpointConnections.html
     */
    rejectVpcEndpointConnections() {
        this.add('ec2:RejectVpcEndpointConnections');
        return this;
    }
    /**
     * Grants permission to reject a VPC peering connection request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RejectVpcPeeringConnection.html
     */
    rejectVpcPeeringConnection() {
        this.add('ec2:RejectVpcPeeringConnection');
        return this;
    }
    /**
     * Grants permission to release an Elastic IP address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseAddress.html
     */
    releaseAddress() {
        this.add('ec2:ReleaseAddress');
        return this;
    }
    /**
     * Grants permission to release one or more On-Demand Dedicated Hosts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReleaseHosts.html
     */
    releaseHosts() {
        this.add('ec2:ReleaseHosts');
        return this;
    }
    /**
     * Grants permission to replace an IAM instance profile for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceIamInstanceProfileAssociation.html
     */
    replaceIamInstanceProfileAssociation() {
        this.add('ec2:ReplaceIamInstanceProfileAssociation');
        return this;
    }
    /**
     * Grants permission to change which network ACL a subnet is associated with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclAssociation.html
     */
    replaceNetworkAclAssociation() {
        this.add('ec2:ReplaceNetworkAclAssociation');
        return this;
    }
    /**
     * Grants permission to replace an entry (rule) in a network ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceNetworkAclEntry.html
     */
    replaceNetworkAclEntry() {
        this.add('ec2:ReplaceNetworkAclEntry');
        return this;
    }
    /**
     * Grants permission to replace a route within a route table in a VPC
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRoute.html
     */
    replaceRoute() {
        this.add('ec2:ReplaceRoute');
        return this;
    }
    /**
     * Grants permission to change the route table that is associated with a subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceRouteTableAssociation.html
     */
    replaceRouteTableAssociation() {
        this.add('ec2:ReplaceRouteTableAssociation');
        return this;
    }
    /**
     * Grants permission to replace a route in a transit gateway route table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReplaceTransitGatewayRoute.html
     */
    replaceTransitGatewayRoute() {
        this.add('ec2:ReplaceTransitGatewayRoute');
        return this;
    }
    /**
     * Grants permission to submit feedback about the status of an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ReportInstanceStatus.html
     */
    reportInstanceStatus() {
        this.add('ec2:ReportInstanceStatus');
        return this;
    }
    /**
     * Grants permission to create a Spot Fleet request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html
     */
    requestSpotFleet() {
        this.add('ec2:RequestSpotFleet');
        return this;
    }
    /**
     * Grants permission to create a Spot Instance request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html
     */
    requestSpotInstances() {
        this.add('ec2:RequestSpotInstances');
        return this;
    }
    /**
     * Grants permission to reset the default customer master key (CMK) for EBS encryption for your account to use the AWS-managed CMK for EBS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetEbsDefaultKmsKeyId.html
     */
    resetEbsDefaultKmsKeyId() {
        this.add('ec2:ResetEbsDefaultKmsKeyId');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon FPGA Image (AFI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetFpgaImageAttribute.html
     */
    resetFpgaImageAttribute() {
        this.add('ec2:ResetFpgaImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an Amazon Machine Image (AMI) to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetImageAttribute.html
     */
    resetImageAttribute() {
        this.add('ec2:ResetImageAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of an instance to its default value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetInstanceAttribute.html
     */
    resetInstanceAttribute() {
        this.add('ec2:ResetInstanceAttribute');
        return this;
    }
    /**
     * Grants permission to reset an attribute of a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetNetworkInterfaceAttribute.html
     */
    resetNetworkInterfaceAttribute() {
        this.add('ec2:ResetNetworkInterfaceAttribute');
        return this;
    }
    /**
     * Grants permission to reset permission settings for a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_ResetSnapshotAttribute.html
     */
    resetSnapshotAttribute() {
        this.add('ec2:ResetSnapshotAttribute');
        return this;
    }
    /**
     * Grants permission to restore an Elastic IP address that was previously moved to the EC2-VPC platform back to the EC2-Classic platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RestoreAddressToClassic.html
     */
    restoreAddressToClassic() {
        this.add('ec2:RestoreAddressToClassic');
        return this;
    }
    /**
     * Grants permission to remove an inbound authorization rule from a Client VPN endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeClientVpnIngress.html
     */
    revokeClientVpnIngress() {
        this.add('ec2:RevokeClientVpnIngress');
        return this;
    }
    /**
     * Grants permission to remove one or more outbound rules from a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupEgress.html
     */
    revokeSecurityGroupEgress() {
        this.add('ec2:RevokeSecurityGroupEgress');
        return this;
    }
    /**
     * Grants permission to remove one or more inbound rules from a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RevokeSecurityGroupIngress.html
     */
    revokeSecurityGroupIngress() {
        this.add('ec2:RevokeSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to launch one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html
     */
    runInstances() {
        this.add('ec2:RunInstances');
        return this;
    }
    /**
     * Grants permission to launch one or more Scheduled Instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunScheduledInstances.html
     */
    runScheduledInstances() {
        this.add('ec2:RunScheduledInstances');
        return this;
    }
    /**
     * Grants permission to search for routes in a local gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchLocalGatewayRoutes.html
     */
    searchLocalGatewayRoutes() {
        this.add('ec2:SearchLocalGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to search for groups, sources, and members in a transit gateway multicast domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayMulticastGroups.html
     */
    searchTransitGatewayMulticastGroups() {
        this.add('ec2:SearchTransitGatewayMulticastGroups');
        return this;
    }
    /**
     * Grants permission to search for routes in a transit gateway route table
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SearchTransitGatewayRoutes.html
     */
    searchTransitGatewayRoutes() {
        this.add('ec2:SearchTransitGatewayRoutes');
        return this;
    }
    /**
     * Grants permission to send a diagnostic interrupt to an Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_SendDiagnosticInterrupt.html
     */
    sendDiagnosticInterrupt() {
        this.add('ec2:SendDiagnosticInterrupt');
        return this;
    }
    /**
     * Grants permission to start a stopped instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartInstances.html
     */
    startInstances() {
        this.add('ec2:StartInstances');
        return this;
    }
    /**
     * Grants permission to start the private DNS verification process for a VPC endpoint service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StartVpcEndpointServicePrivateDnsVerification.html
     */
    startVpcEndpointServicePrivateDnsVerification() {
        this.add('ec2:StartVpcEndpointServicePrivateDnsVerification');
        return this;
    }
    /**
     * Grants permission to stop an Amazon EBS-backed instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_StopInstances.html
     */
    stopInstances() {
        this.add('ec2:StopInstances');
        return this;
    }
    /**
     * Grants permission to terminate active Client VPN endpoint connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateClientVpnConnections.html
     */
    terminateClientVpnConnections() {
        this.add('ec2:TerminateClientVpnConnections');
        return this;
    }
    /**
     * Grants permission to shut down one or more instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_TerminateInstances.html
     */
    terminateInstances() {
        this.add('ec2:TerminateInstances');
        return this;
    }
    /**
     * Grants permission to unassign one or more IPv6 addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignIpv6Addresses.html
     */
    unassignIpv6Addresses() {
        this.add('ec2:UnassignIpv6Addresses');
        return this;
    }
    /**
     * Grants permission to unassign one or more secondary private IP addresses from a network interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnassignPrivateIpAddresses.html
     */
    unassignPrivateIpAddresses() {
        this.add('ec2:UnassignPrivateIpAddresses');
        return this;
    }
    /**
     * Grants permission to disable detailed monitoring for a running instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UnmonitorInstances.html
     */
    unmonitorInstances() {
        this.add('ec2:UnmonitorInstances');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more outbound rules in a VPC security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsEgress.html
     */
    updateSecurityGroupRuleDescriptionsEgress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsEgress');
        return this;
    }
    /**
     * Grants permission to update descriptions for one or more inbound rules in a security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_UpdateSecurityGroupRuleDescriptionsIngress.html
     */
    updateSecurityGroupRuleDescriptionsIngress() {
        this.add('ec2:UpdateSecurityGroupRuleDescriptionsIngress');
        return this;
    }
    /**
     * Grants permission to stop advertising an address range that was provisioned for use in AWS through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_WithdrawByoipCidr.html
     */
    withdrawByoipCidr() {
        this.add('ec2:WithdrawByoipCidr');
        return this;
    }
    /**
     * Adds a resource of type capacity-reservation to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-capacity-reservations.html
     *
     * @param capacityReservationId - Identifier for the capacityReservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onCapacityReservation(capacityReservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:capacity-reservation/${CapacityReservationId}';
        arn = arn.replace('${CapacityReservationId}', capacityReservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type client-vpn-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/what-is.html
     *
     * @param clientVpnEndpointId - Identifier for the clientVpnEndpointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onClientVpnEndpoint(clientVpnEndpointId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:client-vpn-endpoint/${ClientVpnEndpointId}';
        arn = arn.replace('${ClientVpnEndpointId}', clientVpnEndpointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type customer-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param customerGatewayId - Identifier for the customerGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onCustomerGateway(customerGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:customer-gateway/${CustomerGatewayId}';
        arn = arn.replace('${CustomerGatewayId}', customerGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dedicated-host to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/dedicated-hosts-overview.html
     *
     * @param hostId - Identifier for the hostId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:AutoPlacement
     *  - ec2:AvailabilityZone
     *  - ec2:HostRecovery
     *  - ec2:InstanceType
     *  - ec2:Quantity
     *  - ec2:ResourceTag/${TagKey}
     */
    onDedicatedHost(hostId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dedicated-host/${HostId}';
        arn = arn.replace('${HostId}', hostId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dhcp-options to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_DHCP_Options.html
     *
     * @param dhcpOptionsId - Identifier for the dhcpOptionsId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onDhcpOptions(dhcpOptionsId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:dhcp-options/${DhcpOptionsId}';
        arn = arn.replace('${DhcpOptionsId}', dhcpOptionsId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-gpu to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/elastic-gpus.html
     *
     * @param elasticGpuId - Identifier for the elasticGpuId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ec2:ElasticGpuType
     */
    onElasticGpu(elasticGpuId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:elastic-gpu/${ElasticGpuId}';
        arn = arn.replace('${ElasticGpuId}', elasticGpuId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type elastic-inference to the statement
     *
     * https://docs.aws.amazon.com/elastic-inference/latest/developerguide/what-is-ei.html
     *
     * @param elasticInferenceAcceleratorId - Identifier for the elasticInferenceAcceleratorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onElasticInference(elasticInferenceAcceleratorId, account, region, partition) {
        var arn = 'arn:${Partition}:elastic-inference:${Region}:${Account}:elastic-inference-accelerator/${ElasticInferenceAcceleratorId}';
        arn = arn.replace('${ElasticInferenceAcceleratorId}', elasticInferenceAcceleratorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type fpga-image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param fpgaImageId - Identifier for the fpgaImageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Owner
     *  - ec2:Public
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onFpgaImage(fpgaImageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::fpga-image/${FpgaImageId}';
        arn = arn.replace('${FpgaImageId}', fpgaImageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type image to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIs.html
     *
     * @param imageId - Identifier for the imageId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:ImageType
     *  - ec2:Owner
     *  - ec2:Public
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:RootDeviceType
     */
    onImage(imageId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::image/${ImageId}';
        arn = arn.replace('${ImageId}', imageId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Instances.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:AvailabilityZone
     *  - ec2:EbsOptimized
     *  - ec2:InstanceProfile
     *  - ec2:InstanceType
     *  - ec2:PlacementGroup
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:RootDeviceType
     *  - ec2:Tenancy
     */
    onInstance(instanceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type internet-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Internet_Gateway.html
     *
     * @param internetGatewayId - Identifier for the internetGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onInternetGateway(internetGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:internet-gateway/${InternetGatewayId}';
        arn = arn.replace('${InternetGatewayId}', internetGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type key-pair to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-key-pairs.html
     *
     * @param keyPairName - Identifier for the keyPairName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ec2:Region
     */
    onKeyPair(keyPairName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:key-pair/${KeyPairName}';
        arn = arn.replace('${KeyPairName}', keyPairName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type launch-template to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html
     *
     * @param launchTemplateId - Identifier for the launchTemplateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onLaunchTemplate(launchTemplateId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:launch-template/${LaunchTemplateId}';
        arn = arn.replace('${LaunchTemplateId}', launchTemplateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayId - Identifier for the localGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onLocalGateway(localGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway/${LocalGatewayId}';
        arn = arn.replace('${LocalGatewayId}', localGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableId - Identifier for the localGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onLocalGatewayRouteTable(localGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table/${LocalGatewayRouteTableId}';
        arn = arn.replace('${LocalGatewayRouteTableId}', localGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-virtual-interface-group-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVirtualInterfaceGroupAssociationId - Identifier for the localGatewayRouteTableVirtualInterfaceGroupAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onLocalGatewayRouteTableVirtualInterfaceGroupAssociation(localGatewayRouteTableVirtualInterfaceGroupAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-virtual-interface-group-association/${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVirtualInterfaceGroupAssociationId}', localGatewayRouteTableVirtualInterfaceGroupAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-route-table-vpc-association to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayRouteTableVpcAssociationId - Identifier for the localGatewayRouteTableVpcAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onLocalGatewayRouteTableVpcAssociation(localGatewayRouteTableVpcAssociationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-route-table-vpc-association/${LocalGatewayRouteTableVpcAssociationId}';
        arn = arn.replace('${LocalGatewayRouteTableVpcAssociationId}', localGatewayRouteTableVpcAssociationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceId - Identifier for the localGatewayVirtualInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onLocalGatewayVirtualInterface(localGatewayVirtualInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface/${LocalGatewayVirtualInterfaceId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceId}', localGatewayVirtualInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type local-gateway-virtual-interface-group to the statement
     *
     * https://docs.aws.amazon.com/outposts/latest/userguide/outposts-local-gateways.html
     *
     * @param localGatewayVirtualInterfaceGroupId - Identifier for the localGatewayVirtualInterfaceGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onLocalGatewayVirtualInterfaceGroup(localGatewayVirtualInterfaceGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:local-gateway-virtual-interface-group/${LocalGatewayVirtualInterfaceGroupId}';
        arn = arn.replace('${LocalGatewayVirtualInterfaceGroupId}', localGatewayVirtualInterfaceGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-acl to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/vpc-network-acls.html
     *
     * @param naclId - Identifier for the naclId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:Vpc
     */
    onNetworkAcl(naclId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-acl/${NaclId}';
        arn = arn.replace('${NaclId}', naclId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type network-interface to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-eni.html
     *
     * @param networkInterfaceId - Identifier for the networkInterfaceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:AssociatePublicIpAddress
     *  - ec2:AuthorizedService
     *  - ec2:AvailabilityZone
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:Subnet
     *  - ec2:Vpc
     */
    onNetworkInterface(networkInterfaceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:network-interface/${NetworkInterfaceId}';
        arn = arn.replace('${NetworkInterfaceId}', networkInterfaceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type placement-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/placement-groups.html
     *
     * @param placementGroupName - Identifier for the placementGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ec2:PlacementGroupStrategy
     *  - ec2:Region
     */
    onPlacementGroup(placementGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:placement-group/${PlacementGroupName}';
        arn = arn.replace('${PlacementGroupName}', placementGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reserved-instances to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html
     *
     * @param reservationId - Identifier for the reservationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:AvailabilityZone
     *  - ec2:InstanceType
     *  - ec2:Region
     *  - ec2:ReservedInstancesOfferingType
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:Tenancy
     */
    onReservedInstances(reservationId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:reserved-instances/${ReservationId}';
        arn = arn.replace('${ReservationId}', reservationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Route_Tables.html
     *
     * @param routeTableId - Identifier for the routeTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:Vpc
     */
    onRouteTable(routeTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:route-table/${RouteTableId}';
        arn = arn.replace('${RouteTableId}', routeTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type security-group to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-security-groups.html
     *
     * @param securityGroupId - Identifier for the securityGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:Vpc
     */
    onSecurityGroup(securityGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:security-group/${SecurityGroupId}';
        arn = arn.replace('${SecurityGroupId}', securityGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSSnapshots.html
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Owner
     *  - ec2:ParentVolume
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:SnapshotTime
     *  - ec2:VolumeSize
     */
    onSnapshot(snapshotId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}';
        arn = arn.replace('${SnapshotId}', snapshotId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type spot-instance-request to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-spot-instances.html
     *
     * @param spotInstanceRequestId - Identifier for the spotInstanceRequestId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onSpotInstanceRequest(spotInstanceRequestId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:spot-instances-request/${SpotInstanceRequestId}';
        arn = arn.replace('${SpotInstanceRequestId}', spotInstanceRequestId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnet to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param subnetId - Identifier for the subnetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:AvailabilityZone
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:Vpc
     */
    onSubnet(subnetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:subnet/${SubnetId}';
        arn = arn.replace('${SubnetId}', subnetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-session to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-session.html
     *
     * @param trafficMirrorSessionId - Identifier for the trafficMirrorSessionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onTrafficMirrorSession(trafficMirrorSessionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-session/${TrafficMirrorSessionId}';
        arn = arn.replace('${TrafficMirrorSessionId}', trafficMirrorSessionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-target to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-target.html
     *
     * @param trafficMirrorTargetId - Identifier for the trafficMirrorTargetId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onTrafficMirrorTarget(trafficMirrorTargetId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-target/${TrafficMirrorTargetId}';
        arn = arn.replace('${TrafficMirrorTargetId}', trafficMirrorTargetId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterId - Identifier for the trafficMirrorFilterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onTrafficMirrorFilter(trafficMirrorFilterId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter/${TrafficMirrorFilterId}';
        arn = arn.replace('${TrafficMirrorFilterId}', trafficMirrorFilterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type traffic-mirror-filter-rule to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/mirroring/traffic-mirroring-filter.html
     *
     * @param trafficMirrorFilterRuleId - Identifier for the trafficMirrorFilterRuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ec2:Region
     */
    onTrafficMirrorFilterRule(trafficMirrorFilterRuleId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:traffic-mirror-filter-rule/${TrafficMirrorFilterRuleId}';
        arn = arn.replace('${TrafficMirrorFilterRuleId}', trafficMirrorFilterRuleId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-attachment to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayAttachmentId - Identifier for the transitGatewayAttachmentId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onTransitGatewayAttachment(transitGatewayAttachmentId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-attachment/${TransitGatewayAttachmentId}';
        arn = arn.replace('${TransitGatewayAttachmentId}', transitGatewayAttachmentId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-multicast-domain to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/tgw-multicast-overview.html
     *
     * @param transitGatewayMulticastDomainId - Identifier for the transitGatewayMulticastDomainId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onTransitGatewayMulticastDomain(transitGatewayMulticastDomainId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-multicast-domain/${TransitGatewayMulticastDomainId}';
        arn = arn.replace('${TransitGatewayMulticastDomainId}', transitGatewayMulticastDomainId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway-route-table to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayRouteTableId - Identifier for the transitGatewayRouteTableId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onTransitGatewayRouteTable(transitGatewayRouteTableId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway-route-table/${TransitGatewayRouteTableId}';
        arn = arn.replace('${TransitGatewayRouteTableId}', transitGatewayRouteTableId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type transit-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/tgw/how-transit-gateways-work.html
     *
     * @param transitGatewayId - Identifier for the transitGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onTransitGateway(transitGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:transit-gateway/${TransitGatewayId}';
        arn = arn.replace('${TransitGatewayId}', transitGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes.html
     *
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:AvailabilityZone
     *  - ec2:Encrypted
     *  - ec2:ParentSnapshot
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:VolumeIops
     *  - ec2:VolumeSize
     *  - ec2:VolumeType
     */
    onVolume(volumeId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:volume/${VolumeId}';
        arn = arn.replace('${VolumeId}', volumeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/VPC_Subnets.html
     *
     * @param vpcId - Identifier for the vpcId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:Tenancy
     */
    onVpc(vpcId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc/${VpcId}';
        arn = arn.replace('${VpcId}', vpcId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceId - Identifier for the vpceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:VpceServiceName
     *  - ec2:VpceServiceOwner
     */
    onVpcEndpoint(vpceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint/${VpceId}';
        arn = arn.replace('${VpceId}', vpceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-endpoint-service to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-overview.html
     *
     * @param vpceServiceId - Identifier for the vpceServiceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:VpceServicePrivateDnsName
     */
    onVpcEndpointService(vpceServiceId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-endpoint-service/${VpceServiceId}';
        arn = arn.replace('${VpceServiceId}', vpceServiceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-flow-log to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html
     *
     * @param vpcFlowLogId - Identifier for the vpcFlowLogId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:Region
     *  - ec2:ResourceTag/${TagKey}
     */
    onVpcFlowLog(vpcFlowLogId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-flow-log/${VpcFlowLogId}';
        arn = arn.replace('${VpcFlowLogId}', vpcFlowLogId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpc-peering-connection to the statement
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/what-is-vpc-peering.html
     *
     * @param vpcPeeringConnectionId - Identifier for the vpcPeeringConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - ec2:AccepterVpc
     *  - ec2:Region
     *  - ec2:RequesterVpc
     *  - ec2:ResourceTag/${TagKey}
     */
    onVpcPeeringConnection(vpcPeeringConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpc-peering-connection/${VpcPeeringConnectionId}';
        arn = arn.replace('${VpcPeeringConnectionId}', vpcPeeringConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-connection to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnConnectionId - Identifier for the vpnConnectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:RequestTag/${TagKey}
     *  - aws:TagKeys
     *  - ec2:AuthenticationType
     *  - ec2:DPDTimeoutSeconds
     *  - ec2:GatewayType
     *  - ec2:IKEVersions
     *  - ec2:InsideTunnelCidr
     *  - ec2:Phase1DHGroupNumbers
     *  - ec2:Phase1EncryptionAlgorithms
     *  - ec2:Phase1IntegrityAlgorithms
     *  - ec2:Phase1LifetimeSeconds
     *  - ec2:Phase2DHGroupNumbers
     *  - ec2:Phase2EncryptionAlgorithms
     *  - ec2:Phase2IntegrityAlgorithms
     *  - ec2:Phase2LifetimeSeconds
     *  - ec2:PresharedKeys
     *  - ec2:Region
     *  - ec2:RekeyFuzzPercentage
     *  - ec2:RekeyMarginTimeSeconds
     *  - ec2:ResourceTag/${TagKey}
     *  - ec2:RoutingType
     */
    onVpnConnection(vpnConnectionId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-connection/${VpnConnectionId}';
        arn = arn.replace('${VpnConnectionId}', vpnConnectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type vpn-gateway to the statement
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/VPC_VPN.html
     *
     * @param vpnGatewayId - Identifier for the vpnGatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onVpnGateway(vpnGatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}:${Account}:vpn-gateway/${VpnGatewayId}';
        arn = arn.replace('${VpnGatewayId}', vpnGatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the ARN of an accepter VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifAccepterVpc(value, operator) {
        return this.if(`ec2:AccepterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the user wants to associate a public IP address with the instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAssociatePublicIpAddress(value) {
        return this.if(`ec2:AssociatePublicIpAddress`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the authentication type for the VPN tunnel endpoints
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthenticationType(value, operator) {
        return this.if(`ec2:AuthenticationType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS service that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedService(value, operator) {
        return this.if(`ec2:AuthorizedService`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an IAM principal that has permission to use a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthorizedUser(value, operator) {
        return this.if(`ec2:AuthorizedUser`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Auto Placement properties of a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAutoPlacement(value, operator) {
        return this.if(`ec2:AutoPlacement`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of an Availability Zone in an AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAvailabilityZone(value, operator) {
        return this.if(`ec2:AvailabilityZone`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the name of a resource-creating API action
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/supported-iam-actions-tagging.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`ec2:CreateAction`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the duration after which DPD timeout occurs on a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifDPDTimeoutSeconds(value, operator) {
        return this.if(`ec2:DPDTimeoutSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether the instance is enabled for EBS optimization
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEbsOptimized(value) {
        return this.if(`ec2:EbsOptimized`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the type of Elastic Graphics accelerator
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifElasticGpuType(value, operator) {
        return this.if(`ec2:ElasticGpuType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the EBS volume is encrypted
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifEncrypted(value) {
        return this.if(`ec2:Encrypted`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the gateway type for a VPN endpoint on the AWS side of a VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGatewayType(value, operator) {
        return this.if(`ec2:GatewayType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether host recovery is enabled for a Dedicated Host
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifHostRecovery(value, operator) {
        return this.if(`ec2:HostRecovery`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the internet key exchange (IKE) versions that are permitted for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIKEVersions(value, operator) {
        return this.if(`ec2:IKEVersions`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of image (machine, aki, or ari)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImageType(value, operator) {
        return this.if(`ec2:ImageType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the range of inside IP addresses for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInsideTunnelCidr(value, operator) {
        return this.if(`ec2:InsideTunnelCidr`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the market or purchasing option of an instance (on-demand or spot)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceMarketType(value, operator) {
        return this.if(`ec2:InstanceMarketType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of an instance profile
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifInstanceProfile(value, operator) {
        return this.if(`ec2:InstanceProfile`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of instance
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`ec2:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether users are able to override resources that are specified in the launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsLaunchTemplateResource(value) {
        return this.if(`ec2:IsLaunchTemplateResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the ARN of a launch template
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifLaunchTemplate(value, operator) {
        return this.if(`ec2:LaunchTemplate`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by whether the HTTP endpoint is enabled for the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpEndpoint(value, operator) {
        return this.if(`ec2:MetadataHttpEndpoint`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the allowed number of hops when calling the instance metadata service
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMetadataHttpPutResponseHopLimit(value, operator) {
        return this.if(`ec2:MetadataHttpPutResponseHopLimit`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by whether tokens are required when calling the instance metadata service (optional or required)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMetadataHttpTokens(value, operator) {
        return this.if(`ec2:MetadataHttpTokens`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the owner of the resource (amazon, aws-marketplace, or an AWS account ID)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwner(value, operator) {
        return this.if(`ec2:Owner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the parent snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentSnapshot(value, operator) {
        return this.if(`ec2:ParentSnapshot`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the parent volume from which the snapshot was created
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifParentVolume(value, operator) {
        return this.if(`ec2:ParentVolume`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the type of permission for a resource (INSTANCE-ATTACH or EIP-ASSOCIATE)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermission(value, operator) {
        return this.if(`ec2:Permission`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase1DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.htmls
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase1EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 1 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase1IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase1IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 1 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase1LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase1LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the Diffie-Hellman group numbers that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2DHGroupNumbers(value, operator) {
        return this.if(`ec2:Phase2DHGroupNumbers`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the encryption algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2EncryptionAlgorithms(value, operator) {
        return this.if(`ec2:Phase2EncryptionAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the integrity algorithms that are permitted for a VPN tunnel for the phase 2 IKE negotiations
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPhase2IntegrityAlgorithms(value, operator) {
        return this.if(`ec2:Phase2IntegrityAlgorithms`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the lifetime in seconds for phase 2 of the IKE negotiations for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPhase2LifetimeSeconds(value, operator) {
        return this.if(`ec2:Phase2LifetimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of the placement group
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPlacementGroup(value, operator) {
        return this.if(`ec2:PlacementGroup`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the instance placement strategy used by the placement group (cluster, spread, or partition)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPlacementGroupStrategy(value, operator) {
        return this.if(`ec2:PlacementGroupStrategy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the pre-shared key (PSK) used to establish the initial IKE security association between a virtual private gateway and a customer gateway
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPresharedKeys(value, operator) {
        return this.if(`ec2:PresharedKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the product code that is associated with the AMI
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProductCode(value, operator) {
        return this.if(`ec2:ProductCode`, value, operator || 'StringLike');
    }
    /**
     * Filters access by whether the image has public launch permissions
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPublic(value) {
        return this.if(`ec2:Public`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the number of Dedicated Hosts in a request
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifQuantity(value, operator) {
        return this.if(`ec2:Quantity`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the name of the AWS Region
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRegion(value, operator) {
        return this.if(`ec2:Region`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the percentage of increase of the rekey window (determined by the rekey margin time) within which the rekey time is randomly selected for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyFuzzPercentage(value, operator) {
        return this.if(`ec2:RekeyFuzzPercentage`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the margin time before the phase 2 lifetime expires for a VPN tunnel
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRekeyMarginTimeSeconds(value, operator) {
        return this.if(`ec2:RekeyMarginTimeSeconds`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the ARN of a requester VPC in a VPC peering connection
     *
     * https://docs.aws.amazon.com/vpc/latest/peering/security-iam.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifRequesterVpc(value, operator) {
        return this.if(`ec2:RequesterVpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the payment option of the Reserved Instance offering (No Upfront, Partial Upfront, or All Upfront)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-reserved-instances.html#ri-payment-options
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReservedInstancesOfferingType(value, operator) {
        return this.if(`ec2:ReservedInstancesOfferingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`ec2:ResourceTag/`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/control-access-with-tags.html
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ec2:ResourceTag/${tagKey}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the version of the instance metadata service for retrieving IAM role credentials for EC2
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRoleDelivery(value, operator) {
        return this.if(`ec2:RoleDelivery`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the root device type of the instance (ebs or instance-store)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRootDeviceType(value, operator) {
        return this.if(`ec2:RootDeviceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the routing type for the VPN connection
     *
     * https://docs.aws.amazon.com/vpn/latest/s2svpn/vpn-authentication-access-control.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRoutingType(value, operator) {
        return this.if(`ec2:RoutingType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the initiation time of a snapshot
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifSnapshotTime(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`ec2:SnapshotTime`, value, operator || 'DateEquals');
    }
    /**
     * Filters access by the ARN of the instance from which the request originated
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSourceInstanceARN(value, operator) {
        return this.if(`ec2:SourceInstanceARN`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the ARN of the subnet
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifSubnet(value, operator) {
        return this.if(`ec2:Subnet`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the tenancy of the VPC or instance (default, dedicated, or host)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTenancy(value, operator) {
        return this.if(`ec2:Tenancy`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the the number of input/output operations per second (IOPS) provisioned for the volume
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeIops(value, operator) {
        return this.if(`ec2:VolumeIops`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the size of the volume, in GiB.
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeSize(value, operator) {
        return this.if(`ec2:VolumeSize`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the type of volume (gp2, io1, st1, sc1, or standard)
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVolumeType(value, operator) {
        return this.if(`ec2:VolumeType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ARN of the VPC
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifVpc(value, operator) {
        return this.if(`ec2:Vpc`, value, operator || 'ArnEquals');
    }
    /**
     * Filters access by the name of the VPC endpoint service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceName(value, operator) {
        return this.if(`ec2:VpceServiceName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the service owner of the VPC endpoint service (amazon, aws-marketplace, or an AWS account ID)
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServiceOwner(value, operator) {
        return this.if(`ec2:VpceServiceOwner`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the private DNS name of the VPC endpoint service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpceServicePrivateDnsName(value, operator) {
        return this.if(`ec2:VpceServicePrivateDnsName`, value, operator || 'StringLike');
    }
}
exports.Ec2 = Ec2;
//# sourceMappingURL=data:application/json;base64,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