"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticfilesystem = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticfilesystem](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticfilesystem extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticfilesystem](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticfilesystem.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticfilesystem';
        this.actionList = {
            "Backup": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html",
                "description": "Starts a backup job for an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ClientMount": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing read-access to a file system.",
                "accessLevel": "Read",
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "ClientRootAccess": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing root-access to a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "ClientWrite": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html",
                "description": "Permission for allowing write-access to a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "elasticfilesystem:AccessPointArn"
                ]
            },
            "CreateAccessPoint": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateAccessPoint.html",
                "description": "Creates an access point for the specified file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "CreateFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateFileSystem.html",
                "description": "Creates a new, empty file system.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateMountTarget": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateMountTarget.html",
                "description": "Creates a mount target for a file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_CreateTags.html",
                "description": "Creates or overwrites tags associated with a file system.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAccessPoint": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteAccessPoint.html",
                "description": "Deletes the specified access point.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "access-point": {
                        "required": true
                    }
                }
            },
            "DeleteFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystem.html",
                "description": "Deletes a file system, permanently severing access to its contents.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystemPolicy.html",
                "description": "Clears the resource-level policy for a given file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteMountTarget": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteMountTarget.html",
                "description": "Deletes the specified mount target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DeleteTags.html",
                "description": "Deletes the specified tags from a file system.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "DescribeAccessPoints": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeAccessPoints.html",
                "description": "Returns the descriptions of Amazon EFS access points.",
                "accessLevel": "List",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystemPolicy.html",
                "description": "Returns the current resource-level policy for a given file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": false
                    }
                }
            },
            "DescribeFileSystems": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystems.html",
                "description": "Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided; otherwise, returns descriptions of all file systems owned by the caller's AWS account in the AWS region of the endpoint that you're calling.",
                "accessLevel": "List",
                "resourceTypes": {
                    "file-system": {
                        "required": false
                    }
                }
            },
            "DescribeLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeLifecycleConfiguration.html",
                "description": "Returns the current LifecycleConfiguration object for the specified Amazon EFS file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeMountTargetSecurityGroups": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargetSecurityGroups.html",
                "description": "Returns the security groups currently in effect for a mount target.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeMountTargets": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargets.html",
                "description": "Returns the descriptions of all the current mount targets, or a specific mount target, for a file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_DescribeTags.html",
                "description": "Returns the tags associated with a file system.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_ListTagsForResource.html",
                "description": "Returns the tags associated with the specified Amazon EFS resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "ModifyMountTargetSecurityGroups": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_ModifyMountTargetSecurityGroups.html",
                "description": "Modifies the set of security groups in effect for a mount target.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "PutFileSystemPolicy": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_PutFileSystemPolicy.html",
                "description": "Apply a resource-level policy granting and/or restricting actions from given actors for the specified file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "PutLifecycleConfiguration": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_PutLifecycleConfiguration.html",
                "description": "Enables lifecycle management by creating a new LifecycleConfiguration object.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "Restore": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html",
                "description": "Starts a restore job for an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_TagResource.html",
                "description": "Creates or overwrites tags associated with the specified Amazon EFS resource.",
                "accessLevel": "Tagging"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_UntagResource.html",
                "description": "Deletes the specified tags from a specified Amazon EFS resource.",
                "accessLevel": "Tagging"
            },
            "UpdateFileSystem": {
                "url": "https://docs.aws.amazon.com/efs/latest/ug/API_UpdateFileSystem.html",
                "description": "Updates the throughput mode or the amount of provisioned throughput of an existing file system.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "file-system": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "file-system": {
                "name": "file-system",
                "url": "https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:elasticfilesystem:${Region}:${Account}:file-system/${FileSystemId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "access-point": {
                "name": "access-point",
                "url": "https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources",
                "arn": "arn:${Partition}:elasticfilesystem:${Region}:${Account}:access-point/${AccessPointId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Starts a backup job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    backup() {
        this.add('elasticfilesystem:Backup');
        return this;
    }
    /**
     * Permission for allowing read-access to a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    clientMount() {
        this.add('elasticfilesystem:ClientMount');
        return this;
    }
    /**
     * Permission for allowing root-access to a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    clientRootAccess() {
        this.add('elasticfilesystem:ClientRootAccess');
        return this;
    }
    /**
     * Permission for allowing write-access to a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-client-authorization.html
     */
    clientWrite() {
        this.add('elasticfilesystem:ClientWrite');
        return this;
    }
    /**
     * Creates an access point for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateAccessPoint.html
     */
    createAccessPoint() {
        this.add('elasticfilesystem:CreateAccessPoint');
        return this;
    }
    /**
     * Creates a new, empty file system.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateFileSystem.html
     */
    createFileSystem() {
        this.add('elasticfilesystem:CreateFileSystem');
        return this;
    }
    /**
     * Creates a mount target for a file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateMountTarget.html
     */
    createMountTarget() {
        this.add('elasticfilesystem:CreateMountTarget');
        return this;
    }
    /**
     * Creates or overwrites tags associated with a file system.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_CreateTags.html
     */
    createTags() {
        this.add('elasticfilesystem:CreateTags');
        return this;
    }
    /**
     * Deletes the specified access point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteAccessPoint.html
     */
    deleteAccessPoint() {
        this.add('elasticfilesystem:DeleteAccessPoint');
        return this;
    }
    /**
     * Deletes a file system, permanently severing access to its contents.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystem.html
     */
    deleteFileSystem() {
        this.add('elasticfilesystem:DeleteFileSystem');
        return this;
    }
    /**
     * Clears the resource-level policy for a given file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteFileSystemPolicy.html
     */
    deleteFileSystemPolicy() {
        this.add('elasticfilesystem:DeleteFileSystemPolicy');
        return this;
    }
    /**
     * Deletes the specified mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteMountTarget.html
     */
    deleteMountTarget() {
        this.add('elasticfilesystem:DeleteMountTarget');
        return this;
    }
    /**
     * Deletes the specified tags from a file system.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DeleteTags.html
     */
    deleteTags() {
        this.add('elasticfilesystem:DeleteTags');
        return this;
    }
    /**
     * Returns the descriptions of Amazon EFS access points.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeAccessPoints.html
     */
    describeAccessPoints() {
        this.add('elasticfilesystem:DescribeAccessPoints');
        return this;
    }
    /**
     * Returns the current resource-level policy for a given file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystemPolicy.html
     */
    describeFileSystemPolicy() {
        this.add('elasticfilesystem:DescribeFileSystemPolicy');
        return this;
    }
    /**
     * Returns the description of a specific Amazon EFS file system if either the file system CreationToken or the FileSystemId is provided; otherwise, returns descriptions of all file systems owned by the caller's AWS account in the AWS region of the endpoint that you're calling.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeFileSystems.html
     */
    describeFileSystems() {
        this.add('elasticfilesystem:DescribeFileSystems');
        return this;
    }
    /**
     * Returns the current LifecycleConfiguration object for the specified Amazon EFS file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeLifecycleConfiguration.html
     */
    describeLifecycleConfiguration() {
        this.add('elasticfilesystem:DescribeLifecycleConfiguration');
        return this;
    }
    /**
     * Returns the security groups currently in effect for a mount target.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargetSecurityGroups.html
     */
    describeMountTargetSecurityGroups() {
        this.add('elasticfilesystem:DescribeMountTargetSecurityGroups');
        return this;
    }
    /**
     * Returns the descriptions of all the current mount targets, or a specific mount target, for a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeMountTargets.html
     */
    describeMountTargets() {
        this.add('elasticfilesystem:DescribeMountTargets');
        return this;
    }
    /**
     * Returns the tags associated with a file system.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_DescribeTags.html
     */
    describeTags() {
        this.add('elasticfilesystem:DescribeTags');
        return this;
    }
    /**
     * Returns the tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('elasticfilesystem:ListTagsForResource');
        return this;
    }
    /**
     * Modifies the set of security groups in effect for a mount target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_ModifyMountTargetSecurityGroups.html
     */
    modifyMountTargetSecurityGroups() {
        this.add('elasticfilesystem:ModifyMountTargetSecurityGroups');
        return this;
    }
    /**
     * Apply a resource-level policy granting and/or restricting actions from given actors for the specified file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutFileSystemPolicy.html
     */
    putFileSystemPolicy() {
        this.add('elasticfilesystem:PutFileSystemPolicy');
        return this;
    }
    /**
     * Enables lifecycle management by creating a new LifecycleConfiguration object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_PutLifecycleConfiguration.html
     */
    putLifecycleConfiguration() {
        this.add('elasticfilesystem:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Starts a restore job for an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/efs-backup-solutions.html
     */
    restore() {
        this.add('elasticfilesystem:Restore');
        return this;
    }
    /**
     * Creates or overwrites tags associated with the specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_TagResource.html
     */
    tagResource() {
        this.add('elasticfilesystem:TagResource');
        return this;
    }
    /**
     * Deletes the specified tags from a specified Amazon EFS resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UntagResource.html
     */
    untagResource() {
        this.add('elasticfilesystem:UntagResource');
        return this;
    }
    /**
     * Updates the throughput mode or the amount of provisioned throughput of an existing file system.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/efs/latest/ug/API_UpdateFileSystem.html
     */
    updateFileSystem() {
        this.add('elasticfilesystem:UpdateFileSystem');
        return this;
    }
    /**
     * Adds a resource of type file-system to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFileSystem(fileSystemId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:file-system/${FileSystemId}';
        arn = arn.replace('${FileSystemId}', fileSystemId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type access-point to the statement
     *
     * https://docs.aws.amazon.com/efs/latest/ug/access-control-overview.html#access-control-resources
     *
     * @param accessPointId - Identifier for the accessPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onAccessPoint(accessPointId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticfilesystem:${Region}:${Account}:access-point/${AccessPointId}';
        arn = arn.replace('${AccessPointId}', accessPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The ARN of the access point used to mount the file system
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointArn(value, operator) {
        return this.if(`elasticfilesystem:AccessPointArn`, value, operator || 'StringLike');
    }
}
exports.Elasticfilesystem = Elasticfilesystem;
//# sourceMappingURL=data:application/json;base64,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