"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Events = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Events extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'events';
        this.actionList = {
            "ActivateEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html",
                "description": "Activates a partner event source that has been deactivated. Once activated, your matching event bus will start receiving events from the event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "CreateEventBus": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html",
                "description": "Creates a new event bus within your account. This can be a custom event bus which you can use to receive events from your own custom applications and services, or it can be a partner event bus which can be matched to a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-bus": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePartnerEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreatePartnerEventSource.html",
                "description": "Called by an AWS partner to create a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DeactivateEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeactivateEventSource.html",
                "description": "Called by an AWS partner to create a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DeleteEventBus": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource",
                "description": "Deletes the specified custom event bus or partner event bus. All rules associated with this event bus are also deleted. You can't delete your account's default event bus.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-bus": {
                        "required": true
                    }
                }
            },
            "DeletePartnerEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource",
                "description": "Called by an AWS partner to delete a partner event source.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DeleteRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteRule.html",
                "description": "Deletes a rule. You must remove all targets from a rule using RemoveTargets before you can delete the rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DescribeEventBus": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html",
                "description": "Displays the external AWS accounts that are permitted to write events to your account using your account's event bus, and the associated policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    }
                }
            },
            "DescribeEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html",
                "description": "Describes the details of the specified partner event source that is shared with your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DescribePartnerEventSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribePartnerEventSource.html",
                "description": "Called by an AWS partner describe the details of the specified partner event source that they have created.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "DescribeRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeRule.html",
                "description": "Describes the details of the specified rule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DisableRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html",
                "description": "Disables a rule. A disabled rule won't match any events, and won't self-trigger if it has a schedule expression.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "EnableRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_EnableRule.html",
                "description": "Enables a rule. If the rule does not exist, the operation fails.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "ListEventBuses": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventBuses.html",
                "description": "Lists all the event buses in your account, including the default event bus, custom event buses, and partner event buses.",
                "accessLevel": "List"
            },
            "ListEventSources": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventSources.html",
                "description": "Lists the event sources shared with this account.",
                "accessLevel": "List"
            },
            "ListPartnerEventSourceAccounts": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSourceAccounts.html",
                "description": "Called by an AWS partner to display the AWS account ID that the specified partner event source is associated with.",
                "accessLevel": "List",
                "resourceTypes": {
                    "event-source": {
                        "required": true
                    }
                }
            },
            "ListPartnerEventSources": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSources.html",
                "description": "Called by an AWS partner to list all the partner event sources that they have created.",
                "accessLevel": "List"
            },
            "ListRuleNamesByTarget": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRuleNamesByTarget.html",
                "description": "Lists the names of the rules that the given target is put to.",
                "accessLevel": "List"
            },
            "ListRules": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRules.html",
                "description": "Lists the Amazon EventBridge rules in your account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTagsForResource.html",
                "description": "This action lists tags for an Amazon EventBridge resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    }
                }
            },
            "ListTargetsByRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html",
                "description": "Lists of targets assigned to the rule.",
                "accessLevel": "List",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "PutEvents": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html",
                "description": "Sends custom events to Amazon EventBridge so that they can be matched to rules.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "event-bus": {
                        "required": true
                    }
                }
            },
            "PutPartnerEvents": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPartnerEvents.html",
                "description": "Sends custom events to Amazon EventBridge so that they can be matched to rules.",
                "accessLevel": "Write"
            },
            "PutPermission": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html",
                "description": "Running PutPermission permits the specified AWS account to put events to your account's default event bus.",
                "accessLevel": "Write"
            },
            "PutRule": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutRule.html",
                "description": "Creates or updates a rule. Rules are enabled by default, or based on value of the State parameter.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "events:detail.userIdentity.principalId",
                    "events:detail-type",
                    "events:source",
                    "events:detail.service",
                    "events:detail.eventTypeCode",
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "PutTargets": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutTargets.html",
                "description": "Adds target(s) to a rule. Targets are the resources that can be invoked when a rule is triggered.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "events:TargetArn"
                ]
            },
            "RemovePermission": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemovePermission.html",
                "description": "Revokes the permission of another AWS account to be able to put events to your default event bus.",
                "accessLevel": "Write"
            },
            "RemoveTargets": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html",
                "description": "Removes target(s) from a rule so that when the rule is triggered, those targets will no longer be invoked.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html",
                "description": "This action tags an Amazon EventBridge resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "TestEventPattern": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TestEventPattern.html",
                "description": "Tests whether an event pattern matches the provided event.",
                "accessLevel": "Read"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html",
                "description": "This action removes a tag from an Amazon EventBridge resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "event-bus": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "event-source": {
                "name": "event-source",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format",
                "arn": "arn:${Partition}:events:${Region}::event-source/${EventSourceName}",
                "conditionKeys": []
            },
            "event-bus": {
                "name": "event-bus",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format",
                "arn": "arn:${Partition}:events:${Region}:${Account}:event-bus/${EventBusName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format",
                "arn": "arn:${Partition}:events:${Region}:${Account}:rule/${RuleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Activates a partner event source that has been deactivated. Once activated, your matching event bus will start receiving events from the event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html
     */
    activateEventSource() {
        this.add('events:ActivateEventSource');
        return this;
    }
    /**
     * Creates a new event bus within your account. This can be a custom event bus which you can use to receive events from your own custom applications and services, or it can be a partner event bus which can be matched to a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html
     */
    createEventBus() {
        this.add('events:CreateEventBus');
        return this;
    }
    /**
     * Called by an AWS partner to create a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreatePartnerEventSource.html
     */
    createPartnerEventSource() {
        this.add('events:CreatePartnerEventSource');
        return this;
    }
    /**
     * Called by an AWS partner to create a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeactivateEventSource.html
     */
    deactivateEventSource() {
        this.add('events:DeactivateEventSource');
        return this;
    }
    /**
     * Deletes the specified custom event bus or partner event bus. All rules associated with this event bus are also deleted. You can't delete your account's default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    deleteEventBus() {
        this.add('events:DeleteEventBus');
        return this;
    }
    /**
     * Called by an AWS partner to delete a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    deletePartnerEventSource() {
        this.add('events:DeletePartnerEventSource');
        return this;
    }
    /**
     * Deletes a rule. You must remove all targets from a rule using RemoveTargets before you can delete the rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteRule.html
     */
    deleteRule() {
        this.add('events:DeleteRule');
        return this;
    }
    /**
     * Displays the external AWS accounts that are permitted to write events to your account using your account's event bus, and the associated policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html
     */
    describeEventBus() {
        this.add('events:DescribeEventBus');
        return this;
    }
    /**
     * Describes the details of the specified partner event source that is shared with your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html
     */
    describeEventSource() {
        this.add('events:DescribeEventSource');
        return this;
    }
    /**
     * Called by an AWS partner describe the details of the specified partner event source that they have created.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribePartnerEventSource.html
     */
    describePartnerEventSource() {
        this.add('events:DescribePartnerEventSource');
        return this;
    }
    /**
     * Describes the details of the specified rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeRule.html
     */
    describeRule() {
        this.add('events:DescribeRule');
        return this;
    }
    /**
     * Disables a rule. A disabled rule won't match any events, and won't self-trigger if it has a schedule expression.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html
     */
    disableRule() {
        this.add('events:DisableRule');
        return this;
    }
    /**
     * Enables a rule. If the rule does not exist, the operation fails.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_EnableRule.html
     */
    enableRule() {
        this.add('events:EnableRule');
        return this;
    }
    /**
     * Lists all the event buses in your account, including the default event bus, custom event buses, and partner event buses.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventBuses.html
     */
    listEventBuses() {
        this.add('events:ListEventBuses');
        return this;
    }
    /**
     * Lists the event sources shared with this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventSources.html
     */
    listEventSources() {
        this.add('events:ListEventSources');
        return this;
    }
    /**
     * Called by an AWS partner to display the AWS account ID that the specified partner event source is associated with.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSourceAccounts.html
     */
    listPartnerEventSourceAccounts() {
        this.add('events:ListPartnerEventSourceAccounts');
        return this;
    }
    /**
     * Called by an AWS partner to list all the partner event sources that they have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSources.html
     */
    listPartnerEventSources() {
        this.add('events:ListPartnerEventSources');
        return this;
    }
    /**
     * Lists the names of the rules that the given target is put to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRuleNamesByTarget.html
     */
    listRuleNamesByTarget() {
        this.add('events:ListRuleNamesByTarget');
        return this;
    }
    /**
     * Lists the Amazon EventBridge rules in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRules.html
     */
    listRules() {
        this.add('events:ListRules');
        return this;
    }
    /**
     * This action lists tags for an Amazon EventBridge resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('events:ListTagsForResource');
        return this;
    }
    /**
     * Lists of targets assigned to the rule.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html
     */
    listTargetsByRule() {
        this.add('events:ListTargetsByRule');
        return this;
    }
    /**
     * Sends custom events to Amazon EventBridge so that they can be matched to rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html
     */
    putEvents() {
        this.add('events:PutEvents');
        return this;
    }
    /**
     * Sends custom events to Amazon EventBridge so that they can be matched to rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPartnerEvents.html
     */
    putPartnerEvents() {
        this.add('events:PutPartnerEvents');
        return this;
    }
    /**
     * Running PutPermission permits the specified AWS account to put events to your account's default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html
     */
    putPermission() {
        this.add('events:PutPermission');
        return this;
    }
    /**
     * Creates or updates a rule. Rules are enabled by default, or based on value of the State parameter.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutRule.html
     */
    putRule() {
        this.add('events:PutRule');
        return this;
    }
    /**
     * Adds target(s) to a rule. Targets are the resources that can be invoked when a rule is triggered.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutTargets.html
     */
    putTargets() {
        this.add('events:PutTargets');
        return this;
    }
    /**
     * Revokes the permission of another AWS account to be able to put events to your default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemovePermission.html
     */
    removePermission() {
        this.add('events:RemovePermission');
        return this;
    }
    /**
     * Removes target(s) from a rule so that when the rule is triggered, those targets will no longer be invoked.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html
     */
    removeTargets() {
        this.add('events:RemoveTargets');
        return this;
    }
    /**
     * This action tags an Amazon EventBridge resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('events:TagResource');
        return this;
    }
    /**
     * Tests whether an event pattern matches the provided event.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TestEventPattern.html
     */
    testEventPattern() {
        this.add('events:TestEventPattern');
        return this;
    }
    /**
     * This action removes a tag from an Amazon EventBridge resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('events:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type event-source to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventSourceName - Identifier for the eventSourceName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventSource(eventSourceName, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}::event-source/${EventSourceName}';
        arn = arn.replace('${EventSourceName}', eventSourceName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-bus to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventBusName - Identifier for the eventBusName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onEventBus(eventBusName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:event-bus/${EventBusName}';
        arn = arn.replace('${EventBusName}', eventBusName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRule(ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:rule/${RuleName}';
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The ARN of a target that can be put to a rule.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limiting-access-to-targets
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifTargetArn(value, operator) {
        return this.if(`events:TargetArn`, value, operator || 'ArnEquals');
    }
    /**
     * Matches the literal string of the detail-type filed of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-pattern-detail-type
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailType(value, operator) {
        return this.if(`events:detail-type`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.eventTypeCode field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-type-code
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailEventTypeCode(value, operator) {
        return this.if(`events:detail.eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.service field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-service
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailService(value, operator) {
        return this.if(`events:detail.service`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.useridentity.principalid field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#consume-specific-events
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailUserIdentityPrincipalId(value, operator) {
        return this.if(`events:detail.userIdentity.principalId`, value, operator || 'StringLike');
    }
    /**
     * The AWS service or AWS partner event source that generated the event. Matches the literal string of the source field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-limit-access-control
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSource(value, operator) {
        return this.if(`events:source`, value, operator || 'StringLike');
    }
}
exports.Events = Events;
//# sourceMappingURL=data:application/json;base64,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