"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schemas = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [schemas](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridgeschemas.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Schemas extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [schemas](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridgeschemas.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'schemas';
        this.actionList = {
            "CreateDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#CreateDiscoverer",
                "description": "Creates an event schema discoverer. Once created, your events will be automatically map into corresponding schema documents",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "CreateRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#CreateRegistry",
                "description": "Create a new schema registry in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "CreateSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#CreateSchema",
                "description": "Create a new schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DeleteDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DeleteDiscoverer",
                "description": "Deletes discoverer in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "DeleteRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DeleteRegistry",
                "description": "Deletes an existing registry in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#DeleteResourcePolicy",
                "description": "Delete the resource-based policy attached to a given registry.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "DeleteSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DeleteSchema",
                "description": "Deletes an existing schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DeleteSchemaVersion": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-version-schemaversion.html#DeleteSchemaVersion",
                "description": "Deletes a specific version of schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DescribeCodeBinding": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#DescribeCodeBinding",
                "description": "Retrieves metadata for generated code for specific schema in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "DescribeDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DescribeDiscoverer",
                "description": "Retrieves discoverer metadata in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "DescribeRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DescribeRegistry",
                "description": "Describes an existing registry metadata in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "DescribeSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DescribeSchema",
                "description": "Retrieves an existing schema in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "GetCodeBindingSource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language-source.html#GetCodeBindingSource",
                "description": "Retrieves metadata for generated code for specific schema in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "GetDiscoveredSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discover.html#GetDiscoveredSchema",
                "description": "Retrieves schema for the provided list of sample events.",
                "accessLevel": "Read"
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#GetResourcePolicy",
                "description": "Retrieves the resource-based policy attached to a given registry.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "ListDiscoverers": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers.html#ListDiscoverers",
                "description": "Lists all the discoverers in your account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "ListRegistries": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries.html#ListRegistries",
                "description": "List all discoverers in your account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "ListSchemaVersions": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-versions.html#ListSchemaVersions",
                "description": "List all versions of a schema.",
                "accessLevel": "List",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "ListSchemas": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas.html#ListSchemas",
                "description": "List all schemas.",
                "accessLevel": "List",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#ListTagsForResource",
                "description": "This action lists tags for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    },
                    "registry": {
                        "required": true
                    },
                    "schema": {
                        "required": true
                    }
                }
            },
            "PutCodeBinding": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#PutCodeBinding",
                "description": "Generates code for specific schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#PutResourcePolicy",
                "description": "Attach resource-based policy to the specific registry.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "SearchSchemas": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-search.html#SearchSchemas",
                "description": "Searches schemas based on specified keywords in your account.",
                "accessLevel": "List",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            },
            "StartDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StartDiscoverer",
                "description": "Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "StopDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StopDiscoverer",
                "description": "Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#TagResource",
                "description": "This action tags an resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    },
                    "registry": {
                        "required": true
                    },
                    "schema": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#UntagResource",
                "description": "This action removes a tag from on a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    },
                    "registry": {
                        "required": true
                    },
                    "schema": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateDiscoverer": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#UpdateDiscoverer",
                "description": "Updates an existing discoverer in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "discoverer": {
                        "required": true
                    }
                }
            },
            "UpdateRegistry": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#UpdateRegistry",
                "description": "Updates an existing registry metadata in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "registry": {
                        "required": true
                    }
                }
            },
            "UpdateSchema": {
                "url": "https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#UpdateSchema",
                "description": "Updates an existing schema in your account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "schema": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "discoverer": {
                "name": "discoverer",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html",
                "arn": "arn:${Partition}:schemas:${Region}:${Account}:discoverer/${DiscovererId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "registry": {
                "name": "registry",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html",
                "arn": "arn:${Partition}:schemas:${Region}:${Account}:registry/${RegistryName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "schema": {
                "name": "schema",
                "url": "https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html",
                "arn": "arn:${Partition}:schemas:${Region}:${Account}:schema/${RegistryName}/${SchemaName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates an event schema discoverer. Once created, your events will be automatically map into corresponding schema documents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#CreateDiscoverer
     */
    createDiscoverer() {
        this.add('schemas:CreateDiscoverer');
        return this;
    }
    /**
     * Create a new schema registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#CreateRegistry
     */
    createRegistry() {
        this.add('schemas:CreateRegistry');
        return this;
    }
    /**
     * Create a new schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#CreateSchema
     */
    createSchema() {
        this.add('schemas:CreateSchema');
        return this;
    }
    /**
     * Deletes discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DeleteDiscoverer
     */
    deleteDiscoverer() {
        this.add('schemas:DeleteDiscoverer');
        return this;
    }
    /**
     * Deletes an existing registry in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DeleteRegistry
     */
    deleteRegistry() {
        this.add('schemas:DeleteRegistry');
        return this;
    }
    /**
     * Delete the resource-based policy attached to a given registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#DeleteResourcePolicy
     */
    deleteResourcePolicy() {
        this.add('schemas:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DeleteSchema
     */
    deleteSchema() {
        this.add('schemas:DeleteSchema');
        return this;
    }
    /**
     * Deletes a specific version of schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-version-schemaversion.html#DeleteSchemaVersion
     */
    deleteSchemaVersion() {
        this.add('schemas:DeleteSchemaVersion');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#DescribeCodeBinding
     */
    describeCodeBinding() {
        this.add('schemas:DescribeCodeBinding');
        return this;
    }
    /**
     * Retrieves discoverer metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#DescribeDiscoverer
     */
    describeDiscoverer() {
        this.add('schemas:DescribeDiscoverer');
        return this;
    }
    /**
     * Describes an existing registry metadata in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#DescribeRegistry
     */
    describeRegistry() {
        this.add('schemas:DescribeRegistry');
        return this;
    }
    /**
     * Retrieves an existing schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#DescribeSchema
     */
    describeSchema() {
        this.add('schemas:DescribeSchema');
        return this;
    }
    /**
     * Retrieves metadata for generated code for specific schema in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language-source.html#GetCodeBindingSource
     */
    getCodeBindingSource() {
        this.add('schemas:GetCodeBindingSource');
        return this;
    }
    /**
     * Retrieves schema for the provided list of sample events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discover.html#GetDiscoveredSchema
     */
    getDiscoveredSchema() {
        this.add('schemas:GetDiscoveredSchema');
        return this;
    }
    /**
     * Retrieves the resource-based policy attached to a given registry.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#GetResourcePolicy
     */
    getResourcePolicy() {
        this.add('schemas:GetResourcePolicy');
        return this;
    }
    /**
     * Lists all the discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers.html#ListDiscoverers
     */
    listDiscoverers() {
        this.add('schemas:ListDiscoverers');
        return this;
    }
    /**
     * List all discoverers in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries.html#ListRegistries
     */
    listRegistries() {
        this.add('schemas:ListRegistries');
        return this;
    }
    /**
     * List all versions of a schema.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-versions.html#ListSchemaVersions
     */
    listSchemaVersions() {
        this.add('schemas:ListSchemaVersions');
        return this;
    }
    /**
     * List all schemas.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas.html#ListSchemas
     */
    listSchemas() {
        this.add('schemas:ListSchemas');
        return this;
    }
    /**
     * This action lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#ListTagsForResource
     */
    listTagsForResource() {
        this.add('schemas:ListTagsForResource');
        return this;
    }
    /**
     * Generates code for specific schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname-language-language.html#PutCodeBinding
     */
    putCodeBinding() {
        this.add('schemas:PutCodeBinding');
        return this;
    }
    /**
     * Attach resource-based policy to the specific registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-policy.html#PutResourcePolicy
     */
    putResourcePolicy() {
        this.add('schemas:PutResourcePolicy');
        return this;
    }
    /**
     * Searches schemas based on specified keywords in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-search.html#SearchSchemas
     */
    searchSchemas() {
        this.add('schemas:SearchSchemas');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StartDiscoverer
     */
    startDiscoverer() {
        this.add('schemas:StartDiscoverer');
        return this;
    }
    /**
     * Starts the specified discoverer. Once started the discoverer will automatically register schemas for published events to configured source in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#StopDiscoverer
     */
    stopDiscoverer() {
        this.add('schemas:StopDiscoverer');
        return this;
    }
    /**
     * This action tags an resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#TagResource
     */
    tagResource() {
        this.add('schemas:TagResource');
        return this;
    }
    /**
     * This action removes a tag from on a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/tags-resource-arn.html#UntagResource
     */
    untagResource() {
        this.add('schemas:UntagResource');
        return this;
    }
    /**
     * Updates an existing discoverer in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-discoverers-id-discovererid.html#UpdateDiscoverer
     */
    updateDiscoverer() {
        this.add('schemas:UpdateDiscoverer');
        return this;
    }
    /**
     * Updates an existing registry metadata in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname.html#UpdateRegistry
     */
    updateRegistry() {
        this.add('schemas:UpdateRegistry');
        return this;
    }
    /**
     * Updates an existing schema in your account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/schema-reference/v1-registries-name-registryname-schemas-name-schemaname.html#UpdateSchema
     */
    updateSchema() {
        this.add('schemas:UpdateSchema');
        return this;
    }
    /**
     * Adds a resource of type discoverer to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param discovererId - Identifier for the discovererId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDiscoverer(discovererId, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:discoverer/${DiscovererId}';
        arn = arn.replace('${DiscovererId}', discovererId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type registry to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRegistry(registryName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:registry/${RegistryName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type schema to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     *
     * @param registryName - Identifier for the registryName.
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSchema(registryName, schemaName, account, region, partition) {
        var arn = 'arn:${Partition}:schemas:${Region}:${Account}:schema/${RegistryName}/${SchemaName}';
        arn = arn.replace('${RegistryName}', registryName);
        arn = arn.replace('${SchemaName}', schemaName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Schemas = Schemas;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1hem9uZXZlbnRicmlkZ2VzY2hlbWFzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYW1hem9uZXZlbnRicmlkZ2VzY2hlbWFzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHNDQUFvRTtBQUVwRTs7OztHQUlHO0FBQ0gsTUFBYSxPQUFRLFNBQVEsd0JBQWU7SUErVjFDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBcFdOLGtCQUFhLEdBQUcsU0FBUyxDQUFDO1FBQ3ZCLGVBQVUsR0FBWTtZQUM5QixrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLHNIQUFzSDtnQkFDN0gsYUFBYSxFQUFFLDZIQUE2SDtnQkFDNUksYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUU7d0JBQ1osVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLHFIQUFxSDtnQkFDNUgsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLDJJQUEySTtnQkFDbEosYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLHNIQUFzSDtnQkFDN0gsYUFBYSxFQUFFLHFDQUFxQztnQkFDcEQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUU7d0JBQ1osVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLHFIQUFxSDtnQkFDNUgsYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsYUFBYSxFQUFFLGdFQUFnRTtnQkFDL0UsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsS0FBSyxFQUFFLDJJQUEySTtnQkFDbEosYUFBYSxFQUFFLDZDQUE2QztnQkFDNUQsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHdLQUF3SztnQkFDL0ssYUFBYSxFQUFFLHVEQUF1RDtnQkFDdEUsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLG9LQUFvSztnQkFDM0ssYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHdIQUF3SDtnQkFDL0gsYUFBYSxFQUFFLGdEQUFnRDtnQkFDL0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUU7d0JBQ1osVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxrQkFBa0IsRUFBRTtnQkFDbEIsS0FBSyxFQUFFLHVIQUF1SDtnQkFDOUgsYUFBYSxFQUFFLDBEQUEwRDtnQkFDekUsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLDZJQUE2STtnQkFDcEosYUFBYSxFQUFFLCtDQUErQztnQkFDOUQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxzQkFBc0IsRUFBRTtnQkFDdEIsS0FBSyxFQUFFLDRLQUE0SztnQkFDbkwsYUFBYSxFQUFFLDRFQUE0RTtnQkFDM0YsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLHNHQUFzRztnQkFDN0csYUFBYSxFQUFFLDBEQUEwRDtnQkFDekUsYUFBYSxFQUFFLE1BQU07YUFDdEI7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLGtHQUFrRztnQkFDekcsYUFBYSxFQUFFLG1FQUFtRTtnQkFDbEYsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLHFHQUFxRztnQkFDNUcsYUFBYSxFQUFFLDRDQUE0QztnQkFDM0QsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUU7d0JBQ1osVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsS0FBSyxFQUFFLG1HQUFtRztnQkFDMUcsYUFBYSxFQUFFLHVDQUF1QztnQkFDdEQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLDBKQUEwSjtnQkFDakssYUFBYSxFQUFFLGdDQUFnQztnQkFDL0MsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLDBIQUEwSDtnQkFDakksYUFBYSxFQUFFLG1CQUFtQjtnQkFDbEMsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixRQUFRLEVBQUU7d0JBQ1IsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO2lCQUNGO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsS0FBSyxFQUFFLDRHQUE0RztnQkFDbkgsYUFBYSxFQUFFLHdDQUF3QztnQkFDdkQsYUFBYSxFQUFFLE1BQU07Z0JBQ3JCLGVBQWUsRUFBRTtvQkFDZixZQUFZLEVBQUU7d0JBQ1osVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7b0JBQ0QsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSwrSkFBK0o7Z0JBQ3RLLGFBQWEsRUFBRSxxREFBcUQ7Z0JBQ3BFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSxrR0FBa0c7Z0JBQ3pHLGFBQWEsRUFBRSx3REFBd0Q7Z0JBQ3ZFLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxtSUFBbUk7Z0JBQzFJLGFBQWEsRUFBRSwrREFBK0Q7Z0JBQzlFLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSxxSEFBcUg7Z0JBQzVILGFBQWEsRUFBRSw0SkFBNEo7Z0JBQzNLLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxvSEFBb0g7Z0JBQzNILGFBQWEsRUFBRSw0SkFBNEo7Z0JBQzNLLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxvR0FBb0c7Z0JBQzNHLGFBQWEsRUFBRSwrQkFBK0I7Z0JBQzlDLGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7b0JBQ2IsMEJBQTBCO2lCQUMzQjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxzR0FBc0c7Z0JBQzdHLGFBQWEsRUFBRSwrQ0FBK0M7Z0JBQzlELGFBQWEsRUFBRSxTQUFTO2dCQUN4QixlQUFlLEVBQUU7b0JBQ2YsWUFBWSxFQUFFO3dCQUNaLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLElBQUk7cUJBQ2pCO29CQUNELFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7Z0JBQ0QsWUFBWSxFQUFFO29CQUNaLGFBQWE7aUJBQ2Q7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixLQUFLLEVBQUUsc0hBQXNIO2dCQUM3SCxhQUFhLEVBQUUsaURBQWlEO2dCQUNoRSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFlBQVksRUFBRTt3QkFDWixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGdCQUFnQixFQUFFO2dCQUNoQixLQUFLLEVBQUUscUhBQXFIO2dCQUM1SCxhQUFhLEVBQUUsd0RBQXdEO2dCQUN2RSxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFVBQVUsRUFBRTt3QkFDVixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtZQUNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsMklBQTJJO2dCQUNsSixhQUFhLEVBQUUsNkNBQTZDO2dCQUM1RCxhQUFhLEVBQUUsT0FBTztnQkFDdEIsZUFBZSxFQUFFO29CQUNmLFFBQVEsRUFBRTt3QkFDUixVQUFVLEVBQUUsSUFBSTtxQkFDakI7aUJBQ0Y7YUFDRjtTQUNGLENBQUM7UUFDSyxrQkFBYSxHQUFrQjtZQUNwQyxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLEtBQUssRUFBRSw2R0FBNkc7Z0JBQ3BILEtBQUssRUFBRSwwRUFBMEU7Z0JBQ2pGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLFVBQVU7Z0JBQ2xCLEtBQUssRUFBRSw2R0FBNkc7Z0JBQ3BILEtBQUssRUFBRSx3RUFBd0U7Z0JBQy9FLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7WUFDRCxRQUFRLEVBQUU7Z0JBQ1IsTUFBTSxFQUFFLFFBQVE7Z0JBQ2hCLEtBQUssRUFBRSw2R0FBNkc7Z0JBQ3BILEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGVBQWUsRUFBRTtvQkFDZiwyQkFBMkI7aUJBQzVCO2FBQ0Y7U0FDRixDQUFDO0lBU0YsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxHQUFHLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDdEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNyQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxHQUFHLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxZQUFZLENBQUMsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RixJQUFJLEdBQUcsR0FBRywwRUFBMEUsQ0FBQztRQUNyRixHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNuRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFVBQVUsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLElBQUksR0FBRyxHQUFHLHdFQUF3RSxDQUFDO1FBQ25GLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFFBQVEsQ0FBQyxZQUFvQixFQUFFLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0csSUFBSSxHQUFHLEdBQUcsb0ZBQW9GLENBQUM7UUFDL0YsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDbkQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUFueEJELDBCQW14QkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtzY2hlbWFzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25ldmVudGJyaWRnZXNjaGVtYXMuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBTY2hlbWFzIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnc2NoZW1hcyc7XG4gIHByb3RlY3RlZCBhY3Rpb25MaXN0OiBBY3Rpb25zID0ge1xuICAgIFwiQ3JlYXRlRGlzY292ZXJlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1kaXNjb3ZlcmVycy1pZC1kaXNjb3ZlcmVyaWQuaHRtbCNDcmVhdGVEaXNjb3ZlcmVyXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlcyBhbiBldmVudCBzY2hlbWEgZGlzY292ZXJlci4gT25jZSBjcmVhdGVkLCB5b3VyIGV2ZW50cyB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgbWFwIGludG8gY29ycmVzcG9uZGluZyBzY2hlbWEgZG9jdW1lbnRzXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGlzY292ZXJlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiQ3JlYXRlUmVnaXN0cnlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS5odG1sI0NyZWF0ZVJlZ2lzdHJ5XCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlIGEgbmV3IHNjaGVtYSByZWdpc3RyeSBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkNyZWF0ZVNjaGVtYVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLmh0bWwjQ3JlYXRlU2NoZW1hXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiQ3JlYXRlIGEgbmV3IHNjaGVtYSBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic2NoZW1hXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVEaXNjb3ZlcmVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLWRpc2NvdmVyZXJzLWlkLWRpc2NvdmVyZXJpZC5odG1sI0RlbGV0ZURpc2NvdmVyZXJcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGRpc2NvdmVyZXIgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRpc2NvdmVyZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVJlZ2lzdHJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUuaHRtbCNEZWxldGVSZWdpc3RyeVwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYW4gZXhpc3RpbmcgcmVnaXN0cnkgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2lzdHJ5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZWxldGVSZXNvdXJjZVBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1wb2xpY3kuaHRtbCNEZWxldGVSZXNvdXJjZVBvbGljeVwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZSB0aGUgcmVzb3VyY2UtYmFzZWQgcG9saWN5IGF0dGFjaGVkIHRvIGEgZ2l2ZW4gcmVnaXN0cnkuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVNjaGVtYVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLmh0bWwjRGVsZXRlU2NoZW1hXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVsZXRlcyBhbiBleGlzdGluZyBzY2hlbWEgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNjaGVtYVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVsZXRlU2NoZW1hVmVyc2lvblwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLXZlcnNpb24tc2NoZW1hdmVyc2lvbi5odG1sI0RlbGV0ZVNjaGVtYVZlcnNpb25cIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJEZWxldGVzIGEgc3BlY2lmaWMgdmVyc2lvbiBvZiBzY2hlbWEgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNjaGVtYVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiRGVzY3JpYmVDb2RlQmluZGluZ1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLWxhbmd1YWdlLWxhbmd1YWdlLmh0bWwjRGVzY3JpYmVDb2RlQmluZGluZ1wiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBtZXRhZGF0YSBmb3IgZ2VuZXJhdGVkIGNvZGUgZm9yIHNwZWNpZmljIHNjaGVtYSBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzY2hlbWFcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlc2NyaWJlRGlzY292ZXJlclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1kaXNjb3ZlcmVycy1pZC1kaXNjb3ZlcmVyaWQuaHRtbCNEZXNjcmliZURpc2NvdmVyZXJcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXRyaWV2ZXMgZGlzY292ZXJlciBtZXRhZGF0YSBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJkaXNjb3ZlcmVyXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZVJlZ2lzdHJ5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUuaHRtbCNEZXNjcmliZVJlZ2lzdHJ5XCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiRGVzY3JpYmVzIGFuIGV4aXN0aW5nIHJlZ2lzdHJ5IG1ldGFkYXRhIGluIHlvdXIgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2lzdHJ5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJEZXNjcmliZVNjaGVtYVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLmh0bWwjRGVzY3JpYmVTY2hlbWFcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXRyaWV2ZXMgYW4gZXhpc3Rpbmcgc2NoZW1hIGluIHlvdXIgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNjaGVtYVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiR2V0Q29kZUJpbmRpbmdTb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS1zY2hlbWFzLW5hbWUtc2NoZW1hbmFtZS1sYW5ndWFnZS1sYW5ndWFnZS1zb3VyY2UuaHRtbCNHZXRDb2RlQmluZGluZ1NvdXJjZVwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyBtZXRhZGF0YSBmb3IgZ2VuZXJhdGVkIGNvZGUgZm9yIHNwZWNpZmljIHNjaGVtYSBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzY2hlbWFcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldERpc2NvdmVyZWRTY2hlbWFcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtZGlzY292ZXIuaHRtbCNHZXREaXNjb3ZlcmVkU2NoZW1hXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiUmV0cmlldmVzIHNjaGVtYSBmb3IgdGhlIHByb3ZpZGVkIGxpc3Qgb2Ygc2FtcGxlIGV2ZW50cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0UmVzb3VyY2VQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcG9saWN5Lmh0bWwjR2V0UmVzb3VyY2VQb2xpY3lcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJSZXRyaWV2ZXMgdGhlIHJlc291cmNlLWJhc2VkIHBvbGljeSBhdHRhY2hlZCB0byBhIGdpdmVuIHJlZ2lzdHJ5LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3REaXNjb3ZlcmVyc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1kaXNjb3ZlcmVycy5odG1sI0xpc3REaXNjb3ZlcmVyc1wiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3RzIGFsbCB0aGUgZGlzY292ZXJlcnMgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGlzY292ZXJlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdFJlZ2lzdHJpZXNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy5odG1sI0xpc3RSZWdpc3RyaWVzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdCBhbGwgZGlzY292ZXJlcnMgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RTY2hlbWFWZXJzaW9uc1wiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLXZlcnNpb25zLmh0bWwjTGlzdFNjaGVtYVZlcnNpb25zXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdCBhbGwgdmVyc2lvbnMgb2YgYSBzY2hlbWEuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiTGlzdFwiLFxuICAgICAgXCJyZXNvdXJjZVR5cGVzXCI6IHtcbiAgICAgICAgXCJzY2hlbWFcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RTY2hlbWFzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUtc2NoZW1hcy5odG1sI0xpc3RTY2hlbWFzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiTGlzdCBhbGwgc2NoZW1hcy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNjaGVtYVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS90YWdzLXJlc291cmNlLWFybi5odG1sI0xpc3RUYWdzRm9yUmVzb3VyY2VcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJUaGlzIGFjdGlvbiBsaXN0cyB0YWdzIGZvciBhIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGlzY292ZXJlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcInNjaGVtYVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiUHV0Q29kZUJpbmRpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS1zY2hlbWFzLW5hbWUtc2NoZW1hbmFtZS1sYW5ndWFnZS1sYW5ndWFnZS5odG1sI1B1dENvZGVCaW5kaW5nXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR2VuZXJhdGVzIGNvZGUgZm9yIHNwZWNpZmljIHNjaGVtYSBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic2NoZW1hXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJQdXRSZXNvdXJjZVBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1wb2xpY3kuaHRtbCNQdXRSZXNvdXJjZVBvbGljeVwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkF0dGFjaCByZXNvdXJjZS1iYXNlZCBwb2xpY3kgdG8gdGhlIHNwZWNpZmljIHJlZ2lzdHJ5LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInJlZ2lzdHJ5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTZWFyY2hTY2hlbWFzXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUtc2NoZW1hcy1zZWFyY2guaHRtbCNTZWFyY2hTY2hlbWFzXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU2VhcmNoZXMgc2NoZW1hcyBiYXNlZCBvbiBzcGVjaWZpZWQga2V5d29yZHMgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwic2NoZW1hXCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0sXG4gICAgXCJTdGFydERpc2NvdmVyZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtZGlzY292ZXJlcnMtaWQtZGlzY292ZXJlcmlkLmh0bWwjU3RhcnREaXNjb3ZlcmVyXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RhcnRzIHRoZSBzcGVjaWZpZWQgZGlzY292ZXJlci4gT25jZSBzdGFydGVkIHRoZSBkaXNjb3ZlcmVyIHdpbGwgYXV0b21hdGljYWxseSByZWdpc3RlciBzY2hlbWFzIGZvciBwdWJsaXNoZWQgZXZlbnRzIHRvIGNvbmZpZ3VyZWQgc291cmNlIGluIHlvdXIgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRpc2NvdmVyZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlN0b3BEaXNjb3ZlcmVyXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLWRpc2NvdmVyZXJzLWlkLWRpc2NvdmVyZXJpZC5odG1sI1N0b3BEaXNjb3ZlcmVyXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU3RhcnRzIHRoZSBzcGVjaWZpZWQgZGlzY292ZXJlci4gT25jZSBzdGFydGVkIHRoZSBkaXNjb3ZlcmVyIHdpbGwgYXV0b21hdGljYWxseSByZWdpc3RlciBzY2hlbWFzIGZvciBwdWJsaXNoZWQgZXZlbnRzIHRvIGNvbmZpZ3VyZWQgc291cmNlIGluIHlvdXIgYWNjb3VudFwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRpc2NvdmVyZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlRhZ1Jlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3RhZ3MtcmVzb3VyY2UtYXJuLmh0bWwjVGFnUmVzb3VyY2VcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJUaGlzIGFjdGlvbiB0YWdzIGFuIHJlc291cmNlLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlRhZ2dpbmdcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGlzY292ZXJlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH0sXG4gICAgICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcInNjaGVtYVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICBcImNvbmRpdGlvbnNcIjogW1xuICAgICAgICBcImF3czpUYWdLZXlzXCIsXG4gICAgICAgIFwiYXdzOlJlcXVlc3RUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwiVW50YWdSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS90YWdzLXJlc291cmNlLWFybi5odG1sI1VudGFnUmVzb3VyY2VcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJUaGlzIGFjdGlvbiByZW1vdmVzIGEgdGFnIGZyb20gb24gYSByZXNvdXJjZS5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJUYWdnaW5nXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcImRpc2NvdmVyZXJcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBcInJlZ2lzdHJ5XCI6IHtcbiAgICAgICAgICBcInJlcXVpcmVkXCI6IHRydWVcbiAgICAgICAgfSxcbiAgICAgICAgXCJzY2hlbWFcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9LFxuICAgICAgXCJjb25kaXRpb25zXCI6IFtcbiAgICAgICAgXCJhd3M6VGFnS2V5c1wiXG4gICAgICBdXG4gICAgfSxcbiAgICBcIlVwZGF0ZURpc2NvdmVyZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtZGlzY292ZXJlcnMtaWQtZGlzY292ZXJlcmlkLmh0bWwjVXBkYXRlRGlzY292ZXJlclwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlVwZGF0ZXMgYW4gZXhpc3RpbmcgZGlzY292ZXJlciBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwiZGlzY292ZXJlclwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFwiVXBkYXRlUmVnaXN0cnlcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS5odG1sI1VwZGF0ZVJlZ2lzdHJ5XCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhbiBleGlzdGluZyByZWdpc3RyeSBtZXRhZGF0YSBpbiB5b3VyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIlVwZGF0ZVNjaGVtYVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLmh0bWwjVXBkYXRlU2NoZW1hXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiVXBkYXRlcyBhbiBleGlzdGluZyBzY2hlbWEgaW4geW91ciBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCIsXG4gICAgICBcInJlc291cmNlVHlwZXNcIjoge1xuICAgICAgICBcInNjaGVtYVwiOiB7XG4gICAgICAgICAgXCJyZXF1aXJlZFwiOiB0cnVlXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH07XG4gIHB1YmxpYyByZXNvdXJjZVR5cGVzOiBSZXNvdXJjZVR5cGVzID0ge1xuICAgIFwiZGlzY292ZXJlclwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJkaXNjb3ZlcmVyXCIsXG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3QvdXNlcmd1aWRlL2lhbS1pZGVudGl0eS1iYXNlZC1hY2Nlc3MtY29udHJvbC1ldmVudGJyaWRnZS5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06c2NoZW1hczoke1JlZ2lvbn06JHtBY2NvdW50fTpkaXNjb3ZlcmVyLyR7RGlzY292ZXJlcklkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwicmVnaXN0cnlcIjoge1xuICAgICAgXCJuYW1lXCI6IFwicmVnaXN0cnlcIixcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWlkZW50aXR5LWJhc2VkLWFjY2Vzcy1jb250cm9sLWV2ZW50YnJpZGdlLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTpzY2hlbWFzOiR7UmVnaW9ufToke0FjY291bnR9OnJlZ2lzdHJ5LyR7UmVnaXN0cnlOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtcbiAgICAgICAgXCJhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XCJcbiAgICAgIF1cbiAgICB9LFxuICAgIFwic2NoZW1hXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcInNjaGVtYVwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3VzZXJndWlkZS9pYW0taWRlbnRpdHktYmFzZWQtYWNjZXNzLWNvbnRyb2wtZXZlbnRicmlkZ2UuaHRtbFwiLFxuICAgICAgXCJhcm5cIjogXCJhcm46JHtQYXJ0aXRpb259OnNjaGVtYXM6JHtSZWdpb259OiR7QWNjb3VudH06c2NoZW1hLyR7UmVnaXN0cnlOYW1lfS8ke1NjaGVtYU5hbWV9XCIsXG4gICAgICBcImNvbmRpdGlvbktleXNcIjogW1xuICAgICAgICBcImF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cIlxuICAgICAgXVxuICAgIH1cbiAgfTtcblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtzY2hlbWFzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25ldmVudGJyaWRnZXNjaGVtYXMuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhbiBldmVudCBzY2hlbWEgZGlzY292ZXJlci4gT25jZSBjcmVhdGVkLCB5b3VyIGV2ZW50cyB3aWxsIGJlIGF1dG9tYXRpY2FsbHkgbWFwIGludG8gY29ycmVzcG9uZGluZyBzY2hlbWEgZG9jdW1lbnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLWRpc2NvdmVyZXJzLWlkLWRpc2NvdmVyZXJpZC5odG1sI0NyZWF0ZURpc2NvdmVyZXJcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVEaXNjb3ZlcmVyKCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOkNyZWF0ZURpc2NvdmVyZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBuZXcgc2NoZW1hIHJlZ2lzdHJ5IGluIHlvdXIgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS5odG1sI0NyZWF0ZVJlZ2lzdHJ5XG4gICAqL1xuICBwdWJsaWMgY3JlYXRlUmVnaXN0cnkoKSB7XG4gICAgdGhpcy5hZGQoJ3NjaGVtYXM6Q3JlYXRlUmVnaXN0cnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBuZXcgc2NoZW1hIGluIHlvdXIgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS1zY2hlbWFzLW5hbWUtc2NoZW1hbmFtZS5odG1sI0NyZWF0ZVNjaGVtYVxuICAgKi9cbiAgcHVibGljIGNyZWF0ZVNjaGVtYSgpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpDcmVhdGVTY2hlbWEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGRpc2NvdmVyZXIgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1kaXNjb3ZlcmVycy1pZC1kaXNjb3ZlcmVyaWQuaHRtbCNEZWxldGVEaXNjb3ZlcmVyXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlRGlzY292ZXJlcigpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpEZWxldGVEaXNjb3ZlcmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhbiBleGlzdGluZyByZWdpc3RyeSBpbiB5b3VyIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUuaHRtbCNEZWxldGVSZWdpc3RyeVxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVJlZ2lzdHJ5KCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOkRlbGV0ZVJlZ2lzdHJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlIHRoZSByZXNvdXJjZS1iYXNlZCBwb2xpY3kgYXR0YWNoZWQgdG8gYSBnaXZlbiByZWdpc3RyeS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcG9saWN5Lmh0bWwjRGVsZXRlUmVzb3VyY2VQb2xpY3lcbiAgICovXG4gIHB1YmxpYyBkZWxldGVSZXNvdXJjZVBvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpEZWxldGVSZXNvdXJjZVBvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYW4gZXhpc3Rpbmcgc2NoZW1hIGluIHlvdXIgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS1zY2hlbWFzLW5hbWUtc2NoZW1hbmFtZS5odG1sI0RlbGV0ZVNjaGVtYVxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVNjaGVtYSgpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpEZWxldGVTY2hlbWEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgc3BlY2lmaWMgdmVyc2lvbiBvZiBzY2hlbWEgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLXZlcnNpb24tc2NoZW1hdmVyc2lvbi5odG1sI0RlbGV0ZVNjaGVtYVZlcnNpb25cbiAgICovXG4gIHB1YmxpYyBkZWxldGVTY2hlbWFWZXJzaW9uKCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOkRlbGV0ZVNjaGVtYVZlcnNpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgbWV0YWRhdGEgZm9yIGdlbmVyYXRlZCBjb2RlIGZvciBzcGVjaWZpYyBzY2hlbWEgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUtc2NoZW1hcy1uYW1lLXNjaGVtYW5hbWUtbGFuZ3VhZ2UtbGFuZ3VhZ2UuaHRtbCNEZXNjcmliZUNvZGVCaW5kaW5nXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVDb2RlQmluZGluZygpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpEZXNjcmliZUNvZGVCaW5kaW5nJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGRpc2NvdmVyZXIgbWV0YWRhdGEgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLWRpc2NvdmVyZXJzLWlkLWRpc2NvdmVyZXJpZC5odG1sI0Rlc2NyaWJlRGlzY292ZXJlclxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlRGlzY292ZXJlcigpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpEZXNjcmliZURpc2NvdmVyZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYW4gZXhpc3RpbmcgcmVnaXN0cnkgbWV0YWRhdGEgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUuaHRtbCNEZXNjcmliZVJlZ2lzdHJ5XG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVSZWdpc3RyeSgpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpEZXNjcmliZVJlZ2lzdHJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGFuIGV4aXN0aW5nIHNjaGVtYSBpbiB5b3VyIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS1zY2hlbWFzLW5hbWUtc2NoZW1hbmFtZS5odG1sI0Rlc2NyaWJlU2NoZW1hXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmVTY2hlbWEoKSB7XG4gICAgdGhpcy5hZGQoJ3NjaGVtYXM6RGVzY3JpYmVTY2hlbWEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgbWV0YWRhdGEgZm9yIGdlbmVyYXRlZCBjb2RlIGZvciBzcGVjaWZpYyBzY2hlbWEgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUtc2NoZW1hcy1uYW1lLXNjaGVtYW5hbWUtbGFuZ3VhZ2UtbGFuZ3VhZ2Utc291cmNlLmh0bWwjR2V0Q29kZUJpbmRpbmdTb3VyY2VcbiAgICovXG4gIHB1YmxpYyBnZXRDb2RlQmluZGluZ1NvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpHZXRDb2RlQmluZGluZ1NvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBzY2hlbWEgZm9yIHRoZSBwcm92aWRlZCBsaXN0IG9mIHNhbXBsZSBldmVudHMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtZGlzY292ZXIuaHRtbCNHZXREaXNjb3ZlcmVkU2NoZW1hXG4gICAqL1xuICBwdWJsaWMgZ2V0RGlzY292ZXJlZFNjaGVtYSgpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpHZXREaXNjb3ZlcmVkU2NoZW1hJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIHRoZSByZXNvdXJjZS1iYXNlZCBwb2xpY3kgYXR0YWNoZWQgdG8gYSBnaXZlbiByZWdpc3RyeS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1wb2xpY3kuaHRtbCNHZXRSZXNvdXJjZVBvbGljeVxuICAgKi9cbiAgcHVibGljIGdldFJlc291cmNlUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOkdldFJlc291cmNlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYWxsIHRoZSBkaXNjb3ZlcmVycyBpbiB5b3VyIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtZGlzY292ZXJlcnMuaHRtbCNMaXN0RGlzY292ZXJlcnNcbiAgICovXG4gIHB1YmxpYyBsaXN0RGlzY292ZXJlcnMoKSB7XG4gICAgdGhpcy5hZGQoJ3NjaGVtYXM6TGlzdERpc2NvdmVyZXJzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBhbGwgZGlzY292ZXJlcnMgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMuaHRtbCNMaXN0UmVnaXN0cmllc1xuICAgKi9cbiAgcHVibGljIGxpc3RSZWdpc3RyaWVzKCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOkxpc3RSZWdpc3RyaWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdCBhbGwgdmVyc2lvbnMgb2YgYSBzY2hlbWEuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3NjaGVtYS1yZWZlcmVuY2UvdjEtcmVnaXN0cmllcy1uYW1lLXJlZ2lzdHJ5bmFtZS1zY2hlbWFzLW5hbWUtc2NoZW1hbmFtZS12ZXJzaW9ucy5odG1sI0xpc3RTY2hlbWFWZXJzaW9uc1xuICAgKi9cbiAgcHVibGljIGxpc3RTY2hlbWFWZXJzaW9ucygpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpMaXN0U2NoZW1hVmVyc2lvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IGFsbCBzY2hlbWFzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUtc2NoZW1hcy5odG1sI0xpc3RTY2hlbWFzXG4gICAqL1xuICBwdWJsaWMgbGlzdFNjaGVtYXMoKSB7XG4gICAgdGhpcy5hZGQoJ3NjaGVtYXM6TGlzdFNjaGVtYXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGlzIGFjdGlvbiBsaXN0cyB0YWdzIGZvciBhIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3RhZ3MtcmVzb3VyY2UtYXJuLmh0bWwjTGlzdFRhZ3NGb3JSZXNvdXJjZVxuICAgKi9cbiAgcHVibGljIGxpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ3NjaGVtYXM6TGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlcyBjb2RlIGZvciBzcGVjaWZpYyBzY2hlbWEgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLWxhbmd1YWdlLWxhbmd1YWdlLmh0bWwjUHV0Q29kZUJpbmRpbmdcbiAgICovXG4gIHB1YmxpYyBwdXRDb2RlQmluZGluZygpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpQdXRDb2RlQmluZGluZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaCByZXNvdXJjZS1iYXNlZCBwb2xpY3kgdG8gdGhlIHNwZWNpZmljIHJlZ2lzdHJ5LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1wb2xpY3kuaHRtbCNQdXRSZXNvdXJjZVBvbGljeVxuICAgKi9cbiAgcHVibGljIHB1dFJlc291cmNlUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOlB1dFJlc291cmNlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2VhcmNoZXMgc2NoZW1hcyBiYXNlZCBvbiBzcGVjaWZpZWQga2V5d29yZHMgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUtc2NoZW1hcy1zZWFyY2guaHRtbCNTZWFyY2hTY2hlbWFzXG4gICAqL1xuICBwdWJsaWMgc2VhcmNoU2NoZW1hcygpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpTZWFyY2hTY2hlbWFzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3RhcnRzIHRoZSBzcGVjaWZpZWQgZGlzY292ZXJlci4gT25jZSBzdGFydGVkIHRoZSBkaXNjb3ZlcmVyIHdpbGwgYXV0b21hdGljYWxseSByZWdpc3RlciBzY2hlbWFzIGZvciBwdWJsaXNoZWQgZXZlbnRzIHRvIGNvbmZpZ3VyZWQgc291cmNlIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1kaXNjb3ZlcmVycy1pZC1kaXNjb3ZlcmVyaWQuaHRtbCNTdGFydERpc2NvdmVyZXJcbiAgICovXG4gIHB1YmxpYyBzdGFydERpc2NvdmVyZXIoKSB7XG4gICAgdGhpcy5hZGQoJ3NjaGVtYXM6U3RhcnREaXNjb3ZlcmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3RhcnRzIHRoZSBzcGVjaWZpZWQgZGlzY292ZXJlci4gT25jZSBzdGFydGVkIHRoZSBkaXNjb3ZlcmVyIHdpbGwgYXV0b21hdGljYWxseSByZWdpc3RlciBzY2hlbWFzIGZvciBwdWJsaXNoZWQgZXZlbnRzIHRvIGNvbmZpZ3VyZWQgc291cmNlIGluIHlvdXIgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1kaXNjb3ZlcmVycy1pZC1kaXNjb3ZlcmVyaWQuaHRtbCNTdG9wRGlzY292ZXJlclxuICAgKi9cbiAgcHVibGljIHN0b3BEaXNjb3ZlcmVyKCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOlN0b3BEaXNjb3ZlcmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBhY3Rpb24gdGFncyBhbiByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS90YWdzLXJlc291cmNlLWFybi5odG1sI1RhZ1Jlc291cmNlXG4gICAqL1xuICBwdWJsaWMgdGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy5hZGQoJ3NjaGVtYXM6VGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGlzIGFjdGlvbiByZW1vdmVzIGEgdGFnIGZyb20gb24gYSByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS90YWdzLXJlc291cmNlLWFybi5odG1sI1VudGFnUmVzb3VyY2VcbiAgICovXG4gIHB1YmxpYyB1bnRhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOlVudGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGFuIGV4aXN0aW5nIGRpc2NvdmVyZXIgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1kaXNjb3ZlcmVycy1pZC1kaXNjb3ZlcmVyaWQuaHRtbCNVcGRhdGVEaXNjb3ZlcmVyXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlRGlzY292ZXJlcigpIHtcbiAgICB0aGlzLmFkZCgnc2NoZW1hczpVcGRhdGVEaXNjb3ZlcmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhbiBleGlzdGluZyByZWdpc3RyeSBtZXRhZGF0YSBpbiB5b3VyIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC9zY2hlbWEtcmVmZXJlbmNlL3YxLXJlZ2lzdHJpZXMtbmFtZS1yZWdpc3RyeW5hbWUuaHRtbCNVcGRhdGVSZWdpc3RyeVxuICAgKi9cbiAgcHVibGljIHVwZGF0ZVJlZ2lzdHJ5KCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOlVwZGF0ZVJlZ2lzdHJ5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhbiBleGlzdGluZyBzY2hlbWEgaW4geW91ciBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3Qvc2NoZW1hLXJlZmVyZW5jZS92MS1yZWdpc3RyaWVzLW5hbWUtcmVnaXN0cnluYW1lLXNjaGVtYXMtbmFtZS1zY2hlbWFuYW1lLmh0bWwjVXBkYXRlU2NoZW1hXG4gICAqL1xuICBwdWJsaWMgdXBkYXRlU2NoZW1hKCkge1xuICAgIHRoaXMuYWRkKCdzY2hlbWFzOlVwZGF0ZVNjaGVtYScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGRpc2NvdmVyZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZXZlbnRicmlkZ2UvbGF0ZXN0L3VzZXJndWlkZS9pYW0taWRlbnRpdHktYmFzZWQtYWNjZXNzLWNvbnRyb2wtZXZlbnRicmlkZ2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZGlzY292ZXJlcklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRpc2NvdmVyZXJJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uIGtleXM6XG4gICAqICAtIGF3czpSZXNvdXJjZVRhZy8ke1RhZ0tleX1cbiAgICovXG4gIHB1YmxpYyBvbkRpc2NvdmVyZXIoZGlzY292ZXJlcklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnNjaGVtYXM6JHtSZWdpb259OiR7QWNjb3VudH06ZGlzY292ZXJlci8ke0Rpc2NvdmVyZXJJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0Rpc2NvdmVyZXJJZH0nLCBkaXNjb3ZlcmVySWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJlZ2lzdHJ5IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2V2ZW50YnJpZGdlL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWlkZW50aXR5LWJhc2VkLWFjY2Vzcy1jb250cm9sLWV2ZW50YnJpZGdlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlZ2lzdHJ5TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSByZWdpc3RyeU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbiBrZXlzOlxuICAgKiAgLSBhd3M6UmVzb3VyY2VUYWcvJHtUYWdLZXl9XG4gICAqL1xuICBwdWJsaWMgb25SZWdpc3RyeShyZWdpc3RyeU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06c2NoZW1hczoke1JlZ2lvbn06JHtBY2NvdW50fTpyZWdpc3RyeS8ke1JlZ2lzdHJ5TmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lzdHJ5TmFtZX0nLCByZWdpc3RyeU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNjaGVtYSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9ldmVudGJyaWRnZS9sYXRlc3QvdXNlcmd1aWRlL2lhbS1pZGVudGl0eS1iYXNlZC1hY2Nlc3MtY29udHJvbC1ldmVudGJyaWRnZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZWdpc3RyeU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVnaXN0cnlOYW1lLlxuICAgKiBAcGFyYW0gc2NoZW1hTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzY2hlbWFOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb24ga2V5czpcbiAgICogIC0gYXdzOlJlc291cmNlVGFnLyR7VGFnS2V5fVxuICAgKi9cbiAgcHVibGljIG9uU2NoZW1hKHJlZ2lzdHJ5TmFtZTogc3RyaW5nLCBzY2hlbWFOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnNjaGVtYXM6JHtSZWdpb259OiR7QWNjb3VudH06c2NoZW1hLyR7UmVnaXN0cnlOYW1lfS8ke1NjaGVtYU5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpc3RyeU5hbWV9JywgcmVnaXN0cnlOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtTY2hlbWFOYW1lfScsIHNjaGVtYU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG59XG4iXX0=