"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Freertos = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [freertos](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfreertos.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Freertos extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [freertos](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfreertos.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'freertos';
        this.actionList = {
            "CreateSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Creates a software configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Deletes the software configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                }
            },
            "DescribeHardwarePlatform": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Describes the hardware platform.",
                "accessLevel": "Read"
            },
            "DescribeSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Describes the software configuration.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                }
            },
            "GetSoftwareURL": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Get the URL for Amazon FreeRTOS software download.",
                "accessLevel": "Read"
            },
            "GetSoftwareURLForConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Get the URL for Amazon FreeRTOS software download based on the configuration.",
                "accessLevel": "Read"
            },
            "ListFreeRTOSVersions": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists versions of AmazonFreeRTOS.",
                "accessLevel": "List"
            },
            "ListHardwarePlatforms": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists the hardware platforms.",
                "accessLevel": "List"
            },
            "ListHardwareVendors": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists the hardware vendors.",
                "accessLevel": "List"
            },
            "ListSoftwareConfigurations": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Lists the software configurations.",
                "accessLevel": "List"
            },
            "UpdateSoftwareConfiguration": {
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "description": "Updates the software configuration.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "configuration": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "configuration": {
                "name": "configuration",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html",
                "arn": "arn:${Partition}:freertos:${Region}:${Account}:configuration/${ConfigurationName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a software configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    createSoftwareConfiguration() {
        this.add('freertos:CreateSoftwareConfiguration');
        return this;
    }
    /**
     * Deletes the software configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    deleteSoftwareConfiguration() {
        this.add('freertos:DeleteSoftwareConfiguration');
        return this;
    }
    /**
     * Describes the hardware platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    describeHardwarePlatform() {
        this.add('freertos:DescribeHardwarePlatform');
        return this;
    }
    /**
     * Describes the software configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    describeSoftwareConfiguration() {
        this.add('freertos:DescribeSoftwareConfiguration');
        return this;
    }
    /**
     * Get the URL for Amazon FreeRTOS software download.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    getSoftwareURL() {
        this.add('freertos:GetSoftwareURL');
        return this;
    }
    /**
     * Get the URL for Amazon FreeRTOS software download based on the configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    getSoftwareURLForConfiguration() {
        this.add('freertos:GetSoftwareURLForConfiguration');
        return this;
    }
    /**
     * Lists versions of AmazonFreeRTOS.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    listFreeRTOSVersions() {
        this.add('freertos:ListFreeRTOSVersions');
        return this;
    }
    /**
     * Lists the hardware platforms.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    listHardwarePlatforms() {
        this.add('freertos:ListHardwarePlatforms');
        return this;
    }
    /**
     * Lists the hardware vendors.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    listHardwareVendors() {
        this.add('freertos:ListHardwareVendors');
        return this;
    }
    /**
     * Lists the software configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    listSoftwareConfigurations() {
        this.add('freertos:ListSoftwareConfigurations');
        return this;
    }
    /**
     * Updates the software configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     */
    updateSoftwareConfiguration() {
        this.add('freertos:UpdateSoftwareConfiguration');
        return this;
    }
    /**
     * Adds a resource of type configuration to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ocw.html
     *
     * @param configurationName - Identifier for the configurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onConfiguration(configurationName, account, region, partition) {
        var arn = 'arn:${Partition}:freertos:${Region}:${Account}:configuration/${ConfigurationName}';
        arn = arn.replace('${ConfigurationName}', configurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Freertos = Freertos;
//# sourceMappingURL=data:application/json;base64,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