"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Honeycode = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [honeycode](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonhoneycode.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Honeycode extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [honeycode](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonhoneycode.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'honeycode';
        this.actionList = {
            "ApproveTeamAssociation": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#approve-team-association",
                "description": "Grants permission to approve a team association request for your AWS Account",
                "accessLevel": "Write"
            },
            "GetScreenData": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_GetScreenData.html",
                "description": "Grants permission to load the data from a screen",
                "accessLevel": "Read",
                "resourceTypes": {
                    "screen": {
                        "required": true
                    }
                }
            },
            "InvokeScreenAutomation": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_InvokeScreenAutomation.html",
                "description": "Grants permission to invoke a screen automation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "screen-automation": {
                        "required": true
                    }
                }
            },
            "ListTeamAssociations": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#list-team-associations",
                "description": "Grants permission to list all pending and approved team associations with your AWS Account",
                "accessLevel": "List"
            },
            "RejectTeamAssociation": {
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#reject-team-association",
                "description": "Grants permission to reject a team association request for your AWS Account",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "screen": {
                "name": "screen",
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen.html",
                "arn": "arn:${Partition}:honeycode:${Region}:${Account}:screen:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}",
                "conditionKeys": []
            },
            "screen-automation": {
                "name": "screen-automation",
                "url": "https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen-automation.html",
                "arn": "arn:${Partition}:honeycode:${Region}:${Account}:screen-automation:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}/automation/${AutomationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to approve a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#approve-team-association
     */
    approveTeamAssociation() {
        this.add('honeycode:ApproveTeamAssociation');
        return this;
    }
    /**
     * Grants permission to load the data from a screen
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_GetScreenData.html
     */
    getScreenData() {
        this.add('honeycode:GetScreenData');
        return this;
    }
    /**
     * Grants permission to invoke a screen automation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/API_InvokeScreenAutomation.html
     */
    invokeScreenAutomation() {
        this.add('honeycode:InvokeScreenAutomation');
        return this;
    }
    /**
     * Grants permission to list all pending and approved team associations with your AWS Account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#list-team-associations
     */
    listTeamAssociations() {
        this.add('honeycode:ListTeamAssociations');
        return this;
    }
    /**
     * Grants permission to reject a team association request for your AWS Account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/team-association.html#reject-team-association
     */
    rejectTeamAssociation() {
        this.add('honeycode:RejectTeamAssociation');
        return this;
    }
    /**
     * Adds a resource of type screen to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreen(workbookId, appId, screenId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type screen-automation to the statement
     *
     * https://docs.aws.amazon.com/honeycode/latest/UserGuide/resource-screen-automation.html
     *
     * @param workbookId - Identifier for the workbookId.
     * @param appId - Identifier for the appId.
     * @param screenId - Identifier for the screenId.
     * @param automationId - Identifier for the automationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onScreenAutomation(workbookId, appId, screenId, automationId, account, region, partition) {
        var arn = 'arn:${Partition}:honeycode:${Region}:${Account}:screen-automation:workbook/${WorkbookId}/app/${AppId}/screen/${ScreenId}/automation/${AutomationId}';
        arn = arn.replace('${WorkbookId}', workbookId);
        arn = arn.replace('${AppId}', appId);
        arn = arn.replace('${ScreenId}', screenId);
        arn = arn.replace('${AutomationId}', automationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Honeycode = Honeycode;
//# sourceMappingURL=data:application/json;base64,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