"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lex = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [lex](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlex.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Lex extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [lex](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonlex.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'lex';
        this.actionList = {
            "CreateBotVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateBotVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "CreateIntentVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateIntentVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "CreateSlotTypeVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_CreateSlotTypeVersion.html",
                "description": "Creates a new version based on the $LATEST version of the specified slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBot.html",
                "description": "Deletes all versions of a bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "DeleteBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotAlias.html",
                "description": "Deletes an alias for a specific bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                }
            },
            "DeleteBotChannelAssociation": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotChannelAssociation.html",
                "description": "Deletes the association between a Amazon Lex bot alias and a messaging platform.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "DeleteBotVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotVersion.html",
                "description": "Deletes a specific version of a bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "DeleteIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntent.html",
                "description": "Deletes all versions of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "DeleteIntentVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntentVersion.html",
                "description": "Deletes a specific version of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "DeleteSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_DeleteSession.html",
                "description": "Removes session information for a specified bot, alias, and user ID.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "DeleteSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotType.html",
                "description": "Deletes all versions of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteSlotTypeVersion": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotTypeVersion.html",
                "description": "Deletes a specific version of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "DeleteUtterances": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_DeleteUtterances.html",
                "description": "Deletes the information Amazon Lex maintains for utterances on a specific bot and userId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBot.html",
                "description": "Returns information for a specific bot. In addition to the bot name, the bot version or alias is required.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "GetBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAlias.html",
                "description": "Returns information about a Amazon Lex bot alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                }
            },
            "GetBotAliases": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAliases.html",
                "description": "Returns a list of aliases for a given Amazon Lex bot.",
                "accessLevel": "List"
            },
            "GetBotChannelAssociation": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociation.html",
                "description": "Returns information about the association between a Amazon Lex bot and a messaging platform.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "GetBotChannelAssociations": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociations.html",
                "description": "Returns a list of all of the channels associated with a single bot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "channel": {
                        "required": true
                    }
                }
            },
            "GetBotVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBotVersions.html",
                "description": "Returns information for all versions of a specific bot.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetBots": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBots.html",
                "description": "Returns information for the $LATEST version of all bots, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetBuiltinIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntent.html",
                "description": "Returns information about a built-in intent.",
                "accessLevel": "Read"
            },
            "GetBuiltinIntents": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntents.html",
                "description": "Gets a list of built-in intents that meet the specified criteria.",
                "accessLevel": "Read"
            },
            "GetBuiltinSlotTypes": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinSlotTypes.html",
                "description": "Gets a list of built-in slot types that meet the specified criteria.",
                "accessLevel": "Read"
            },
            "GetExport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetExport.html",
                "description": "Exports Amazon Lex Resource in a requested format.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "GetImport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetImport.html",
                "description": "Gets information about an import job started with StartImport.",
                "accessLevel": "Read"
            },
            "GetIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntent.html",
                "description": "Returns information for a specific intent. In addition to the intent name, you must also specify the intent version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "GetIntentVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntentVersions.html",
                "description": "Returns information for all versions of a specific intent.",
                "accessLevel": "List",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "GetIntents": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetIntents.html",
                "description": "Returns information for the $LATEST version of all intents, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_GetSession.html",
                "description": "Returns session information for a specified bot, alias, and user ID.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "GetSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotType.html",
                "description": "Returns information about a specific version of a slot type. In addition to specifying the slot type name, you must also specify the slot type version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "GetSlotTypeVersions": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypeVersions.html",
                "description": "Returns information for all versions of a specific slot type.",
                "accessLevel": "List",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "GetSlotTypes": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypes.html",
                "description": "Returns information for the $LATEST version of all slot types, subject to filters provided by the client.",
                "accessLevel": "List"
            },
            "GetUtterancesView": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_GetUtterancesView.html",
                "description": "Returns a view of aggregate utterance data for versions of a bot for a recent time period.",
                "accessLevel": "List",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_ListTagsForResource.html",
                "description": "Lists tags for a Lex resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                }
            },
            "PostContent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostContent.html",
                "description": "Sends user input (text or speech) to Amazon Lex.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PostText": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html",
                "description": "Sends user input (text-only) to Amazon Lex.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PutBot": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutBot.html",
                "description": "Creates or updates the $LATEST version of a Amazon Lex conversational bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot version": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutBotAlias": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutBotAlias.html",
                "description": "Creates or updates an alias for the specific bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "PutIntent": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutIntent.html",
                "description": "Creates or updates the $LATEST version of an intent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "intent version": {
                        "required": true
                    }
                }
            },
            "PutSession": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PutSession.html",
                "description": "Creates a new session or modifies an existing session with an Amazon Lex bot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bot alias": {
                        "required": false
                    },
                    "bot version": {
                        "required": false
                    }
                }
            },
            "PutSlotType": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_PutSlotType.html",
                "description": "Creates or updates the $LATEST version of a slot type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "slottype version": {
                        "required": true
                    }
                }
            },
            "StartImport": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_StartImport.html",
                "description": "Starts a job to import a resource to Amazon Lex.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_TagResource.html",
                "description": "Adds or overwrites tags to a Lex resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_UntagResource.html",
                "description": "Removes tags from a Lex resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "bot": {
                        "required": false
                    },
                    "bot alias": {
                        "required": false
                    },
                    "channel": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            }
        };
        this.resourceTypes = {
            "bot": {
                "name": "bot",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "bot version": {
                "name": "bot version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotVersion}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "bot alias": {
                "name": "bot alias",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotAliasMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotAlias}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "channel": {
                "name": "channel",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_BotChannelAssociation.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:bot-channel:${BotName}:${BotAlias}:${ChannelName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "intent version": {
                "name": "intent version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_Intent.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:intent:${IntentName}:${IntentVersion}",
                "conditionKeys": []
            },
            "slottype version": {
                "name": "slottype version",
                "url": "https://docs.aws.amazon.com/lex/latest/dg/API_SlotTypeMetadata.html",
                "arn": "arn:${Partition}:lex:${Region}:${Account}:slottype:${SlotName}:${SlotVersion}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a new version based on the $LATEST version of the specified bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateBotVersion.html
     */
    createBotVersion() {
        this.add('lex:CreateBotVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateIntentVersion.html
     */
    createIntentVersion() {
        this.add('lex:CreateIntentVersion');
        return this;
    }
    /**
     * Creates a new version based on the $LATEST version of the specified slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_CreateSlotTypeVersion.html
     */
    createSlotTypeVersion() {
        this.add('lex:CreateSlotTypeVersion');
        return this;
    }
    /**
     * Deletes all versions of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBot.html
     */
    deleteBot() {
        this.add('lex:DeleteBot');
        return this;
    }
    /**
     * Deletes an alias for a specific bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotAlias.html
     */
    deleteBotAlias() {
        this.add('lex:DeleteBotAlias');
        return this;
    }
    /**
     * Deletes the association between a Amazon Lex bot alias and a messaging platform.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotChannelAssociation.html
     */
    deleteBotChannelAssociation() {
        this.add('lex:DeleteBotChannelAssociation');
        return this;
    }
    /**
     * Deletes a specific version of a bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteBotVersion.html
     */
    deleteBotVersion() {
        this.add('lex:DeleteBotVersion');
        return this;
    }
    /**
     * Deletes all versions of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntent.html
     */
    deleteIntent() {
        this.add('lex:DeleteIntent');
        return this;
    }
    /**
     * Deletes a specific version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteIntentVersion.html
     */
    deleteIntentVersion() {
        this.add('lex:DeleteIntentVersion');
        return this;
    }
    /**
     * Removes session information for a specified bot, alias, and user ID.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_DeleteSession.html
     */
    deleteSession() {
        this.add('lex:DeleteSession');
        return this;
    }
    /**
     * Deletes all versions of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotType.html
     */
    deleteSlotType() {
        this.add('lex:DeleteSlotType');
        return this;
    }
    /**
     * Deletes a specific version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteSlotTypeVersion.html
     */
    deleteSlotTypeVersion() {
        this.add('lex:DeleteSlotTypeVersion');
        return this;
    }
    /**
     * Deletes the information Amazon Lex maintains for utterances on a specific bot and userId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_DeleteUtterances.html
     */
    deleteUtterances() {
        this.add('lex:DeleteUtterances');
        return this;
    }
    /**
     * Returns information for a specific bot. In addition to the bot name, the bot version or alias is required.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBot.html
     */
    getBot() {
        this.add('lex:GetBot');
        return this;
    }
    /**
     * Returns information about a Amazon Lex bot alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAlias.html
     */
    getBotAlias() {
        this.add('lex:GetBotAlias');
        return this;
    }
    /**
     * Returns a list of aliases for a given Amazon Lex bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotAliases.html
     */
    getBotAliases() {
        this.add('lex:GetBotAliases');
        return this;
    }
    /**
     * Returns information about the association between a Amazon Lex bot and a messaging platform.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociation.html
     */
    getBotChannelAssociation() {
        this.add('lex:GetBotChannelAssociation');
        return this;
    }
    /**
     * Returns a list of all of the channels associated with a single bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotChannelAssociations.html
     */
    getBotChannelAssociations() {
        this.add('lex:GetBotChannelAssociations');
        return this;
    }
    /**
     * Returns information for all versions of a specific bot.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBotVersions.html
     */
    getBotVersions() {
        this.add('lex:GetBotVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all bots, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBots.html
     */
    getBots() {
        this.add('lex:GetBots');
        return this;
    }
    /**
     * Returns information about a built-in intent.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntent.html
     */
    getBuiltinIntent() {
        this.add('lex:GetBuiltinIntent');
        return this;
    }
    /**
     * Gets a list of built-in intents that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinIntents.html
     */
    getBuiltinIntents() {
        this.add('lex:GetBuiltinIntents');
        return this;
    }
    /**
     * Gets a list of built-in slot types that meet the specified criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetBuiltinSlotTypes.html
     */
    getBuiltinSlotTypes() {
        this.add('lex:GetBuiltinSlotTypes');
        return this;
    }
    /**
     * Exports Amazon Lex Resource in a requested format.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetExport.html
     */
    getExport() {
        this.add('lex:GetExport');
        return this;
    }
    /**
     * Gets information about an import job started with StartImport.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetImport.html
     */
    getImport() {
        this.add('lex:GetImport');
        return this;
    }
    /**
     * Returns information for a specific intent. In addition to the intent name, you must also specify the intent version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntent.html
     */
    getIntent() {
        this.add('lex:GetIntent');
        return this;
    }
    /**
     * Returns information for all versions of a specific intent.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntentVersions.html
     */
    getIntentVersions() {
        this.add('lex:GetIntentVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all intents, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetIntents.html
     */
    getIntents() {
        this.add('lex:GetIntents');
        return this;
    }
    /**
     * Returns session information for a specified bot, alias, and user ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_GetSession.html
     */
    getSession() {
        this.add('lex:GetSession');
        return this;
    }
    /**
     * Returns information about a specific version of a slot type. In addition to specifying the slot type name, you must also specify the slot type version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotType.html
     */
    getSlotType() {
        this.add('lex:GetSlotType');
        return this;
    }
    /**
     * Returns information for all versions of a specific slot type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypeVersions.html
     */
    getSlotTypeVersions() {
        this.add('lex:GetSlotTypeVersions');
        return this;
    }
    /**
     * Returns information for the $LATEST version of all slot types, subject to filters provided by the client.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetSlotTypes.html
     */
    getSlotTypes() {
        this.add('lex:GetSlotTypes');
        return this;
    }
    /**
     * Returns a view of aggregate utterance data for versions of a bot for a recent time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_GetUtterancesView.html
     */
    getUtterancesView() {
        this.add('lex:GetUtterancesView');
        return this;
    }
    /**
     * Lists tags for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('lex:ListTagsForResource');
        return this;
    }
    /**
     * Sends user input (text or speech) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostContent.html
     */
    postContent() {
        this.add('lex:PostContent');
        return this;
    }
    /**
     * Sends user input (text-only) to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PostText.html
     */
    postText() {
        this.add('lex:PostText');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a Amazon Lex conversational bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBot.html
     */
    putBot() {
        this.add('lex:PutBot');
        return this;
    }
    /**
     * Creates or updates an alias for the specific bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutBotAlias.html
     */
    putBotAlias() {
        this.add('lex:PutBotAlias');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of an intent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutIntent.html
     */
    putIntent() {
        this.add('lex:PutIntent');
        return this;
    }
    /**
     * Creates a new session or modifies an existing session with an Amazon Lex bot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_runtime_PutSession.html
     */
    putSession() {
        this.add('lex:PutSession');
        return this;
    }
    /**
     * Creates or updates the $LATEST version of a slot type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_PutSlotType.html
     */
    putSlotType() {
        this.add('lex:PutSlotType');
        return this;
    }
    /**
     * Starts a job to import a resource to Amazon Lex.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_StartImport.html
     */
    startImport() {
        this.add('lex:StartImport');
        return this;
    }
    /**
     * Adds or overwrites tags to a Lex resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_TagResource.html
     */
    tagResource() {
        this.add('lex:TagResource');
        return this;
    }
    /**
     * Removes tags from a Lex resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_UntagResource.html
     */
    untagResource() {
        this.add('lex:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type bot to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBot(botName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botVersion - Identifier for the botVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBotVersion(botName, botVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotVersion}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotVersion}', botVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bot alias to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotAliasMetadata.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBotAlias(botName, botAlias, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot:${BotName}:${BotAlias}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_BotChannelAssociation.html
     *
     * @param botName - Identifier for the botName.
     * @param botAlias - Identifier for the botAlias.
     * @param channelName - Identifier for the channelName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onChannel(botName, botAlias, channelName, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:bot-channel:${BotName}:${BotAlias}:${ChannelName}';
        arn = arn.replace('${BotName}', botName);
        arn = arn.replace('${BotAlias}', botAlias);
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type intent version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_Intent.html
     *
     * @param intentName - Identifier for the intentName.
     * @param intentVersion - Identifier for the intentVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIntentVersion(intentName, intentVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:intent:${IntentName}:${IntentVersion}';
        arn = arn.replace('${IntentName}', intentName);
        arn = arn.replace('${IntentVersion}', intentVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type slottype version to the statement
     *
     * https://docs.aws.amazon.com/lex/latest/dg/API_SlotTypeMetadata.html
     *
     * @param slotName - Identifier for the slotName.
     * @param slotVersion - Identifier for the slotVersion.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSlottypeVersion(slotName, slotVersion, account, region, partition) {
        var arn = 'arn:${Partition}:lex:${Region}:${Account}:slottype:${SlotName}:${SlotVersion}';
        arn = arn.replace('${SlotName}', slotName);
        arn = arn.replace('${SlotVersion}', slotVersion);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Enables you to control access based on the intents included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedIntents(value, operator) {
        return this.if(`lex:associatedIntents`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the slot types included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedSlotTypes(value, operator) {
        return this.if(`lex:associatedSlotTypes`, value, operator || 'StringLike');
    }
    /**
     * Enables you to control access based on the channel type included in the request.
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChannelType(value, operator) {
        return this.if(`lex:channelType`, value, operator || 'StringLike');
    }
}
exports.Lex = Lex;
//# sourceMappingURL=data:application/json;base64,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