"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Managedblockchain = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [managedblockchain](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedblockchain.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Managedblockchain extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [managedblockchain](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedblockchain.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'managedblockchain';
        this.actionList = {
            "CreateMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateMember.html",
                "description": "Grants permission to create a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "CreateNetwork": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNetwork.html",
                "description": "Grants permission to create an Amazon Managed Blockchain network.",
                "accessLevel": "Write"
            },
            "CreateNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNode.html",
                "description": "Grants permission to create a node within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "CreateProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateProposal.html",
                "description": "Grants permission to create a proposal that other blockchain network members can vote on to add or remove a member in an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "DeleteMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteMember.html",
                "description": "Grants permission to delete a member and all associated resources from an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "DeleteNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteNode.html",
                "description": "Grants permission to delete a node from a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "GetMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetMember.html",
                "description": "Grants permission to return detailed information about a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "GetNetwork": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNetwork.html",
                "description": "Grants permission to return detailed information about an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "GetNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNode.html",
                "description": "Grants permission to return detailed information about a node within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "GetProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetProposal.html",
                "description": "Grants permission to return detailed information about a proposal of an Amazon Managed Blockchain network.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListInvitations.html",
                "description": "Grants permission to list the invitations extended to the active AWS account from any Managed Blockchain network.",
                "accessLevel": "List"
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListMembers.html",
                "description": "Grants permission to list the members of an Amazon Managed Blockchain network and the properties of their memberships.",
                "accessLevel": "List",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "ListNetworks": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNetworks.html",
                "description": "Grants permission to return information about the Amazon Managed Blockchain networks in which the current AWS account has members.",
                "accessLevel": "List"
            },
            "ListNodes": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNodes.html",
                "description": "Grants permission to list the nodes within a member of an Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "ListProposalVotes": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposalVotes.html",
                "description": "Grants permission to list all votes for a proposal, including the value of the vote and the unique identifier of the member that cast the vote for the given Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            },
            "ListProposals": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposals.html",
                "description": "Grants permission to list proposals for the given Amazon Managed Blockchain network.",
                "accessLevel": "List",
                "resourceTypes": {
                    "network": {
                        "required": true
                    }
                }
            },
            "RejectInvitation": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_RejectInvitation.html",
                "description": "Grants permission to reject the invitation to join the blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "invitation": {
                        "required": true
                    }
                }
            },
            "UpdateMember": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateMember.html",
                "description": "Grants permission to update a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "member": {
                        "required": true
                    }
                }
            },
            "UpdateNode": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateNode.html",
                "description": "Grants permission to update a node from a member of an Amazon Managed Blockchain network.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "node": {
                        "required": true
                    }
                }
            },
            "VoteOnProposal": {
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_VoteOnProposal.html",
                "description": "Grants permission to cast a vote for a proposal on behalf of the blockchain network member specified.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "proposal": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "network": {
                "name": "network",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Network.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}::networks/${NetworkId}",
                "conditionKeys": []
            },
            "member": {
                "name": "member",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Member.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:members/${MemberId}",
                "conditionKeys": []
            },
            "node": {
                "name": "node",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Node.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:nodes/${NodeId}",
                "conditionKeys": []
            },
            "proposal": {
                "name": "proposal",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Proposal.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}::proposals/${ProposalId}",
                "conditionKeys": []
            },
            "invitation": {
                "name": "invitation",
                "url": "https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Invitation.html",
                "arn": "arn:${Partition}:managedblockchain:${Region}:${Account}:invitations/${InvitationId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to create a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateMember.html
     */
    createMember() {
        this.add('managedblockchain:CreateMember');
        return this;
    }
    /**
     * Grants permission to create an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNetwork.html
     */
    createNetwork() {
        this.add('managedblockchain:CreateNetwork');
        return this;
    }
    /**
     * Grants permission to create a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateNode.html
     */
    createNode() {
        this.add('managedblockchain:CreateNode');
        return this;
    }
    /**
     * Grants permission to create a proposal that other blockchain network members can vote on to add or remove a member in an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_CreateProposal.html
     */
    createProposal() {
        this.add('managedblockchain:CreateProposal');
        return this;
    }
    /**
     * Grants permission to delete a member and all associated resources from an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteMember.html
     */
    deleteMember() {
        this.add('managedblockchain:DeleteMember');
        return this;
    }
    /**
     * Grants permission to delete a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_DeleteNode.html
     */
    deleteNode() {
        this.add('managedblockchain:DeleteNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetMember.html
     */
    getMember() {
        this.add('managedblockchain:GetMember');
        return this;
    }
    /**
     * Grants permission to return detailed information about an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNetwork.html
     */
    getNetwork() {
        this.add('managedblockchain:GetNetwork');
        return this;
    }
    /**
     * Grants permission to return detailed information about a node within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetNode.html
     */
    getNode() {
        this.add('managedblockchain:GetNode');
        return this;
    }
    /**
     * Grants permission to return detailed information about a proposal of an Amazon Managed Blockchain network.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_GetProposal.html
     */
    getProposal() {
        this.add('managedblockchain:GetProposal');
        return this;
    }
    /**
     * Grants permission to list the invitations extended to the active AWS account from any Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListInvitations.html
     */
    listInvitations() {
        this.add('managedblockchain:ListInvitations');
        return this;
    }
    /**
     * Grants permission to list the members of an Amazon Managed Blockchain network and the properties of their memberships.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListMembers.html
     */
    listMembers() {
        this.add('managedblockchain:ListMembers');
        return this;
    }
    /**
     * Grants permission to return information about the Amazon Managed Blockchain networks in which the current AWS account has members.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNetworks.html
     */
    listNetworks() {
        this.add('managedblockchain:ListNetworks');
        return this;
    }
    /**
     * Grants permission to list the nodes within a member of an Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListNodes.html
     */
    listNodes() {
        this.add('managedblockchain:ListNodes');
        return this;
    }
    /**
     * Grants permission to list all votes for a proposal, including the value of the vote and the unique identifier of the member that cast the vote for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposalVotes.html
     */
    listProposalVotes() {
        this.add('managedblockchain:ListProposalVotes');
        return this;
    }
    /**
     * Grants permission to list proposals for the given Amazon Managed Blockchain network.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_ListProposals.html
     */
    listProposals() {
        this.add('managedblockchain:ListProposals');
        return this;
    }
    /**
     * Grants permission to reject the invitation to join the blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_RejectInvitation.html
     */
    rejectInvitation() {
        this.add('managedblockchain:RejectInvitation');
        return this;
    }
    /**
     * Grants permission to update a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateMember.html
     */
    updateMember() {
        this.add('managedblockchain:UpdateMember');
        return this;
    }
    /**
     * Grants permission to update a node from a member of an Amazon Managed Blockchain network.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_UpdateNode.html
     */
    updateNode() {
        this.add('managedblockchain:UpdateNode');
        return this;
    }
    /**
     * Grants permission to cast a vote for a proposal on behalf of the blockchain network member specified.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_VoteOnProposal.html
     */
    voteOnProposal() {
        this.add('managedblockchain:VoteOnProposal');
        return this;
    }
    /**
     * Adds a resource of type network to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Network.html
     *
     * @param networkId - Identifier for the networkId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNetwork(networkId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::networks/${NetworkId}';
        arn = arn.replace('${NetworkId}', networkId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type member to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Member.html
     *
     * @param memberId - Identifier for the memberId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMember(memberId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:members/${MemberId}';
        arn = arn.replace('${MemberId}', memberId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type node to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Node.html
     *
     * @param nodeId - Identifier for the nodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onNode(nodeId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:nodes/${NodeId}';
        arn = arn.replace('${NodeId}', nodeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type proposal to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Proposal.html
     *
     * @param proposalId - Identifier for the proposalId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProposal(proposalId, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}::proposals/${ProposalId}';
        arn = arn.replace('${ProposalId}', proposalId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type invitation to the statement
     *
     * https://docs.aws.amazon.com/managed-blockchain/latest/APIReference/API_Invitation.html
     *
     * @param invitationId - Identifier for the invitationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInvitation(invitationId, account, region, partition) {
        var arn = 'arn:${Partition}:managedblockchain:${Region}:${Account}:invitations/${InvitationId}';
        arn = arn.replace('${InvitationId}', invitationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Managedblockchain = Managedblockchain;
//# sourceMappingURL=data:application/json;base64,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