"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kafka = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kafka](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedstreamingforapachekafka.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kafka extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kafka](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmanagedstreamingforapachekafka.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kafka';
        this.actionList = {
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster",
                "description": "Grants permission to create a cluster.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration",
                "description": "Grants permission to create a configuration.",
                "accessLevel": "Write"
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster",
                "description": "Grants permission to delete a cluster.",
                "accessLevel": "Write"
            },
            "DescribeCluster": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster",
                "description": "Grants permission to describe a cluster.",
                "accessLevel": "Read"
            },
            "DescribeClusterOperation": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation",
                "description": "Returns a description of the cluster operation specified by the ARN.",
                "accessLevel": "Read"
            },
            "DescribeConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn.html#DescribeConfiguration",
                "description": "Grants permission to describe a configuration.",
                "accessLevel": "Read"
            },
            "DescribeConfigurationRevision": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn-revision.html#DescribeConfigurationRevision",
                "description": "Grants permission to describe a configuration revision.",
                "accessLevel": "Read"
            },
            "GetBootstrapBrokers": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers",
                "description": "Grants permission to get connection details for the broker nodes in a cluster.",
                "accessLevel": "Read"
            },
            "GetCompatibleKafkaVersions": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions",
                "description": "Returns a list of the Apache Kafka versions to which you can update this cluster.",
                "accessLevel": "List"
            },
            "ListClusterOperations": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations",
                "description": "Returns a list of all the operations that have been performed on the specified MSK cluster.",
                "accessLevel": "Read"
            },
            "ListClusters": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters",
                "description": "Grants permission to return a list of all clusters in the current account.",
                "accessLevel": "List"
            },
            "ListConfigurations": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration",
                "description": "Grants permission to return a list of all configurations in the current account.",
                "accessLevel": "List"
            },
            "ListNodes": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes",
                "description": "Grants permission to return a list of nodes in a cluster.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource",
                "description": "Grants permission to list tags of a MSK resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource",
                "description": "Grants permission to tag a MSK resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource",
                "description": "Grants permission to remove tags from a MSK resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateBrokerCount": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount",
                "description": "Updates the number of broker nodes of the cluster.",
                "accessLevel": "Write"
            },
            "UpdateBrokerStorage": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage",
                "description": "Updates the storage size of the broker nodes of the cluster",
                "accessLevel": "Write"
            },
            "UpdateClusterConfiguration": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration",
                "description": "Update Kafka configuration running on a cluster.",
                "accessLevel": "Write"
            },
            "UpdateClusterKafkaVersion": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion",
                "description": "Updates the cluster to the specified Apache Kafka version.",
                "accessLevel": "Write"
            },
            "UpdateMonitoring": {
                "url": "https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring",
                "description": "Updates the monitoring settings for the cluster.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "cluster": {
                "name": "cluster",
                "url": "",
                "arn": "arn:${Partition}:kafka:${Region}:${Account}:cluster/${ClusterName}/${UUID}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#CreateCluster
     */
    createCluster() {
        this.add('kafka:CreateCluster');
        return this;
    }
    /**
     * Grants permission to create a configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    createConfiguration() {
        this.add('kafka:CreateConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DeleteCluster
     */
    deleteCluster() {
        this.add('kafka:DeleteCluster');
        return this;
    }
    /**
     * Grants permission to describe a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn.html#DescribeCluster
     */
    describeCluster() {
        this.add('kafka:DescribeCluster');
        return this;
    }
    /**
     * Returns a description of the cluster operation specified by the ARN.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/operations-clusteroperationarn.html#DescribeClusterOperation
     */
    describeClusterOperation() {
        this.add('kafka:DescribeClusterOperation');
        return this;
    }
    /**
     * Grants permission to describe a configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn.html#DescribeConfiguration
     */
    describeConfiguration() {
        this.add('kafka:DescribeConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a configuration revision.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations-configurationarn-revision.html#DescribeConfigurationRevision
     */
    describeConfigurationRevision() {
        this.add('kafka:DescribeConfigurationRevision');
        return this;
    }
    /**
     * Grants permission to get connection details for the broker nodes in a cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-bootstrap-brokers.html#GetBootstrapBrokers
     */
    getBootstrapBrokers() {
        this.add('kafka:GetBootstrapBrokers');
        return this;
    }
    /**
     * Returns a list of the Apache Kafka versions to which you can update this cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/compatible-kafka-versions.html#GetCompatibleKafkaVersions
     */
    getCompatibleKafkaVersions() {
        this.add('kafka:GetCompatibleKafkaVersions');
        return this;
    }
    /**
     * Returns a list of all the operations that have been performed on the specified MSK cluster.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-operations.html#ListClusterOperations
     */
    listClusterOperations() {
        this.add('kafka:ListClusterOperations');
        return this;
    }
    /**
     * Grants permission to return a list of all clusters in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters.html#ListClusters
     */
    listClusters() {
        this.add('kafka:ListClusters');
        return this;
    }
    /**
     * Grants permission to return a list of all configurations in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/configurations.html#CreateConfiguration
     */
    listConfigurations() {
        this.add('kafka:ListConfigurations');
        return this;
    }
    /**
     * Grants permission to return a list of nodes in a cluster.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes.html#ListNodes
     */
    listNodes() {
        this.add('kafka:ListNodes');
        return this;
    }
    /**
     * Grants permission to list tags of a MSK resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#ListTagsForResource
     */
    listTagsForResource() {
        this.add('kafka:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to tag a MSK resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#TagResource
     */
    tagResource() {
        this.add('kafka:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a MSK resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/tags-resourcearn.html#UntagResource
     */
    untagResource() {
        this.add('kafka:UntagResource');
        return this;
    }
    /**
     * Updates the number of broker nodes of the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-count.html#UpdateBrokerCount
     */
    updateBrokerCount() {
        this.add('kafka:UpdateBrokerCount');
        return this;
    }
    /**
     * Updates the storage size of the broker nodes of the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-nodes-storage.html#UpdateBrokerStorage
     */
    updateBrokerStorage() {
        this.add('kafka:UpdateBrokerStorage');
        return this;
    }
    /**
     * Update Kafka configuration running on a cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-configuration.html#UpdateClusterConfiguration
     */
    updateClusterConfiguration() {
        this.add('kafka:UpdateClusterConfiguration');
        return this;
    }
    /**
     * Updates the cluster to the specified Apache Kafka version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-version.html#UpdateClusterKafkaVersion
     */
    updateClusterKafkaVersion() {
        this.add('kafka:UpdateClusterKafkaVersion');
        return this;
    }
    /**
     * Updates the monitoring settings for the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/msk/1.0/apireference/clusters-clusterarn-monitoring.html#UpdateMonitoring
     */
    updateMonitoring() {
        this.add('kafka:UpdateMonitoring');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * @param clusterName - Identifier for the clusterName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCluster(clusterName, uUID, account, region, partition) {
        var arn = 'arn:${Partition}:kafka:${Region}:${Account}:cluster/${ClusterName}/${UUID}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${UUID}', uUID);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kafka = Kafka;
//# sourceMappingURL=data:application/json;base64,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