"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Redshift = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [redshift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonredshift.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Redshift extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [redshift](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonredshift.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'redshift';
        this.actionList = {
            "AcceptReservedNodeExchange": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_AcceptReservedNodeExchange.html",
                "description": "Grants permission to exchange a DC1 reserved node for a DC2 reserved node with no changes to the configuration",
                "accessLevel": "Write"
            },
            "AuthorizeClusterSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeClusterSecurityGroupIngress.html",
                "description": "Grants permission to add an inbound (ingress) rule to an Amazon Redshift security group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "securitygroup": {
                        "required": true
                    },
                    "securitygroupingress-ec2securitygroup": {
                        "required": true
                    }
                }
            },
            "AuthorizeSnapshotAccess": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeSnapshotAccess.html",
                "description": "Grants permission to the specified AWS account to restore a snapshot",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "BatchDeleteClusterSnapshots": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchDeleteClusterSnapshots.html",
                "description": "Grants permission to delete snapshots in a batch of size upto 100",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "BatchModifyClusterSnapshots": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchModifyClusterSnapshots.html",
                "description": "Grants permission to modify settings for a list of snapshots",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "CancelQuery": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to cancel a query through the Amazon Redshift console",
                "accessLevel": "Write"
            },
            "CancelQuerySession": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to see queries in the Amazon Redshift console",
                "accessLevel": "Write"
            },
            "CancelResize": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CancelResize.html",
                "description": "Grants permission to cancel a resize operation",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "CopyClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CopyClusterSnapshot.html",
                "description": "Grants permission to copy a cluster snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateCluster.html",
                "description": "Grants permission to create a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterParameterGroup.html",
                "description": "Grants permission to create an Amazon Redshift parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parametergroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateClusterSecurityGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSecurityGroup.html",
                "description": "Grants permission to create an Amazon Redshift security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securitygroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSnapshot.html",
                "description": "Grants permission to create a manual snapshot of the specified cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateClusterSubnetGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSubnetGroup.html",
                "description": "Grants permission to create an Amazon Redshift subnet group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnetgroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateClusterUser": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/generating-iam-credentials-role-permissions.html",
                "description": "Grants permission to automatically create the specified Amazon Redshift user if it does not exist",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "dbuser": {
                        "required": true
                    }
                },
                "conditions": [
                    "redshift:DbUser"
                ]
            },
            "CreateEventSubscription": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateEventSubscription.html",
                "description": "Grants permission to create an Amazon Redshift event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "eventsubscription": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateHsmClientCertificate": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmClientCertificate.html",
                "description": "Grants permission to create an HSM client certificate that a cluster uses to connect to an HSM",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hsmclientcertificate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateHsmConfiguration": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmConfiguration.html",
                "description": "Grants permission to create an HSM configuration that contains information required by a cluster to store and use database encryption keys in a hardware security module (HSM)",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hsmconfiguration": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSavedQuery": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to create saved SQL queries through the Amazon Redshift console",
                "accessLevel": "Write"
            },
            "CreateScheduledAction": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateScheduledAction.html",
                "description": "Grants permission to create an Amazon Redshift scheduled action",
                "accessLevel": "Write"
            },
            "CreateSnapshotCopyGrant": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotCopyGrant.html",
                "description": "Grants permission to create a snapshot copy grant and encrypt copied snapshots in a destination AWS Region",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "snapshotcopygrant": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSnapshotSchedule": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotSchedule.html",
                "description": "Grants permission to create a snapshot schedule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshotschedule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTags": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateTags.html",
                "description": "Grants permission to add one or more tags to a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "dbgroup": {
                        "required": false
                    },
                    "dbname": {
                        "required": false
                    },
                    "dbuser": {
                        "required": false
                    },
                    "eventsubscription": {
                        "required": false
                    },
                    "hsmclientcertificate": {
                        "required": false
                    },
                    "hsmconfiguration": {
                        "required": false
                    },
                    "parametergroup": {
                        "required": false
                    },
                    "securitygroup": {
                        "required": false
                    },
                    "securitygroupingress-cidr": {
                        "required": false
                    },
                    "securitygroupingress-ec2securitygroup": {
                        "required": false
                    },
                    "snapshot": {
                        "required": false
                    },
                    "snapshotcopygrant": {
                        "required": false
                    },
                    "snapshotschedule": {
                        "required": false
                    },
                    "subnetgroup": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteCluster.html",
                "description": "Grants permission to delete a previously provisioned cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DeleteClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterParameterGroup.html",
                "description": "Grants permission to delete an Amazon Redshift parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parametergroup": {
                        "required": true
                    }
                }
            },
            "DeleteClusterSecurityGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSecurityGroup.html",
                "description": "Grants permission to delete an Amazon Redshift security group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securitygroup": {
                        "required": true
                    }
                }
            },
            "DeleteClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSnapshot.html",
                "description": "Grants permission to delete a manual snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "DeleteClusterSubnetGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSubnetGroup.html",
                "description": "Grants permission to delete a cluster subnet group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnetgroup": {
                        "required": true
                    }
                }
            },
            "DeleteEventSubscription": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteEventSubscription.html",
                "description": "Grants permission to delete an Amazon Redshift event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "eventsubscription": {
                        "required": true
                    }
                }
            },
            "DeleteHsmClientCertificate": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmClientCertificate.html",
                "description": "Grants permission to delete an HSM client certificate",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hsmclientcertificate": {
                        "required": true
                    }
                }
            },
            "DeleteHsmConfiguration": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmConfiguration.html",
                "description": "Grants permission to delete an Amazon Redshift HSM configuration",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hsmconfiguration": {
                        "required": true
                    }
                }
            },
            "DeleteSavedQueries": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to delete saved SQL queries through the Amazon Redshift console",
                "accessLevel": "Write"
            },
            "DeleteScheduledAction": {
                "url": "",
                "description": "Grants permission to delete an Amazon Redshift scheduled action",
                "accessLevel": "Write"
            },
            "DeleteSnapshotCopyGrant": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotCopyGrant.html",
                "description": "Grants permission to delete a snapshot copy grant",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshotcopygrant": {
                        "required": true
                    }
                }
            },
            "DeleteSnapshotSchedule": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotSchedule.html",
                "description": "Grants permission to delete a snapshot schedule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshotschedule": {
                        "required": true
                    }
                }
            },
            "DeleteTags": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteTags.html",
                "description": "Grants permission to delete a tag or tags from a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "dbgroup": {
                        "required": false
                    },
                    "dbname": {
                        "required": false
                    },
                    "dbuser": {
                        "required": false
                    },
                    "eventsubscription": {
                        "required": false
                    },
                    "hsmclientcertificate": {
                        "required": false
                    },
                    "hsmconfiguration": {
                        "required": false
                    },
                    "parametergroup": {
                        "required": false
                    },
                    "securitygroup": {
                        "required": false
                    },
                    "securitygroupingress-cidr": {
                        "required": false
                    },
                    "securitygroupingress-ec2securitygroup": {
                        "required": false
                    },
                    "snapshot": {
                        "required": false
                    },
                    "snapshotcopygrant": {
                        "required": false
                    },
                    "snapshotschedule": {
                        "required": false
                    },
                    "subnetgroup": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "DescribeAccountAttributes": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeAccountAttributes.html",
                "description": "Grants permission to describe attributes attached to the specified AWS account",
                "accessLevel": "Read"
            },
            "DescribeClusterDbRevisions": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterDbRevisions.html",
                "description": "Grants permission to describe database revisions for a cluster",
                "accessLevel": "List"
            },
            "DescribeClusterParameterGroups": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameterGroups.html",
                "description": "Grants permission to describe Amazon Redshift parameter groups, including parameter groups you created and the default parameter group",
                "accessLevel": "Read"
            },
            "DescribeClusterParameters": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameters.html",
                "description": "Grants permission to describe parameters contained within an Amazon Redshift parameter group",
                "accessLevel": "Read",
                "resourceTypes": {
                    "parametergroup": {
                        "required": true
                    }
                }
            },
            "DescribeClusterSecurityGroups": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSecurityGroups.html",
                "description": "Grants permission to describe Amazon Redshift security groups",
                "accessLevel": "Read"
            },
            "DescribeClusterSnapshots": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSnapshots.html",
                "description": "Grants permission to describe one or more snapshot objects, which contain metadata about your cluster snapshots",
                "accessLevel": "Read"
            },
            "DescribeClusterSubnetGroups": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSubnetGroups.html",
                "description": "Grants permission to describe one or more cluster subnet group objects, which contain metadata about your cluster subnet groups",
                "accessLevel": "Read"
            },
            "DescribeClusterTracks": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterTracks.html",
                "description": "Grants permission to describe available maintenance tracks",
                "accessLevel": "List"
            },
            "DescribeClusterVersions": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterVersions.html",
                "description": "Grants permission to describe available Amazon Redshift cluster versions",
                "accessLevel": "Read"
            },
            "DescribeClusters": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusters.html",
                "description": "Grants permission to describe properties of provisioned clusters",
                "accessLevel": "List"
            },
            "DescribeDefaultClusterParameters": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeDefaultClusterParameters.html",
                "description": "Grants permission to describe parameter settings for a parameter group family",
                "accessLevel": "Read"
            },
            "DescribeEventCategories": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventCategories.html",
                "description": "Grants permission to describe event categories for all event source types, or for a specified source type",
                "accessLevel": "Read"
            },
            "DescribeEventSubscriptions": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventSubscriptions.html",
                "description": "Grants permission to describe Amazon Redshift event notification subscriptions for the specified AWS account",
                "accessLevel": "Read"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEvents.html",
                "description": "Grants permission to describe events related to clusters, security groups, snapshots, and parameter groups for the past 14 days",
                "accessLevel": "List"
            },
            "DescribeHsmClientCertificates": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmClientCertificates.html",
                "description": "Grants permission to describe HSM client certificates",
                "accessLevel": "Read"
            },
            "DescribeHsmConfigurations": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmConfigurations.html",
                "description": "Grants permission to describe Amazon Redshift HSM configurations",
                "accessLevel": "Read"
            },
            "DescribeLoggingStatus": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeLoggingStatus.html",
                "description": "Grants permission to describe whether information, such as queries and connection attempts, is being logged for a cluster",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DescribeNodeConfigurationOptions": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeNodeConfigurationOptions.html",
                "description": "Grants permission to describe properties of possible node configurations such as node type, number of nodes, and disk usage for the specified action type",
                "accessLevel": "List"
            },
            "DescribeOrderableClusterOptions": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeOrderableClusterOptions.html",
                "description": "Grants permission to describe orderable cluster options",
                "accessLevel": "Read"
            },
            "DescribeQuery": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to describe a query through the Amazon Redshift console",
                "accessLevel": "Read"
            },
            "DescribeReservedNodeOfferings": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodeOfferings.html",
                "description": "Grants permission to describe available reserved node offerings by Amazon Redshift",
                "accessLevel": "Read"
            },
            "DescribeReservedNodes": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodes.html",
                "description": "Grants permission to describe the reserved nodes",
                "accessLevel": "Read"
            },
            "DescribeResize": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeResize.html",
                "description": "Grants permission to describe the last resize operation for a cluster",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DescribeSavedQueries": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to describe saved queries through the Amazon Redshift console",
                "accessLevel": "Read"
            },
            "DescribeScheduledActions": {
                "url": "",
                "description": "Grants permission to describe created Amazon Redshift scheduled actions",
                "accessLevel": "Read"
            },
            "DescribeSnapshotCopyGrants": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotCopyGrants.html",
                "description": "Grants permission to describe snapshot copy grants owned by the specified AWS account in the destination AWS Region",
                "accessLevel": "Read"
            },
            "DescribeSnapshotSchedules": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotSchedules.html",
                "description": "Grants permission to describe snapshot schedules",
                "accessLevel": "Read",
                "resourceTypes": {
                    "snapshotschedule": {
                        "required": true
                    }
                }
            },
            "DescribeStorage": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeStorage.html",
                "description": "Grants permission to describe account level backups storage size and provisional storage",
                "accessLevel": "Read"
            },
            "DescribeTable": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to describe a table through the Amazon Redshift console",
                "accessLevel": "Read"
            },
            "DescribeTableRestoreStatus": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTableRestoreStatus.html",
                "description": "Grants permission to describe status of one or more table restore requests made using the RestoreTableFromClusterSnapshot API action",
                "accessLevel": "Read"
            },
            "DescribeTags": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTags.html",
                "description": "Grants permission to describe tags",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cluster": {
                        "required": false
                    },
                    "dbgroup": {
                        "required": false
                    },
                    "dbname": {
                        "required": false
                    },
                    "dbuser": {
                        "required": false
                    },
                    "eventsubscription": {
                        "required": false
                    },
                    "hsmclientcertificate": {
                        "required": false
                    },
                    "hsmconfiguration": {
                        "required": false
                    },
                    "parametergroup": {
                        "required": false
                    },
                    "securitygroup": {
                        "required": false
                    },
                    "securitygroupingress-cidr": {
                        "required": false
                    },
                    "securitygroupingress-ec2securitygroup": {
                        "required": false
                    },
                    "snapshot": {
                        "required": false
                    },
                    "snapshotcopygrant": {
                        "required": false
                    },
                    "snapshotschedule": {
                        "required": false
                    },
                    "subnetgroup": {
                        "required": false
                    }
                }
            },
            "DisableLogging": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableLogging.html",
                "description": "Grants permission to disable logging information, such as queries and connection attempts, for a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DisableSnapshotCopy": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableSnapshotCopy.html",
                "description": "Grants permission to disable the automatic copy of snapshots for a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "EnableLogging": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableLogging.html",
                "description": "Grants permission to enable logging information, such as queries and connection attempts, for a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "EnableSnapshotCopy": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableSnapshotCopy.html",
                "description": "Grants permission to enable the automatic copy of snapshots for a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ExecuteQuery": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to execute a query through the Amazon Redshift console",
                "accessLevel": "Write"
            },
            "FetchResults": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to fetch query results through the Amazon Redshift console",
                "accessLevel": "Read"
            },
            "GetClusterCredentials": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html",
                "description": "Grants permission to get temporary credentials to access an Amazon Redshift database by the specified AWS account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dbuser": {
                        "required": true
                    },
                    "dbgroup": {
                        "required": false
                    },
                    "dbname": {
                        "required": false
                    }
                },
                "conditions": [
                    "redshift:DbName",
                    "redshift:DbUser",
                    "redshift:DurationSeconds"
                ]
            },
            "GetReservedNodeExchangeOfferings": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetReservedNodeExchangeOfferings.html",
                "description": "Grants permission to get an array of DC2 ReservedNodeOfferings that matches the payment type, term, and usage price of the given DC1 reserved node",
                "accessLevel": "Read"
            },
            "JoinGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html",
                "description": "Grants permission to join the specified Amazon Redshift group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "dbgroup": {
                        "required": true
                    }
                }
            },
            "ListDatabases": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to list databases through the Amazon Redshift console",
                "accessLevel": "List"
            },
            "ListSavedQueries": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to list saved queries through the Amazon Redshift console",
                "accessLevel": "List"
            },
            "ListSchemas": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to list schemas through the Amazon Redshift console",
                "accessLevel": "List"
            },
            "ListTables": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to list tables through the Amazon Redshift console",
                "accessLevel": "List"
            },
            "ModifyCluster": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyCluster.html",
                "description": "Grants permission to modify the settings of a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ModifyClusterDbRevision": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterDbRevision.html",
                "description": "Grants permission to modify the database revision of a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ModifyClusterIamRoles": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterIamRoles.html",
                "description": "Grants permission to modify the list of AWS Identity and Access Management (IAM) roles that can be used by a cluster to access other AWS services",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ModifyClusterMaintenance": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterMaintenance.html",
                "description": "Grants permission to modify the maintenance settings of a cluster",
                "accessLevel": "Write"
            },
            "ModifyClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterParameterGroup.html",
                "description": "Grants permission to modify the parameters of a parameter group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parametergroup": {
                        "required": true
                    }
                }
            },
            "ModifyClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshot.html",
                "description": "Grants permission to modify the settings of a snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ModifyClusterSnapshotSchedule": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshotSchedule.html",
                "description": "Grants permission to modify a snapshot schedule for a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ModifyClusterSubnetGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSubnetGroup.html",
                "description": "Grants permission to modify a cluster subnet group to include the specified list of VPC subnets",
                "accessLevel": "Write",
                "resourceTypes": {
                    "subnetgroup": {
                        "required": true
                    }
                }
            },
            "ModifyEventSubscription": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyEventSubscription.html",
                "description": "Grants permission to modify an existing Amazon Redshift event notification subscription",
                "accessLevel": "Write",
                "resourceTypes": {
                    "eventsubscription": {
                        "required": true
                    }
                }
            },
            "ModifySavedQuery": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to modify an existing saved query through the Amazon Redshift console",
                "accessLevel": "Write"
            },
            "ModifyScheduledAction": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyScheduledAction.html",
                "description": "Grants permission to modify an existing Amazon Redshift scheduled action",
                "accessLevel": "Write"
            },
            "ModifySnapshotCopyRetentionPeriod": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotCopyRetentionPeriod.html",
                "description": "Grants permission to modify the number of days to retain snapshots in the destination AWS Region after they are copied from the source AWS Region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ModifySnapshotSchedule": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotSchedule.html",
                "description": "Grants permission to modify a snapshot schedule",
                "accessLevel": "Write",
                "resourceTypes": {
                    "snapshotschedule": {
                        "required": true
                    }
                }
            },
            "PauseCluster": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_PauseCluster.html",
                "description": "Grants permission to pause a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "PurchaseReservedNodeOffering": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_PurchaseReservedNodeOffering.html",
                "description": "Grants permission to purchase a reserved node",
                "accessLevel": "Write"
            },
            "RebootCluster": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_RebootCluster.html",
                "description": "Grants permission to reboot a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ResetClusterParameterGroup": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResetClusterParameterGroup.html",
                "description": "Grants permission to set one or more parameters of a parameter group to their default values and set the source values of the parameters to \"engine-default\"",
                "accessLevel": "Write",
                "resourceTypes": {
                    "parametergroup": {
                        "required": true
                    }
                }
            },
            "ResizeCluster": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResizeCluster.html",
                "description": "Grants permission to change the size of a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "RestoreFromClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreFromClusterSnapshot.html",
                "description": "Grants permission to create a cluster from a snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "RestoreTableFromClusterSnapshot": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreTableFromClusterSnapshot.html",
                "description": "Grants permission to create a table from a table in an Amazon Redshift cluster snapshot",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    },
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "ResumeCluster": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResumeCluster.html",
                "description": "Grants permission to resume a cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "RevokeClusterSecurityGroupIngress": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeClusterSecurityGroupIngress.html",
                "description": "Grants permission to revoke an ingress rule in an Amazon Redshift security group for a previously authorized IP range or Amazon EC2 security group",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "securitygroup": {
                        "required": true
                    },
                    "securitygroupingress-ec2securitygroup": {
                        "required": true
                    }
                }
            },
            "RevokeSnapshotAccess": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeSnapshotAccess.html",
                "description": "Grants permission to revoke access from the specified AWS account to restore a snapshot",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "snapshot": {
                        "required": true
                    }
                }
            },
            "RotateEncryptionKey": {
                "url": "https://docs.aws.amazon.com/redshift/latest/APIReference/API_RotateEncryptionKey.html",
                "description": "Grants permission to rotate an encryption key for a cluster",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ViewQueriesFromConsole": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to view query results through the Amazon Redshift console",
                "accessLevel": "List"
            },
            "ViewQueriesInConsole": {
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html",
                "description": "Grants permission to terminate running queries and loads through the Amazon Redshift console",
                "accessLevel": "List"
            }
        };
        this.resourceTypes = {
            "cluster": {
                "name": "cluster",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-clusters.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:cluster:${ClusterName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dbgroup": {
                "name": "dbgroup",
                "url": "https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_GROUP.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:dbgroup:${ClusterName}/${DbGroup}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dbname": {
                "name": "dbname",
                "url": "https://docs.aws.amazon.com/redshift/latest/dg/t_creating_database.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:dbname:${ClusterName}/${DbName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dbuser": {
                "name": "dbuser",
                "url": "https://docs.aws.amazon.com/redshift/latest/dg/r_Users.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:dbuser:${ClusterName}/${DbUser}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "eventsubscription": {
                "name": "eventsubscription",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-events.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:eventsubscription:${EventSubscriptionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "hsmclientcertificate": {
                "name": "hsmclientcertificate",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:hsmclientcertificate:${HSMClientCertificateId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "hsmconfiguration": {
                "name": "hsmconfiguration",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:hsmconfiguration:${HSMConfigurationId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "parametergroup": {
                "name": "parametergroup",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-parameter-groups.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:parametergroup:${ParameterGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "securitygroup": {
                "name": "securitygroup",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:securitygroup:${SecurityGroupName}/ec2securitygroup/${Owner}/${Ec2SecurityGroupId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "securitygroupingress-cidr": {
                "name": "securitygroupingress-cidr",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:securitygroupingress:${SecurityGroupName}/cidrip/${IpRange}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "securitygroupingress-ec2securitygroup": {
                "name": "securitygroupingress-ec2securitygroup",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:securitygroupingress:${SecurityGroupName}/ec2securitygroup/${Owner}/${Ece2SecuritygroupId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "snapshot": {
                "name": "snapshot",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:snapshot:${ClusterName}/${SnapshotName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "snapshotcopygrant": {
                "name": "snapshotcopygrant",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#configure-snapshot-copy-grant",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:snapshotcopygrant:${SnapshotCopyGrantName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "snapshotschedule": {
                "name": "snapshotschedule",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:snapshotschedule:${ParameterGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "subnetgroup": {
                "name": "subnetgroup",
                "url": "https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-cluster-subnet-groups.html",
                "arn": "arn:${Partition}:redshift:${Region}:${Account}:subnetgroup:${SubnetGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to exchange a DC1 reserved node for a DC2 reserved node with no changes to the configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AcceptReservedNodeExchange.html
     */
    acceptReservedNodeExchange() {
        this.add('redshift:AcceptReservedNodeExchange');
        return this;
    }
    /**
     * Grants permission to add an inbound (ingress) rule to an Amazon Redshift security group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeClusterSecurityGroupIngress.html
     */
    authorizeClusterSecurityGroupIngress() {
        this.add('redshift:AuthorizeClusterSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to the specified AWS account to restore a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_AuthorizeSnapshotAccess.html
     */
    authorizeSnapshotAccess() {
        this.add('redshift:AuthorizeSnapshotAccess');
        return this;
    }
    /**
     * Grants permission to delete snapshots in a batch of size upto 100
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchDeleteClusterSnapshots.html
     */
    batchDeleteClusterSnapshots() {
        this.add('redshift:BatchDeleteClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to modify settings for a list of snapshots
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_BatchModifyClusterSnapshots.html
     */
    batchModifyClusterSnapshots() {
        this.add('redshift:BatchModifyClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to cancel a query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    cancelQuery() {
        this.add('redshift:CancelQuery');
        return this;
    }
    /**
     * Grants permission to see queries in the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    cancelQuerySession() {
        this.add('redshift:CancelQuerySession');
        return this;
    }
    /**
     * Grants permission to cancel a resize operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CancelResize.html
     */
    cancelResize() {
        this.add('redshift:CancelResize');
        return this;
    }
    /**
     * Grants permission to copy a cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CopyClusterSnapshot.html
     */
    copyClusterSnapshot() {
        this.add('redshift:CopyClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateCluster.html
     */
    createCluster() {
        this.add('redshift:CreateCluster');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterParameterGroup.html
     */
    createClusterParameterGroup() {
        this.add('redshift:CreateClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSecurityGroup.html
     */
    createClusterSecurityGroup() {
        this.add('redshift:CreateClusterSecurityGroup');
        return this;
    }
    /**
     * Grants permission to create a manual snapshot of the specified cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSnapshot.html
     */
    createClusterSnapshot() {
        this.add('redshift:CreateClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateClusterSubnetGroup.html
     */
    createClusterSubnetGroup() {
        this.add('redshift:CreateClusterSubnetGroup');
        return this;
    }
    /**
     * Grants permission to automatically create the specified Amazon Redshift user if it does not exist
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/generating-iam-credentials-role-permissions.html
     */
    createClusterUser() {
        this.add('redshift:CreateClusterUser');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateEventSubscription.html
     */
    createEventSubscription() {
        this.add('redshift:CreateEventSubscription');
        return this;
    }
    /**
     * Grants permission to create an HSM client certificate that a cluster uses to connect to an HSM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmClientCertificate.html
     */
    createHsmClientCertificate() {
        this.add('redshift:CreateHsmClientCertificate');
        return this;
    }
    /**
     * Grants permission to create an HSM configuration that contains information required by a cluster to store and use database encryption keys in a hardware security module (HSM)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateHsmConfiguration.html
     */
    createHsmConfiguration() {
        this.add('redshift:CreateHsmConfiguration');
        return this;
    }
    /**
     * Grants permission to create saved SQL queries through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    createSavedQuery() {
        this.add('redshift:CreateSavedQuery');
        return this;
    }
    /**
     * Grants permission to create an Amazon Redshift scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateScheduledAction.html
     */
    createScheduledAction() {
        this.add('redshift:CreateScheduledAction');
        return this;
    }
    /**
     * Grants permission to create a snapshot copy grant and encrypt copied snapshots in a destination AWS Region
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotCopyGrant.html
     */
    createSnapshotCopyGrant() {
        this.add('redshift:CreateSnapshotCopyGrant');
        return this;
    }
    /**
     * Grants permission to create a snapshot schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateSnapshotSchedule.html
     */
    createSnapshotSchedule() {
        this.add('redshift:CreateSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a specified resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_CreateTags.html
     */
    createTags() {
        this.add('redshift:CreateTags');
        return this;
    }
    /**
     * Grants permission to delete a previously provisioned cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteCluster.html
     */
    deleteCluster() {
        this.add('redshift:DeleteCluster');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterParameterGroup.html
     */
    deleteClusterParameterGroup() {
        this.add('redshift:DeleteClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSecurityGroup.html
     */
    deleteClusterSecurityGroup() {
        this.add('redshift:DeleteClusterSecurityGroup');
        return this;
    }
    /**
     * Grants permission to delete a manual snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSnapshot.html
     */
    deleteClusterSnapshot() {
        this.add('redshift:DeleteClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to delete a cluster subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteClusterSubnetGroup.html
     */
    deleteClusterSubnetGroup() {
        this.add('redshift:DeleteClusterSubnetGroup');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteEventSubscription.html
     */
    deleteEventSubscription() {
        this.add('redshift:DeleteEventSubscription');
        return this;
    }
    /**
     * Grants permission to delete an HSM client certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmClientCertificate.html
     */
    deleteHsmClientCertificate() {
        this.add('redshift:DeleteHsmClientCertificate');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift HSM configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteHsmConfiguration.html
     */
    deleteHsmConfiguration() {
        this.add('redshift:DeleteHsmConfiguration');
        return this;
    }
    /**
     * Grants permission to delete saved SQL queries through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    deleteSavedQueries() {
        this.add('redshift:DeleteSavedQueries');
        return this;
    }
    /**
     * Grants permission to delete an Amazon Redshift scheduled action
     *
     * Access Level: Write
     */
    deleteScheduledAction() {
        this.add('redshift:DeleteScheduledAction');
        return this;
    }
    /**
     * Grants permission to delete a snapshot copy grant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotCopyGrant.html
     */
    deleteSnapshotCopyGrant() {
        this.add('redshift:DeleteSnapshotCopyGrant');
        return this;
    }
    /**
     * Grants permission to delete a snapshot schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteSnapshotSchedule.html
     */
    deleteSnapshotSchedule() {
        this.add('redshift:DeleteSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to delete a tag or tags from a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DeleteTags.html
     */
    deleteTags() {
        this.add('redshift:DeleteTags');
        return this;
    }
    /**
     * Grants permission to describe attributes attached to the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeAccountAttributes.html
     */
    describeAccountAttributes() {
        this.add('redshift:DescribeAccountAttributes');
        return this;
    }
    /**
     * Grants permission to describe database revisions for a cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterDbRevisions.html
     */
    describeClusterDbRevisions() {
        this.add('redshift:DescribeClusterDbRevisions');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift parameter groups, including parameter groups you created and the default parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameterGroups.html
     */
    describeClusterParameterGroups() {
        this.add('redshift:DescribeClusterParameterGroups');
        return this;
    }
    /**
     * Grants permission to describe parameters contained within an Amazon Redshift parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterParameters.html
     */
    describeClusterParameters() {
        this.add('redshift:DescribeClusterParameters');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift security groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSecurityGroups.html
     */
    describeClusterSecurityGroups() {
        this.add('redshift:DescribeClusterSecurityGroups');
        return this;
    }
    /**
     * Grants permission to describe one or more snapshot objects, which contain metadata about your cluster snapshots
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSnapshots.html
     */
    describeClusterSnapshots() {
        this.add('redshift:DescribeClusterSnapshots');
        return this;
    }
    /**
     * Grants permission to describe one or more cluster subnet group objects, which contain metadata about your cluster subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterSubnetGroups.html
     */
    describeClusterSubnetGroups() {
        this.add('redshift:DescribeClusterSubnetGroups');
        return this;
    }
    /**
     * Grants permission to describe available maintenance tracks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterTracks.html
     */
    describeClusterTracks() {
        this.add('redshift:DescribeClusterTracks');
        return this;
    }
    /**
     * Grants permission to describe available Amazon Redshift cluster versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusterVersions.html
     */
    describeClusterVersions() {
        this.add('redshift:DescribeClusterVersions');
        return this;
    }
    /**
     * Grants permission to describe properties of provisioned clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeClusters.html
     */
    describeClusters() {
        this.add('redshift:DescribeClusters');
        return this;
    }
    /**
     * Grants permission to describe parameter settings for a parameter group family
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeDefaultClusterParameters.html
     */
    describeDefaultClusterParameters() {
        this.add('redshift:DescribeDefaultClusterParameters');
        return this;
    }
    /**
     * Grants permission to describe event categories for all event source types, or for a specified source type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventCategories.html
     */
    describeEventCategories() {
        this.add('redshift:DescribeEventCategories');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift event notification subscriptions for the specified AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    describeEventSubscriptions() {
        this.add('redshift:DescribeEventSubscriptions');
        return this;
    }
    /**
     * Grants permission to describe events related to clusters, security groups, snapshots, and parameter groups for the past 14 days
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeEvents.html
     */
    describeEvents() {
        this.add('redshift:DescribeEvents');
        return this;
    }
    /**
     * Grants permission to describe HSM client certificates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmClientCertificates.html
     */
    describeHsmClientCertificates() {
        this.add('redshift:DescribeHsmClientCertificates');
        return this;
    }
    /**
     * Grants permission to describe Amazon Redshift HSM configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeHsmConfigurations.html
     */
    describeHsmConfigurations() {
        this.add('redshift:DescribeHsmConfigurations');
        return this;
    }
    /**
     * Grants permission to describe whether information, such as queries and connection attempts, is being logged for a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeLoggingStatus.html
     */
    describeLoggingStatus() {
        this.add('redshift:DescribeLoggingStatus');
        return this;
    }
    /**
     * Grants permission to describe properties of possible node configurations such as node type, number of nodes, and disk usage for the specified action type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeNodeConfigurationOptions.html
     */
    describeNodeConfigurationOptions() {
        this.add('redshift:DescribeNodeConfigurationOptions');
        return this;
    }
    /**
     * Grants permission to describe orderable cluster options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeOrderableClusterOptions.html
     */
    describeOrderableClusterOptions() {
        this.add('redshift:DescribeOrderableClusterOptions');
        return this;
    }
    /**
     * Grants permission to describe a query through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    describeQuery() {
        this.add('redshift:DescribeQuery');
        return this;
    }
    /**
     * Grants permission to describe available reserved node offerings by Amazon Redshift
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodeOfferings.html
     */
    describeReservedNodeOfferings() {
        this.add('redshift:DescribeReservedNodeOfferings');
        return this;
    }
    /**
     * Grants permission to describe the reserved nodes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeReservedNodes.html
     */
    describeReservedNodes() {
        this.add('redshift:DescribeReservedNodes');
        return this;
    }
    /**
     * Grants permission to describe the last resize operation for a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeResize.html
     */
    describeResize() {
        this.add('redshift:DescribeResize');
        return this;
    }
    /**
     * Grants permission to describe saved queries through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    describeSavedQueries() {
        this.add('redshift:DescribeSavedQueries');
        return this;
    }
    /**
     * Grants permission to describe created Amazon Redshift scheduled actions
     *
     * Access Level: Read
     */
    describeScheduledActions() {
        this.add('redshift:DescribeScheduledActions');
        return this;
    }
    /**
     * Grants permission to describe snapshot copy grants owned by the specified AWS account in the destination AWS Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotCopyGrants.html
     */
    describeSnapshotCopyGrants() {
        this.add('redshift:DescribeSnapshotCopyGrants');
        return this;
    }
    /**
     * Grants permission to describe snapshot schedules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeSnapshotSchedules.html
     */
    describeSnapshotSchedules() {
        this.add('redshift:DescribeSnapshotSchedules');
        return this;
    }
    /**
     * Grants permission to describe account level backups storage size and provisional storage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeStorage.html
     */
    describeStorage() {
        this.add('redshift:DescribeStorage');
        return this;
    }
    /**
     * Grants permission to describe a table through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    describeTable() {
        this.add('redshift:DescribeTable');
        return this;
    }
    /**
     * Grants permission to describe status of one or more table restore requests made using the RestoreTableFromClusterSnapshot API action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTableRestoreStatus.html
     */
    describeTableRestoreStatus() {
        this.add('redshift:DescribeTableRestoreStatus');
        return this;
    }
    /**
     * Grants permission to describe tags
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DescribeTags.html
     */
    describeTags() {
        this.add('redshift:DescribeTags');
        return this;
    }
    /**
     * Grants permission to disable logging information, such as queries and connection attempts, for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableLogging.html
     */
    disableLogging() {
        this.add('redshift:DisableLogging');
        return this;
    }
    /**
     * Grants permission to disable the automatic copy of snapshots for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_DisableSnapshotCopy.html
     */
    disableSnapshotCopy() {
        this.add('redshift:DisableSnapshotCopy');
        return this;
    }
    /**
     * Grants permission to enable logging information, such as queries and connection attempts, for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableLogging.html
     */
    enableLogging() {
        this.add('redshift:EnableLogging');
        return this;
    }
    /**
     * Grants permission to enable the automatic copy of snapshots for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_EnableSnapshotCopy.html
     */
    enableSnapshotCopy() {
        this.add('redshift:EnableSnapshotCopy');
        return this;
    }
    /**
     * Grants permission to execute a query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    executeQuery() {
        this.add('redshift:ExecuteQuery');
        return this;
    }
    /**
     * Grants permission to fetch query results through the Amazon Redshift console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    fetchResults() {
        this.add('redshift:FetchResults');
        return this;
    }
    /**
     * Grants permission to get temporary credentials to access an Amazon Redshift database by the specified AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html
     */
    getClusterCredentials() {
        this.add('redshift:GetClusterCredentials');
        return this;
    }
    /**
     * Grants permission to get an array of DC2 ReservedNodeOfferings that matches the payment type, term, and usage price of the given DC1 reserved node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetReservedNodeExchangeOfferings.html
     */
    getReservedNodeExchangeOfferings() {
        this.add('redshift:GetReservedNodeExchangeOfferings');
        return this;
    }
    /**
     * Grants permission to join the specified Amazon Redshift group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_GetClusterCredentials.html
     */
    joinGroup() {
        this.add('redshift:JoinGroup');
        return this;
    }
    /**
     * Grants permission to list databases through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listDatabases() {
        this.add('redshift:ListDatabases');
        return this;
    }
    /**
     * Grants permission to list saved queries through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listSavedQueries() {
        this.add('redshift:ListSavedQueries');
        return this;
    }
    /**
     * Grants permission to list schemas through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listSchemas() {
        this.add('redshift:ListSchemas');
        return this;
    }
    /**
     * Grants permission to list tables through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    listTables() {
        this.add('redshift:ListTables');
        return this;
    }
    /**
     * Grants permission to modify the settings of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyCluster.html
     */
    modifyCluster() {
        this.add('redshift:ModifyCluster');
        return this;
    }
    /**
     * Grants permission to modify the database revision of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterDbRevision.html
     */
    modifyClusterDbRevision() {
        this.add('redshift:ModifyClusterDbRevision');
        return this;
    }
    /**
     * Grants permission to modify the list of AWS Identity and Access Management (IAM) roles that can be used by a cluster to access other AWS services
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterIamRoles.html
     */
    modifyClusterIamRoles() {
        this.add('redshift:ModifyClusterIamRoles');
        return this;
    }
    /**
     * Grants permission to modify the maintenance settings of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterMaintenance.html
     */
    modifyClusterMaintenance() {
        this.add('redshift:ModifyClusterMaintenance');
        return this;
    }
    /**
     * Grants permission to modify the parameters of a parameter group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterParameterGroup.html
     */
    modifyClusterParameterGroup() {
        this.add('redshift:ModifyClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to modify the settings of a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshot.html
     */
    modifyClusterSnapshot() {
        this.add('redshift:ModifyClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to modify a snapshot schedule for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSnapshotSchedule.html
     */
    modifyClusterSnapshotSchedule() {
        this.add('redshift:ModifyClusterSnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to modify a cluster subnet group to include the specified list of VPC subnets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyClusterSubnetGroup.html
     */
    modifyClusterSubnetGroup() {
        this.add('redshift:ModifyClusterSubnetGroup');
        return this;
    }
    /**
     * Grants permission to modify an existing Amazon Redshift event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyEventSubscription.html
     */
    modifyEventSubscription() {
        this.add('redshift:ModifyEventSubscription');
        return this;
    }
    /**
     * Grants permission to modify an existing saved query through the Amazon Redshift console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    modifySavedQuery() {
        this.add('redshift:ModifySavedQuery');
        return this;
    }
    /**
     * Grants permission to modify an existing Amazon Redshift scheduled action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifyScheduledAction.html
     */
    modifyScheduledAction() {
        this.add('redshift:ModifyScheduledAction');
        return this;
    }
    /**
     * Grants permission to modify the number of days to retain snapshots in the destination AWS Region after they are copied from the source AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotCopyRetentionPeriod.html
     */
    modifySnapshotCopyRetentionPeriod() {
        this.add('redshift:ModifySnapshotCopyRetentionPeriod');
        return this;
    }
    /**
     * Grants permission to modify a snapshot schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ModifySnapshotSchedule.html
     */
    modifySnapshotSchedule() {
        this.add('redshift:ModifySnapshotSchedule');
        return this;
    }
    /**
     * Grants permission to pause a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_PauseCluster.html
     */
    pauseCluster() {
        this.add('redshift:PauseCluster');
        return this;
    }
    /**
     * Grants permission to purchase a reserved node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_PurchaseReservedNodeOffering.html
     */
    purchaseReservedNodeOffering() {
        this.add('redshift:PurchaseReservedNodeOffering');
        return this;
    }
    /**
     * Grants permission to reboot a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RebootCluster.html
     */
    rebootCluster() {
        this.add('redshift:RebootCluster');
        return this;
    }
    /**
     * Grants permission to set one or more parameters of a parameter group to their default values and set the source values of the parameters to "engine-default"
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResetClusterParameterGroup.html
     */
    resetClusterParameterGroup() {
        this.add('redshift:ResetClusterParameterGroup');
        return this;
    }
    /**
     * Grants permission to change the size of a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResizeCluster.html
     */
    resizeCluster() {
        this.add('redshift:ResizeCluster');
        return this;
    }
    /**
     * Grants permission to create a cluster from a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreFromClusterSnapshot.html
     */
    restoreFromClusterSnapshot() {
        this.add('redshift:RestoreFromClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to create a table from a table in an Amazon Redshift cluster snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RestoreTableFromClusterSnapshot.html
     */
    restoreTableFromClusterSnapshot() {
        this.add('redshift:RestoreTableFromClusterSnapshot');
        return this;
    }
    /**
     * Grants permission to resume a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_ResumeCluster.html
     */
    resumeCluster() {
        this.add('redshift:ResumeCluster');
        return this;
    }
    /**
     * Grants permission to revoke an ingress rule in an Amazon Redshift security group for a previously authorized IP range or Amazon EC2 security group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeClusterSecurityGroupIngress.html
     */
    revokeClusterSecurityGroupIngress() {
        this.add('redshift:RevokeClusterSecurityGroupIngress');
        return this;
    }
    /**
     * Grants permission to revoke access from the specified AWS account to restore a snapshot
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RevokeSnapshotAccess.html
     */
    revokeSnapshotAccess() {
        this.add('redshift:RevokeSnapshotAccess');
        return this;
    }
    /**
     * Grants permission to rotate an encryption key for a cluster
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/redshift/latest/APIReference/API_RotateEncryptionKey.html
     */
    rotateEncryptionKey() {
        this.add('redshift:RotateEncryptionKey');
        return this;
    }
    /**
     * Grants permission to view query results through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    viewQueriesFromConsole() {
        this.add('redshift:ViewQueriesFromConsole');
        return this;
    }
    /**
     * Grants permission to terminate running queries and loads through the Amazon Redshift console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-policy-resources.resource-permissions.html
     */
    viewQueriesInConsole() {
        this.add('redshift:ViewQueriesInConsole');
        return this;
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCluster(clusterName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:cluster:${ClusterName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dbgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_GROUP.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbGroup - Identifier for the dbGroup.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDbgroup(clusterName, dbGroup, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:dbgroup:${ClusterName}/${DbGroup}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${DbGroup}', dbGroup);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dbname to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/t_creating_database.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbName - Identifier for the dbName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDbname(clusterName, dbName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:dbname:${ClusterName}/${DbName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${DbName}', dbName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dbuser to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/dg/r_Users.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param dbUser - Identifier for the dbUser.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDbuser(clusterName, dbUser, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:dbuser:${ClusterName}/${DbUser}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${DbUser}', dbUser);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type eventsubscription to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-events.html
     *
     * @param eventSubscriptionName - Identifier for the eventSubscriptionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onEventsubscription(eventSubscriptionName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:eventsubscription:${EventSubscriptionName}';
        arn = arn.replace('${EventSubscriptionName}', eventSubscriptionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hsmclientcertificate to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM
     *
     * @param hSMClientCertificateId - Identifier for the hSMClientCertificateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onHsmclientcertificate(hSMClientCertificateId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:hsmclientcertificate:${HSMClientCertificateId}';
        arn = arn.replace('${HSMClientCertificateId}', hSMClientCertificateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hsmconfiguration to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#working-with-HSM
     *
     * @param hSMConfigurationId - Identifier for the hSMConfigurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onHsmconfiguration(hSMConfigurationId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:hsmconfiguration:${HSMConfigurationId}';
        arn = arn.replace('${HSMConfigurationId}', hSMConfigurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-parameter-groups.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onParametergroup(parameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:parametergroup:${ParameterGroupName}';
        arn = arn.replace('${ParameterGroupName}', parameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param owner - Identifier for the owner.
     * @param ec2SecurityGroupId - Identifier for the ec2SecurityGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSecuritygroup(securityGroupName, owner, ec2SecurityGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:securitygroup:${SecurityGroupName}/ec2securitygroup/${Owner}/${Ec2SecurityGroupId}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${Owner}', owner);
        arn = arn.replace('${Ec2SecurityGroupId}', ec2SecurityGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroupingress-cidr to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param ipRange - Identifier for the ipRange.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSecuritygroupingressCidr(securityGroupName, ipRange, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:securitygroupingress:${SecurityGroupName}/cidrip/${IpRange}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${IpRange}', ipRange);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroupingress-ec2securitygroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-security-groups.html
     *
     * @param securityGroupName - Identifier for the securityGroupName.
     * @param owner - Identifier for the owner.
     * @param ece2SecuritygroupId - Identifier for the ece2SecuritygroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSecuritygroupingressEc2securitygroup(securityGroupName, owner, ece2SecuritygroupId, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:securitygroupingress:${SecurityGroupName}/ec2securitygroup/${Owner}/${Ece2SecuritygroupId}';
        arn = arn.replace('${SecurityGroupName}', securityGroupName);
        arn = arn.replace('${Owner}', owner);
        arn = arn.replace('${Ece2SecuritygroupId}', ece2SecuritygroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSnapshot(clusterName, snapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:snapshot:${ClusterName}/${SnapshotName}';
        arn = arn.replace('${ClusterName}', clusterName);
        arn = arn.replace('${SnapshotName}', snapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshotcopygrant to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-db-encryption.html#configure-snapshot-copy-grant
     *
     * @param snapshotCopyGrantName - Identifier for the snapshotCopyGrantName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSnapshotcopygrant(snapshotCopyGrantName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:snapshotcopygrant:${SnapshotCopyGrantName}';
        arn = arn.replace('${SnapshotCopyGrantName}', snapshotCopyGrantName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshotschedule to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-snapshots.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSnapshotschedule(parameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:snapshotschedule:${ParameterGroupName}';
        arn = arn.replace('${ParameterGroupName}', parameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/working-with-cluster-subnet-groups.html
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSubnetgroup(subnetGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:redshift:${Region}:${Account}:subnetgroup:${SubnetGroupName}';
        arn = arn.replace('${SubnetGroupName}', subnetGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the database name
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbName(value, operator) {
        return this.if(`redshift:DbName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the database user name
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDbUser(value, operator) {
        return this.if(`redshift:DbUser`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the number of seconds until a temporary credential set expires
     *
     * https://docs.aws.amazon.com/redshift/latest/mgmt/redshift-iam-access-control-overview.html#redshift-policy-resources.conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDurationSeconds(value, operator) {
        return this.if(`redshift:DurationSeconds`, value, operator || 'StringLike');
    }
}
exports.Redshift = Redshift;
//# sourceMappingURL=data:application/json;base64,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