"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ses = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ses](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ses extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ses](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonses.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ses';
        this.actionList = {
            "CloneReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html",
                "description": "Creates a receipt rule set by cloning an existing one",
                "accessLevel": "Write"
            },
            "CreateConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html",
                "description": "Creates a new configuration set",
                "accessLevel": "Write"
            },
            "CreateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html",
                "description": "Creates a configuration set event destination",
                "accessLevel": "Write"
            },
            "CreateConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html",
                "description": "Creates an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "CreateCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html",
                "description": "Creates a new custom verification email template",
                "accessLevel": "Write"
            },
            "CreateReceiptFilter": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html",
                "description": "Creates a new IP address filter",
                "accessLevel": "Write"
            },
            "CreateReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html",
                "description": "Creates a receipt rule",
                "accessLevel": "Write"
            },
            "CreateReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html",
                "description": "Creates an empty receipt rule set",
                "accessLevel": "Write"
            },
            "CreateTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html",
                "description": "Creates an email template",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html",
                "description": "Deletes the configuration set",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html",
                "description": "Deletes a configuration set event destination",
                "accessLevel": "Write"
            },
            "DeleteConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html",
                "description": "Deletes an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "DeleteCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html",
                "description": "Deletes an existing custom verification email template",
                "accessLevel": "Write"
            },
            "DeleteIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html",
                "description": "Deletes the specified identity (an email address or a domain) from the list of verified identities",
                "accessLevel": "Write"
            },
            "DeleteIdentityPolicy": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html",
                "description": "Deletes the specified identity (an email address or a domain) from the list of verified identities",
                "accessLevel": "Write"
            },
            "DeleteReceiptFilter": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html",
                "description": "Deletes the specified IP address filter",
                "accessLevel": "Write"
            },
            "DeleteReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html",
                "description": "Deletes the specified receipt rule",
                "accessLevel": "Write"
            },
            "DeleteReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html",
                "description": "Deletes the specified receipt rule set and all of the receipt rules it contains",
                "accessLevel": "Write"
            },
            "DeleteTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html",
                "description": "Deletes an email template",
                "accessLevel": "Write"
            },
            "DeleteVerifiedEmailAddress": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html",
                "description": "Deletes the specified email address from the list of verified addresses",
                "accessLevel": "Write"
            },
            "DescribeActiveReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html",
                "description": "Returns the metadata and receipt rules for the receipt rule set that is currently active",
                "accessLevel": "Read"
            },
            "DescribeConfigurationSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html",
                "description": "Returns the details of the specified configuration set",
                "accessLevel": "Read"
            },
            "DescribeReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html",
                "description": "Returns the details of the specified receipt rule",
                "accessLevel": "Read"
            },
            "DescribeReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html",
                "description": "Returns the details of the specified receipt rule set",
                "accessLevel": "Read"
            },
            "GetAccountSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html",
                "description": "Returns the email sending status of the Amazon SES account for the current region",
                "accessLevel": "Read"
            },
            "GetCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html",
                "description": "Returns the custom email verification template for the template name you specify",
                "accessLevel": "Read"
            },
            "GetIdentityDkimAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html",
                "description": "Returns the current status of Easy DKIM signing for an entity",
                "accessLevel": "Read"
            },
            "GetIdentityMailFromDomainAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html",
                "description": "Returns the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)",
                "accessLevel": "Read"
            },
            "GetIdentityNotificationAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html",
                "description": "Given a list of verified identities (email addresses and/or domains), returns a structure describing identity notification attributes",
                "accessLevel": "Read"
            },
            "GetIdentityPolicies": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html",
                "description": "Returns the requested sending authorization policies for the given identity (an email address or a domain)",
                "accessLevel": "Read"
            },
            "GetIdentityVerificationAttributes": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html",
                "description": "Given a list of identities (email addresses and/or domains), returns the verification status and (for domain identities) the verification token for each identity",
                "accessLevel": "Read"
            },
            "GetSendQuota": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html",
                "description": "Returns the user's current sending limits",
                "accessLevel": "Read"
            },
            "GetSendStatistics": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html",
                "description": "Returns the user's sending statistics. The result is a list of data points, representing the last two weeks of sending activity",
                "accessLevel": "Read"
            },
            "GetTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html",
                "description": "Returns the template object (which includes the Subject line, HTML part and text part) for the template you specify",
                "accessLevel": "Read"
            },
            "ListConfigurationSets": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html",
                "description": "Returns a list of the configuration sets associated with your Amazon SES account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListCustomVerificationEmailTemplates": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html",
                "description": "Lists the existing custom verification email templates for your account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListIdentities": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html",
                "description": "Returns a list containing all of the identities (email addresses and domains) for your AWS account, regardless of verification status",
                "accessLevel": "List"
            },
            "ListIdentityPolicies": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html",
                "description": "Returns a list of sending authorization policies that are attached to the given identity (an email address or a domain)",
                "accessLevel": "List"
            },
            "ListReceiptFilters": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html",
                "description": "Lists the IP address filters associated with your AWS account",
                "accessLevel": "List"
            },
            "ListReceiptRuleSets": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html",
                "description": "Lists the receipt rule sets that exist under your AWS account",
                "accessLevel": "List"
            },
            "ListTemplates": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html",
                "description": "Lists the email templates present in your Amazon SES account in the current AWS Region",
                "accessLevel": "List"
            },
            "ListVerifiedEmailAddresses": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html",
                "description": "Returns a list containing all of the email addresses that have been verified",
                "accessLevel": "List"
            },
            "PutIdentityPolicy": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html",
                "description": "Adds or updates a sending authorization policy for the specified identity (an email address or a domain)",
                "accessLevel": "Write"
            },
            "ReorderReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html",
                "description": "Reorders the receipt rules within a receipt rule set",
                "accessLevel": "Write"
            },
            "SendBounce": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html",
                "description": "Generates and sends a bounce message to the sender of an email you received through Amazon SES",
                "accessLevel": "Write",
                "conditions": [
                    "ses:FromAddress"
                ]
            },
            "SendBulkTemplatedEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html",
                "description": "Composes an email message to multiple destinations",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendCustomVerificationEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html",
                "description": "Adds an email address to the list of identities for your Amazon SES account in the current AWS Region and attempts to verify it",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html",
                "description": "Composes an email message based on input data, and then immediately queues the message for sending",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendRawEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html",
                "description": "Sends an email message, with header and content specified by the client",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SendTemplatedEmail": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html",
                "description": "Composes an email message using an email template and immediately queues it for sending",
                "accessLevel": "Write",
                "resourceTypes": {
                    "identity": {
                        "required": true
                    }
                },
                "conditions": [
                    "ses:FeedbackAddress",
                    "ses:FromAddress",
                    "ses:FromDisplayName",
                    "ses:Recipients"
                ]
            },
            "SetActiveReceiptRuleSet": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html",
                "description": "Sets the specified receipt rule set as the active receipt rule set",
                "accessLevel": "Write"
            },
            "SetIdentityDkimEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html",
                "description": "Enables or disables Easy DKIM signing of email sent from an identity",
                "accessLevel": "Write"
            },
            "SetIdentityFeedbackForwardingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html",
                "description": "Given an identity (an email address or a domain), enables or disables whether Amazon SES forwards bounce and complaint notifications as email",
                "accessLevel": "Write"
            },
            "SetIdentityHeadersInNotificationsEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html",
                "description": "Given an identity (an email address or a domain), sets whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type",
                "accessLevel": "Write"
            },
            "SetIdentityMailFromDomain": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html",
                "description": "Enables or disables the custom MAIL FROM domain setup for a verified identity (an email address or a domain)",
                "accessLevel": "Write"
            },
            "SetIdentityNotificationTopic": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html",
                "description": "Given an identity (an email address or a domain), sets the Amazon Simple Notification Service (Amazon SNS) topic to which Amazon SES will publish bounce, complaint, and/or delivery notifications for emails sent with that identity as the Source",
                "accessLevel": "Write"
            },
            "SetReceiptRulePosition": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html",
                "description": "Sets the position of the specified receipt rule in the receipt rule set",
                "accessLevel": "Write"
            },
            "TestRenderTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html",
                "description": "Creates a preview of the MIME content of an email when provided with a template and a set of replacement data",
                "accessLevel": "Write"
            },
            "UpdateAccountSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html",
                "description": "Enables or disables email sending across your entire Amazon SES account in the current AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetEventDestination": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html",
                "description": "Updates the event destination of a configuration set",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetReputationMetricsEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html",
                "description": "Enables or disables the publishing of reputation metrics for emails sent using a specific configuration set in a given AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetSendingEnabled": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html",
                "description": "Enables or disables email sending for messages sent using a specific configuration set in a given AWS Region",
                "accessLevel": "Write"
            },
            "UpdateConfigurationSetTrackingOptions": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html",
                "description": "Modifies an association between a configuration set and a custom domain for open and click event tracking",
                "accessLevel": "Write"
            },
            "UpdateCustomVerificationEmailTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html",
                "description": "Updates an existing custom verification email template",
                "accessLevel": "Write"
            },
            "UpdateReceiptRule": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html",
                "description": "Updates a receipt rule",
                "accessLevel": "Write"
            },
            "UpdateTemplate": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html",
                "description": "Updates an email template",
                "accessLevel": "Write"
            },
            "VerifyDomainDkim": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html",
                "description": "Returns a set of DKIM tokens for a domain",
                "accessLevel": "Read"
            },
            "VerifyDomainIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html",
                "description": "Verifies a domain",
                "accessLevel": "Read"
            },
            "VerifyEmailAddress": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html",
                "description": "Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second",
                "accessLevel": "Read"
            },
            "VerifyEmailIdentity": {
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html",
                "description": "Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "configuration-set": {
                "name": "configuration-set",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}",
                "conditionKeys": []
            },
            "custom-verification-email-template": {
                "name": "custom-verification-email-template",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:custom-verification-email-template/${CustomVerificationEmailTemplateName}",
                "conditionKeys": []
            },
            "event-destination": {
                "name": "event-destination",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_EventDestination.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}",
                "conditionKeys": []
            },
            "identity": {
                "name": "identity",
                "url": "",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}",
                "conditionKeys": []
            },
            "receipt-filter": {
                "name": "receipt-filter",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptFilter.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-filter/${ReceiptFilterName}",
                "conditionKeys": []
            },
            "receipt-rule": {
                "name": "receipt-rule",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRule.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}:receipt-rule/${ReceiptRuleName}",
                "conditionKeys": []
            },
            "receipt-rule-set": {
                "name": "receipt-rule-set",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRuleSetMetadata.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}",
                "conditionKeys": []
            },
            "template": {
                "name": "template",
                "url": "https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html",
                "arn": "arn:${Partition}:ses:${Region}:${Account}:template/${TemplateName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a receipt rule set by cloning an existing one
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CloneReceiptRuleSet.html
     */
    cloneReceiptRuleSet() {
        this.add('ses:CloneReceiptRuleSet');
        return this;
    }
    /**
     * Creates a new configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSet.html
     */
    createConfigurationSet() {
        this.add('ses:CreateConfigurationSet');
        return this;
    }
    /**
     * Creates a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetEventDestination.html
     */
    createConfigurationSetEventDestination() {
        this.add('ses:CreateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Creates an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateConfigurationSetTrackingOptions.html
     */
    createConfigurationSetTrackingOptions() {
        this.add('ses:CreateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Creates a new custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateCustomVerificationEmailTemplate.html
     */
    createCustomVerificationEmailTemplate() {
        this.add('ses:CreateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Creates a new IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptFilter.html
     */
    createReceiptFilter() {
        this.add('ses:CreateReceiptFilter');
        return this;
    }
    /**
     * Creates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRule.html
     */
    createReceiptRule() {
        this.add('ses:CreateReceiptRule');
        return this;
    }
    /**
     * Creates an empty receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateReceiptRuleSet.html
     */
    createReceiptRuleSet() {
        this.add('ses:CreateReceiptRuleSet');
        return this;
    }
    /**
     * Creates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CreateTemplate.html
     */
    createTemplate() {
        this.add('ses:CreateTemplate');
        return this;
    }
    /**
     * Deletes the configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSet.html
     */
    deleteConfigurationSet() {
        this.add('ses:DeleteConfigurationSet');
        return this;
    }
    /**
     * Deletes a configuration set event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetEventDestination.html
     */
    deleteConfigurationSetEventDestination() {
        this.add('ses:DeleteConfigurationSetEventDestination');
        return this;
    }
    /**
     * Deletes an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteConfigurationSetTrackingOptions.html
     */
    deleteConfigurationSetTrackingOptions() {
        this.add('ses:DeleteConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Deletes an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteCustomVerificationEmailTemplate.html
     */
    deleteCustomVerificationEmailTemplate() {
        this.add('ses:DeleteCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentity.html
     */
    deleteIdentity() {
        this.add('ses:DeleteIdentity');
        return this;
    }
    /**
     * Deletes the specified identity (an email address or a domain) from the list of verified identities
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteIdentityPolicy.html
     */
    deleteIdentityPolicy() {
        this.add('ses:DeleteIdentityPolicy');
        return this;
    }
    /**
     * Deletes the specified IP address filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptFilter.html
     */
    deleteReceiptFilter() {
        this.add('ses:DeleteReceiptFilter');
        return this;
    }
    /**
     * Deletes the specified receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRule.html
     */
    deleteReceiptRule() {
        this.add('ses:DeleteReceiptRule');
        return this;
    }
    /**
     * Deletes the specified receipt rule set and all of the receipt rules it contains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteReceiptRuleSet.html
     */
    deleteReceiptRuleSet() {
        this.add('ses:DeleteReceiptRuleSet');
        return this;
    }
    /**
     * Deletes an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteTemplate.html
     */
    deleteTemplate() {
        this.add('ses:DeleteTemplate');
        return this;
    }
    /**
     * Deletes the specified email address from the list of verified addresses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DeleteVerifiedEmailAddress.html
     */
    deleteVerifiedEmailAddress() {
        this.add('ses:DeleteVerifiedEmailAddress');
        return this;
    }
    /**
     * Returns the metadata and receipt rules for the receipt rule set that is currently active
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeActiveReceiptRuleSet.html
     */
    describeActiveReceiptRuleSet() {
        this.add('ses:DescribeActiveReceiptRuleSet');
        return this;
    }
    /**
     * Returns the details of the specified configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeConfigurationSet.html
     */
    describeConfigurationSet() {
        this.add('ses:DescribeConfigurationSet');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRule.html
     */
    describeReceiptRule() {
        this.add('ses:DescribeReceiptRule');
        return this;
    }
    /**
     * Returns the details of the specified receipt rule set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_DescribeReceiptRuleSet.html
     */
    describeReceiptRuleSet() {
        this.add('ses:DescribeReceiptRuleSet');
        return this;
    }
    /**
     * Returns the email sending status of the Amazon SES account for the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetAccountSendingEnabled.html
     */
    getAccountSendingEnabled() {
        this.add('ses:GetAccountSendingEnabled');
        return this;
    }
    /**
     * Returns the custom email verification template for the template name you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetCustomVerificationEmailTemplate.html
     */
    getCustomVerificationEmailTemplate() {
        this.add('ses:GetCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Returns the current status of Easy DKIM signing for an entity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityDkimAttributes.html
     */
    getIdentityDkimAttributes() {
        this.add('ses:GetIdentityDkimAttributes');
        return this;
    }
    /**
     * Returns the custom MAIL FROM attributes for a list of identities (email addresses and/or domains)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityMailFromDomainAttributes.html
     */
    getIdentityMailFromDomainAttributes() {
        this.add('ses:GetIdentityMailFromDomainAttributes');
        return this;
    }
    /**
     * Given a list of verified identities (email addresses and/or domains), returns a structure describing identity notification attributes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityNotificationAttributes.html
     */
    getIdentityNotificationAttributes() {
        this.add('ses:GetIdentityNotificationAttributes');
        return this;
    }
    /**
     * Returns the requested sending authorization policies for the given identity (an email address or a domain)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityPolicies.html
     */
    getIdentityPolicies() {
        this.add('ses:GetIdentityPolicies');
        return this;
    }
    /**
     * Given a list of identities (email addresses and/or domains), returns the verification status and (for domain identities) the verification token for each identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetIdentityVerificationAttributes.html
     */
    getIdentityVerificationAttributes() {
        this.add('ses:GetIdentityVerificationAttributes');
        return this;
    }
    /**
     * Returns the user's current sending limits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendQuota.html
     */
    getSendQuota() {
        this.add('ses:GetSendQuota');
        return this;
    }
    /**
     * Returns the user's sending statistics. The result is a list of data points, representing the last two weeks of sending activity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetSendStatistics.html
     */
    getSendStatistics() {
        this.add('ses:GetSendStatistics');
        return this;
    }
    /**
     * Returns the template object (which includes the Subject line, HTML part and text part) for the template you specify
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_GetTemplate.html
     */
    getTemplate() {
        this.add('ses:GetTemplate');
        return this;
    }
    /**
     * Returns a list of the configuration sets associated with your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListConfigurationSets.html
     */
    listConfigurationSets() {
        this.add('ses:ListConfigurationSets');
        return this;
    }
    /**
     * Lists the existing custom verification email templates for your account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListCustomVerificationEmailTemplates.html
     */
    listCustomVerificationEmailTemplates() {
        this.add('ses:ListCustomVerificationEmailTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the identities (email addresses and domains) for your AWS account, regardless of verification status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentities.html
     */
    listIdentities() {
        this.add('ses:ListIdentities');
        return this;
    }
    /**
     * Returns a list of sending authorization policies that are attached to the given identity (an email address or a domain)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListIdentityPolicies.html
     */
    listIdentityPolicies() {
        this.add('ses:ListIdentityPolicies');
        return this;
    }
    /**
     * Lists the IP address filters associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptFilters.html
     */
    listReceiptFilters() {
        this.add('ses:ListReceiptFilters');
        return this;
    }
    /**
     * Lists the receipt rule sets that exist under your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListReceiptRuleSets.html
     */
    listReceiptRuleSets() {
        this.add('ses:ListReceiptRuleSets');
        return this;
    }
    /**
     * Lists the email templates present in your Amazon SES account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListTemplates.html
     */
    listTemplates() {
        this.add('ses:ListTemplates');
        return this;
    }
    /**
     * Returns a list containing all of the email addresses that have been verified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ListVerifiedEmailAddresses.html
     */
    listVerifiedEmailAddresses() {
        this.add('ses:ListVerifiedEmailAddresses');
        return this;
    }
    /**
     * Adds or updates a sending authorization policy for the specified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_PutIdentityPolicy.html
     */
    putIdentityPolicy() {
        this.add('ses:PutIdentityPolicy');
        return this;
    }
    /**
     * Reorders the receipt rules within a receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReorderReceiptRuleSet.html
     */
    reorderReceiptRuleSet() {
        this.add('ses:ReorderReceiptRuleSet');
        return this;
    }
    /**
     * Generates and sends a bounce message to the sender of an email you received through Amazon SES
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBounce.html
     */
    sendBounce() {
        this.add('ses:SendBounce');
        return this;
    }
    /**
     * Composes an email message to multiple destinations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendBulkTemplatedEmail.html
     */
    sendBulkTemplatedEmail() {
        this.add('ses:SendBulkTemplatedEmail');
        return this;
    }
    /**
     * Adds an email address to the list of identities for your Amazon SES account in the current AWS Region and attempts to verify it
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendCustomVerificationEmail.html
     */
    sendCustomVerificationEmail() {
        this.add('ses:SendCustomVerificationEmail');
        return this;
    }
    /**
     * Composes an email message based on input data, and then immediately queues the message for sending
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendEmail.html
     */
    sendEmail() {
        this.add('ses:SendEmail');
        return this;
    }
    /**
     * Sends an email message, with header and content specified by the client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendRawEmail.html
     */
    sendRawEmail() {
        this.add('ses:SendRawEmail');
        return this;
    }
    /**
     * Composes an email message using an email template and immediately queues it for sending
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SendTemplatedEmail.html
     */
    sendTemplatedEmail() {
        this.add('ses:SendTemplatedEmail');
        return this;
    }
    /**
     * Sets the specified receipt rule set as the active receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetActiveReceiptRuleSet.html
     */
    setActiveReceiptRuleSet() {
        this.add('ses:SetActiveReceiptRuleSet');
        return this;
    }
    /**
     * Enables or disables Easy DKIM signing of email sent from an identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityDkimEnabled.html
     */
    setIdentityDkimEnabled() {
        this.add('ses:SetIdentityDkimEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), enables or disables whether Amazon SES forwards bounce and complaint notifications as email
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityFeedbackForwardingEnabled.html
     */
    setIdentityFeedbackForwardingEnabled() {
        this.add('ses:SetIdentityFeedbackForwardingEnabled');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets whether Amazon SES includes the original email headers in the Amazon Simple Notification Service (Amazon SNS) notifications of a specified type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityHeadersInNotificationsEnabled.html
     */
    setIdentityHeadersInNotificationsEnabled() {
        this.add('ses:SetIdentityHeadersInNotificationsEnabled');
        return this;
    }
    /**
     * Enables or disables the custom MAIL FROM domain setup for a verified identity (an email address or a domain)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityMailFromDomain.html
     */
    setIdentityMailFromDomain() {
        this.add('ses:SetIdentityMailFromDomain');
        return this;
    }
    /**
     * Given an identity (an email address or a domain), sets the Amazon Simple Notification Service (Amazon SNS) topic to which Amazon SES will publish bounce, complaint, and/or delivery notifications for emails sent with that identity as the Source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetIdentityNotificationTopic.html
     */
    setIdentityNotificationTopic() {
        this.add('ses:SetIdentityNotificationTopic');
        return this;
    }
    /**
     * Sets the position of the specified receipt rule in the receipt rule set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_SetReceiptRulePosition.html
     */
    setReceiptRulePosition() {
        this.add('ses:SetReceiptRulePosition');
        return this;
    }
    /**
     * Creates a preview of the MIME content of an email when provided with a template and a set of replacement data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_TestRenderTemplate.html
     */
    testRenderTemplate() {
        this.add('ses:TestRenderTemplate');
        return this;
    }
    /**
     * Enables or disables email sending across your entire Amazon SES account in the current AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateAccountSendingEnabled.html
     */
    updateAccountSendingEnabled() {
        this.add('ses:UpdateAccountSendingEnabled');
        return this;
    }
    /**
     * Updates the event destination of a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetEventDestination.html
     */
    updateConfigurationSetEventDestination() {
        this.add('ses:UpdateConfigurationSetEventDestination');
        return this;
    }
    /**
     * Enables or disables the publishing of reputation metrics for emails sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetReputationMetricsEnabled.html
     */
    updateConfigurationSetReputationMetricsEnabled() {
        this.add('ses:UpdateConfigurationSetReputationMetricsEnabled');
        return this;
    }
    /**
     * Enables or disables email sending for messages sent using a specific configuration set in a given AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetSendingEnabled.html
     */
    updateConfigurationSetSendingEnabled() {
        this.add('ses:UpdateConfigurationSetSendingEnabled');
        return this;
    }
    /**
     * Modifies an association between a configuration set and a custom domain for open and click event tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateConfigurationSetTrackingOptions.html
     */
    updateConfigurationSetTrackingOptions() {
        this.add('ses:UpdateConfigurationSetTrackingOptions');
        return this;
    }
    /**
     * Updates an existing custom verification email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateCustomVerificationEmailTemplate.html
     */
    updateCustomVerificationEmailTemplate() {
        this.add('ses:UpdateCustomVerificationEmailTemplate');
        return this;
    }
    /**
     * Updates a receipt rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateReceiptRule.html
     */
    updateReceiptRule() {
        this.add('ses:UpdateReceiptRule');
        return this;
    }
    /**
     * Updates an email template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_UpdateTemplate.html
     */
    updateTemplate() {
        this.add('ses:UpdateTemplate');
        return this;
    }
    /**
     * Returns a set of DKIM tokens for a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainDkim.html
     */
    verifyDomainDkim() {
        this.add('ses:VerifyDomainDkim');
        return this;
    }
    /**
     * Verifies a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyDomainIdentity.html
     */
    verifyDomainIdentity() {
        this.add('ses:VerifyDomainIdentity');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailAddress.html
     */
    verifyEmailAddress() {
        this.add('ses:VerifyEmailAddress');
        return this;
    }
    /**
     * Verifies an email address. This action causes a confirmation email message to be sent to the specified address. This action is throttled at one request per second
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_VerifyEmailIdentity.html
     */
    verifyEmailIdentity() {
        this.add('ses:VerifyEmailIdentity');
        return this;
    }
    /**
     * Adds a resource of type configuration-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type custom-verification-email-template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_CustomVerificationEmailTemplate.html
     *
     * @param customVerificationEmailTemplateName - Identifier for the customVerificationEmailTemplateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCustomVerificationEmailTemplate(customVerificationEmailTemplateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:custom-verification-email-template/${CustomVerificationEmailTemplateName}';
        arn = arn.replace('${CustomVerificationEmailTemplateName}', customVerificationEmailTemplateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-destination to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_EventDestination.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param eventDestinationName - Identifier for the eventDestinationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventDestination(configurationSetName, eventDestinationName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:configuration-set/${ConfigurationSetName}:event-destination/${EventDestinationName}';
        arn = arn.replace('${ConfigurationSetName}', configurationSetName);
        arn = arn.replace('${EventDestinationName}', eventDestinationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * @param identityName - Identifier for the identityName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIdentity(identityName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:identity/${IdentityName}';
        arn = arn.replace('${IdentityName}', identityName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-filter to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptFilter.html
     *
     * @param receiptFilterName - Identifier for the receiptFilterName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptFilter(receiptFilterName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-filter/${ReceiptFilterName}';
        arn = arn.replace('${ReceiptFilterName}', receiptFilterName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRule.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param receiptRuleName - Identifier for the receiptRuleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRule(receiptRuleSetName, receiptRuleName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}:receipt-rule/${ReceiptRuleName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${ReceiptRuleName}', receiptRuleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type receipt-rule-set to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_ReceiptRuleSetMetadata.html
     *
     * @param receiptRuleSetName - Identifier for the receiptRuleSetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReceiptRuleSet(receiptRuleSetName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:receipt-rule-set/${ReceiptRuleSetName}';
        arn = arn.replace('${ReceiptRuleSetName}', receiptRuleSetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type template to the statement
     *
     * https://docs.aws.amazon.com/ses/latest/APIReference/API_Template.html
     *
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTemplate(templateName, account, region, partition) {
        var arn = 'arn:${Partition}:ses:${Region}:${Account}:template/${TemplateName}';
        arn = arn.replace('${TemplateName}', templateName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The "Return-Path" address, which specifies where bounces and complaints are sent by email feedback forwarding.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFeedbackAddress(value, operator) {
        return this.if(`ses:FeedbackAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromAddress(value, operator) {
        return this.if(`ses:FromAddress`, value, operator || 'StringLike');
    }
    /**
     * The "From" address that is used as the display name of a message.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFromDisplayName(value, operator) {
        return this.if(`ses:FromDisplayName`, value, operator || 'StringLike');
    }
    /**
     * The recipient addresses of a message, which include the "To", "CC", and "BCC" addresses.
     *
     * https://docs.aws.amazon.com/ses/latest/DeveloperGuide/email-format.html#email-header
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipients(value, operator) {
        return this.if(`ses:Recipients`, value, operator || 'StringLike');
    }
}
exports.Ses = Ses;
//# sourceMappingURL=data:application/json;base64,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