"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Swf = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [swf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpleworkflowservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Swf extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [swf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonsimpleworkflowservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'swf';
        this.actionList = {
            "CancelTimer": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelTimer.html",
                "description": "Description for CancelTimer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CancelWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelWorkflowExecution.html",
                "description": "Description for CancelWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CompleteWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CompleteWorkflowExecution.html",
                "description": "Description for CompleteWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ContinueAsNewWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ContinueAsNewWorkflowExecution.html",
                "description": "Description for ContinueAsNewWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "CountClosedWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html",
                "description": "Returns the number of closed workflow executions within the given domain that meet the specified filtering criteria.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "CountOpenWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html",
                "description": "Returns the number of open workflow executions within the given domain that meet the specified filtering criteria.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "CountPendingActivityTasks": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html",
                "description": "Returns the estimated number of activity tasks in the specified task list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "CountPendingDecisionTasks": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html",
                "description": "Returns the estimated number of decision tasks in the specified task list.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "DeprecateActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html",
                "description": "Deprecates the specified activity type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version"
                ]
            },
            "DeprecateDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html",
                "description": "Deprecates the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DeprecateWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html",
                "description": "Deprecates the specified workflow type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "DescribeActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html",
                "description": "Returns information about the specified activity type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version"
                ]
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html",
                "description": "Returns information about the specified domain, including description and status.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html",
                "description": "Returns information about the specified workflow execution including its type and some statistics.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "DescribeWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html",
                "description": "Returns information about the specified workflow type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "FailWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_FailWorkflowExecution.html",
                "description": "Description for FailWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "GetWorkflowExecutionHistory": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html",
                "description": "Returns the history of the specified workflow execution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListActivityTypes": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html",
                "description": "Returns information about all activities registered in the specified domain that match the specified name and registration status.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ListClosedWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html",
                "description": "Returns a list of closed workflow executions in the specified domain that meet the filtering criteria.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html",
                "description": "Returns the list of domains registered in the account.",
                "accessLevel": "List"
            },
            "ListOpenWorkflowExecutions": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html",
                "description": "Returns a list of open workflow executions in the specified domain that meet the filtering criteria.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagFilter.tag",
                    "swf:typeFilter.name",
                    "swf:typeFilter.version"
                ]
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html",
                "description": "This action lists tags for an AWS SWF resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                }
            },
            "ListWorkflowTypes": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html",
                "description": "Returns information about workflow types in the specified domain.",
                "accessLevel": "List",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "PollForActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html",
                "description": "Used by workers to get an ActivityTask from the specified activity taskList.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "PollForDecisionTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html",
                "description": "Used by deciders to get a DecisionTask from the specified decision taskList.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:taskList.name"
                ]
            },
            "RecordActivityTaskHeartbeat": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html",
                "description": "Used by activity workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RecordMarker": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordMarker.html",
                "description": "Description for RecordMarker",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RegisterActivityType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html",
                "description": "Registers a new activity type along with its configuration settings in the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:defaultTaskList.name",
                    "swf:name",
                    "swf:version"
                ]
            },
            "RegisterDomain": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html",
                "description": "Registers a new domain.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "RegisterWorkflowType": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html",
                "description": "Registers a new workflow type and its configuration settings in the specified domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:defaultTaskList.name",
                    "swf:name",
                    "swf:version"
                ]
            },
            "RequestCancelActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelActivityTask.html",
                "description": "Description for RequestCancelActivityTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RequestCancelExternalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelExternalWorkflowExecution.html",
                "description": "Description for RequestCancelExternalWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RequestCancelWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html",
                "description": "Records a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondActivityTaskCanceled": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondActivityTaskCompleted": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:activityType.name",
                    "swf:activityType.version",
                    "swf:tagList.member.0",
                    "swf:tagList.member.1",
                    "swf:tagList.member.2",
                    "swf:tagList.member.3",
                    "swf:tagList.member.4",
                    "swf:taskList.name",
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "RespondActivityTaskFailed": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html",
                "description": "Used by workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified).",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "RespondDecisionTaskCompleted": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html",
                "description": "Used by deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "ScheduleActivityTask": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ScheduleActivityTask.html",
                "description": "Description for ScheduleActivityTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "SignalExternalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalExternalWorkflowExecution.html",
                "description": "Description for SignalExternalWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "SignalWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html",
                "description": "Records a WorkflowExecutionSignaled event in the workflow execution history and creates a decision task for the workflow execution identified by the given domain, workflowId and runId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartChildWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartChildWorkflowExecution.html",
                "description": "Description for StartChildWorkflowExecution",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartTimer": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartTimer.html",
                "description": "Description for StartTimer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "StartWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html",
                "description": "Starts an execution of the workflow type in the specified domain using the provided workflowId and input data.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                },
                "conditions": [
                    "swf:tagList.member.0",
                    "swf:tagList.member.1",
                    "swf:tagList.member.2",
                    "swf:tagList.member.3",
                    "swf:tagList.member.4",
                    "swf:taskList.name",
                    "swf:workflowType.name",
                    "swf:workflowType.version"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html",
                "description": "This action tags an AWS SWF resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}"
                ]
            },
            "TerminateWorkflowExecution": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html",
                "description": "Records a WorkflowExecutionTerminated event and forces closure of the workflow execution identified by the given domain, runId, and workflowId.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "domain": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html",
                "description": "This action removes a tag from an AWS SWF resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "domain": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "domain": {
                "name": "domain",
                "url": "https://docs.aws.amazon.com/swf/latest/developerguide/swf-dev-domains.html",
                "arn": "arn:${Partition}:swf::${Account}:domain/${DomainName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Description for CancelTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelTimer.html
     */
    cancelTimer() {
        this.add('swf:CancelTimer');
        return this;
    }
    /**
     * Description for CancelWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CancelWorkflowExecution.html
     */
    cancelWorkflowExecution() {
        this.add('swf:CancelWorkflowExecution');
        return this;
    }
    /**
     * Description for CompleteWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CompleteWorkflowExecution.html
     */
    completeWorkflowExecution() {
        this.add('swf:CompleteWorkflowExecution');
        return this;
    }
    /**
     * Description for ContinueAsNewWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ContinueAsNewWorkflowExecution.html
     */
    continueAsNewWorkflowExecution() {
        this.add('swf:ContinueAsNewWorkflowExecution');
        return this;
    }
    /**
     * Returns the number of closed workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountClosedWorkflowExecutions.html
     */
    countClosedWorkflowExecutions() {
        this.add('swf:CountClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the number of open workflow executions within the given domain that meet the specified filtering criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountOpenWorkflowExecutions.html
     */
    countOpenWorkflowExecutions() {
        this.add('swf:CountOpenWorkflowExecutions');
        return this;
    }
    /**
     * Returns the estimated number of activity tasks in the specified task list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingActivityTasks.html
     */
    countPendingActivityTasks() {
        this.add('swf:CountPendingActivityTasks');
        return this;
    }
    /**
     * Returns the estimated number of decision tasks in the specified task list.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_CountPendingDecisionTasks.html
     */
    countPendingDecisionTasks() {
        this.add('swf:CountPendingDecisionTasks');
        return this;
    }
    /**
     * Deprecates the specified activity type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateActivityType.html
     */
    deprecateActivityType() {
        this.add('swf:DeprecateActivityType');
        return this;
    }
    /**
     * Deprecates the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateDomain.html
     */
    deprecateDomain() {
        this.add('swf:DeprecateDomain');
        return this;
    }
    /**
     * Deprecates the specified workflow type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DeprecateWorkflowType.html
     */
    deprecateWorkflowType() {
        this.add('swf:DeprecateWorkflowType');
        return this;
    }
    /**
     * Returns information about the specified activity type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeActivityType.html
     */
    describeActivityType() {
        this.add('swf:DescribeActivityType');
        return this;
    }
    /**
     * Returns information about the specified domain, including description and status.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeDomain.html
     */
    describeDomain() {
        this.add('swf:DescribeDomain');
        return this;
    }
    /**
     * Returns information about the specified workflow execution including its type and some statistics.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowExecution.html
     */
    describeWorkflowExecution() {
        this.add('swf:DescribeWorkflowExecution');
        return this;
    }
    /**
     * Returns information about the specified workflow type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_DescribeWorkflowType.html
     */
    describeWorkflowType() {
        this.add('swf:DescribeWorkflowType');
        return this;
    }
    /**
     * Description for FailWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_FailWorkflowExecution.html
     */
    failWorkflowExecution() {
        this.add('swf:FailWorkflowExecution');
        return this;
    }
    /**
     * Returns the history of the specified workflow execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_GetWorkflowExecutionHistory.html
     */
    getWorkflowExecutionHistory() {
        this.add('swf:GetWorkflowExecutionHistory');
        return this;
    }
    /**
     * Returns information about all activities registered in the specified domain that match the specified name and registration status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListActivityTypes.html
     */
    listActivityTypes() {
        this.add('swf:ListActivityTypes');
        return this;
    }
    /**
     * Returns a list of closed workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListClosedWorkflowExecutions.html
     */
    listClosedWorkflowExecutions() {
        this.add('swf:ListClosedWorkflowExecutions');
        return this;
    }
    /**
     * Returns the list of domains registered in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListDomains.html
     */
    listDomains() {
        this.add('swf:ListDomains');
        return this;
    }
    /**
     * Returns a list of open workflow executions in the specified domain that meet the filtering criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListOpenWorkflowExecutions.html
     */
    listOpenWorkflowExecutions() {
        this.add('swf:ListOpenWorkflowExecutions');
        return this;
    }
    /**
     * This action lists tags for an AWS SWF resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('swf:ListTagsForResource');
        return this;
    }
    /**
     * Returns information about workflow types in the specified domain.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ListWorkflowTypes.html
     */
    listWorkflowTypes() {
        this.add('swf:ListWorkflowTypes');
        return this;
    }
    /**
     * Used by workers to get an ActivityTask from the specified activity taskList.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForActivityTask.html
     */
    pollForActivityTask() {
        this.add('swf:PollForActivityTask');
        return this;
    }
    /**
     * Used by deciders to get a DecisionTask from the specified decision taskList.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_PollForDecisionTask.html
     */
    pollForDecisionTask() {
        this.add('swf:PollForDecisionTask');
        return this;
    }
    /**
     * Used by activity workers to report to the service that the ActivityTask represented by the specified taskToken is still making progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordActivityTaskHeartbeat.html
     */
    recordActivityTaskHeartbeat() {
        this.add('swf:RecordActivityTaskHeartbeat');
        return this;
    }
    /**
     * Description for RecordMarker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RecordMarker.html
     */
    recordMarker() {
        this.add('swf:RecordMarker');
        return this;
    }
    /**
     * Registers a new activity type along with its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterActivityType.html
     */
    registerActivityType() {
        this.add('swf:RegisterActivityType');
        return this;
    }
    /**
     * Registers a new domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterDomain.html
     */
    registerDomain() {
        this.add('swf:RegisterDomain');
        return this;
    }
    /**
     * Registers a new workflow type and its configuration settings in the specified domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RegisterWorkflowType.html
     */
    registerWorkflowType() {
        this.add('swf:RegisterWorkflowType');
        return this;
    }
    /**
     * Description for RequestCancelActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelActivityTask.html
     */
    requestCancelActivityTask() {
        this.add('swf:RequestCancelActivityTask');
        return this;
    }
    /**
     * Description for RequestCancelExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelExternalWorkflowExecution.html
     */
    requestCancelExternalWorkflowExecution() {
        this.add('swf:RequestCancelExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionCancelRequested event in the currently running workflow execution identified by the given domain, workflowId, and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RequestCancelWorkflowExecution.html
     */
    requestCancelWorkflowExecution() {
        this.add('swf:RequestCancelWorkflowExecution');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken was successfully canceled.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCanceled.html
     */
    respondActivityTaskCanceled() {
        this.add('swf:RespondActivityTaskCanceled');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken completed successfully with a result (if provided).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskCompleted.html
     */
    respondActivityTaskCompleted() {
        this.add('swf:RespondActivityTaskCompleted');
        return this;
    }
    /**
     * Used by workers to tell the service that the ActivityTask identified by the taskToken has failed with reason (if specified).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondActivityTaskFailed.html
     */
    respondActivityTaskFailed() {
        this.add('swf:RespondActivityTaskFailed');
        return this;
    }
    /**
     * Used by deciders to tell the service that the DecisionTask identified by the taskToken has successfully completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_RespondDecisionTaskCompleted.html
     */
    respondDecisionTaskCompleted() {
        this.add('swf:RespondDecisionTaskCompleted');
        return this;
    }
    /**
     * Description for ScheduleActivityTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_ScheduleActivityTask.html
     */
    scheduleActivityTask() {
        this.add('swf:ScheduleActivityTask');
        return this;
    }
    /**
     * Description for SignalExternalWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalExternalWorkflowExecution.html
     */
    signalExternalWorkflowExecution() {
        this.add('swf:SignalExternalWorkflowExecution');
        return this;
    }
    /**
     * Records a WorkflowExecutionSignaled event in the workflow execution history and creates a decision task for the workflow execution identified by the given domain, workflowId and runId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_SignalWorkflowExecution.html
     */
    signalWorkflowExecution() {
        this.add('swf:SignalWorkflowExecution');
        return this;
    }
    /**
     * Description for StartChildWorkflowExecution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartChildWorkflowExecution.html
     */
    startChildWorkflowExecution() {
        this.add('swf:StartChildWorkflowExecution');
        return this;
    }
    /**
     * Description for StartTimer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartTimer.html
     */
    startTimer() {
        this.add('swf:StartTimer');
        return this;
    }
    /**
     * Starts an execution of the workflow type in the specified domain using the provided workflowId and input data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_StartWorkflowExecution.html
     */
    startWorkflowExecution() {
        this.add('swf:StartWorkflowExecution');
        return this;
    }
    /**
     * This action tags an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('swf:TagResource');
        return this;
    }
    /**
     * Records a WorkflowExecutionTerminated event and forces closure of the workflow execution identified by the given domain, runId, and workflowId.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_TerminateWorkflowExecution.html
     */
    terminateWorkflowExecution() {
        this.add('swf:TerminateWorkflowExecution');
        return this;
    }
    /**
     * This action removes a tag from an AWS SWF resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/amazonswf/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('swf:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/swf/latest/developerguide/swf-dev-domains.html
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDomain(domainName, account, partition) {
        var arn = 'arn:${Partition}:swf::${Account}:domain/${DomainName}';
        arn = arn.replace('${DomainName}', domainName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Constrains the policy statement to only an activity type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeName(value, operator) {
        return this.if(`swf:activityType.name`, value, operator || 'StringLike');
    }
    /**
     * Contstrains the policy statement to only an activity type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifActivityTypeVersion(value, operator) {
        return this.if(`swf:activityType.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching defaultTaskList name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDefaultTaskListName(value, operator) {
        return this.if(`swf:defaultTaskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifName(value, operator) {
        return this.if(`swf:name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a matching tagFilter.tag value.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagFilterTag(value, operator) {
        return this.if(`swf:tagFilter.tag`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember0(value, operator) {
        return this.if(`swf:tagList.member.0`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember1(value, operator) {
        return this.if(`swf:tagList.member.1`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember2(value, operator) {
        return this.if(`swf:tagList.member.2`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember3(value, operator) {
        return this.if(`swf:tagList.member.3`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that contain the specified tag.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTagListMember4(value, operator) {
        return this.if(`swf:tagList.member.4`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a tasklist with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskListName(value, operator) {
        return this.if(`swf:taskList.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterName(value, operator) {
        return this.if(`swf:typeFilter.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a type filter with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTypeFilterVersion(value, operator) {
        return this.if(`swf:typeFilter.version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only activities or workflows with the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersion(value, operator) {
        return this.if(`swf:version`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified name.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeName(value, operator) {
        return this.if(`swf:workflowType.name`, value, operator || 'StringLike');
    }
    /**
     * Constrains the policy statement to only requests that specify a workflow type of the specified version.
     *
     * https://docs.aws.amazon.com/amazonswf/latest/developerguide/swf-dev-iam.html##swf-dev-iam.api
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWorkflowTypeVersion(value, operator) {
        return this.if(`swf:workflowType.version`, value, operator || 'StringLike');
    }
}
exports.Swf = Swf;
//# sourceMappingURL=data:application/json;base64,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