"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Worklink = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [worklink](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworklink.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Worklink extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [worklink](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonworklink.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'worklink';
        this.actionList = {
            "AssociateDomain": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_AssociateDomain.html",
                "description": "Grants permission to associate a domain with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "AssociateWebsiteAuthorizationProvider": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteAuthorizationProvider.html",
                "description": "Grants permission to associate a website authorization provider with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "AssociateWebsiteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteCertificateAuthority.html",
                "description": "Grants permission to associate a website certificate authority with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "CreateFleet": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_CreateFleet.html",
                "description": "Grants permission to create an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteFleet": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DeleteFleet.html",
                "description": "Grants permission to delete an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeAuditStreamConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeAuditStreamConfiguration.html",
                "description": "Grants permission to describe the audit stream configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeCompanyNetworkConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeCompanyNetworkConfiguration.html",
                "description": "Grants permission to describe the company network configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeDevice": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevice.html",
                "description": "Grants permission to describe details of a device associated with an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeDevicePolicyConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevicePolicyConfiguration.html",
                "description": "Grants permission to describe the device policy configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeDomain": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDomain.html",
                "description": "Grants permission to describe details about a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeFleetMetadata": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeFleetMetadata.html",
                "description": "Grants permission to describe metadata of an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeIdentityProviderConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeIdentityProviderConfiguration.html",
                "description": "Grants permission to describe the identity provider configuration for an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DescribeWebsiteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DescribeWebsiteCertificateAuthority.html",
                "description": "Grants permission to describe a website certificate authority associated with an Amazon WorkLink fleet",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DisassociateDomain": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateDomain.html",
                "description": "Grants permission to disassociate a domain from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DisassociateWebsiteAuthorizationProvider": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteAuthorizationProvider.html",
                "description": "Grants permission to disassociate a website authorization provider from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "DisassociateWebsiteCertificateAuthority": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteCertificateAuthority.html",
                "description": "Grants permission to disassociate a website certificate authority from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListDevices": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListDevices.html",
                "description": "Grants permission to list the devices associated with an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListDomains": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListDomains.html",
                "description": "Grants permission to list the associated domains for an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListFleets": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListFleets.html",
                "description": "Grants permission to list the Amazon WorkLink fleets associated with the account",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListTagsForResource.html",
                "description": "Grants permission to list tags for a resource",
                "accessLevel": "Read",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListWebsiteAuthorizationProviders": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteAuthorizationProviders.html",
                "description": "Grants permission to list the website authorization providers for an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "ListWebsiteCertificateAuthorities": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteCertificateAuthorities.html",
                "description": "Grants permission to list the website certificate authorities associated with an Amazon WorkLink fleet",
                "accessLevel": "List",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "RestoreDomainAccess": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_RestoreDomainAccess.html",
                "description": "Grants permission to restore access to a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "RevokeDomainAccess": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_RevokeDomainAccess.html",
                "description": "Grants permission to revoke access to a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "SignOutUser": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_SignOutUser.html",
                "description": "Grants permission to sign out a user from an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_TagResource.html",
                "description": "Grants permission to add one or more tags to a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UntagResource.html",
                "description": "Grants permission to remove one or more tags from a resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAuditStreamConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateAuditStreamConfiguration.html",
                "description": "Grants permission to update the audit stream configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateCompanyNetworkConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateCompanyNetworkConfiguration.html",
                "description": "Grants permission to update the company network configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateDevicePolicyConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDevicePolicyConfiguration.html",
                "description": "Grants permission to update the device policy configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateDomainMetadata": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDomainMetadata.html",
                "description": "Grants permission to update the metadata for a domain associated with an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateFleetMetadata": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateFleetMetadata.html",
                "description": "Grants permission to update the metadata of an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            },
            "UpdateIdentityProviderConfiguration": {
                "url": "https://docs.aws.amazon.com/worklink/latest/api/API_UpdateIdentityProviderConfiguration.html",
                "description": "Grants permission to update the identity provider configuration for an Amazon WorkLink fleet",
                "accessLevel": "Write",
                "resourceTypes": {
                    "fleet": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "fleet": {
                "name": "fleet",
                "url": "https://docs.aws.amazon.com/worklink/latest/api/worklink-resources.html#Fleet",
                "arn": "arn:${Partition}:worklink::${Account}:fleet/${FleetName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to associate a domain with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateDomain.html
     */
    associateDomain() {
        this.add('worklink:AssociateDomain');
        return this;
    }
    /**
     * Grants permission to associate a website authorization provider with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteAuthorizationProvider.html
     */
    associateWebsiteAuthorizationProvider() {
        this.add('worklink:AssociateWebsiteAuthorizationProvider');
        return this;
    }
    /**
     * Grants permission to associate a website certificate authority with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_AssociateWebsiteCertificateAuthority.html
     */
    associateWebsiteCertificateAuthority() {
        this.add('worklink:AssociateWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to create an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_CreateFleet.html
     */
    createFleet() {
        this.add('worklink:CreateFleet');
        return this;
    }
    /**
     * Grants permission to delete an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DeleteFleet.html
     */
    deleteFleet() {
        this.add('worklink:DeleteFleet');
        return this;
    }
    /**
     * Grants permission to describe the audit stream configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeAuditStreamConfiguration.html
     */
    describeAuditStreamConfiguration() {
        this.add('worklink:DescribeAuditStreamConfiguration');
        return this;
    }
    /**
     * Grants permission to describe the company network configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeCompanyNetworkConfiguration.html
     */
    describeCompanyNetworkConfiguration() {
        this.add('worklink:DescribeCompanyNetworkConfiguration');
        return this;
    }
    /**
     * Grants permission to describe details of a device associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevice.html
     */
    describeDevice() {
        this.add('worklink:DescribeDevice');
        return this;
    }
    /**
     * Grants permission to describe the device policy configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDevicePolicyConfiguration.html
     */
    describeDevicePolicyConfiguration() {
        this.add('worklink:DescribeDevicePolicyConfiguration');
        return this;
    }
    /**
     * Grants permission to describe details about a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeDomain.html
     */
    describeDomain() {
        this.add('worklink:DescribeDomain');
        return this;
    }
    /**
     * Grants permission to describe metadata of an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeFleetMetadata.html
     */
    describeFleetMetadata() {
        this.add('worklink:DescribeFleetMetadata');
        return this;
    }
    /**
     * Grants permission to describe the identity provider configuration for an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeIdentityProviderConfiguration.html
     */
    describeIdentityProviderConfiguration() {
        this.add('worklink:DescribeIdentityProviderConfiguration');
        return this;
    }
    /**
     * Grants permission to describe a website certificate authority associated with an Amazon WorkLink fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DescribeWebsiteCertificateAuthority.html
     */
    describeWebsiteCertificateAuthority() {
        this.add('worklink:DescribeWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to disassociate a domain from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateDomain.html
     */
    disassociateDomain() {
        this.add('worklink:DisassociateDomain');
        return this;
    }
    /**
     * Grants permission to disassociate a website authorization provider from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteAuthorizationProvider.html
     */
    disassociateWebsiteAuthorizationProvider() {
        this.add('worklink:DisassociateWebsiteAuthorizationProvider');
        return this;
    }
    /**
     * Grants permission to disassociate a website certificate authority from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_DisassociateWebsiteCertificateAuthority.html
     */
    disassociateWebsiteCertificateAuthority() {
        this.add('worklink:DisassociateWebsiteCertificateAuthority');
        return this;
    }
    /**
     * Grants permission to list the devices associated with an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListDevices.html
     */
    listDevices() {
        this.add('worklink:ListDevices');
        return this;
    }
    /**
     * Grants permission to list the associated domains for an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListDomains.html
     */
    listDomains() {
        this.add('worklink:ListDomains');
        return this;
    }
    /**
     * Grants permission to list the Amazon WorkLink fleets associated with the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListFleets.html
     */
    listFleets() {
        this.add('worklink:ListFleets');
        return this;
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('worklink:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to list the website authorization providers for an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteAuthorizationProviders.html
     */
    listWebsiteAuthorizationProviders() {
        this.add('worklink:ListWebsiteAuthorizationProviders');
        return this;
    }
    /**
     * Grants permission to list the website certificate authorities associated with an Amazon WorkLink fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_ListWebsiteCertificateAuthorities.html
     */
    listWebsiteCertificateAuthorities() {
        this.add('worklink:ListWebsiteCertificateAuthorities');
        return this;
    }
    /**
     * Grants permission to restore access to a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_RestoreDomainAccess.html
     */
    restoreDomainAccess() {
        this.add('worklink:RestoreDomainAccess');
        return this;
    }
    /**
     * Grants permission to revoke access to a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_RevokeDomainAccess.html
     */
    revokeDomainAccess() {
        this.add('worklink:RevokeDomainAccess');
        return this;
    }
    /**
     * Grants permission to sign out a user from an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_SignOutUser.html
     */
    signOutUser() {
        this.add('worklink:SignOutUser');
        return this;
    }
    /**
     * Grants permission to add one or more tags to a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_TagResource.html
     */
    tagResource() {
        this.add('worklink:TagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UntagResource.html
     */
    untagResource() {
        this.add('worklink:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the audit stream configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateAuditStreamConfiguration.html
     */
    updateAuditStreamConfiguration() {
        this.add('worklink:UpdateAuditStreamConfiguration');
        return this;
    }
    /**
     * Grants permission to update the company network configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateCompanyNetworkConfiguration.html
     */
    updateCompanyNetworkConfiguration() {
        this.add('worklink:UpdateCompanyNetworkConfiguration');
        return this;
    }
    /**
     * Grants permission to update the device policy configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDevicePolicyConfiguration.html
     */
    updateDevicePolicyConfiguration() {
        this.add('worklink:UpdateDevicePolicyConfiguration');
        return this;
    }
    /**
     * Grants permission to update the metadata for a domain associated with an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateDomainMetadata.html
     */
    updateDomainMetadata() {
        this.add('worklink:UpdateDomainMetadata');
        return this;
    }
    /**
     * Grants permission to update the metadata of an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateFleetMetadata.html
     */
    updateFleetMetadata() {
        this.add('worklink:UpdateFleetMetadata');
        return this;
    }
    /**
     * Grants permission to update the identity provider configuration for an Amazon WorkLink fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/worklink/latest/api/API_UpdateIdentityProviderConfiguration.html
     */
    updateIdentityProviderConfiguration() {
        this.add('worklink:UpdateIdentityProviderConfiguration');
        return this;
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/worklink/latest/api/worklink-resources.html#Fleet
     *
     * @param fleetName - Identifier for the fleetName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onFleet(fleetName, account, partition) {
        var arn = 'arn:${Partition}:worklink::${Account}:fleet/${FleetName}';
        arn = arn.replace('${FleetName}', fleetName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Worklink = Worklink;
//# sourceMappingURL=data:application/json;base64,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