"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Backup = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [backup](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbackup.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Backup extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [backup](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbackup.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'backup';
        this.actionList = {
            "CopyIntoBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CopyIntoBackupVault.html",
                "description": "Copy into a backup vault",
                "accessLevel": "Write"
            },
            "CreateBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html",
                "description": "Creates a new backup plan",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html",
                "description": "Creates a new resource assignment in a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "CreateBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html",
                "description": "Creates a new backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html",
                "description": "Deletes a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "DeleteBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html",
                "description": "Deletes a resource assignment from a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html",
                "description": "Deletes a backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html",
                "description": "Deletes backup vault access policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html",
                "description": "Remove notifications from backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DeleteRecoveryPoint": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html",
                "description": "Deletes a recovery point from a backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "DescribeBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html",
                "description": "Describes a backup job",
                "accessLevel": "Read"
            },
            "DescribeBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html",
                "description": "Creates a new backup vault with the specified name.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "DescribeCopyJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html",
                "description": "Describes a copy job",
                "accessLevel": "Read",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DescribeProtectedResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html",
                "description": "Describes a protected resource.",
                "accessLevel": "Read"
            },
            "DescribeRecoveryPoint": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html",
                "description": "Describes a recovery point.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "DescribeRestoreJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html",
                "description": "Describes a restore job.",
                "accessLevel": "Read"
            },
            "ExportBackupPlanTemplate": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html",
                "description": "Exports a backup plan as a JSON.",
                "accessLevel": "Read"
            },
            "GetBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html",
                "description": "Gets a backup plan.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "GetBackupPlanFromJSON": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html",
                "description": "Transforms a JSON to a backup plan.",
                "accessLevel": "Read"
            },
            "GetBackupPlanFromTemplate": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html",
                "description": "Transforms a template to a backup plan.",
                "accessLevel": "Read"
            },
            "GetBackupSelection": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html",
                "description": "Gets a backup plan resource assignment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "GetBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html",
                "description": "Gets backup vault access policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "GetBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html",
                "description": "Gets backup vault notifications.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "GetRecoveryPointRestoreMetadata": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html",
                "description": "Gets recovery point restore metadata.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "GetSupportedResourceTypes": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html",
                "description": "Gets supported resource types.",
                "accessLevel": "Read"
            },
            "ListBackupJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html",
                "description": "Lists backup jobs.",
                "accessLevel": "List"
            },
            "ListBackupPlanTemplates": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html",
                "description": "Lists backup plan templates provided by AWS Backup.",
                "accessLevel": "List"
            },
            "ListBackupPlanVersions": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html",
                "description": "Lists backup plan versions.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "ListBackupPlans": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html",
                "description": "Lists backup plans.",
                "accessLevel": "List"
            },
            "ListBackupSelections": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html",
                "description": "Lists resource assignments for a specific backup plan.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "ListBackupVaults": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html",
                "description": "Lists backup vaults.",
                "accessLevel": "List"
            },
            "ListCopyJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html",
                "description": "List copy jobs",
                "accessLevel": "List"
            },
            "ListProtectedResources": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html",
                "description": "Lists protected resources by AWS Backup.",
                "accessLevel": "List"
            },
            "ListRecoveryPointsByBackupVault": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html",
                "description": "Lists recovery points inside a backup vault.",
                "accessLevel": "List",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "ListRecoveryPointsByResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html",
                "description": "Lists recovery points for a resource.",
                "accessLevel": "List"
            },
            "ListRestoreJobs": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html",
                "description": "Lists restore jobs.",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html",
                "description": "Lists tags for a resource.",
                "accessLevel": "List"
            },
            "PutBackupVaultAccessPolicy": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html",
                "description": "Adds an access policy to the backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "PutBackupVaultNotifications": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html",
                "description": "Adds an SNS topic to the backup vault.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "StartBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html",
                "description": "Starts a new backup job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupVault": {
                        "required": true
                    }
                }
            },
            "StartCopyJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html",
                "description": "Copy a backup from a source region to a destination region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "StartRestoreJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html",
                "description": "Starts a new restore job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            },
            "StopBackupJob": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html",
                "description": "Stops a backup job.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html",
                "description": "Tags a resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html",
                "description": "Untags a resource.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateBackupPlan": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html",
                "description": "Updates a backup plan.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backupPlan": {
                        "required": true
                    }
                }
            },
            "UpdateRecoveryPointLifecycle": {
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html",
                "description": "Updates the lifecycle of the recovery point.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "recoveryPoint": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "backupVault": {
                "name": "backupVault",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:backup-vault:${BackupVaultName}",
                "conditionKeys": []
            },
            "backupPlan": {
                "name": "backupPlan",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:backup-plan:${BackupPlanId}",
                "conditionKeys": []
            },
            "recoveryPoint": {
                "name": "recoveryPoint",
                "url": "https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html",
                "arn": "arn:${Partition}:backup:${Region}:${Account}:recovery-point:${RecoveryPointId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Copy into a backup vault
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CopyIntoBackupVault.html
     */
    copyIntoBackupVault() {
        this.add('backup:CopyIntoBackupVault');
        return this;
    }
    /**
     * Creates a new backup plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupPlan.html
     */
    createBackupPlan() {
        this.add('backup:CreateBackupPlan');
        return this;
    }
    /**
     * Creates a new resource assignment in a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupSelection.html
     */
    createBackupSelection() {
        this.add('backup:CreateBackupSelection');
        return this;
    }
    /**
     * Creates a new backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_CreateBackupVault.html
     */
    createBackupVault() {
        this.add('backup:CreateBackupVault');
        return this;
    }
    /**
     * Deletes a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupPlan.html
     */
    deleteBackupPlan() {
        this.add('backup:DeleteBackupPlan');
        return this;
    }
    /**
     * Deletes a resource assignment from a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupSelection.html
     */
    deleteBackupSelection() {
        this.add('backup:DeleteBackupSelection');
        return this;
    }
    /**
     * Deletes a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVault.html
     */
    deleteBackupVault() {
        this.add('backup:DeleteBackupVault');
        return this;
    }
    /**
     * Deletes backup vault access policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultAccessPolicy.html
     */
    deleteBackupVaultAccessPolicy() {
        this.add('backup:DeleteBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Remove notifications from backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteBackupVaultNotifications.html
     */
    deleteBackupVaultNotifications() {
        this.add('backup:DeleteBackupVaultNotifications');
        return this;
    }
    /**
     * Deletes a recovery point from a backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DeleteRecoveryPoint.html
     */
    deleteRecoveryPoint() {
        this.add('backup:DeleteRecoveryPoint');
        return this;
    }
    /**
     * Describes a backup job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupJob.html
     */
    describeBackupJob() {
        this.add('backup:DescribeBackupJob');
        return this;
    }
    /**
     * Creates a new backup vault with the specified name.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeBackupVault.html
     */
    describeBackupVault() {
        this.add('backup:DescribeBackupVault');
        return this;
    }
    /**
     * Describes a copy job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeCopyJob.html
     */
    describeCopyJob() {
        this.add('backup:DescribeCopyJob');
        return this;
    }
    /**
     * Describes a protected resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeProtectedResource.html
     */
    describeProtectedResource() {
        this.add('backup:DescribeProtectedResource');
        return this;
    }
    /**
     * Describes a recovery point.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRecoveryPoint.html
     */
    describeRecoveryPoint() {
        this.add('backup:DescribeRecoveryPoint');
        return this;
    }
    /**
     * Describes a restore job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_DescribeRestoreJob.html
     */
    describeRestoreJob() {
        this.add('backup:DescribeRestoreJob');
        return this;
    }
    /**
     * Exports a backup plan as a JSON.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ExportBackupPlanTemplate.html
     */
    exportBackupPlanTemplate() {
        this.add('backup:ExportBackupPlanTemplate');
        return this;
    }
    /**
     * Gets a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlan.html
     */
    getBackupPlan() {
        this.add('backup:GetBackupPlan');
        return this;
    }
    /**
     * Transforms a JSON to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromJSON.html
     */
    getBackupPlanFromJSON() {
        this.add('backup:GetBackupPlanFromJSON');
        return this;
    }
    /**
     * Transforms a template to a backup plan.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupPlanFromTemplate.html
     */
    getBackupPlanFromTemplate() {
        this.add('backup:GetBackupPlanFromTemplate');
        return this;
    }
    /**
     * Gets a backup plan resource assignment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupSelection.html
     */
    getBackupSelection() {
        this.add('backup:GetBackupSelection');
        return this;
    }
    /**
     * Gets backup vault access policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultAccessPolicy.html
     */
    getBackupVaultAccessPolicy() {
        this.add('backup:GetBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Gets backup vault notifications.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetBackupVaultNotifications.html
     */
    getBackupVaultNotifications() {
        this.add('backup:GetBackupVaultNotifications');
        return this;
    }
    /**
     * Gets recovery point restore metadata.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetRecoveryPointRestoreMetadata.html
     */
    getRecoveryPointRestoreMetadata() {
        this.add('backup:GetRecoveryPointRestoreMetadata');
        return this;
    }
    /**
     * Gets supported resource types.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_GetSupportedResourceTypes.html
     */
    getSupportedResourceTypes() {
        this.add('backup:GetSupportedResourceTypes');
        return this;
    }
    /**
     * Lists backup jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html
     */
    listBackupJobs() {
        this.add('backup:ListBackupJobs');
        return this;
    }
    /**
     * Lists backup plan templates provided by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanTemplates.html
     */
    listBackupPlanTemplates() {
        this.add('backup:ListBackupPlanTemplates');
        return this;
    }
    /**
     * Lists backup plan versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlanVersions.html
     */
    listBackupPlanVersions() {
        this.add('backup:ListBackupPlanVersions');
        return this;
    }
    /**
     * Lists backup plans.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupPlans.html
     */
    listBackupPlans() {
        this.add('backup:ListBackupPlans');
        return this;
    }
    /**
     * Lists resource assignments for a specific backup plan.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupSelections.html
     */
    listBackupSelections() {
        this.add('backup:ListBackupSelections');
        return this;
    }
    /**
     * Lists backup vaults.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupVaults.html
     */
    listBackupVaults() {
        this.add('backup:ListBackupVaults');
        return this;
    }
    /**
     * List copy jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListCopyJobs.html
     */
    listCopyJobs() {
        this.add('backup:ListCopyJobs');
        return this;
    }
    /**
     * Lists protected resources by AWS Backup.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListProtectedResources.html
     */
    listProtectedResources() {
        this.add('backup:ListProtectedResources');
        return this;
    }
    /**
     * Lists recovery points inside a backup vault.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByBackupVault.html
     */
    listRecoveryPointsByBackupVault() {
        this.add('backup:ListRecoveryPointsByBackupVault');
        return this;
    }
    /**
     * Lists recovery points for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRecoveryPointsByResource.html
     */
    listRecoveryPointsByResource() {
        this.add('backup:ListRecoveryPointsByResource');
        return this;
    }
    /**
     * Lists restore jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListRestoreJobs.html
     */
    listRestoreJobs() {
        this.add('backup:ListRestoreJobs');
        return this;
    }
    /**
     * Lists tags for a resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListTags.html
     */
    listTags() {
        this.add('backup:ListTags');
        return this;
    }
    /**
     * Adds an access policy to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultAccessPolicy.html
     */
    putBackupVaultAccessPolicy() {
        this.add('backup:PutBackupVaultAccessPolicy');
        return this;
    }
    /**
     * Adds an SNS topic to the backup vault.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_PutBackupVaultNotifications.html
     */
    putBackupVaultNotifications() {
        this.add('backup:PutBackupVaultNotifications');
        return this;
    }
    /**
     * Starts a new backup job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartBackupJob.html
     */
    startBackupJob() {
        this.add('backup:StartBackupJob');
        return this;
    }
    /**
     * Copy a backup from a source region to a destination region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartCopyJob.html
     */
    startCopyJob() {
        this.add('backup:StartCopyJob');
        return this;
    }
    /**
     * Starts a new restore job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StartRestoreJob.html
     */
    startRestoreJob() {
        this.add('backup:StartRestoreJob');
        return this;
    }
    /**
     * Stops a backup job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_StopBackupJob.html
     */
    stopBackupJob() {
        this.add('backup:StopBackupJob');
        return this;
    }
    /**
     * Tags a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_TagResource.html
     */
    tagResource() {
        this.add('backup:TagResource');
        return this;
    }
    /**
     * Untags a resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UntagResource.html
     */
    untagResource() {
        this.add('backup:UntagResource');
        return this;
    }
    /**
     * Updates a backup plan.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateBackupPlan.html
     */
    updateBackupPlan() {
        this.add('backup:UpdateBackupPlan');
        return this;
    }
    /**
     * Updates the lifecycle of the recovery point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/API_UpdateRecoveryPointLifecycle.html
     */
    updateRecoveryPointLifecycle() {
        this.add('backup:UpdateRecoveryPointLifecycle');
        return this;
    }
    /**
     * Adds a resource of type backupVault to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/vaults.html
     *
     * @param backupVaultName - Identifier for the backupVaultName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBackupVault(backupVaultName, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-vault:${BackupVaultName}';
        arn = arn.replace('${BackupVaultName}', backupVaultName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type backupPlan to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/about-backup-plans.html
     *
     * @param backupPlanId - Identifier for the backupPlanId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBackupPlan(backupPlanId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:backup-plan:${BackupPlanId}';
        arn = arn.replace('${BackupPlanId}', backupPlanId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type recoveryPoint to the statement
     *
     * https://docs.aws.amazon.com/aws-backup/latest/devguide/recovery-points.html
     *
     * @param recoveryPointId - Identifier for the recoveryPointId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRecoveryPoint(recoveryPointId, account, region, partition) {
        var arn = 'arn:${Partition}:backup:${Region}:${Account}:recovery-point:${RecoveryPointId}';
        arn = arn.replace('${RecoveryPointId}', recoveryPointId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Backup = Backup;
//# sourceMappingURL=data:application/json;base64,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