"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloud9 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloud9](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloud9 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloud9](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloud9';
        this.actionList = {
            "CreateEnvironmentEC2": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentEC2.html",
                "description": "Grants permission to create an AWS Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and then hosts the environment on the instance.",
                "accessLevel": "Write",
                "conditions": [
                    "cloud9:EnvironmentName",
                    "cloud9:InstanceType",
                    "cloud9:SubnetId",
                    "cloud9:UserArn"
                ]
            },
            "CreateEnvironmentMembership": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentMembership.html",
                "description": "Grants permission to add an environment member to an AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloud9:UserArn",
                    "cloud9:EnvironmentId",
                    "cloud9:Permissions"
                ]
            },
            "DeleteEnvironment": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironment.html",
                "description": "Grants permission to delete an AWS Cloud9 development environment. If the environment is hosted on an Amazon Elastic Compute Cloud (Amazon EC2) instance, also terminates the instance.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "DeleteEnvironmentMembership": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironmentMembership.html",
                "description": "Grants permission to delete an environment member from an AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "DescribeEnvironmentMemberships": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentMemberships.html",
                "description": "Grants permission to get information about environment members for an AWS Cloud9 development environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloud9:UserArn",
                    "cloud9:EnvironmentId"
                ]
            },
            "DescribeEnvironmentStatus": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentStatus.html",
                "description": "Grants permission to get status information for an AWS Cloud9 development environment.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "DescribeEnvironments": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironments.html",
                "description": "Grants permission to get information about AWS Cloud9 development environments.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "GetUserSettings": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html",
                "description": "Grants permission to get IDE-specific settings of an AWS Cloud9 user.",
                "accessLevel": "Read"
            },
            "ListEnvironments": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListEnvironments.html",
                "description": "Grants permission to get a list of AWS Cloud9 development environment identifiers.",
                "accessLevel": "Read"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListTagsForResource.html",
                "description": "Lists tags for a cloud9 environment",
                "accessLevel": "Read",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_TagResource.html",
                "description": "Adds tags to a cloud9 environment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UntagResource.html",
                "description": "Removes tags from a cloud9 environment",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateEnvironment": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironment.html",
                "description": "Grants permission to change the settings of an existing AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                }
            },
            "UpdateEnvironmentMembership": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironmentMembership.html",
                "description": "Grants permission to change the settings of an existing environment member for an AWS Cloud9 development environment.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "environment": {
                        "required": true
                    }
                },
                "conditions": [
                    "cloud9:UserArn",
                    "cloud9:EnvironmentId",
                    "cloud9:Permissions"
                ]
            },
            "UpdateUserSettings": {
                "url": "https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html",
                "description": "Grants permission to update IDE-specific settings of an AWS Cloud9 user.",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "environment": {
                "name": "environment",
                "url": "https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-environment",
                "arn": "arn:${Partition}:cloud9:${Region}:${Account}:environment:${ResourceId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to create an AWS Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and then hosts the environment on the instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentEC2.html
     */
    createEnvironmentEC2() {
        this.add('cloud9:CreateEnvironmentEC2');
        return this;
    }
    /**
     * Grants permission to add an environment member to an AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentMembership.html
     */
    createEnvironmentMembership() {
        this.add('cloud9:CreateEnvironmentMembership');
        return this;
    }
    /**
     * Grants permission to delete an AWS Cloud9 development environment. If the environment is hosted on an Amazon Elastic Compute Cloud (Amazon EC2) instance, also terminates the instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironment.html
     */
    deleteEnvironment() {
        this.add('cloud9:DeleteEnvironment');
        return this;
    }
    /**
     * Grants permission to delete an environment member from an AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironmentMembership.html
     */
    deleteEnvironmentMembership() {
        this.add('cloud9:DeleteEnvironmentMembership');
        return this;
    }
    /**
     * Grants permission to get information about environment members for an AWS Cloud9 development environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentMemberships.html
     */
    describeEnvironmentMemberships() {
        this.add('cloud9:DescribeEnvironmentMemberships');
        return this;
    }
    /**
     * Grants permission to get status information for an AWS Cloud9 development environment.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentStatus.html
     */
    describeEnvironmentStatus() {
        this.add('cloud9:DescribeEnvironmentStatus');
        return this;
    }
    /**
     * Grants permission to get information about AWS Cloud9 development environments.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironments.html
     */
    describeEnvironments() {
        this.add('cloud9:DescribeEnvironments');
        return this;
    }
    /**
     * Grants permission to get IDE-specific settings of an AWS Cloud9 user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html
     */
    getUserSettings() {
        this.add('cloud9:GetUserSettings');
        return this;
    }
    /**
     * Grants permission to get a list of AWS Cloud9 development environment identifiers.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListEnvironments.html
     */
    listEnvironments() {
        this.add('cloud9:ListEnvironments');
        return this;
    }
    /**
     * Lists tags for a cloud9 environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('cloud9:ListTagsForResource');
        return this;
    }
    /**
     * Adds tags to a cloud9 environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('cloud9:TagResource');
        return this;
    }
    /**
     * Removes tags from a cloud9 environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('cloud9:UntagResource');
        return this;
    }
    /**
     * Grants permission to change the settings of an existing AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironment.html
     */
    updateEnvironment() {
        this.add('cloud9:UpdateEnvironment');
        return this;
    }
    /**
     * Grants permission to change the settings of an existing environment member for an AWS Cloud9 development environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironmentMembership.html
     */
    updateEnvironmentMembership() {
        this.add('cloud9:UpdateEnvironmentMembership');
        return this;
    }
    /**
     * Grants permission to update IDE-specific settings of an AWS Cloud9 user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/settings-user.html
     */
    updateUserSettings() {
        this.add('cloud9:UpdateUserSettings');
        return this;
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-environment
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onEnvironment(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:cloud9:${Region}:${Account}:environment:${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the AWS Cloud9 environment ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_EnvironmentId
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentId(value, operator) {
        return this.if(`cloud9:EnvironmentId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS Cloud9 environment name
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_EnvironmentName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentName(value, operator) {
        return this.if(`cloud9:EnvironmentName`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the instance type of the AWS Cloud9 environment's Amazon EC2 instance
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_InstanceType
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`cloud9:InstanceType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the type of AWS Cloud9 permissions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_Permissions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissions(value, operator) {
        return this.if(`cloud9:Permissions`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the subnet ID that the AWS Cloud9 environment will be created in
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_SubnetId
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnetId(value, operator) {
        return this.if(`cloud9:SubnetId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the user ARN specified
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html#awscloud9-cloud9_UserArn
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifUserArn(value, operator) {
        return this.if(`cloud9:UserArn`, value, operator || 'ArnEquals');
    }
}
exports.Cloud9 = Cloud9;
//# sourceMappingURL=data:application/json;base64,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