"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudhsm = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudhsm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudhsm.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudhsm extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudhsm](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudhsm.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudhsm';
        this.actionList = {
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_AddTagsToResource.html",
                "description": "Adds or overwrites one or more tags for the specified AWS CloudHSM resource",
                "accessLevel": "Tagging"
            },
            "CopyBackupToRegion": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CopyBackupToRegion.html",
                "description": "Creates a copy of a backup in the specified region",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCluster": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateCluster.html",
                "description": "Creates a new AWS CloudHSM cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backup": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateHapg": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_CreateHapg.html",
                "description": "Creates a high-availability partition group",
                "accessLevel": "Write"
            },
            "CreateHsm": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html",
                "description": "Creates a new hardware security module (HSM) in the specified AWS CloudHSM cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "CreateLunaClient": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_CreateLunaClient.html",
                "description": "Creates an HSM client",
                "accessLevel": "Write"
            },
            "DeleteBackup": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteBackup.html",
                "description": "Deletes the specified CloudHSM backup",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backup": {
                        "required": true
                    }
                }
            },
            "DeleteCluster": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteCluster.html",
                "description": "Deletes the specified AWS CloudHSM cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "DeleteHapg": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DeleteHapg.html",
                "description": "Deletes a high-availability partition group",
                "accessLevel": "Write"
            },
            "DeleteHsm": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteHsm.html",
                "description": "Deletes the specified HSM",
                "accessLevel": "Write"
            },
            "DeleteLunaClient": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DeleteLunaClient.html",
                "description": "Deletes a client",
                "accessLevel": "Write"
            },
            "DescribeBackups": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeBackups.html",
                "description": "Gets information about backups of AWS CloudHSM clusters",
                "accessLevel": "Read"
            },
            "DescribeClusters": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeClusters.html",
                "description": "Gets information about AWS CloudHSM clusters",
                "accessLevel": "Read"
            },
            "DescribeHapg": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeHapg.html",
                "description": "Retrieves information about a high-availability partition group",
                "accessLevel": "Read"
            },
            "DescribeHsm": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeHsm.html",
                "description": "Retrieves information about an HSM. You can identify the HSM by its ARN or its serial number",
                "accessLevel": "Read"
            },
            "DescribeLunaClient": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeLunaClient.html",
                "description": "Retrieves information about an HSM client",
                "accessLevel": "Read"
            },
            "GetConfig": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_GetConfig.html",
                "description": "Gets the configuration files necessary to connect to all high availability partition groups the client is associated with",
                "accessLevel": "Read"
            },
            "InitializeCluster": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_InitializeCluster.html",
                "description": "Claims an AWS CloudHSM cluster",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cluster": {
                        "required": true
                    }
                }
            },
            "ListAvailableZones": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListAvailableZones.html",
                "description": "Lists the Availability Zones that have available AWS CloudHSM capacity",
                "accessLevel": "List"
            },
            "ListHapgs": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListHapgs.html",
                "description": "Lists the high-availability partition groups for the account",
                "accessLevel": "List"
            },
            "ListHsms": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListHsms.html",
                "description": "Retrieves the identifiers of all of the HSMs provisioned for the current customer",
                "accessLevel": "List"
            },
            "ListLunaClients": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListLunaClients.html",
                "description": "Lists all of the clients",
                "accessLevel": "List"
            },
            "ListTags": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_ListTags.html",
                "description": "Gets a list of tags for the specified AWS CloudHSM cluster",
                "accessLevel": "Read",
                "resourceTypes": {
                    "backup": {
                        "required": false
                    },
                    "cluster": {
                        "required": false
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListTagsForResource.html",
                "description": "Returns a list of all tags for the specified AWS CloudHSM resource",
                "accessLevel": "Read"
            },
            "ModifyHapg": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyHapg.html",
                "description": "Modifies an existing high-availability partition group",
                "accessLevel": "Write"
            },
            "ModifyHsm": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyHsm.html",
                "description": "Modifies an HSM",
                "accessLevel": "Write"
            },
            "ModifyLunaClient": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyLunaClient.html",
                "description": "Modifies the certificate used by the client",
                "accessLevel": "Write"
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_RemoveTagsFromResource.html",
                "description": "Removes one or more tags from the specified AWS CloudHSM resource",
                "accessLevel": "Tagging"
            },
            "RestoreBackup": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_RestoreBackup.html",
                "description": "Restores the specified CloudHSM backup",
                "accessLevel": "Write",
                "resourceTypes": {
                    "backup": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_TagResource.html",
                "description": "Adds or overwrites one or more tags for the specified AWS CloudHSM cluster",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "backup": {
                        "required": false
                    },
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_UntagResource.html",
                "description": "Removes the specified tag or tags from the specified AWS CloudHSM cluster",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "backup": {
                        "required": false
                    },
                    "cluster": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "backup": {
                "name": "backup",
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/userguide/backups.html",
                "arn": "arn:${Partition}:cloudhsm:${Region}:${Account}:backup/${CloudHsmBackupInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "cluster": {
                "name": "cluster",
                "url": "https://docs.aws.amazon.com/cloudhsm/latest/userguide/clusters.html",
                "arn": "arn:${Partition}:cloudhsm:${Region}:${Account}:cluster/${CloudHsmClusterInstanceName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Adds or overwrites one or more tags for the specified AWS CloudHSM resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_AddTagsToResource.html
     */
    addTagsToResource() {
        this.add('cloudhsm:AddTagsToResource');
        return this;
    }
    /**
     * Creates a copy of a backup in the specified region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CopyBackupToRegion.html
     */
    copyBackupToRegion() {
        this.add('cloudhsm:CopyBackupToRegion');
        return this;
    }
    /**
     * Creates a new AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateCluster.html
     */
    createCluster() {
        this.add('cloudhsm:CreateCluster');
        return this;
    }
    /**
     * Creates a high-availability partition group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_CreateHapg.html
     */
    createHapg() {
        this.add('cloudhsm:CreateHapg');
        return this;
    }
    /**
     * Creates a new hardware security module (HSM) in the specified AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_CreateHsm.html
     */
    createHsm() {
        this.add('cloudhsm:CreateHsm');
        return this;
    }
    /**
     * Creates an HSM client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_CreateLunaClient.html
     */
    createLunaClient() {
        this.add('cloudhsm:CreateLunaClient');
        return this;
    }
    /**
     * Deletes the specified CloudHSM backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteBackup.html
     */
    deleteBackup() {
        this.add('cloudhsm:DeleteBackup');
        return this;
    }
    /**
     * Deletes the specified AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteCluster.html
     */
    deleteCluster() {
        this.add('cloudhsm:DeleteCluster');
        return this;
    }
    /**
     * Deletes a high-availability partition group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DeleteHapg.html
     */
    deleteHapg() {
        this.add('cloudhsm:DeleteHapg');
        return this;
    }
    /**
     * Deletes the specified HSM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DeleteHsm.html
     */
    deleteHsm() {
        this.add('cloudhsm:DeleteHsm');
        return this;
    }
    /**
     * Deletes a client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DeleteLunaClient.html
     */
    deleteLunaClient() {
        this.add('cloudhsm:DeleteLunaClient');
        return this;
    }
    /**
     * Gets information about backups of AWS CloudHSM clusters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeBackups.html
     */
    describeBackups() {
        this.add('cloudhsm:DescribeBackups');
        return this;
    }
    /**
     * Gets information about AWS CloudHSM clusters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_DescribeClusters.html
     */
    describeClusters() {
        this.add('cloudhsm:DescribeClusters');
        return this;
    }
    /**
     * Retrieves information about a high-availability partition group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeHapg.html
     */
    describeHapg() {
        this.add('cloudhsm:DescribeHapg');
        return this;
    }
    /**
     * Retrieves information about an HSM. You can identify the HSM by its ARN or its serial number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeHsm.html
     */
    describeHsm() {
        this.add('cloudhsm:DescribeHsm');
        return this;
    }
    /**
     * Retrieves information about an HSM client
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_DescribeLunaClient.html
     */
    describeLunaClient() {
        this.add('cloudhsm:DescribeLunaClient');
        return this;
    }
    /**
     * Gets the configuration files necessary to connect to all high availability partition groups the client is associated with
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_GetConfig.html
     */
    getConfig() {
        this.add('cloudhsm:GetConfig');
        return this;
    }
    /**
     * Claims an AWS CloudHSM cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_InitializeCluster.html
     */
    initializeCluster() {
        this.add('cloudhsm:InitializeCluster');
        return this;
    }
    /**
     * Lists the Availability Zones that have available AWS CloudHSM capacity
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListAvailableZones.html
     */
    listAvailableZones() {
        this.add('cloudhsm:ListAvailableZones');
        return this;
    }
    /**
     * Lists the high-availability partition groups for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListHapgs.html
     */
    listHapgs() {
        this.add('cloudhsm:ListHapgs');
        return this;
    }
    /**
     * Retrieves the identifiers of all of the HSMs provisioned for the current customer
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListHsms.html
     */
    listHsms() {
        this.add('cloudhsm:ListHsms');
        return this;
    }
    /**
     * Lists all of the clients
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListLunaClients.html
     */
    listLunaClients() {
        this.add('cloudhsm:ListLunaClients');
        return this;
    }
    /**
     * Gets a list of tags for the specified AWS CloudHSM cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_ListTags.html
     */
    listTags() {
        this.add('cloudhsm:ListTags');
        return this;
    }
    /**
     * Returns a list of all tags for the specified AWS CloudHSM resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('cloudhsm:ListTagsForResource');
        return this;
    }
    /**
     * Modifies an existing high-availability partition group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyHapg.html
     */
    modifyHapg() {
        this.add('cloudhsm:ModifyHapg');
        return this;
    }
    /**
     * Modifies an HSM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyHsm.html
     */
    modifyHsm() {
        this.add('cloudhsm:ModifyHsm');
        return this;
    }
    /**
     * Modifies the certificate used by the client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_ModifyLunaClient.html
     */
    modifyLunaClient() {
        this.add('cloudhsm:ModifyLunaClient');
        return this;
    }
    /**
     * Removes one or more tags from the specified AWS CloudHSM resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudhsm/classic/APIReference/API_RemoveTagsFromResource.html
     */
    removeTagsFromResource() {
        this.add('cloudhsm:RemoveTagsFromResource');
        return this;
    }
    /**
     * Restores the specified CloudHSM backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_RestoreBackup.html
     */
    restoreBackup() {
        this.add('cloudhsm:RestoreBackup');
        return this;
    }
    /**
     * Adds or overwrites one or more tags for the specified AWS CloudHSM cluster
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('cloudhsm:TagResource');
        return this;
    }
    /**
     * Removes the specified tag or tags from the specified AWS CloudHSM cluster
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('cloudhsm:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/userguide/backups.html
     *
     * @param cloudHsmBackupInstanceName - Identifier for the cloudHsmBackupInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBackup(cloudHsmBackupInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudhsm:${Region}:${Account}:backup/${CloudHsmBackupInstanceName}';
        arn = arn.replace('${CloudHsmBackupInstanceName}', cloudHsmBackupInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/cloudhsm/latest/userguide/clusters.html
     *
     * @param cloudHsmClusterInstanceName - Identifier for the cloudHsmClusterInstanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onCluster(cloudHsmClusterInstanceName, account, region, partition) {
        var arn = 'arn:${Partition}:cloudhsm:${Region}:${Account}:cluster/${CloudHsmClusterInstanceName}';
        arn = arn.replace('${CloudHsmClusterInstanceName}', cloudHsmClusterInstanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Cloudhsm = Cloudhsm;
//# sourceMappingURL=data:application/json;base64,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