"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codebuild = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codebuild](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodebuild.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codebuild extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codebuild](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodebuild.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codebuild';
        this.actionList = {
            "BatchDeleteBuilds": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchDeleteBuilds.html",
                "description": "Deletes one or more builds.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchGetBuilds": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuilds.html",
                "description": "Gets information about one or more builds.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchGetProjects": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetProjects.html",
                "description": "Gets information about one or more build projects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "BatchGetReportGroups": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReportGroups.html",
                "description": "Returns an array of ReportGroup objects that are specified by the input reportGroupArns parameter.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "BatchGetReports": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReports.html",
                "description": "Returns an array of the Report objects specified by the input reportArns parameter.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "BatchPutTestCases": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Adds or updates information about a report.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "CreateProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateProject.html",
                "description": "Creates a build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateReport": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Creates a report. A report is created when tests specified in the buildspec file for a report groups run during the build of a project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "CreateReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateReportGroup.html",
                "description": "Creates a report group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWebhook": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateWebhook.html",
                "description": "For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteOAuthToken": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Deletes an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.",
                "accessLevel": "Write"
            },
            "DeleteProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteProject.html",
                "description": "Deletes a build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DeleteReport": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html",
                "description": "Deletes a report.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "DeleteReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReportGroup.html",
                "description": "Deletes a report group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "DeleteResourcePolicy": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteResourcePolicy.html",
                "description": "Deletes a resource policy for the associated project or report group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": false
                    },
                    "report-group": {
                        "required": false
                    }
                }
            },
            "DeleteSourceCredentials": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteSourceCredentials.html",
                "description": "Deletes a set of GitHub, GitHub Enterprise, or Bitbucket source credentials.",
                "accessLevel": "Write"
            },
            "DeleteWebhook": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteWebhook.html",
                "description": "For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "DescribeTestCases": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeTestCases.html",
                "description": "Returns an array of TestCase objects.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "GetResourcePolicy": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetResourcePolicy.html",
                "description": "Returns a resource policy for the specified project or report group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "project": {
                        "required": false
                    },
                    "report-group": {
                        "required": false
                    }
                }
            },
            "ImportSourceCredentials": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ImportSourceCredentials.html",
                "description": "Imports the source repository credentials for an AWS CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository.",
                "accessLevel": "Write"
            },
            "InvalidateProjectCache": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_InvalidateProjectCache.html",
                "description": "Resets the cache for a project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListBuilds": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuilds.html",
                "description": "Gets a list of build IDs, with each build ID representing a single build.",
                "accessLevel": "List"
            },
            "ListBuildsForProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildsForProject.html",
                "description": "Gets a list of build IDs for the specified build project, with each build ID representing a single build.",
                "accessLevel": "List",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "ListConnectedOAuthAccounts": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Lists connected third-party OAuth providers. Only used in the AWS CodeBuild console.",
                "accessLevel": "List"
            },
            "ListCuratedEnvironmentImages": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCuratedEnvironmentImages.html",
                "description": "Gets information about Docker images that are managed by AWS CodeBuild.",
                "accessLevel": "List"
            },
            "ListProjects": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListProjects.html",
                "description": "Gets a list of build project names, with each build project name representing a single build project.",
                "accessLevel": "List"
            },
            "ListReportGroups": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportGroups.html",
                "description": "Returns a list of report group ARNs. Each report group ARN represents one report group.",
                "accessLevel": "List"
            },
            "ListReports": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReports.html",
                "description": "Returns a list of report ARNs. Each report ARN representing one report.",
                "accessLevel": "List"
            },
            "ListReportsForReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html",
                "description": "Returns a list of report ARNs that belong to the specified report group. Each report ARN represents one report.",
                "accessLevel": "List",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "ListRepositories": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Lists source code repositories from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.",
                "accessLevel": "List"
            },
            "ListSharedProjects": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedProjects.html",
                "description": "Returns a list of project ARNs that have been shared with the requester. Each project ARN represents one project.",
                "accessLevel": "List"
            },
            "ListSharedReportGroups": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedReportGroups.html",
                "description": "Returns a list of report group ARNs that have been shared with the requester. Each report group ARN represents one report group.",
                "accessLevel": "List"
            },
            "ListSourceCredentials": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSourceCredentials.html",
                "description": "Returns a list of SourceCredentialsInfo objects.",
                "accessLevel": "List"
            },
            "PersistOAuthToken": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Saves an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.",
                "accessLevel": "Write"
            },
            "PutResourcePolicy": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_PutResourcePolicy.html",
                "description": "Creates a resource policy for the associated project or report group.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "project": {
                        "required": false
                    },
                    "report-group": {
                        "required": false
                    }
                }
            },
            "StartBuild": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html",
                "description": "Starts running a build.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "StopBuild": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuild.html",
                "description": "Attempts to stop running a build.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            },
            "UpdateProject": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProject.html",
                "description": "Changes the settings of an existing build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateReport": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies",
                "description": "Updates information about a report.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                }
            },
            "UpdateReportGroup": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateReportGroup.html",
                "description": "Changes the settings of an existing report group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "report-group": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UpdateWebhook": {
                "url": "https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateWebhook.html",
                "description": "Updates the webhook associated with an AWS CodeBuild build project.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "project": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "build": {
                "name": "build",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:build/${BuildId}",
                "conditionKeys": []
            },
            "project": {
                "name": "project",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:project/${ProjectName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "report-group": {
                "name": "report-group",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:report-group/${ReportGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "report": {
                "name": "report",
                "url": "https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats",
                "arn": "arn:${Partition}:codebuild:${Region}:${Account}:report/${ReportGroupName}:${ReportId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Deletes one or more builds.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchDeleteBuilds.html
     */
    batchDeleteBuilds() {
        this.add('codebuild:BatchDeleteBuilds');
        return this;
    }
    /**
     * Gets information about one or more builds.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuilds.html
     */
    batchGetBuilds() {
        this.add('codebuild:BatchGetBuilds');
        return this;
    }
    /**
     * Gets information about one or more build projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetProjects.html
     */
    batchGetProjects() {
        this.add('codebuild:BatchGetProjects');
        return this;
    }
    /**
     * Returns an array of ReportGroup objects that are specified by the input reportGroupArns parameter.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReportGroups.html
     */
    batchGetReportGroups() {
        this.add('codebuild:BatchGetReportGroups');
        return this;
    }
    /**
     * Returns an array of the Report objects specified by the input reportArns parameter.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReports.html
     */
    batchGetReports() {
        this.add('codebuild:BatchGetReports');
        return this;
    }
    /**
     * Adds or updates information about a report.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    batchPutTestCases() {
        this.add('codebuild:BatchPutTestCases');
        return this;
    }
    /**
     * Creates a build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateProject.html
     */
    createProject() {
        this.add('codebuild:CreateProject');
        return this;
    }
    /**
     * Creates a report. A report is created when tests specified in the buildspec file for a report groups run during the build of a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    createReport() {
        this.add('codebuild:CreateReport');
        return this;
    }
    /**
     * Creates a report group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateReportGroup.html
     */
    createReportGroup() {
        this.add('codebuild:CreateReportGroup');
        return this;
    }
    /**
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateWebhook.html
     */
    createWebhook() {
        this.add('codebuild:CreateWebhook');
        return this;
    }
    /**
     * Deletes an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    deleteOAuthToken() {
        this.add('codebuild:DeleteOAuthToken');
        return this;
    }
    /**
     * Deletes a build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteProject.html
     */
    deleteProject() {
        this.add('codebuild:DeleteProject');
        return this;
    }
    /**
     * Deletes a report.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html
     */
    deleteReport() {
        this.add('codebuild:DeleteReport');
        return this;
    }
    /**
     * Deletes a report group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReportGroup.html
     */
    deleteReportGroup() {
        this.add('codebuild:DeleteReportGroup');
        return this;
    }
    /**
     * Deletes a resource policy for the associated project or report group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteResourcePolicy.html
     */
    deleteResourcePolicy() {
        this.add('codebuild:DeleteResourcePolicy');
        return this;
    }
    /**
     * Deletes a set of GitHub, GitHub Enterprise, or Bitbucket source credentials.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteSourceCredentials.html
     */
    deleteSourceCredentials() {
        this.add('codebuild:DeleteSourceCredentials');
        return this;
    }
    /**
     * For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteWebhook.html
     */
    deleteWebhook() {
        this.add('codebuild:DeleteWebhook');
        return this;
    }
    /**
     * Returns an array of TestCase objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeTestCases.html
     */
    describeTestCases() {
        this.add('codebuild:DescribeTestCases');
        return this;
    }
    /**
     * Returns a resource policy for the specified project or report group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetResourcePolicy.html
     */
    getResourcePolicy() {
        this.add('codebuild:GetResourcePolicy');
        return this;
    }
    /**
     * Imports the source repository credentials for an AWS CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ImportSourceCredentials.html
     */
    importSourceCredentials() {
        this.add('codebuild:ImportSourceCredentials');
        return this;
    }
    /**
     * Resets the cache for a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_InvalidateProjectCache.html
     */
    invalidateProjectCache() {
        this.add('codebuild:InvalidateProjectCache');
        return this;
    }
    /**
     * Gets a list of build IDs, with each build ID representing a single build.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuilds.html
     */
    listBuilds() {
        this.add('codebuild:ListBuilds');
        return this;
    }
    /**
     * Gets a list of build IDs for the specified build project, with each build ID representing a single build.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildsForProject.html
     */
    listBuildsForProject() {
        this.add('codebuild:ListBuildsForProject');
        return this;
    }
    /**
     * Lists connected third-party OAuth providers. Only used in the AWS CodeBuild console.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    listConnectedOAuthAccounts() {
        this.add('codebuild:ListConnectedOAuthAccounts');
        return this;
    }
    /**
     * Gets information about Docker images that are managed by AWS CodeBuild.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCuratedEnvironmentImages.html
     */
    listCuratedEnvironmentImages() {
        this.add('codebuild:ListCuratedEnvironmentImages');
        return this;
    }
    /**
     * Gets a list of build project names, with each build project name representing a single build project.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListProjects.html
     */
    listProjects() {
        this.add('codebuild:ListProjects');
        return this;
    }
    /**
     * Returns a list of report group ARNs. Each report group ARN represents one report group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportGroups.html
     */
    listReportGroups() {
        this.add('codebuild:ListReportGroups');
        return this;
    }
    /**
     * Returns a list of report ARNs. Each report ARN representing one report.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReports.html
     */
    listReports() {
        this.add('codebuild:ListReports');
        return this;
    }
    /**
     * Returns a list of report ARNs that belong to the specified report group. Each report ARN represents one report.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html
     */
    listReportsForReportGroup() {
        this.add('codebuild:ListReportsForReportGroup');
        return this;
    }
    /**
     * Lists source code repositories from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    listRepositories() {
        this.add('codebuild:ListRepositories');
        return this;
    }
    /**
     * Returns a list of project ARNs that have been shared with the requester. Each project ARN represents one project.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedProjects.html
     */
    listSharedProjects() {
        this.add('codebuild:ListSharedProjects');
        return this;
    }
    /**
     * Returns a list of report group ARNs that have been shared with the requester. Each report group ARN represents one report group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedReportGroups.html
     */
    listSharedReportGroups() {
        this.add('codebuild:ListSharedReportGroups');
        return this;
    }
    /**
     * Returns a list of SourceCredentialsInfo objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSourceCredentials.html
     */
    listSourceCredentials() {
        this.add('codebuild:ListSourceCredentials');
        return this;
    }
    /**
     * Saves an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    persistOAuthToken() {
        this.add('codebuild:PersistOAuthToken');
        return this;
    }
    /**
     * Creates a resource policy for the associated project or report group.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_PutResourcePolicy.html
     */
    putResourcePolicy() {
        this.add('codebuild:PutResourcePolicy');
        return this;
    }
    /**
     * Starts running a build.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html
     */
    startBuild() {
        this.add('codebuild:StartBuild');
        return this;
    }
    /**
     * Attempts to stop running a build.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuild.html
     */
    stopBuild() {
        this.add('codebuild:StopBuild');
        return this;
    }
    /**
     * Changes the settings of an existing build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProject.html
     */
    updateProject() {
        this.add('codebuild:UpdateProject');
        return this;
    }
    /**
     * Updates information about a report.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    updateReport() {
        this.add('codebuild:UpdateReport');
        return this;
    }
    /**
     * Changes the settings of an existing report group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateReportGroup.html
     */
    updateReportGroup() {
        this.add('codebuild:UpdateReportGroup');
        return this;
    }
    /**
     * Updates the webhook associated with an AWS CodeBuild build project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateWebhook.html
     */
    updateWebhook() {
        this.add('codebuild:UpdateWebhook');
        return this;
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildId - Identifier for the buildId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBuild(buildId, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:build/${BuildId}';
        arn = arn.replace('${BuildId}', buildId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onProject(projectName, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:project/${ProjectName}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type report-group to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onReportGroup(reportGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:report-group/${ReportGroupName}';
        arn = arn.replace('${ReportGroupName}', reportGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type report to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReport(reportGroupName, reportId, account, region, partition) {
        var arn = 'arn:${Partition}:codebuild:${Region}:${Account}:report/${ReportGroupName}:${ReportId}';
        arn = arn.replace('${ReportGroupName}', reportGroupName);
        arn = arn.replace('${ReportId}', reportId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codebuild = Codebuild;
//# sourceMappingURL=data:application/json;base64,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