"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deepcomposer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [deepcomposer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepcomposer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deepcomposer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [deepcomposer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdeepcomposer.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'deepcomposer';
        this.actionList = {
            "AssociateCoupon": {
                "url": "",
                "description": "Associates a DeepComposer coupon (or DSN) with the account associated with the sender of the request.",
                "accessLevel": "Write"
            },
            "CreateAudio": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Creates an audio file by converting the midi composition into a wav or mp3 file.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "composition": {
                        "required": true
                    }
                }
            },
            "CreateComposition": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Creates a multi-track midi composition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateModel": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html",
                "description": "Starts creating/training a generative-model that is able to perform inference against the user-provided piano-melody to create a multi-track midi composition.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteComposition": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Deletes the composition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "composition": {
                        "required": true
                    }
                }
            },
            "DeleteModel": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html",
                "description": "Deletes the model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            },
            "GetComposition": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Returns information about the composition.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "composition": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetModel": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html",
                "description": "Returns information about the model.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "GetSampleModel": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Returns information about the sample/pre-trained DeepComposer model.",
                "accessLevel": "Read"
            },
            "ListCompositions": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Returns a list of all the compositions owned by the sender of the request.",
                "accessLevel": "List"
            },
            "ListModels": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html",
                "description": "Returns a list of all the models owned by the sender of the request.",
                "accessLevel": "List"
            },
            "ListSampleModels": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Returns a list of all the sample/pre-trained models provided by the DeepComposer service.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "",
                "description": "Grants permission to lists tag for a resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "composition": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "ListTrainingTopics": {
                "url": "",
                "description": "Returns a list of all the training options or topic for creating/training a model.",
                "accessLevel": "List"
            },
            "TagResource": {
                "url": "",
                "description": "Grants permission to tag a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "composition": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UntagResource": {
                "url": "",
                "description": "Grants permission to untag a resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "composition": {
                        "required": false
                    },
                    "model": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys",
                    "aws:RequestTag/${TagKey}",
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "UpdateComposition": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "description": "Modifies the mutable properties associated with a composition.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "composition": {
                        "required": true
                    }
                }
            },
            "UpdateModel": {
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html",
                "description": "Modifies the mutable properties associated with a model.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "model": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "model": {
                "name": "model",
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html",
                "arn": "arn:${Partition}:deepcomposer:${Region}:${Account}:model/${ModelId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "composition": {
                "name": "composition",
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "arn": "arn:${Partition}:deepcomposer:${Region}:${Account}:composition/${CompositionId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "audio": {
                "name": "audio",
                "url": "https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html",
                "arn": "arn:${Partition}:deepcomposer:${Region}:${Account}:audio/${AudioId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associates a DeepComposer coupon (or DSN) with the account associated with the sender of the request.
     *
     * Access Level: Write
     */
    associateCoupon() {
        this.add('deepcomposer:AssociateCoupon');
        return this;
    }
    /**
     * Creates an audio file by converting the midi composition into a wav or mp3 file.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    createAudio() {
        this.add('deepcomposer:CreateAudio');
        return this;
    }
    /**
     * Creates a multi-track midi composition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    createComposition() {
        this.add('deepcomposer:CreateComposition');
        return this;
    }
    /**
     * Starts creating/training a generative-model that is able to perform inference against the user-provided piano-melody to create a multi-track midi composition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    createModel() {
        this.add('deepcomposer:CreateModel');
        return this;
    }
    /**
     * Deletes the composition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    deleteComposition() {
        this.add('deepcomposer:DeleteComposition');
        return this;
    }
    /**
     * Deletes the model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    deleteModel() {
        this.add('deepcomposer:DeleteModel');
        return this;
    }
    /**
     * Returns information about the composition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    getComposition() {
        this.add('deepcomposer:GetComposition');
        return this;
    }
    /**
     * Returns information about the model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    getModel() {
        this.add('deepcomposer:GetModel');
        return this;
    }
    /**
     * Returns information about the sample/pre-trained DeepComposer model.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    getSampleModel() {
        this.add('deepcomposer:GetSampleModel');
        return this;
    }
    /**
     * Returns a list of all the compositions owned by the sender of the request.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    listCompositions() {
        this.add('deepcomposer:ListCompositions');
        return this;
    }
    /**
     * Returns a list of all the models owned by the sender of the request.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    listModels() {
        this.add('deepcomposer:ListModels');
        return this;
    }
    /**
     * Returns a list of all the sample/pre-trained models provided by the DeepComposer service.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    listSampleModels() {
        this.add('deepcomposer:ListSampleModels');
        return this;
    }
    /**
     * Grants permission to lists tag for a resource.
     *
     * Access Level: List
     */
    listTagsForResource() {
        this.add('deepcomposer:ListTagsForResource');
        return this;
    }
    /**
     * Returns a list of all the training options or topic for creating/training a model.
     *
     * Access Level: List
     */
    listTrainingTopics() {
        this.add('deepcomposer:ListTrainingTopics');
        return this;
    }
    /**
     * Grants permission to tag a resource.
     *
     * Access Level: Tagging
     */
    tagResource() {
        this.add('deepcomposer:TagResource');
        return this;
    }
    /**
     * Grants permission to untag a resource.
     *
     * Access Level: Tagging
     */
    untagResource() {
        this.add('deepcomposer:UntagResource');
        return this;
    }
    /**
     * Modifies the mutable properties associated with a composition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     */
    updateComposition() {
        this.add('deepcomposer:UpdateComposition');
        return this;
    }
    /**
     * Modifies the mutable properties associated with a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     */
    updateModel() {
        this.add('deepcomposer:UpdateModel');
        return this;
    }
    /**
     * Adds a resource of type model to the statement
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-train-custom-model.html
     *
     * @param modelId - Identifier for the modelId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onModel(modelId, account, region, partition) {
        var arn = 'arn:${Partition}:deepcomposer:${Region}:${Account}:model/${ModelId}';
        arn = arn.replace('${ModelId}', modelId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type composition to the statement
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     *
     * @param compositionId - Identifier for the compositionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onComposition(compositionId, account, region, partition) {
        var arn = 'arn:${Partition}:deepcomposer:${Region}:${Account}:composition/${CompositionId}';
        arn = arn.replace('${CompositionId}', compositionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type audio to the statement
     *
     * https://docs.aws.amazon.com/latest/devguide/get-started.htmlget-started-compose-with-trained-model.html
     *
     * @param audioId - Identifier for the audioId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAudio(audioId, account, region, partition) {
        var arn = 'arn:${Partition}:deepcomposer:${Region}:${Account}:audio/${AudioId}';
        arn = arn.replace('${AudioId}', audioId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Deepcomposer = Deepcomposer;
//# sourceMappingURL=data:application/json;base64,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