"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ds = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ds extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ds](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsdirectoryservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ds';
        this.actionList = {
            "AcceptSharedDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html",
                "description": "Accepts a directory sharing request that was sent from the directory owner account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AddIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html",
                "description": "Adds a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "AddTagsToResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html",
                "description": "Adds or overwrites one or more tags for the specified Amazon Directory Services directory.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "AuthorizeApplication": {
                "url": "",
                "description": "Authorizes an application for your AWS Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CancelSchemaExtension": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html",
                "description": "Cancels an in-progress schema extension to a Microsoft AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CheckAlias": {
                "url": "",
                "description": "Verifies that the alias is available for use.",
                "accessLevel": "Read"
            },
            "ConnectDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html",
                "description": "Creates an AD Connector to connect to an on-premises directory.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html",
                "description": "Creates an alias for a directory and assigns the alias to the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateComputer": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html",
                "description": "Creates a computer account in the specified directory, and joins the computer to the directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html",
                "description": "Creates a conditional forwarder associated with your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html",
                "description": "Creates a Simple AD directory.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateIdentityPoolDirectory": {
                "url": "",
                "description": "Creates a IdentityPool Directory in the AWS cloud.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateLogSubscription": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html",
                "description": "Creates a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateMicrosoftAD": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html",
                "description": "Creates a Microsoft AD in the AWS cloud.",
                "accessLevel": "Tagging",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html",
                "description": "Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "CreateTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html",
                "description": "Initiates the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html",
                "description": "Deletes a conditional forwarder that has been set up for your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html",
                "description": "Deletes an AWS Directory Service directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteLogSubscription": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html",
                "description": "Deletes the specified log subscription.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html",
                "description": "Deletes a directory snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeleteTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/DeleteTrust.html",
                "description": "Deletes an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeregisterCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html",
                "description": "Deletes from the system the certificate that was registered for a secured LDAP connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DeregisterEventTopic": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html",
                "description": "Removes the specified directory as a publisher to the specified SNS topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html",
                "description": "Displays information about the certificate registered for a secured LDAP connection.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeConditionalForwarders": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html",
                "description": "Obtains information about the conditional forwarders for this account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html",
                "description": "Obtains information about the directories that belong to this account.",
                "accessLevel": "List"
            },
            "DescribeDomainControllers": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html",
                "description": "Provides information about any domain controllers in your directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeEventTopics": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html",
                "description": "Obtains information about which SNS topics receive status messages from the specified directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeLDAPSSettings": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html",
                "description": "Describes the status of LDAP security for the specified directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeSharedDirectories": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html",
                "description": "Returns the shared directories in your account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DescribeSnapshots": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html",
                "description": "Obtains information about the directory snapshots that belong to this account.",
                "accessLevel": "Read"
            },
            "DescribeTrusts": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html",
                "description": "Obtains information about the trust relationships for this account.",
                "accessLevel": "Read"
            },
            "DisableLDAPS": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html",
                "description": "Deactivates LDAP secure calls for the specified directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html",
                "description": "Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "DisableSso": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html",
                "description": "Disables single-sign on for a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableLDAPS": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html",
                "description": "Activates the switch for the specific directory to always use LDAP secure calls.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html",
                "description": "Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "EnableSso": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html",
                "description": "Enables single-sign on for a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetAuthorizedApplicationDetails": {
                "url": "",
                "description": "",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "GetDirectoryLimits": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html",
                "description": "Obtains directory limit information for the current region.",
                "accessLevel": "Read"
            },
            "GetSnapshotLimits": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html",
                "description": "Obtains the manual snapshot limits for a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListAuthorizedApplications": {
                "url": "",
                "description": "Obtains the aws applications authorized for a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html",
                "description": "For the specified directory, lists all the certificates registered for a secured LDAP connection.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html",
                "description": "Lists the address blocks that you have added to a directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListLogSubscriptions": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html",
                "description": "Lists the active log subscriptions for the AWS account.",
                "accessLevel": "Read"
            },
            "ListSchemaExtensions": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html",
                "description": "Lists all schema extensions applied to a Microsoft AD Directory.",
                "accessLevel": "List",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html",
                "description": "Lists all tags on an Amazon Directory Services directory.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RegisterCertificate": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html",
                "description": "Registers a certificate for secured LDAP connection.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RegisterEventTopic": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html",
                "description": "Associates a directory with an SNS topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RejectSharedDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html",
                "description": "Rejects a directory sharing request that was sent from the directory owner account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RemoveIpRoutes": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html",
                "description": "Removes IP address blocks from a directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RemoveTagsFromResource": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html",
                "description": "Removes tags from an Amazon Directory Services directory.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "ResetUserPassword": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html",
                "description": "Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "RestoreFromSnapshot": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html",
                "description": "Restores a directory using an existing directory snapshot.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "ShareDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html",
                "description": "Shares a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "StartSchemaExtension": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html",
                "description": "Applies a schema extension to a Microsoft AD directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UnauthorizeApplication": {
                "url": "",
                "description": "Unauthorizes an application from your AWS Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UnshareDirectory": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html",
                "description": "Stops the directory sharing between the directory owner and consumer accounts.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateConditionalForwarder": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html",
                "description": "Updates a conditional forwarder that has been set up for your AWS directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateNumberOfDomainControllers": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html",
                "description": "Adds or removes domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateRadius": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html",
                "description": "Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "UpdateTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html",
                "description": "Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            },
            "VerifyTrust": {
                "url": "https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html",
                "description": "Verifies a trust relationship between your Microsoft AD in the AWS cloud and an external domain.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "directory": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "directory": {
                "name": "directory",
                "url": "",
                "arn": "arn:${Partition}:ds:${Region}:${Account}:directory/${DirectoryId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Accepts a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html
     */
    acceptSharedDirectory() {
        this.add('ds:AcceptSharedDirectory');
        return this;
    }
    /**
     * Adds a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html
     */
    addIpRoutes() {
        this.add('ds:AddIpRoutes');
        return this;
    }
    /**
     * Adds or overwrites one or more tags for the specified Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html
     */
    addTagsToResource() {
        this.add('ds:AddTagsToResource');
        return this;
    }
    /**
     * Authorizes an application for your AWS Directory.
     *
     * Access Level: Write
     */
    authorizeApplication() {
        this.add('ds:AuthorizeApplication');
        return this;
    }
    /**
     * Cancels an in-progress schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html
     */
    cancelSchemaExtension() {
        this.add('ds:CancelSchemaExtension');
        return this;
    }
    /**
     * Verifies that the alias is available for use.
     *
     * Access Level: Read
     */
    checkAlias() {
        this.add('ds:CheckAlias');
        return this;
    }
    /**
     * Creates an AD Connector to connect to an on-premises directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html
     */
    connectDirectory() {
        this.add('ds:ConnectDirectory');
        return this;
    }
    /**
     * Creates an alias for a directory and assigns the alias to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html
     */
    createAlias() {
        this.add('ds:CreateAlias');
        return this;
    }
    /**
     * Creates a computer account in the specified directory, and joins the computer to the directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html
     */
    createComputer() {
        this.add('ds:CreateComputer');
        return this;
    }
    /**
     * Creates a conditional forwarder associated with your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html
     */
    createConditionalForwarder() {
        this.add('ds:CreateConditionalForwarder');
        return this;
    }
    /**
     * Creates a Simple AD directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html
     */
    createDirectory() {
        this.add('ds:CreateDirectory');
        return this;
    }
    /**
     * Creates a IdentityPool Directory in the AWS cloud.
     *
     * Access Level: Tagging
     */
    createIdentityPoolDirectory() {
        this.add('ds:CreateIdentityPoolDirectory');
        return this;
    }
    /**
     * Creates a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html
     */
    createLogSubscription() {
        this.add('ds:CreateLogSubscription');
        return this;
    }
    /**
     * Creates a Microsoft AD in the AWS cloud.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html
     */
    createMicrosoftAD() {
        this.add('ds:CreateMicrosoftAD');
        return this;
    }
    /**
     * Creates a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html
     */
    createSnapshot() {
        this.add('ds:CreateSnapshot');
        return this;
    }
    /**
     * Initiates the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html
     */
    createTrust() {
        this.add('ds:CreateTrust');
        return this;
    }
    /**
     * Deletes a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html
     */
    deleteConditionalForwarder() {
        this.add('ds:DeleteConditionalForwarder');
        return this;
    }
    /**
     * Deletes an AWS Directory Service directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html
     */
    deleteDirectory() {
        this.add('ds:DeleteDirectory');
        return this;
    }
    /**
     * Deletes the specified log subscription.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html
     */
    deleteLogSubscription() {
        this.add('ds:DeleteLogSubscription');
        return this;
    }
    /**
     * Deletes a directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html
     */
    deleteSnapshot() {
        this.add('ds:DeleteSnapshot');
        return this;
    }
    /**
     * Deletes an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/DeleteTrust.html
     */
    deleteTrust() {
        this.add('ds:DeleteTrust');
        return this;
    }
    /**
     * Deletes from the system the certificate that was registered for a secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html
     */
    deregisterCertificate() {
        this.add('ds:DeregisterCertificate');
        return this;
    }
    /**
     * Removes the specified directory as a publisher to the specified SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html
     */
    deregisterEventTopic() {
        this.add('ds:DeregisterEventTopic');
        return this;
    }
    /**
     * Displays information about the certificate registered for a secured LDAP connection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html
     */
    describeCertificate() {
        this.add('ds:DescribeCertificate');
        return this;
    }
    /**
     * Obtains information about the conditional forwarders for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html
     */
    describeConditionalForwarders() {
        this.add('ds:DescribeConditionalForwarders');
        return this;
    }
    /**
     * Obtains information about the directories that belong to this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html
     */
    describeDirectories() {
        this.add('ds:DescribeDirectories');
        return this;
    }
    /**
     * Provides information about any domain controllers in your directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html
     */
    describeDomainControllers() {
        this.add('ds:DescribeDomainControllers');
        return this;
    }
    /**
     * Obtains information about which SNS topics receive status messages from the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html
     */
    describeEventTopics() {
        this.add('ds:DescribeEventTopics');
        return this;
    }
    /**
     * Describes the status of LDAP security for the specified directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html
     */
    describeLDAPSSettings() {
        this.add('ds:DescribeLDAPSSettings');
        return this;
    }
    /**
     * Returns the shared directories in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html
     */
    describeSharedDirectories() {
        this.add('ds:DescribeSharedDirectories');
        return this;
    }
    /**
     * Obtains information about the directory snapshots that belong to this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html
     */
    describeSnapshots() {
        this.add('ds:DescribeSnapshots');
        return this;
    }
    /**
     * Obtains information about the trust relationships for this account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html
     */
    describeTrusts() {
        this.add('ds:DescribeTrusts');
        return this;
    }
    /**
     * Deactivates LDAP secure calls for the specified directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html
     */
    disableLDAPS() {
        this.add('ds:DisableLDAPS');
        return this;
    }
    /**
     * Disables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html
     */
    disableRadius() {
        this.add('ds:DisableRadius');
        return this;
    }
    /**
     * Disables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html
     */
    disableSso() {
        this.add('ds:DisableSso');
        return this;
    }
    /**
     * Activates the switch for the specific directory to always use LDAP secure calls.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html
     */
    enableLDAPS() {
        this.add('ds:EnableLDAPS');
        return this;
    }
    /**
     * Enables multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html
     */
    enableRadius() {
        this.add('ds:EnableRadius');
        return this;
    }
    /**
     * Enables single-sign on for a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html
     */
    enableSso() {
        this.add('ds:EnableSso');
        return this;
    }
    /**
     *
     *
     * Access Level: Read
     */
    getAuthorizedApplicationDetails() {
        this.add('ds:GetAuthorizedApplicationDetails');
        return this;
    }
    /**
     * Obtains directory limit information for the current region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html
     */
    getDirectoryLimits() {
        this.add('ds:GetDirectoryLimits');
        return this;
    }
    /**
     * Obtains the manual snapshot limits for a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html
     */
    getSnapshotLimits() {
        this.add('ds:GetSnapshotLimits');
        return this;
    }
    /**
     * Obtains the aws applications authorized for a directory.
     *
     * Access Level: Read
     */
    listAuthorizedApplications() {
        this.add('ds:ListAuthorizedApplications');
        return this;
    }
    /**
     * For the specified directory, lists all the certificates registered for a secured LDAP connection.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html
     */
    listCertificates() {
        this.add('ds:ListCertificates');
        return this;
    }
    /**
     * Lists the address blocks that you have added to a directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html
     */
    listIpRoutes() {
        this.add('ds:ListIpRoutes');
        return this;
    }
    /**
     * Lists the active log subscriptions for the AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html
     */
    listLogSubscriptions() {
        this.add('ds:ListLogSubscriptions');
        return this;
    }
    /**
     * Lists all schema extensions applied to a Microsoft AD Directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html
     */
    listSchemaExtensions() {
        this.add('ds:ListSchemaExtensions');
        return this;
    }
    /**
     * Lists all tags on an Amazon Directory Services directory.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('ds:ListTagsForResource');
        return this;
    }
    /**
     * Registers a certificate for secured LDAP connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html
     */
    registerCertificate() {
        this.add('ds:RegisterCertificate');
        return this;
    }
    /**
     * Associates a directory with an SNS topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html
     */
    registerEventTopic() {
        this.add('ds:RegisterEventTopic');
        return this;
    }
    /**
     * Rejects a directory sharing request that was sent from the directory owner account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html
     */
    rejectSharedDirectory() {
        this.add('ds:RejectSharedDirectory');
        return this;
    }
    /**
     * Removes IP address blocks from a directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html
     */
    removeIpRoutes() {
        this.add('ds:RemoveIpRoutes');
        return this;
    }
    /**
     * Removes tags from an Amazon Directory Services directory.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html
     */
    removeTagsFromResource() {
        this.add('ds:RemoveTagsFromResource');
        return this;
    }
    /**
     * Resets the password for any user in your AWS Managed Microsoft AD or Simple AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html
     */
    resetUserPassword() {
        this.add('ds:ResetUserPassword');
        return this;
    }
    /**
     * Restores a directory using an existing directory snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html
     */
    restoreFromSnapshot() {
        this.add('ds:RestoreFromSnapshot');
        return this;
    }
    /**
     * Shares a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html
     */
    shareDirectory() {
        this.add('ds:ShareDirectory');
        return this;
    }
    /**
     * Applies a schema extension to a Microsoft AD directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html
     */
    startSchemaExtension() {
        this.add('ds:StartSchemaExtension');
        return this;
    }
    /**
     * Unauthorizes an application from your AWS Directory.
     *
     * Access Level: Write
     */
    unauthorizeApplication() {
        this.add('ds:UnauthorizeApplication');
        return this;
    }
    /**
     * Stops the directory sharing between the directory owner and consumer accounts.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html
     */
    unshareDirectory() {
        this.add('ds:UnshareDirectory');
        return this;
    }
    /**
     * Updates a conditional forwarder that has been set up for your AWS directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html
     */
    updateConditionalForwarder() {
        this.add('ds:UpdateConditionalForwarder');
        return this;
    }
    /**
     * Adds or removes domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html
     */
    updateNumberOfDomainControllers() {
        this.add('ds:UpdateNumberOfDomainControllers');
        return this;
    }
    /**
     * Updates the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html
     */
    updateRadius() {
        this.add('ds:UpdateRadius');
        return this;
    }
    /**
     * Updates the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html
     */
    updateTrust() {
        this.add('ds:UpdateTrust');
        return this;
    }
    /**
     * Verifies a trust relationship between your Microsoft AD in the AWS cloud and an external domain.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html
     */
    verifyTrust() {
        this.add('ds:VerifyTrust');
        return this;
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDirectory(directoryId, account, region, partition) {
        var arn = 'arn:${Partition}:ds:${Region}:${Account}:directory/${DirectoryId}';
        arn = arn.replace('${DirectoryId}', directoryId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Ds = Ds;
//# sourceMappingURL=data:application/json;base64,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