"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediastore = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mediastore](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediastore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediastore extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mediastore](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awselementalmediastore.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mediastore';
        this.actionList = {
            "CreateContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html",
                "description": "Grants permission to create containers.",
                "accessLevel": "Write"
            },
            "DeleteContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html",
                "description": "Grants permission to delete any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html",
                "description": "Grants permission to delete the access policy of any container in the current account.",
                "accessLevel": "Permissions management"
            },
            "DeleteCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html",
                "description": "Grants permission to delete the CORS policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html",
                "description": "Grants permission to delete the lifecycle policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html",
                "description": "Grants permission to delete the metric policy from any container in the current account.",
                "accessLevel": "Write"
            },
            "DeleteObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html",
                "description": "Grants permission to delete objects.",
                "accessLevel": "Write"
            },
            "DescribeContainer": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html",
                "description": "Grants permission to retrieve details on any container in the current account.",
                "accessLevel": "List"
            },
            "DescribeObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html",
                "description": "Grants permission to retrieve object metadata.",
                "accessLevel": "List"
            },
            "GetContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html",
                "description": "Grants permission to retrieve the access policy of any container in the current account.",
                "accessLevel": "Read"
            },
            "GetCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html",
                "description": "Grants permission to retrieve the CORS policy of any container in the current account.",
                "accessLevel": "Read"
            },
            "GetLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html",
                "description": "Grants permission to retrieve the lifecycle policy that is assigned to any container in the current account.",
                "accessLevel": "Read"
            },
            "GetMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html",
                "description": "Grants permission to retrieve the metric policy that is assigned to any container in the current account.",
                "accessLevel": "Read"
            },
            "GetObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html",
                "description": "Grants permission to retrieve objects.",
                "accessLevel": "Read"
            },
            "ListContainers": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html",
                "description": "Grants permission to retrieve a list of containers in the current account.",
                "accessLevel": "List"
            },
            "ListItems": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html",
                "description": "Grants permission to retrieve a list of objects and folders in the current account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html",
                "description": "Grants permission to list tags on any container in the current account.",
                "accessLevel": "Read"
            },
            "PutContainerPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html",
                "description": "Grants permission to create or replace the access policy of any container in the current account.",
                "accessLevel": "Permissions management"
            },
            "PutCorsPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html",
                "description": "Grants permission to add or modify the CORS policy of any container in the current account.",
                "accessLevel": "Write"
            },
            "PutLifecyclePolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html",
                "description": "Grants permission to add or modify the lifecycle policy that is assigned to any container in the current account.",
                "accessLevel": "Write"
            },
            "PutMetricPolicy": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html",
                "description": "Grants permission to add or modify the metric policy that is assigned to any container in the current account.",
                "accessLevel": "Write"
            },
            "PutObject": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html",
                "description": "Grants permission to upload objects.",
                "accessLevel": "Write"
            },
            "StartAccessLogging": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html",
                "description": "Grants permission to enable access logging on any container in the current account.",
                "accessLevel": "Write"
            },
            "StopAccessLogging": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html",
                "description": "Grants permission to disable access logging on any container in the current account.",
                "accessLevel": "Write"
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html",
                "description": "Grants permission to add tags to any container in the current account.",
                "accessLevel": "Tagging"
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html",
                "description": "Grants permission to remove tags from any container in the current account.",
                "accessLevel": "Tagging"
            }
        };
        this.resourceTypes = {
            "container": {
                "name": "container",
                "url": "https://docs.aws.amazon.com/mediastore/latest/ug/containers.html",
                "arn": "arn:${Partition}:mediastore:${Region}:${Account}:container/${ContainerName}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Grants permission to create containers.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html
     */
    createContainer() {
        this.add('mediastore:CreateContainer');
        return this;
    }
    /**
     * Grants permission to delete any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html
     */
    deleteContainer() {
        this.add('mediastore:DeleteContainer');
        return this;
    }
    /**
     * Grants permission to delete the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html
     */
    deleteContainerPolicy() {
        this.add('mediastore:DeleteContainerPolicy');
        return this;
    }
    /**
     * Grants permission to delete the CORS policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html
     */
    deleteCorsPolicy() {
        this.add('mediastore:DeleteCorsPolicy');
        return this;
    }
    /**
     * Grants permission to delete the lifecycle policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html
     */
    deleteLifecyclePolicy() {
        this.add('mediastore:DeleteLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to delete the metric policy from any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html
     */
    deleteMetricPolicy() {
        this.add('mediastore:DeleteMetricPolicy');
        return this;
    }
    /**
     * Grants permission to delete objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html
     */
    deleteObject() {
        this.add('mediastore:DeleteObject');
        return this;
    }
    /**
     * Grants permission to retrieve details on any container in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html
     */
    describeContainer() {
        this.add('mediastore:DescribeContainer');
        return this;
    }
    /**
     * Grants permission to retrieve object metadata.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html
     */
    describeObject() {
        this.add('mediastore:DescribeObject');
        return this;
    }
    /**
     * Grants permission to retrieve the access policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html
     */
    getContainerPolicy() {
        this.add('mediastore:GetContainerPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the CORS policy of any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html
     */
    getCorsPolicy() {
        this.add('mediastore:GetCorsPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html
     */
    getLifecyclePolicy() {
        this.add('mediastore:GetLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html
     */
    getMetricPolicy() {
        this.add('mediastore:GetMetricPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve objects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html
     */
    getObject() {
        this.add('mediastore:GetObject');
        return this;
    }
    /**
     * Grants permission to retrieve a list of containers in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html
     */
    listContainers() {
        this.add('mediastore:ListContainers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of objects and folders in the current account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html
     */
    listItems() {
        this.add('mediastore:ListItems');
        return this;
    }
    /**
     * Grants permission to list tags on any container in the current account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('mediastore:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to create or replace the access policy of any container in the current account.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html
     */
    putContainerPolicy() {
        this.add('mediastore:PutContainerPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the CORS policy of any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html
     */
    putCorsPolicy() {
        this.add('mediastore:PutCorsPolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the lifecycle policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html
     */
    putLifecyclePolicy() {
        this.add('mediastore:PutLifecyclePolicy');
        return this;
    }
    /**
     * Grants permission to add or modify the metric policy that is assigned to any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html
     */
    putMetricPolicy() {
        this.add('mediastore:PutMetricPolicy');
        return this;
    }
    /**
     * Grants permission to upload objects.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html
     */
    putObject() {
        this.add('mediastore:PutObject');
        return this;
    }
    /**
     * Grants permission to enable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html
     */
    startAccessLogging() {
        this.add('mediastore:StartAccessLogging');
        return this;
    }
    /**
     * Grants permission to disable access logging on any container in the current account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html
     */
    stopAccessLogging() {
        this.add('mediastore:StopAccessLogging');
        return this;
    }
    /**
     * Grants permission to add tags to any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html
     */
    tagResource() {
        this.add('mediastore:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from any container in the current account.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html
     */
    untagResource() {
        this.add('mediastore:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type container to the statement
     *
     * https://docs.aws.amazon.com/mediastore/latest/ug/containers.html
     *
     * @param containerName - Identifier for the containerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onContainer(containerName, account, region, partition) {
        var arn = 'arn:${Partition}:mediastore:${Region}:${Account}:container/${ContainerName}';
        arn = arn.replace('${ContainerName}', containerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mediastore = Mediastore;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzZWxlbWVudGFsbWVkaWFzdG9yZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImF3c2VsZW1lbnRhbG1lZGlhc3RvcmUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsc0NBQW9FO0FBRXBFOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSx3QkFBZTtJQStJN0M7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFwSk4sa0JBQWEsR0FBRyxZQUFZLENBQUM7UUFDMUIsZUFBVSxHQUFZO1lBQzlCLGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUseUNBQXlDO2dCQUN4RCxhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELGlCQUFpQixFQUFFO2dCQUNqQixLQUFLLEVBQUUscUZBQXFGO2dCQUM1RixhQUFhLEVBQUUsbUVBQW1FO2dCQUNsRixhQUFhLEVBQUUsT0FBTzthQUN2QjtZQUNELHVCQUF1QixFQUFFO2dCQUN2QixLQUFLLEVBQUUsMkZBQTJGO2dCQUNsRyxhQUFhLEVBQUUsd0ZBQXdGO2dCQUN2RyxhQUFhLEVBQUUsd0JBQXdCO2FBQ3hDO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCLEtBQUssRUFBRSxzRkFBc0Y7Z0JBQzdGLGFBQWEsRUFBRSx3RkFBd0Y7Z0JBQ3ZHLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSw2RkFBNkY7Z0JBQzVHLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLEtBQUssRUFBRSwyRkFBMkY7Z0JBQ2xHLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QsbUJBQW1CLEVBQUU7Z0JBQ25CLEtBQUssRUFBRSx1RkFBdUY7Z0JBQzlGLGFBQWEsRUFBRSxnRkFBZ0Y7Z0JBQy9GLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSw2RkFBNkY7Z0JBQ3BHLGFBQWEsRUFBRSxnREFBZ0Q7Z0JBQy9ELGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSwwRkFBMEY7Z0JBQ3pHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSx3RkFBd0Y7Z0JBQ3ZHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSw4R0FBOEc7Z0JBQzdILGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSwyR0FBMkc7Z0JBQzFILGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSx3Q0FBd0M7Z0JBQ3ZELGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSw0RUFBNEU7Z0JBQzNGLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSxxRkFBcUY7Z0JBQ3BHLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSx5RkFBeUY7Z0JBQ2hHLGFBQWEsRUFBRSx5RUFBeUU7Z0JBQ3hGLGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1lBQ0Qsb0JBQW9CLEVBQUU7Z0JBQ3BCLEtBQUssRUFBRSx3RkFBd0Y7Z0JBQy9GLGFBQWEsRUFBRSxtR0FBbUc7Z0JBQ2xILGFBQWEsRUFBRSx3QkFBd0I7YUFDeEM7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLDZGQUE2RjtnQkFDNUcsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLG1IQUFtSDtnQkFDbEksYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxpQkFBaUIsRUFBRTtnQkFDakIsS0FBSyxFQUFFLHFGQUFxRjtnQkFDNUYsYUFBYSxFQUFFLGdIQUFnSDtnQkFDL0gsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLHNDQUFzQztnQkFDckQsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxvQkFBb0IsRUFBRTtnQkFDcEIsS0FBSyxFQUFFLHdGQUF3RjtnQkFDL0YsYUFBYSxFQUFFLHFGQUFxRjtnQkFDcEcsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxtQkFBbUIsRUFBRTtnQkFDbkIsS0FBSyxFQUFFLHVGQUF1RjtnQkFDOUYsYUFBYSxFQUFFLHNGQUFzRjtnQkFDckcsYUFBYSxFQUFFLE9BQU87YUFDdkI7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsS0FBSyxFQUFFLGlGQUFpRjtnQkFDeEYsYUFBYSxFQUFFLHdFQUF3RTtnQkFDdkYsYUFBYSxFQUFFLFNBQVM7YUFDekI7WUFDRCxlQUFlLEVBQUU7Z0JBQ2YsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsYUFBYSxFQUFFLDZFQUE2RTtnQkFDNUYsYUFBYSxFQUFFLFNBQVM7YUFDekI7U0FDRixDQUFDO1FBQ0ssa0JBQWEsR0FBa0I7WUFDcEMsV0FBVyxFQUFFO2dCQUNYLE1BQU0sRUFBRSxXQUFXO2dCQUNuQixLQUFLLEVBQUUsa0VBQWtFO2dCQUN6RSxLQUFLLEVBQUUsNkVBQTZFO2dCQUNwRixlQUFlLEVBQUUsRUFBRTthQUNwQjtTQUNGLENBQUM7SUFTRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxHQUFHLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxHQUFHLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxHQUFHLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFNBQVM7UUFDZCxJQUFJLENBQUMsR0FBRyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLElBQUksQ0FBQyxHQUFHLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLDRCQUE0QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksU0FBUztRQUNkLElBQUksQ0FBQyxHQUFHLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixJQUFJLENBQUMsR0FBRyxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksV0FBVyxDQUFDLGFBQXFCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0YsSUFBSSxHQUFHLEdBQUcsNkVBQTZFLENBQUM7UUFDeEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDckQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7Q0FDRjtBQWxlRCxnQ0FrZUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY3Rpb25zLCBQb2xpY3lTdGF0ZW1lbnQsIFJlc291cmNlVHlwZXMgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFttZWRpYXN0b3JlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NlbGVtZW50YWxtZWRpYXN0b3JlLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgTWVkaWFzdG9yZSBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ21lZGlhc3RvcmUnO1xuICBwcm90ZWN0ZWQgYWN0aW9uTGlzdDogQWN0aW9ucyA9IHtcbiAgICBcIkNyZWF0ZUNvbnRhaW5lclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0NyZWF0ZUNvbnRhaW5lci5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGNvbnRhaW5lcnMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZWxldGVDb250YWluZXJcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVDb250YWluZXIuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZWxldGVDb250YWluZXJQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVDb250YWluZXJQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgYWNjZXNzIHBvbGljeSBvZiBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiXG4gICAgfSxcbiAgICBcIkRlbGV0ZUNvcnNQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVDb3JzUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIENPUlMgcG9saWN5IGZyb20gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiRGVsZXRlTGlmZWN5Y2xlUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlTGlmZWN5Y2xlUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIGxpZmVjeWNsZSBwb2xpY3kgZnJvbSBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZWxldGVNZXRyaWNQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVNZXRyaWNQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgbWV0cmljIHBvbGljeSBmcm9tIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJXcml0ZVwiXG4gICAgfSxcbiAgICBcIkRlbGV0ZU9iamVjdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX29ianN0b3JlX0RlbGV0ZU9iamVjdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIG9iamVjdHMuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJEZXNjcmliZUNvbnRhaW5lclwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29udGFpbmVyLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBkZXRhaWxzIG9uIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiRGVzY3JpYmVPYmplY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9vYmpzdG9yZV9EZXNjcmliZU9iamVjdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgb2JqZWN0IG1ldGFkYXRhLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJHZXRDb250YWluZXJQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9HZXRDb250YWluZXJQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBhY2Nlc3MgcG9saWN5IG9mIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiR2V0Q29yc1BvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldENvcnNQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBDT1JTIHBvbGljeSBvZiBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldExpZmVjeWNsZVBvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldExpZmVjeWNsZVBvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGxpZmVjeWNsZSBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldE1ldHJpY1BvbGljeVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldE1ldHJpY1BvbGljeS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIG1ldHJpYyBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUmVhZFwiXG4gICAgfSxcbiAgICBcIkdldE9iamVjdFwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX29ianN0b3JlX0dldE9iamVjdC5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgb2JqZWN0cy5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiTGlzdENvbnRhaW5lcnNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9MaXN0Q29udGFpbmVycy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGNvbnRhaW5lcnMgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJMaXN0XCJcbiAgICB9LFxuICAgIFwiTGlzdEl0ZW1zXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfb2Jqc3RvcmVfTGlzdEl0ZW1zLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2Ygb2JqZWN0cyBhbmQgZm9sZGVycyBpbiB0aGUgY3VycmVudCBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH0sXG4gICAgXCJMaXN0VGFnc0ZvclJlc291cmNlXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0YWdzIG9uIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJSZWFkXCJcbiAgICB9LFxuICAgIFwiUHV0Q29udGFpbmVyUG9saWN5XCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfUHV0Q29udGFpbmVyUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgb3IgcmVwbGFjZSB0aGUgYWNjZXNzIHBvbGljeSBvZiBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiUGVybWlzc2lvbnMgbWFuYWdlbWVudFwiXG4gICAgfSxcbiAgICBcIlB1dENvcnNQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9QdXRDb3JzUG9saWN5Lmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgb3IgbW9kaWZ5IHRoZSBDT1JTIHBvbGljeSBvZiBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRMaWZlY3ljbGVQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9QdXRMaWZlY3ljbGVQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBvciBtb2RpZnkgdGhlIGxpZmVjeWNsZSBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRNZXRyaWNQb2xpY3lcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9QdXRNZXRyaWNQb2xpY3kuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBvciBtb2RpZnkgdGhlIG1ldHJpYyBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJQdXRPYmplY3RcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9vYmpzdG9yZV9QdXRPYmplY3QuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkdyYW50cyBwZXJtaXNzaW9uIHRvIHVwbG9hZCBvYmplY3RzLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiU3RhcnRBY2Nlc3NMb2dnaW5nXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfU3RhcnRBY2Nlc3NMb2dnaW5nLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgYWNjZXNzIGxvZ2dpbmcgb24gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiU3RvcEFjY2Vzc0xvZ2dpbmdcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9TdG9wQWNjZXNzTG9nZ2luZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYWJsZSBhY2Nlc3MgbG9nZ2luZyBvbiBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIlxuICAgIH0sXG4gICAgXCJUYWdSZXNvdXJjZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgdGFncyB0byBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiVGFnZ2luZ1wiXG4gICAgfSxcbiAgICBcIlVudGFnUmVzb3VyY2VcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgdGFncyBmcm9tIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cIixcbiAgICAgIFwiYWNjZXNzTGV2ZWxcIjogXCJUYWdnaW5nXCJcbiAgICB9XG4gIH07XG4gIHB1YmxpYyByZXNvdXJjZVR5cGVzOiBSZXNvdXJjZVR5cGVzID0ge1xuICAgIFwiY29udGFpbmVyXCI6IHtcbiAgICAgIFwibmFtZVwiOiBcImNvbnRhaW5lclwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvdWcvY29udGFpbmVycy5odG1sXCIsXG4gICAgICBcImFyblwiOiBcImFybjoke1BhcnRpdGlvbn06bWVkaWFzdG9yZToke1JlZ2lvbn06JHtBY2NvdW50fTpjb250YWluZXIvJHtDb250YWluZXJOYW1lfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW21lZGlhc3RvcmVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2VsZW1lbnRhbG1lZGlhc3RvcmUuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGNvbnRhaW5lcnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlQ29udGFpbmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVDb250YWluZXIoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6Q3JlYXRlQ29udGFpbmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVDb250YWluZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZUNvbnRhaW5lcigpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpEZWxldGVDb250YWluZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIGFjY2VzcyBwb2xpY3kgb2YgYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlQ29udGFpbmVyUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVDb250YWluZXJQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6RGVsZXRlQ29udGFpbmVyUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBDT1JTIHBvbGljeSBmcm9tIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVDb3JzUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVDb3JzUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYXN0b3JlOkRlbGV0ZUNvcnNQb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIGxpZmVjeWNsZSBwb2xpY3kgZnJvbSBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlTGlmZWN5Y2xlUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBkZWxldGVMaWZlY3ljbGVQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6RGVsZXRlTGlmZWN5Y2xlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBtZXRyaWMgcG9saWN5IGZyb20gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0RlbGV0ZU1ldHJpY1BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlTWV0cmljUG9saWN5KCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYXN0b3JlOkRlbGV0ZU1ldHJpY1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBvYmplY3RzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX29ianN0b3JlX0RlbGV0ZU9iamVjdC5odG1sXG4gICAqL1xuICBwdWJsaWMgZGVsZXRlT2JqZWN0KCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYXN0b3JlOkRlbGV0ZU9iamVjdCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGRldGFpbHMgb24gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVzY3JpYmVDb250YWluZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlQ29udGFpbmVyKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYXN0b3JlOkRlc2NyaWJlQ29udGFpbmVyJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgb2JqZWN0IG1ldGFkYXRhLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfb2Jqc3RvcmVfRGVzY3JpYmVPYmplY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlc2NyaWJlT2JqZWN0KCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYXN0b3JlOkRlc2NyaWJlT2JqZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGFjY2VzcyBwb2xpY3kgb2YgYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfR2V0Q29udGFpbmVyUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRDb250YWluZXJQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6R2V0Q29udGFpbmVyUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIENPUlMgcG9saWN5IG9mIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldENvcnNQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldENvcnNQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6R2V0Q29yc1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBsaWZlY3ljbGUgcG9saWN5IHRoYXQgaXMgYXNzaWduZWQgdG8gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfR2V0TGlmZWN5Y2xlUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRMaWZlY3ljbGVQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6R2V0TGlmZWN5Y2xlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIG1ldHJpYyBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9HZXRNZXRyaWNQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldE1ldHJpY1BvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpHZXRNZXRyaWNQb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBvYmplY3RzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfb2Jqc3RvcmVfR2V0T2JqZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRPYmplY3QoKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6R2V0T2JqZWN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGNvbnRhaW5lcnMgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RDb250YWluZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBsaXN0Q29udGFpbmVycygpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpMaXN0Q29udGFpbmVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBvYmplY3RzIGFuZCBmb2xkZXJzIGluIHRoZSBjdXJyZW50IGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9vYmpzdG9yZV9MaXN0SXRlbXMuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RJdGVtcygpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpMaXN0SXRlbXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3Mgb24gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgbGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIG9yIHJlcGxhY2UgdGhlIGFjY2VzcyBwb2xpY3kgb2YgYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfUHV0Q29udGFpbmVyUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBwdXRDb250YWluZXJQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6UHV0Q29udGFpbmVyUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9yIG1vZGlmeSB0aGUgQ09SUyBwb2xpY3kgb2YgYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1B1dENvcnNQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dENvcnNQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6UHV0Q29yc1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBvciBtb2RpZnkgdGhlIGxpZmVjeWNsZSBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhbnkgY29udGFpbmVyIGluIHRoZSBjdXJyZW50IGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfUHV0TGlmZWN5Y2xlUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyBwdXRMaWZlY3ljbGVQb2xpY3koKSB7XG4gICAgdGhpcy5hZGQoJ21lZGlhc3RvcmU6UHV0TGlmZWN5Y2xlUG9saWN5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIG9yIG1vZGlmeSB0aGUgbWV0cmljIHBvbGljeSB0aGF0IGlzIGFzc2lnbmVkIHRvIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9QdXRNZXRyaWNQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dE1ldHJpY1BvbGljeSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpQdXRNZXRyaWNQb2xpY3knKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGxvYWQgb2JqZWN0cy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9vYmpzdG9yZV9QdXRPYmplY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHB1dE9iamVjdCgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpQdXRPYmplY3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgYWNjZXNzIGxvZ2dpbmcgb24gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0QWNjZXNzTG9nZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgc3RhcnRBY2Nlc3NMb2dnaW5nKCkge1xuICAgIHRoaXMuYWRkKCdtZWRpYXN0b3JlOlN0YXJ0QWNjZXNzTG9nZ2luZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2FibGUgYWNjZXNzIGxvZ2dpbmcgb24gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1N0b3BBY2Nlc3NMb2dnaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyBzdG9wQWNjZXNzTG9nZ2luZygpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpTdG9wQWNjZXNzTG9nZ2luZycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCB0YWdzIHRvIGFueSBjb250YWluZXIgaW4gdGhlIGN1cnJlbnQgYWNjb3VudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0YWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0YWdzIGZyb20gYW55IGNvbnRhaW5lciBpbiB0aGUgY3VycmVudCBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfVW50YWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdW50YWdSZXNvdXJjZSgpIHtcbiAgICB0aGlzLmFkZCgnbWVkaWFzdG9yZTpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY29udGFpbmVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L3VnL2NvbnRhaW5lcnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY29udGFpbmVyTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjb250YWluZXJOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uQ29udGFpbmVyKGNvbnRhaW5lck5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06bWVkaWFzdG9yZToke1JlZ2lvbn06JHtBY2NvdW50fTpjb250YWluZXIvJHtDb250YWluZXJOYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7Q29udGFpbmVyTmFtZX0nLCBjb250YWluZXJOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19