"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fms = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [fms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsfirewallmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Fms extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [fms](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsfirewallmanager.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'fms';
        this.actionList = {
            "AssociateAdminAccount": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AssociateAdminAccount.html",
                "description": "Sets the AWS Firewall Manager administrator account and enables the service in all organization accounts",
                "accessLevel": "Write"
            },
            "DeleteNotificationChannel": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteNotificationChannel.html",
                "description": "Deletes an AWS Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic that is used to notify the FM administrator about major FM events and errors across the organization.",
                "accessLevel": "Write"
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeletePolicy.html",
                "description": "Permanently deletes an AWS Firewall Manager policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "DisassociateAdminAccount": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DisassociateAdminAccount.html",
                "description": "Disassociates the account that has been set as the AWS Firewall Manager administrator account and and disables the service in all organization accounts",
                "accessLevel": "Write"
            },
            "GetAdminAccount": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAdminAccount.html",
                "description": "Returns the AWS Organizations master account that is associated with AWS Firewall Manager as the AWS Firewall Manager administrator.",
                "accessLevel": "Read"
            },
            "GetComplianceDetail": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetComplianceDetail.html",
                "description": "Returns detailed compliance information about the specified member account. Details include resources that are in and out of compliance with the specified policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetNotificationChannel": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetNotificationChannel.html",
                "description": "Returns information about the Amazon Simple Notification Service (SNS) topic that is used to record AWS Firewall Manager SNS logs.",
                "accessLevel": "Read"
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetPolicy.html",
                "description": "Returns information about the specified AWS Firewall Manager policy.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetProtectionStatus": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtectionStatus.html",
                "description": "Returns policy-level attack summary information in the event of a potential DDoS attack.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListComplianceStatus": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListComplianceStatus.html",
                "description": "Returns an array of PolicyComplianceStatus objects in the response. Use PolicyComplianceStatus to get a summary of which member accounts are protected by the specified policy.",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListMemberAccounts": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListMemberAccounts.html",
                "description": "Returns an array of member account ids if the caller is FMS admin account.",
                "accessLevel": "List"
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListPolicies.html",
                "description": "Returns an array of PolicySummary objects in the response.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListTagsForResource.html",
                "description": "Lists the Tags for a given resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "PutNotificationChannel": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutNotificationChannel.html",
                "description": "Designates the IAM role and Amazon Simple Notification Service (SNS) topic that AWS Firewall Manager (FM) could use to notify the FM administrator about major FM events and errors across the organization.",
                "accessLevel": "Write"
            },
            "PutPolicy": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutPolicy.html",
                "description": "Creates an AWS Firewall Manager policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_TagResource.html",
                "description": "Adds a Tag to a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_UntagResource.html",
                "description": "Removes a Tag from a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            }
        };
        this.resourceTypes = {
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_Policy.html",
                "arn": "arn:${Partition}:fms:${Region}:${Account}:policy/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Sets the AWS Firewall Manager administrator account and enables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_AssociateAdminAccount.html
     */
    associateAdminAccount() {
        this.add('fms:AssociateAdminAccount');
        return this;
    }
    /**
     * Deletes an AWS Firewall Manager association with the IAM role and the Amazon Simple Notification Service (SNS) topic that is used to notify the FM administrator about major FM events and errors across the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeleteNotificationChannel.html
     */
    deleteNotificationChannel() {
        this.add('fms:DeleteNotificationChannel');
        return this;
    }
    /**
     * Permanently deletes an AWS Firewall Manager policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('fms:DeletePolicy');
        return this;
    }
    /**
     * Disassociates the account that has been set as the AWS Firewall Manager administrator account and and disables the service in all organization accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_DisassociateAdminAccount.html
     */
    disassociateAdminAccount() {
        this.add('fms:DisassociateAdminAccount');
        return this;
    }
    /**
     * Returns the AWS Organizations master account that is associated with AWS Firewall Manager as the AWS Firewall Manager administrator.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetAdminAccount.html
     */
    getAdminAccount() {
        this.add('fms:GetAdminAccount');
        return this;
    }
    /**
     * Returns detailed compliance information about the specified member account. Details include resources that are in and out of compliance with the specified policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetComplianceDetail.html
     */
    getComplianceDetail() {
        this.add('fms:GetComplianceDetail');
        return this;
    }
    /**
     * Returns information about the Amazon Simple Notification Service (SNS) topic that is used to record AWS Firewall Manager SNS logs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetNotificationChannel.html
     */
    getNotificationChannel() {
        this.add('fms:GetNotificationChannel');
        return this;
    }
    /**
     * Returns information about the specified AWS Firewall Manager policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetPolicy.html
     */
    getPolicy() {
        this.add('fms:GetPolicy');
        return this;
    }
    /**
     * Returns policy-level attack summary information in the event of a potential DDoS attack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_GetProtectionStatus.html
     */
    getProtectionStatus() {
        this.add('fms:GetProtectionStatus');
        return this;
    }
    /**
     * Returns an array of PolicyComplianceStatus objects in the response. Use PolicyComplianceStatus to get a summary of which member accounts are protected by the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListComplianceStatus.html
     */
    listComplianceStatus() {
        this.add('fms:ListComplianceStatus');
        return this;
    }
    /**
     * Returns an array of member account ids if the caller is FMS admin account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListMemberAccounts.html
     */
    listMemberAccounts() {
        this.add('fms:ListMemberAccounts');
        return this;
    }
    /**
     * Returns an array of PolicySummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListPolicies.html
     */
    listPolicies() {
        this.add('fms:ListPolicies');
        return this;
    }
    /**
     * Lists the Tags for a given resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('fms:ListTagsForResource');
        return this;
    }
    /**
     * Designates the IAM role and Amazon Simple Notification Service (SNS) topic that AWS Firewall Manager (FM) could use to notify the FM administrator about major FM events and errors across the organization.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutNotificationChannel.html
     */
    putNotificationChannel() {
        this.add('fms:PutNotificationChannel');
        return this;
    }
    /**
     * Creates an AWS Firewall Manager policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_PutPolicy.html
     */
    putPolicy() {
        this.add('fms:PutPolicy');
        return this;
    }
    /**
     * Adds a Tag to a given resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('fms:TagResource');
        return this;
    }
    /**
     * Removes a Tag from a given resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('fms:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/fms/2018-01-01/APIReference/API_Policy.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onPolicy(id, account, region, partition) {
        var arn = 'arn:${Partition}:fms:${Region}:${Account}:policy/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Fms = Fms;
//# sourceMappingURL=data:application/json;base64,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