"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Health = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [health](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Health extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [health](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awshealthapisandnotifications.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'health';
        this.actionList = {
            "DescribeAffectedAccountsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html",
                "description": "Gets a list of accounts that have been affected by the specified events in organization.",
                "accessLevel": "Read"
            },
            "DescribeAffectedEntities": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html",
                "description": "Gets a list of entities that have been affected by the specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeAffectedEntitiesForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html",
                "description": "Gets a list of entities that have been affected by the specified events and accounts in organization.",
                "accessLevel": "Read"
            },
            "DescribeEntityAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html",
                "description": "Returns the number of entities that are affected by each of the specified events.",
                "accessLevel": "Read"
            },
            "DescribeEventAggregates": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html",
                "description": "Returns the number of events of each event type (issue, scheduled change, and account notification).",
                "accessLevel": "Read"
            },
            "DescribeEventDetails": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html",
                "description": "Returns detailed information about one or more specified events.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "event": {
                        "required": true
                    }
                },
                "conditions": [
                    "health:eventTypeCode",
                    "health:service"
                ]
            },
            "DescribeEventDetailsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html",
                "description": "Returns detailed information about one or more specified events for provided accounts in organization.",
                "accessLevel": "Read"
            },
            "DescribeEventTypes": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html",
                "description": "Returns the event types that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEvents": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html",
                "description": "Returns information about events that meet the specified filter criteria.",
                "accessLevel": "Read"
            },
            "DescribeEventsForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html",
                "description": "Returns information about events that meet the specified filter criteria in organization.",
                "accessLevel": "Read"
            },
            "DescribeHealthServiceStatusForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html",
                "description": "Returns the status of enabling or disabling the Organizational View feature",
                "accessLevel": "Permissions management"
            },
            "DisableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html",
                "description": "Disables the Organizational View feature.",
                "accessLevel": "Permissions management"
            },
            "EnableHealthServiceAccessForOrganization": {
                "url": "https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html",
                "description": "Enables the Organizational View feature.",
                "accessLevel": "Permissions management"
            }
        };
        this.resourceTypes = {
            "event": {
                "name": "event",
                "url": "https://docs.aws.amazon.com/health/latest/ug/supported-operations.html",
                "arn": "arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Gets a list of accounts that have been affected by the specified events in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedAccountsForOrganization.html
     */
    describeAffectedAccountsForOrganization() {
        this.add('health:DescribeAffectedAccountsForOrganization');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntities.html
     */
    describeAffectedEntities() {
        this.add('health:DescribeAffectedEntities');
        return this;
    }
    /**
     * Gets a list of entities that have been affected by the specified events and accounts in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeAffectedEntitiesForOrganization.html
     */
    describeAffectedEntitiesForOrganization() {
        this.add('health:DescribeAffectedEntitiesForOrganization');
        return this;
    }
    /**
     * Returns the number of entities that are affected by each of the specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEntityAggregates.html
     */
    describeEntityAggregates() {
        this.add('health:DescribeEntityAggregates');
        return this;
    }
    /**
     * Returns the number of events of each event type (issue, scheduled change, and account notification).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventAggregates.html
     */
    describeEventAggregates() {
        this.add('health:DescribeEventAggregates');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetails.html
     */
    describeEventDetails() {
        this.add('health:DescribeEventDetails');
        return this;
    }
    /**
     * Returns detailed information about one or more specified events for provided accounts in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventDetailsForOrganization.html
     */
    describeEventDetailsForOrganization() {
        this.add('health:DescribeEventDetailsForOrganization');
        return this;
    }
    /**
     * Returns the event types that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventTypes.html
     */
    describeEventTypes() {
        this.add('health:DescribeEventTypes');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEvents.html
     */
    describeEvents() {
        this.add('health:DescribeEvents');
        return this;
    }
    /**
     * Returns information about events that meet the specified filter criteria in organization.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeEventsForOrganization.html
     */
    describeEventsForOrganization() {
        this.add('health:DescribeEventsForOrganization');
        return this;
    }
    /**
     * Returns the status of enabling or disabling the Organizational View feature
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DescribeHealthServiceStatusForOrganization.html
     */
    describeHealthServiceStatusForOrganization() {
        this.add('health:DescribeHealthServiceStatusForOrganization');
        return this;
    }
    /**
     * Disables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_DisableHealthServiceAccessForOrganization.html
     */
    disableHealthServiceAccessForOrganization() {
        this.add('health:DisableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Enables the Organizational View feature.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/health/latest/APIReference/API_EnableHealthServiceAccessForOrganization.html
     */
    enableHealthServiceAccessForOrganization() {
        this.add('health:EnableHealthServiceAccessForOrganization');
        return this;
    }
    /**
     * Adds a resource of type event to the statement
     *
     * https://docs.aws.amazon.com/health/latest/ug/supported-operations.html
     *
     * @param service - Identifier for the service.
     * @param eventTypeCode - Identifier for the eventTypeCode.
     * @param eventTypePlusId - Identifier for the eventTypePlusId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEvent(service, eventTypeCode, eventTypePlusId, account, region, partition) {
        var arn = 'arn:${Partition}:health:${Region}:${Account}:event/${Service}/${EventTypeCode}/${EventTypePlusId}';
        arn = arn.replace('${Service}', service);
        arn = arn.replace('${EventTypeCode}', eventTypeCode);
        arn = arn.replace('${EventTypePlusId}', eventTypePlusId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The type of event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEventTypeCode(value, operator) {
        return this.if(`health:eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * The service of the event.
     *
     * https://docs.aws.amazon.com/health/latest/ug/controlling-access.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`health:service`, value, operator || 'StringLike');
    }
}
exports.Health = Health;
//# sourceMappingURL=data:application/json;base64,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