"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iot = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [iot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iot extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [iot](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsiot.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'iot';
        this.actionList = {
            "AcceptCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AcceptCertificateTransfer.html",
                "description": "Accepts a pending certificate transfer.",
                "accessLevel": "Write"
            },
            "AddThingToBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToBillingGroup.html",
                "description": "Adds a thing to the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "AddThingToThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToThingGroup.html",
                "description": "Adds a thing to the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "AssociateTargetsWithJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AssociateTargetsWithJob.html",
                "description": "Associates a group with a continuous job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "AttachPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPolicy.html",
                "description": "Attaches a policy to the specified target.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "AttachPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPrincipalPolicy.html",
                "description": "Attaches the specified policy to the specified principal (certificate or other credential).",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "AttachSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachSecurityProfile.html",
                "description": "Associates a Device Defender security profile with a thing group or with this account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "AttachThingPrincipal": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_AttachThingPrincipal.html",
                "description": "Attaches the specified principal to the specified thing.",
                "accessLevel": "Write"
            },
            "CancelAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditMitigationActionsTask.html",
                "description": "Cancels a mitigation action task that is in progress.",
                "accessLevel": "Write"
            },
            "CancelAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditTask.html",
                "description": "Cancels an audit that is in progress. The audit can be either scheduled or on-demand.",
                "accessLevel": "Write"
            },
            "CancelCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelCertificateTransfer.html",
                "description": "Cancels a pending transfer for the specified certificate.",
                "accessLevel": "Write"
            },
            "CancelJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJob.html",
                "description": "Cancels a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "CancelJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJobExecution.html",
                "description": "Cancels a job execution on a particular device.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "ClearDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ClearDefaultAuthorizer.html",
                "description": "Clears the default authorizer.",
                "accessLevel": "Write"
            },
            "CloseTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CloseTunnel.html",
                "description": "Closes a tunnel.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "tunnel": {
                        "required": true
                    }
                },
                "conditions": [
                    "iot:Delete"
                ]
            },
            "Connect": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Connect as the specified client",
                "accessLevel": "Write",
                "resourceTypes": {
                    "client": {
                        "required": true
                    }
                }
            },
            "CreateAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateAuthorizer.html",
                "description": "Creates an authorizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "CreateBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateBillingGroup.html",
                "description": "Creates a billing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateCertificateFromCsr": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html",
                "description": "Creates an X.509 certificate using the specified certificate signing request.",
                "accessLevel": "Write"
            },
            "CreateDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDimension.html",
                "description": "Defines a dimension that can be used to to limit the scope of a metric used in a security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDynamicThingGroup.html",
                "description": "Creates a Dynamic Thing Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateJob.html",
                "description": "Creates a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateKeysAndCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html",
                "description": "Creates a 2048 bit RSA key pair and issues an X.509 certificate using the issued public key.",
                "accessLevel": "Write"
            },
            "CreateMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateMitigationAction.html",
                "description": "Defines an action that can be applied to audit findings by using StartAuditMitigationActionsTask.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateOTAUpdate.html",
                "description": "Creates an OTA update job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreatePolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicy.html",
                "description": "Creates an AWS IoT policy.",
                "accessLevel": "Write"
            },
            "CreatePolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicyVersion.html",
                "description": "Creates a new version of the specified AWS IoT policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningClaim": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningClaim.html",
                "description": "Creates a provisioning claim.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplate.html",
                "description": "Creates a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplateVersion.html",
                "description": "Creates a new version of a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "CreateRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateRoleAlias.html",
                "description": "Creates a role alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "CreateScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateScheduledAudit.html",
                "description": "Creates a scheduled audit that is run at a specified time interval.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateSecurityProfile.html",
                "description": "Creates a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateStream.html",
                "description": "Creates a new AWS IoT stream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThing.html",
                "description": "Creates a thing in the thing registry.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "billinggroup": {
                        "required": false
                    }
                }
            },
            "CreateThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingGroup.html",
                "description": "Creates a thing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingType.html",
                "description": "Creates a new thing type.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_CreateTopicRule.html",
                "description": "Creates a rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "DeleteAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAccountAuditConfiguration.html",
                "description": "Deletes the audit configuration associated with the account.",
                "accessLevel": "Write"
            },
            "DeleteAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAuthorizer.html",
                "description": "Deletes the specified authorizer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "DeleteBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteBillingGroup.html",
                "description": "Deletes the specified billing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCACertificate.html",
                "description": "Deletes a registered CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "DeleteCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCertificate.html",
                "description": "Deletes the specified certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "DeleteDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDimension.html",
                "description": "Removes the specified dimension from your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "DeleteDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDynamicThingGroup.html",
                "description": "Deletes the specified Dynamic Thing Group",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJob.html",
                "description": "Deletes a job and its related job executions.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "DeleteJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJobExecution.html",
                "description": "Deletes a job execution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteMitigationAction.html",
                "description": "Deletes a defined mitigation action from your AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "DeleteOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteOTAUpdate.html",
                "description": "Deletes an OTA update job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                }
            },
            "DeletePolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicy.html",
                "description": "Deletes the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeletePolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicyVersion.html",
                "description": "Deletes the specified version of the specified policy.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "DeleteProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplate.html",
                "description": "Deletes a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DeleteProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplateVersion.html",
                "description": "Deletes a fleet provisioning template version.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DeleteRegistrationCode": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRegistrationCode.html",
                "description": "Deletes a CA certificate registration code.",
                "accessLevel": "Write"
            },
            "DeleteRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRoleAlias.html",
                "description": "Deletes the specified role alias.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "DeleteScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteScheduledAudit.html",
                "description": "Deletes a scheduled audit.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "DeleteSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteSecurityProfile.html",
                "description": "Deletes a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "DeleteStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteStream.html",
                "description": "Deletes a specified stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DeleteThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html",
                "description": "Deletes the specified thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingGroup.html",
                "description": "Deletes the specified thing group.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "DeleteThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Deletes the specified thing shadow.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DeleteThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingType.html",
                "description": "Deletes the specified thing type.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DeleteTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteTopicRule.html",
                "description": "Deletes the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DeleteV2LoggingLevel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteV2LoggingLevel.html",
                "description": "Deletes the specified v2 logging level.",
                "accessLevel": "Write"
            },
            "DeprecateThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DeprecateThingType.html",
                "description": "Deprecates the specified thing type.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DescribeAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAccountAuditConfiguration.html",
                "description": "Gets information about audit configurations for the account.",
                "accessLevel": "Read"
            },
            "DescribeAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditMitigationActionsTask.html",
                "description": "Gets information about an audit mitigation task that is used to apply mitigation actions to a set of audit findings.",
                "accessLevel": "Read"
            },
            "DescribeAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditTask.html",
                "description": "Gets information about a Device Defender audit.",
                "accessLevel": "Read"
            },
            "DescribeAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuthorizer.html",
                "description": "Describes an authorizer.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "DescribeBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeBillingGroup.html",
                "description": "Gets information about the specified billing group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "DescribeCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCACertificate.html",
                "description": "Describes a registered CA certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "DescribeCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCertificate.html",
                "description": "Gets information about the specified certificate.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "DescribeDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDefaultAuthorizer.html",
                "description": "Describes the default authorizer.",
                "accessLevel": "Read"
            },
            "DescribeDimension": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDimension.html",
                "description": "Provides details about a dimension that is defined in your AWS account.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "DescribeEndpoint": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEndpoint.html",
                "description": "Returns a unique endpoint specific to the AWS account making the call.",
                "accessLevel": "Read"
            },
            "DescribeEventConfigurations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEventConfigurations.html",
                "description": "Returns account event configurations.",
                "accessLevel": "Read"
            },
            "DescribeIndex": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeIndex.html",
                "description": "Gets information about the specified index.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "DescribeJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJob.html",
                "description": "Describes a job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "DescribeJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJobExecution.html",
                "description": "Describes a job execution.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "DescribeMitigationAction": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeMitigationAction.html",
                "description": "Gets information about a mitigation action.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "DescribeProvisioningTemplate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplate.html",
                "description": "Returns information about a fleet provisioning template.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DescribeProvisioningTemplateVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplateVersion.html",
                "description": "Returns information about a fleet provisioning template version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "DescribeRoleAlias": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeRoleAlias.html",
                "description": "Describes a role alias.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "DescribeScheduledAudit": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeScheduledAudit.html",
                "description": "Gets information about a scheduled audit.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "DescribeSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeSecurityProfile.html",
                "description": "Gets information about a Device Defender security profile.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    }
                }
            },
            "DescribeStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeStream.html",
                "description": "Gets information about the specified stream.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "DescribeThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThing.html",
                "description": "Gets information about the specified thing.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "DescribeThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingGroup.html",
                "description": "Gets information about the specified thing group.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "DescribeThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingRegistrationTask.html",
                "description": "Gets information about the bulk thing registration task.",
                "accessLevel": "Read"
            },
            "DescribeThingType": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingType.html",
                "description": "Gets information about the specified thing type.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thingtype": {
                        "required": true
                    }
                }
            },
            "DescribeTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeTunnel.html",
                "description": "Describes a tunnel.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "tunnel": {
                        "required": true
                    }
                }
            },
            "DetachPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPolicy.html",
                "description": "Detaches a policy from the specified target.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "DetachPrincipalPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPrincipalPolicy.html",
                "description": "Removes the specified policy from the specified certificate.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "DetachSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachSecurityProfile.html",
                "description": "Disassociates a Device Defender security profile from a thing group or from this account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "DetachThingPrincipal": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DetachThingPrincipal.html",
                "description": "Detaches the specified principal from the specified thing.",
                "accessLevel": "Write"
            },
            "DisableTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_DisableTopicRule.html",
                "description": "Disables the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "EnableTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_EnableTopicRule.html",
                "description": "Enables the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetCardinality": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetCardinality.html",
                "description": "Get cardinality for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetEffectivePolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetEffectivePolicies.html",
                "description": "Gets effective policies.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "GetIndexingConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetIndexingConfiguration.html",
                "description": "Gets current fleet indexing configuration",
                "accessLevel": "Read"
            },
            "GetJobDocument": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetJobDocument.html",
                "description": "Gets a job document.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "GetLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetLoggingOptions.html",
                "description": "Gets the logging options.",
                "accessLevel": "Read"
            },
            "GetOTAUpdate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetOTAUpdate.html",
                "description": "Gets the information about the OTA update job.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "otaupdate": {
                        "required": true
                    }
                }
            },
            "GetPendingJobExecutions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPendingJobExecutions.html",
                "description": "Gets the list of all jobs for a thing that are not in a terminal state.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "GetPercentiles": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPercentiles.html",
                "description": "Get percentiles for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicy.html",
                "description": "Gets information about the specified policy with the policy document of the default version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetPolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicyVersion.html",
                "description": "Gets information about the specified policy version.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "GetRegistrationCode": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetRegistrationCode.html",
                "description": "Gets a registration code used to register a CA certificate with AWS IoT.",
                "accessLevel": "Read"
            },
            "GetStatistics": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetStatistics.html",
                "description": "Get statistics for IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "GetThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Gets the thing shadow.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "GetTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetTopicRule.html",
                "description": "Gets information about the specified rule.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetV2LoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_GetV2LoggingOptions.html",
                "description": "Gets v2 logging options.",
                "accessLevel": "Read"
            },
            "ListActiveViolations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListActiveViolations.html",
                "description": "Lists the active violations for a given Device Defender security profile or Thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "ListAttachedPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAttachedPolicies.html",
                "description": "Lists the policies attached to the specified thing group.",
                "accessLevel": "List"
            },
            "ListAuditFindings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditFindings.html",
                "description": "Lists the findings (results) of a Device Defender audit or of the audits performed during a specified time period.",
                "accessLevel": "List"
            },
            "ListAuditMitigationActionsExecutions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsExecutions.html",
                "description": "Gets the status of audit mitigation action tasks that were executed.",
                "accessLevel": "List"
            },
            "ListAuditMitigationActionsTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsTasks.html",
                "description": "Gets a list of audit mitigation action tasks that match the specified filters.",
                "accessLevel": "List"
            },
            "ListAuditTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditTasks.html",
                "description": "Lists the Device Defender audits that have been performed during a given time period.",
                "accessLevel": "List"
            },
            "ListAuthorizers": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuthorizers.html",
                "description": "Lists the authorizers registered in your account.",
                "accessLevel": "List"
            },
            "ListBillingGroups": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListBillingGroups.html",
                "description": "Lists all billing groups.",
                "accessLevel": "List"
            },
            "ListCACertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCACertificates.html",
                "description": "Lists the CA certificates registered for your AWS account.",
                "accessLevel": "List"
            },
            "ListCertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificates.html",
                "description": "Lists your certificates.",
                "accessLevel": "List"
            },
            "ListCertificatesByCA": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificatesByCA.html",
                "description": "List the device certificates signed by the specified CA certificate.",
                "accessLevel": "List"
            },
            "ListDimensions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListDimensions.html",
                "description": "Lists the dimensions that are defined for your AWS account.",
                "accessLevel": "List"
            },
            "ListIndices": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListIndices.html",
                "description": "Lists all indices for fleet index",
                "accessLevel": "List"
            },
            "ListJobExecutionsForJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForJob.html",
                "description": "Lists the job executions for a job.",
                "accessLevel": "List",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "ListJobExecutionsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForThing.html",
                "description": "Lists the job executions for the specified thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListJobs": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobs.html",
                "description": "Lists jobs.",
                "accessLevel": "List"
            },
            "ListMitigationActions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListMitigationActions.html",
                "description": "Gets a list of all mitigation actions that match the specified filter criteria.",
                "accessLevel": "List"
            },
            "ListNamedShadowsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListNamedShadowsForThing.html",
                "description": "Lists all named shadows for a given thing.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListOTAUpdates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListOTAUpdates.html",
                "description": "Lists OTA update jobs in the account.",
                "accessLevel": "List"
            },
            "ListOutgoingCertificates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListOutgoingCertificates.html",
                "description": "Lists certificates that are being transfered but not yet accepted.",
                "accessLevel": "List"
            },
            "ListPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicies.html",
                "description": "Lists your policies.",
                "accessLevel": "List"
            },
            "ListPolicyPrincipals": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyPrincipals.html",
                "description": "Lists the principals associated with the specified policy.",
                "accessLevel": "List"
            },
            "ListPolicyVersions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyVersions.html",
                "description": "Lists the versions of the specified policy, and identifies the default version.",
                "accessLevel": "List"
            },
            "ListPrincipalPolicies": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalPolicies.html",
                "description": "Lists the policies attached to the specified principal. If you use an Amazon Cognito identity, the ID needs to be in Amazon Cognito Identity format.",
                "accessLevel": "List"
            },
            "ListPrincipalThings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalThings.html",
                "description": "Lists the things associated with the specified principal.",
                "accessLevel": "List"
            },
            "ListProvisioningTemplateVersions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplateVersions.html",
                "description": "A list of fleet provisioning template versions.",
                "accessLevel": "List",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "ListProvisioningTemplates": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplates.html",
                "description": "Lists the fleet provisioning templates in your AWS account.",
                "accessLevel": "List"
            },
            "ListRoleAliases": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListRoleAliases.html",
                "description": "Lists role aliases.",
                "accessLevel": "List"
            },
            "ListScheduledAudits": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListScheduledAudits.html",
                "description": "Lists all of your scheduled audits.",
                "accessLevel": "List"
            },
            "ListSecurityProfiles": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfiles.html",
                "description": "Lists the Device Defender security profiles you have created.",
                "accessLevel": "List",
                "resourceTypes": {
                    "dimension": {
                        "required": false
                    }
                }
            },
            "ListSecurityProfilesForTarget": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfilesForTarget.html",
                "description": "Lists the Device Defender security profiles attached to a target.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "ListStreams": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListStreams.html",
                "description": "Lists the streams in your account.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTagsForResource.html",
                "description": "Lists all tags for a given resource.",
                "accessLevel": "List",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                }
            },
            "ListTargetsForPolicy": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForPolicy.html",
                "description": "List targets for the specified policy.",
                "accessLevel": "List",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "ListTargetsForSecurityProfile": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForSecurityProfile.html",
                "description": "Lists the targets associated with a given Device Defender security profile.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    }
                }
            },
            "ListThingGroups": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroups.html",
                "description": "Lists all thing groups.",
                "accessLevel": "List"
            },
            "ListThingGroupsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroupsForThing.html",
                "description": "List thing groups to which the specified thing belongs.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ListThingPrincipals": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingPrincipals.html",
                "description": "Lists the principals associated with the specified thing.",
                "accessLevel": "List"
            },
            "ListThingRegistrationTaskReports": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTaskReports.html",
                "description": "Lists information about bulk thing registration tasks.",
                "accessLevel": "List"
            },
            "ListThingRegistrationTasks": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTasks.html",
                "description": "Lists bulk thing registration tasks.",
                "accessLevel": "List"
            },
            "ListThingTypes": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingTypes.html",
                "description": "Lists all thing types.",
                "accessLevel": "List"
            },
            "ListThings": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThings.html",
                "description": "Lists all things.",
                "accessLevel": "List"
            },
            "ListThingsInBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInBillingGroup.html",
                "description": "Lists all things in the specified billing group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "ListThingsInThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInThingGroup.html",
                "description": "Lists all things in the specified thing group.",
                "accessLevel": "List",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "ListTopicRules": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTopicRules.html",
                "description": "Lists the rules for the specific topic.",
                "accessLevel": "List"
            },
            "ListTunnels": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListTunnels.html",
                "description": "Lists tunnels.",
                "accessLevel": "List"
            },
            "ListV2LoggingLevels": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListV2LoggingLevels.html",
                "description": "Lists the v2 logging levels.",
                "accessLevel": "List"
            },
            "ListViolationEvents": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ListViolationEvents.html",
                "description": "Lists the Device Defender security profile violations discovered during the given time period.",
                "accessLevel": "List",
                "resourceTypes": {
                    "securityprofile": {
                        "required": false
                    },
                    "thing": {
                        "required": false
                    }
                }
            },
            "OpenTunnel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_OpenTunnel.html",
                "description": "Opens a tunnel.",
                "accessLevel": "Write",
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys",
                    "iot:ThingGroupArn",
                    "iot:TunnelDestinationService"
                ]
            },
            "Publish": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Publish to the specified topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "Receive": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Receive from the specified topic.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topic": {
                        "required": true
                    }
                }
            },
            "RegisterCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCACertificate.html",
                "description": "Registers a CA certificate with AWS IoT.",
                "accessLevel": "Write"
            },
            "RegisterCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html",
                "description": "Registers a device certificate with AWS IoT.",
                "accessLevel": "Write"
            },
            "RegisterCertificateWithoutCA": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html",
                "description": "Registers a device certificate with AWS IoT without a registered CA (certificate authority).",
                "accessLevel": "Write"
            },
            "RegisterThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterThing.html",
                "description": "Registers your thing.",
                "accessLevel": "Write"
            },
            "RejectCertificateTransfer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RejectCertificateTransfer.html",
                "description": "Rejects a pending certificate transfer.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "RemoveThingFromBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromBillingGroup.html",
                "description": "Removes thing from the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    },
                    "thing": {
                        "required": true
                    }
                }
            },
            "RemoveThingFromThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromThingGroup.html",
                "description": "Removes thing from the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "ReplaceTopicRule": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_ReplaceTopicRule.html",
                "description": "Replaces the specified rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "SearchIndex": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SearchIndex.html",
                "description": "Search IoT fleet index",
                "accessLevel": "Read",
                "resourceTypes": {
                    "index": {
                        "required": true
                    }
                }
            },
            "SetDefaultAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultAuthorizer.html",
                "description": "Sets the default authorizer. This will be used if a websocket connection is made without specifying an authorizer.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "SetDefaultPolicyVersion": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultPolicyVersion.html",
                "description": "Sets the specified version of the specified policy as the policy's default (operative) version.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "policy": {
                        "required": true
                    }
                }
            },
            "SetLoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetLoggingOptions.html",
                "description": "Sets the logging options.",
                "accessLevel": "Write"
            },
            "SetV2LoggingLevel": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingLevel.html",
                "description": "Sets the v2 logging level.",
                "accessLevel": "Write"
            },
            "SetV2LoggingOptions": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingOptions.html",
                "description": "Sets the v2 logging options.",
                "accessLevel": "Write"
            },
            "StartAuditMitigationActionsTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartAuditMitigationActionsTask.html",
                "description": "Starts a task that applies a set of mitigation actions to the specified target.",
                "accessLevel": "Write"
            },
            "StartNextPendingJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartNextPendingJobExecution.html",
                "description": "Gets and starts the next pending job execution for a thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "StartOnDemandAuditTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartOnDemandAuditTask.html",
                "description": "Starts an on-demand Device Defender audit.",
                "accessLevel": "Write"
            },
            "StartThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StartThingRegistrationTask.html",
                "description": "Starts a bulk thing registration task.",
                "accessLevel": "Write"
            },
            "StopThingRegistrationTask": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_StopThingRegistrationTask.html",
                "description": "Stops a bulk thing registration task.",
                "accessLevel": "Write"
            },
            "Subscribe": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Subscribe to the specified TopicFilter.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "topicfilter": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "",
                "description": "Tag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "TestAuthorization": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TestAuthorization.html",
                "description": "Test the policies evaluation for group policies",
                "accessLevel": "Read",
                "resourceTypes": {
                    "cert": {
                        "required": false
                    }
                }
            },
            "TestInvokeAuthorizer": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TestInvokeAuthorizer.html",
                "description": "Invoke the specified custom authorizer for testing purposes.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "TransferCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_TransferCertificate.html",
                "description": "Transfers the specified certificate to the specified AWS account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "",
                "description": "Untag a specified resource",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "billinggroup": {
                        "required": false
                    },
                    "dimension": {
                        "required": false
                    },
                    "dynamicthinggroup": {
                        "required": false
                    },
                    "job": {
                        "required": false
                    },
                    "mitigationaction": {
                        "required": false
                    },
                    "otaupdate": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "scheduledaudit": {
                        "required": false
                    },
                    "securityprofile": {
                        "required": false
                    },
                    "stream": {
                        "required": false
                    },
                    "thinggroup": {
                        "required": false
                    },
                    "thingtype": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateAccountAuditConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateAccountAuditConfiguration.html",
                "description": "Configures or reconfigures the Device Defender audit settings for this account.",
                "accessLevel": "Write"
            },
            "UpdateAuthorizer": {
                "url": "",
                "description": "Updates an authorizer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "authorizer": {
                        "required": true
                    }
                }
            },
            "UpdateBillingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateBillingGroup.html",
                "description": "Updates information associated with the specified billing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "billinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateCACertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCACertificate.html",
                "description": "Updates a registered CA certificate.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cacert": {
                        "required": true
                    }
                }
            },
            "UpdateCertificate": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCertificate.html",
                "description": "Updates the status of the specified certificate. This operation is idempotent.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "cert": {
                        "required": true
                    }
                }
            },
            "UpdateDimension": {
                "url": "",
                "description": "Updates the definition for a dimension.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dimension": {
                        "required": true
                    }
                }
            },
            "UpdateDynamicThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateDynamicThingGroup.html",
                "description": "Updates a Dynamic Thing Group",
                "accessLevel": "Write",
                "resourceTypes": {
                    "dynamicthinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateEventConfigurations": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateEventConfigurations.html",
                "description": "Updates event configurations.",
                "accessLevel": "Write"
            },
            "UpdateIndexingConfiguration": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateIndexingConfiguration.html",
                "description": "Updates fleet indexing configuration",
                "accessLevel": "Write"
            },
            "UpdateJob": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJob.html",
                "description": "Updates a job.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "job": {
                        "required": true
                    }
                }
            },
            "UpdateJobExecution": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJobExecution.html",
                "description": "Updates a job execution.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "UpdateMitigationAction": {
                "url": "",
                "description": "Updates the definition for the specified mitigation action.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "mitigationaction": {
                        "required": true
                    }
                }
            },
            "UpdateProvisioningTemplate": {
                "url": "",
                "description": "Updates a fleet provisioning template.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "provisioningtemplate": {
                        "required": true
                    }
                }
            },
            "UpdateRoleAlias": {
                "url": "",
                "description": "Updates the role alias",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rolealias": {
                        "required": true
                    }
                }
            },
            "UpdateScheduledAudit": {
                "url": "",
                "description": "Updates a scheduled audit, including what checks are performed and how often the audit takes place.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "scheduledaudit": {
                        "required": true
                    }
                }
            },
            "UpdateSecurityProfile": {
                "url": "",
                "description": "Updates a Device Defender security profile.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "securityprofile": {
                        "required": true
                    },
                    "dimension": {
                        "required": false
                    }
                }
            },
            "UpdateStream": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateStream.html",
                "description": "Updates the data for a stream.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "stream": {
                        "required": true
                    }
                }
            },
            "UpdateThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThing.html",
                "description": "Updates information associated with the specified thing.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "UpdateThingGroup": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroup.html",
                "description": "Updates information associated with the specified thing group.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thinggroup": {
                        "required": true
                    }
                }
            },
            "UpdateThingGroupsForThing": {
                "url": "https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroupsForThing.html",
                "description": "Updates the thing groups to which the thing belongs.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    },
                    "thinggroup": {
                        "required": false
                    }
                }
            },
            "UpdateThingShadow": {
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html",
                "description": "Updates the thing shadow.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "thing": {
                        "required": true
                    }
                }
            },
            "ValidateSecurityProfileBehaviors": {
                "url": "",
                "description": "Validates a Device Defender security profile behaviors specification.",
                "accessLevel": "Read"
            }
        };
        this.resourceTypes = {
            "client": {
                "name": "client",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:client/${ClientId}",
                "conditionKeys": []
            },
            "index": {
                "name": "index",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-indexing.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:index/${IndexName}",
                "conditionKeys": []
            },
            "job": {
                "name": "job",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-jobs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:job/${JobId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "tunnel": {
                "name": "tunnel",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-tunnels.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:tunnel/${TunnelId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "thing": {
                "name": "thing",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}",
                "conditionKeys": []
            },
            "thinggroup": {
                "name": "thinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "billinggroup": {
                "name": "billinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/billing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:billinggroup/${BillingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dynamicthinggroup": {
                "name": "dynamicthinggroup",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/dynamic-thing-groups.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "thingtype": {
                "name": "thingtype",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/thing-types.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:thingtype/${ThingTypeName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "topic": {
                "name": "topic",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:topic/${TopicName}",
                "conditionKeys": []
            },
            "topicfilter": {
                "name": "topicfilter",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/topics.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:topicfilter/${TopicFilter}",
                "conditionKeys": []
            },
            "rolealias": {
                "name": "rolealias",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/authorizing-direct-aws.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:rolealias/${RoleAlias}",
                "conditionKeys": []
            },
            "authorizer": {
                "name": "authorizer",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/custom-authorizer.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:authorizer/${AuthorizerName}",
                "conditionKeys": []
            },
            "policy": {
                "name": "policy",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-policies.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:policy/${PolicyName}",
                "conditionKeys": []
            },
            "cert": {
                "name": "cert",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:cert/${Certificate}",
                "conditionKeys": []
            },
            "cacert": {
                "name": "cacert",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:cacert/${CACertificate}",
                "conditionKeys": []
            },
            "stream": {
                "name": "stream",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:stream/${StreamId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "otaupdate": {
                "name": "otaupdate",
                "url": "https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:otaupdate/${OtaUpdateId}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "scheduledaudit": {
                "name": "scheduledaudit",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-audit.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:scheduledaudit/${ScheduleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "mitigationaction": {
                "name": "mitigationaction",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-mitigation-actions.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:mitigationaction/${MitigationActionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "securityprofile": {
                "name": "securityprofile",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:securityprofile/${SecurityProfileName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "dimension": {
                "name": "dimension",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:dimension/${DimensionName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:rule/${RuleName}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "provisioningtemplate": {
                "name": "provisioningtemplate",
                "url": "https://docs.aws.amazon.com/iot/latest/developerguide/provision-template.html",
                "arn": "arn:${Partition}:iot:${Region}:${Account}:provisioningtemplate/${ProvisioningTemplate}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Accepts a pending certificate transfer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AcceptCertificateTransfer.html
     */
    acceptCertificateTransfer() {
        this.add('iot:AcceptCertificateTransfer');
        return this;
    }
    /**
     * Adds a thing to the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToBillingGroup.html
     */
    addThingToBillingGroup() {
        this.add('iot:AddThingToBillingGroup');
        return this;
    }
    /**
     * Adds a thing to the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AddThingToThingGroup.html
     */
    addThingToThingGroup() {
        this.add('iot:AddThingToThingGroup');
        return this;
    }
    /**
     * Associates a group with a continuous job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AssociateTargetsWithJob.html
     */
    associateTargetsWithJob() {
        this.add('iot:AssociateTargetsWithJob');
        return this;
    }
    /**
     * Attaches a policy to the specified target.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPolicy.html
     */
    attachPolicy() {
        this.add('iot:AttachPolicy');
        return this;
    }
    /**
     * Attaches the specified policy to the specified principal (certificate or other credential).
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachPrincipalPolicy.html
     */
    attachPrincipalPolicy() {
        this.add('iot:AttachPrincipalPolicy');
        return this;
    }
    /**
     * Associates a Device Defender security profile with a thing group or with this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachSecurityProfile.html
     */
    attachSecurityProfile() {
        this.add('iot:AttachSecurityProfile');
        return this;
    }
    /**
     * Attaches the specified principal to the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_AttachThingPrincipal.html
     */
    attachThingPrincipal() {
        this.add('iot:AttachThingPrincipal');
        return this;
    }
    /**
     * Cancels a mitigation action task that is in progress.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditMitigationActionsTask.html
     */
    cancelAuditMitigationActionsTask() {
        this.add('iot:CancelAuditMitigationActionsTask');
        return this;
    }
    /**
     * Cancels an audit that is in progress. The audit can be either scheduled or on-demand.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelAuditTask.html
     */
    cancelAuditTask() {
        this.add('iot:CancelAuditTask');
        return this;
    }
    /**
     * Cancels a pending transfer for the specified certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelCertificateTransfer.html
     */
    cancelCertificateTransfer() {
        this.add('iot:CancelCertificateTransfer');
        return this;
    }
    /**
     * Cancels a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJob.html
     */
    cancelJob() {
        this.add('iot:CancelJob');
        return this;
    }
    /**
     * Cancels a job execution on a particular device.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CancelJobExecution.html
     */
    cancelJobExecution() {
        this.add('iot:CancelJobExecution');
        return this;
    }
    /**
     * Clears the default authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ClearDefaultAuthorizer.html
     */
    clearDefaultAuthorizer() {
        this.add('iot:ClearDefaultAuthorizer');
        return this;
    }
    /**
     * Closes a tunnel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CloseTunnel.html
     */
    closeTunnel() {
        this.add('iot:CloseTunnel');
        return this;
    }
    /**
     * Connect as the specified client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    connect() {
        this.add('iot:Connect');
        return this;
    }
    /**
     * Creates an authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateAuthorizer.html
     */
    createAuthorizer() {
        this.add('iot:CreateAuthorizer');
        return this;
    }
    /**
     * Creates a billing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateBillingGroup.html
     */
    createBillingGroup() {
        this.add('iot:CreateBillingGroup');
        return this;
    }
    /**
     * Creates an X.509 certificate using the specified certificate signing request.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateCertificateFromCsr.html
     */
    createCertificateFromCsr() {
        this.add('iot:CreateCertificateFromCsr');
        return this;
    }
    /**
     * Defines a dimension that can be used to to limit the scope of a metric used in a security profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDimension.html
     */
    createDimension() {
        this.add('iot:CreateDimension');
        return this;
    }
    /**
     * Creates a Dynamic Thing Group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateDynamicThingGroup.html
     */
    createDynamicThingGroup() {
        this.add('iot:CreateDynamicThingGroup');
        return this;
    }
    /**
     * Creates a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateJob.html
     */
    createJob() {
        this.add('iot:CreateJob');
        return this;
    }
    /**
     * Creates a 2048 bit RSA key pair and issues an X.509 certificate using the issued public key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateKeysAndCertificate.html
     */
    createKeysAndCertificate() {
        this.add('iot:CreateKeysAndCertificate');
        return this;
    }
    /**
     * Defines an action that can be applied to audit findings by using StartAuditMitigationActionsTask.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateMitigationAction.html
     */
    createMitigationAction() {
        this.add('iot:CreateMitigationAction');
        return this;
    }
    /**
     * Creates an OTA update job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateOTAUpdate.html
     */
    createOTAUpdate() {
        this.add('iot:CreateOTAUpdate');
        return this;
    }
    /**
     * Creates an AWS IoT policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicy.html
     */
    createPolicy() {
        this.add('iot:CreatePolicy');
        return this;
    }
    /**
     * Creates a new version of the specified AWS IoT policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreatePolicyVersion.html
     */
    createPolicyVersion() {
        this.add('iot:CreatePolicyVersion');
        return this;
    }
    /**
     * Creates a provisioning claim.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningClaim.html
     */
    createProvisioningClaim() {
        this.add('iot:CreateProvisioningClaim');
        return this;
    }
    /**
     * Creates a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplate.html
     */
    createProvisioningTemplate() {
        this.add('iot:CreateProvisioningTemplate');
        return this;
    }
    /**
     * Creates a new version of a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateProvisioningTemplateVersion.html
     */
    createProvisioningTemplateVersion() {
        this.add('iot:CreateProvisioningTemplateVersion');
        return this;
    }
    /**
     * Creates a role alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateRoleAlias.html
     */
    createRoleAlias() {
        this.add('iot:CreateRoleAlias');
        return this;
    }
    /**
     * Creates a scheduled audit that is run at a specified time interval.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateScheduledAudit.html
     */
    createScheduledAudit() {
        this.add('iot:CreateScheduledAudit');
        return this;
    }
    /**
     * Creates a Device Defender security profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateSecurityProfile.html
     */
    createSecurityProfile() {
        this.add('iot:CreateSecurityProfile');
        return this;
    }
    /**
     * Creates a new AWS IoT stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateStream.html
     */
    createStream() {
        this.add('iot:CreateStream');
        return this;
    }
    /**
     * Creates a thing in the thing registry.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThing.html
     */
    createThing() {
        this.add('iot:CreateThing');
        return this;
    }
    /**
     * Creates a thing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingGroup.html
     */
    createThingGroup() {
        this.add('iot:CreateThingGroup');
        return this;
    }
    /**
     * Creates a new thing type.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateThingType.html
     */
    createThingType() {
        this.add('iot:CreateThingType');
        return this;
    }
    /**
     * Creates a rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_CreateTopicRule.html
     */
    createTopicRule() {
        this.add('iot:CreateTopicRule');
        return this;
    }
    /**
     * Deletes the audit configuration associated with the account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAccountAuditConfiguration.html
     */
    deleteAccountAuditConfiguration() {
        this.add('iot:DeleteAccountAuditConfiguration');
        return this;
    }
    /**
     * Deletes the specified authorizer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteAuthorizer.html
     */
    deleteAuthorizer() {
        this.add('iot:DeleteAuthorizer');
        return this;
    }
    /**
     * Deletes the specified billing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteBillingGroup.html
     */
    deleteBillingGroup() {
        this.add('iot:DeleteBillingGroup');
        return this;
    }
    /**
     * Deletes a registered CA certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCACertificate.html
     */
    deleteCACertificate() {
        this.add('iot:DeleteCACertificate');
        return this;
    }
    /**
     * Deletes the specified certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteCertificate.html
     */
    deleteCertificate() {
        this.add('iot:DeleteCertificate');
        return this;
    }
    /**
     * Removes the specified dimension from your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDimension.html
     */
    deleteDimension() {
        this.add('iot:DeleteDimension');
        return this;
    }
    /**
     * Deletes the specified Dynamic Thing Group
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteDynamicThingGroup.html
     */
    deleteDynamicThingGroup() {
        this.add('iot:DeleteDynamicThingGroup');
        return this;
    }
    /**
     * Deletes a job and its related job executions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJob.html
     */
    deleteJob() {
        this.add('iot:DeleteJob');
        return this;
    }
    /**
     * Deletes a job execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteJobExecution.html
     */
    deleteJobExecution() {
        this.add('iot:DeleteJobExecution');
        return this;
    }
    /**
     * Deletes a defined mitigation action from your AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteMitigationAction.html
     */
    deleteMitigationAction() {
        this.add('iot:DeleteMitigationAction');
        return this;
    }
    /**
     * Deletes an OTA update job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteOTAUpdate.html
     */
    deleteOTAUpdate() {
        this.add('iot:DeleteOTAUpdate');
        return this;
    }
    /**
     * Deletes the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicy.html
     */
    deletePolicy() {
        this.add('iot:DeletePolicy');
        return this;
    }
    /**
     * Deletes the specified version of the specified policy.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeletePolicyVersion.html
     */
    deletePolicyVersion() {
        this.add('iot:DeletePolicyVersion');
        return this;
    }
    /**
     * Deletes a fleet provisioning template.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplate.html
     */
    deleteProvisioningTemplate() {
        this.add('iot:DeleteProvisioningTemplate');
        return this;
    }
    /**
     * Deletes a fleet provisioning template version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteProvisioningTemplateVersion.html
     */
    deleteProvisioningTemplateVersion() {
        this.add('iot:DeleteProvisioningTemplateVersion');
        return this;
    }
    /**
     * Deletes a CA certificate registration code.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRegistrationCode.html
     */
    deleteRegistrationCode() {
        this.add('iot:DeleteRegistrationCode');
        return this;
    }
    /**
     * Deletes the specified role alias.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteRoleAlias.html
     */
    deleteRoleAlias() {
        this.add('iot:DeleteRoleAlias');
        return this;
    }
    /**
     * Deletes a scheduled audit.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteScheduledAudit.html
     */
    deleteScheduledAudit() {
        this.add('iot:DeleteScheduledAudit');
        return this;
    }
    /**
     * Deletes a Device Defender security profile.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteSecurityProfile.html
     */
    deleteSecurityProfile() {
        this.add('iot:DeleteSecurityProfile');
        return this;
    }
    /**
     * Deletes a specified stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteStream.html
     */
    deleteStream() {
        this.add('iot:DeleteStream');
        return this;
    }
    /**
     * Deletes the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThing.html
     */
    deleteThing() {
        this.add('iot:DeleteThing');
        return this;
    }
    /**
     * Deletes the specified thing group.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingGroup.html
     */
    deleteThingGroup() {
        this.add('iot:DeleteThingGroup');
        return this;
    }
    /**
     * Deletes the specified thing shadow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    deleteThingShadow() {
        this.add('iot:DeleteThingShadow');
        return this;
    }
    /**
     * Deletes the specified thing type.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteThingType.html
     */
    deleteThingType() {
        this.add('iot:DeleteThingType');
        return this;
    }
    /**
     * Deletes the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteTopicRule.html
     */
    deleteTopicRule() {
        this.add('iot:DeleteTopicRule');
        return this;
    }
    /**
     * Deletes the specified v2 logging level.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeleteV2LoggingLevel.html
     */
    deleteV2LoggingLevel() {
        this.add('iot:DeleteV2LoggingLevel');
        return this;
    }
    /**
     * Deprecates the specified thing type.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DeprecateThingType.html
     */
    deprecateThingType() {
        this.add('iot:DeprecateThingType');
        return this;
    }
    /**
     * Gets information about audit configurations for the account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAccountAuditConfiguration.html
     */
    describeAccountAuditConfiguration() {
        this.add('iot:DescribeAccountAuditConfiguration');
        return this;
    }
    /**
     * Gets information about an audit mitigation task that is used to apply mitigation actions to a set of audit findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditMitigationActionsTask.html
     */
    describeAuditMitigationActionsTask() {
        this.add('iot:DescribeAuditMitigationActionsTask');
        return this;
    }
    /**
     * Gets information about a Device Defender audit.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuditTask.html
     */
    describeAuditTask() {
        this.add('iot:DescribeAuditTask');
        return this;
    }
    /**
     * Describes an authorizer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeAuthorizer.html
     */
    describeAuthorizer() {
        this.add('iot:DescribeAuthorizer');
        return this;
    }
    /**
     * Gets information about the specified billing group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeBillingGroup.html
     */
    describeBillingGroup() {
        this.add('iot:DescribeBillingGroup');
        return this;
    }
    /**
     * Describes a registered CA certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCACertificate.html
     */
    describeCACertificate() {
        this.add('iot:DescribeCACertificate');
        return this;
    }
    /**
     * Gets information about the specified certificate.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeCertificate.html
     */
    describeCertificate() {
        this.add('iot:DescribeCertificate');
        return this;
    }
    /**
     * Describes the default authorizer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDefaultAuthorizer.html
     */
    describeDefaultAuthorizer() {
        this.add('iot:DescribeDefaultAuthorizer');
        return this;
    }
    /**
     * Provides details about a dimension that is defined in your AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeDimension.html
     */
    describeDimension() {
        this.add('iot:DescribeDimension');
        return this;
    }
    /**
     * Returns a unique endpoint specific to the AWS account making the call.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEndpoint.html
     */
    describeEndpoint() {
        this.add('iot:DescribeEndpoint');
        return this;
    }
    /**
     * Returns account event configurations.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeEventConfigurations.html
     */
    describeEventConfigurations() {
        this.add('iot:DescribeEventConfigurations');
        return this;
    }
    /**
     * Gets information about the specified index.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeIndex.html
     */
    describeIndex() {
        this.add('iot:DescribeIndex');
        return this;
    }
    /**
     * Describes a job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJob.html
     */
    describeJob() {
        this.add('iot:DescribeJob');
        return this;
    }
    /**
     * Describes a job execution.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeJobExecution.html
     */
    describeJobExecution() {
        this.add('iot:DescribeJobExecution');
        return this;
    }
    /**
     * Gets information about a mitigation action.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeMitigationAction.html
     */
    describeMitigationAction() {
        this.add('iot:DescribeMitigationAction');
        return this;
    }
    /**
     * Returns information about a fleet provisioning template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplate.html
     */
    describeProvisioningTemplate() {
        this.add('iot:DescribeProvisioningTemplate');
        return this;
    }
    /**
     * Returns information about a fleet provisioning template version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeProvisioningTemplateVersion.html
     */
    describeProvisioningTemplateVersion() {
        this.add('iot:DescribeProvisioningTemplateVersion');
        return this;
    }
    /**
     * Describes a role alias.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeRoleAlias.html
     */
    describeRoleAlias() {
        this.add('iot:DescribeRoleAlias');
        return this;
    }
    /**
     * Gets information about a scheduled audit.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeScheduledAudit.html
     */
    describeScheduledAudit() {
        this.add('iot:DescribeScheduledAudit');
        return this;
    }
    /**
     * Gets information about a Device Defender security profile.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeSecurityProfile.html
     */
    describeSecurityProfile() {
        this.add('iot:DescribeSecurityProfile');
        return this;
    }
    /**
     * Gets information about the specified stream.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeStream.html
     */
    describeStream() {
        this.add('iot:DescribeStream');
        return this;
    }
    /**
     * Gets information about the specified thing.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThing.html
     */
    describeThing() {
        this.add('iot:DescribeThing');
        return this;
    }
    /**
     * Gets information about the specified thing group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingGroup.html
     */
    describeThingGroup() {
        this.add('iot:DescribeThingGroup');
        return this;
    }
    /**
     * Gets information about the bulk thing registration task.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingRegistrationTask.html
     */
    describeThingRegistrationTask() {
        this.add('iot:DescribeThingRegistrationTask');
        return this;
    }
    /**
     * Gets information about the specified thing type.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeThingType.html
     */
    describeThingType() {
        this.add('iot:DescribeThingType');
        return this;
    }
    /**
     * Describes a tunnel.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DescribeTunnel.html
     */
    describeTunnel() {
        this.add('iot:DescribeTunnel');
        return this;
    }
    /**
     * Detaches a policy from the specified target.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPolicy.html
     */
    detachPolicy() {
        this.add('iot:DetachPolicy');
        return this;
    }
    /**
     * Removes the specified policy from the specified certificate.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachPrincipalPolicy.html
     */
    detachPrincipalPolicy() {
        this.add('iot:DetachPrincipalPolicy');
        return this;
    }
    /**
     * Disassociates a Device Defender security profile from a thing group or from this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachSecurityProfile.html
     */
    detachSecurityProfile() {
        this.add('iot:DetachSecurityProfile');
        return this;
    }
    /**
     * Detaches the specified principal from the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DetachThingPrincipal.html
     */
    detachThingPrincipal() {
        this.add('iot:DetachThingPrincipal');
        return this;
    }
    /**
     * Disables the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_DisableTopicRule.html
     */
    disableTopicRule() {
        this.add('iot:DisableTopicRule');
        return this;
    }
    /**
     * Enables the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_EnableTopicRule.html
     */
    enableTopicRule() {
        this.add('iot:EnableTopicRule');
        return this;
    }
    /**
     * Get cardinality for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetCardinality.html
     */
    getCardinality() {
        this.add('iot:GetCardinality');
        return this;
    }
    /**
     * Gets effective policies.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetEffectivePolicies.html
     */
    getEffectivePolicies() {
        this.add('iot:GetEffectivePolicies');
        return this;
    }
    /**
     * Gets current fleet indexing configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetIndexingConfiguration.html
     */
    getIndexingConfiguration() {
        this.add('iot:GetIndexingConfiguration');
        return this;
    }
    /**
     * Gets a job document.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetJobDocument.html
     */
    getJobDocument() {
        this.add('iot:GetJobDocument');
        return this;
    }
    /**
     * Gets the logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetLoggingOptions.html
     */
    getLoggingOptions() {
        this.add('iot:GetLoggingOptions');
        return this;
    }
    /**
     * Gets the information about the OTA update job.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetOTAUpdate.html
     */
    getOTAUpdate() {
        this.add('iot:GetOTAUpdate');
        return this;
    }
    /**
     * Gets the list of all jobs for a thing that are not in a terminal state.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPendingJobExecutions.html
     */
    getPendingJobExecutions() {
        this.add('iot:GetPendingJobExecutions');
        return this;
    }
    /**
     * Get percentiles for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPercentiles.html
     */
    getPercentiles() {
        this.add('iot:GetPercentiles');
        return this;
    }
    /**
     * Gets information about the specified policy with the policy document of the default version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicy.html
     */
    getPolicy() {
        this.add('iot:GetPolicy');
        return this;
    }
    /**
     * Gets information about the specified policy version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetPolicyVersion.html
     */
    getPolicyVersion() {
        this.add('iot:GetPolicyVersion');
        return this;
    }
    /**
     * Gets a registration code used to register a CA certificate with AWS IoT.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetRegistrationCode.html
     */
    getRegistrationCode() {
        this.add('iot:GetRegistrationCode');
        return this;
    }
    /**
     * Get statistics for IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetStatistics.html
     */
    getStatistics() {
        this.add('iot:GetStatistics');
        return this;
    }
    /**
     * Gets the thing shadow.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    getThingShadow() {
        this.add('iot:GetThingShadow');
        return this;
    }
    /**
     * Gets information about the specified rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetTopicRule.html
     */
    getTopicRule() {
        this.add('iot:GetTopicRule');
        return this;
    }
    /**
     * Gets v2 logging options.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_GetV2LoggingOptions.html
     */
    getV2LoggingOptions() {
        this.add('iot:GetV2LoggingOptions');
        return this;
    }
    /**
     * Lists the active violations for a given Device Defender security profile or Thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListActiveViolations.html
     */
    listActiveViolations() {
        this.add('iot:ListActiveViolations');
        return this;
    }
    /**
     * Lists the policies attached to the specified thing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAttachedPolicies.html
     */
    listAttachedPolicies() {
        this.add('iot:ListAttachedPolicies');
        return this;
    }
    /**
     * Lists the findings (results) of a Device Defender audit or of the audits performed during a specified time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditFindings.html
     */
    listAuditFindings() {
        this.add('iot:ListAuditFindings');
        return this;
    }
    /**
     * Gets the status of audit mitigation action tasks that were executed.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsExecutions.html
     */
    listAuditMitigationActionsExecutions() {
        this.add('iot:ListAuditMitigationActionsExecutions');
        return this;
    }
    /**
     * Gets a list of audit mitigation action tasks that match the specified filters.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditMitigationActionsTasks.html
     */
    listAuditMitigationActionsTasks() {
        this.add('iot:ListAuditMitigationActionsTasks');
        return this;
    }
    /**
     * Lists the Device Defender audits that have been performed during a given time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuditTasks.html
     */
    listAuditTasks() {
        this.add('iot:ListAuditTasks');
        return this;
    }
    /**
     * Lists the authorizers registered in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListAuthorizers.html
     */
    listAuthorizers() {
        this.add('iot:ListAuthorizers');
        return this;
    }
    /**
     * Lists all billing groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListBillingGroups.html
     */
    listBillingGroups() {
        this.add('iot:ListBillingGroups');
        return this;
    }
    /**
     * Lists the CA certificates registered for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCACertificates.html
     */
    listCACertificates() {
        this.add('iot:ListCACertificates');
        return this;
    }
    /**
     * Lists your certificates.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificates.html
     */
    listCertificates() {
        this.add('iot:ListCertificates');
        return this;
    }
    /**
     * List the device certificates signed by the specified CA certificate.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListCertificatesByCA.html
     */
    listCertificatesByCA() {
        this.add('iot:ListCertificatesByCA');
        return this;
    }
    /**
     * Lists the dimensions that are defined for your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListDimensions.html
     */
    listDimensions() {
        this.add('iot:ListDimensions');
        return this;
    }
    /**
     * Lists all indices for fleet index
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListIndices.html
     */
    listIndices() {
        this.add('iot:ListIndices');
        return this;
    }
    /**
     * Lists the job executions for a job.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForJob.html
     */
    listJobExecutionsForJob() {
        this.add('iot:ListJobExecutionsForJob');
        return this;
    }
    /**
     * Lists the job executions for the specified thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobExecutionsForThing.html
     */
    listJobExecutionsForThing() {
        this.add('iot:ListJobExecutionsForThing');
        return this;
    }
    /**
     * Lists jobs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListJobs.html
     */
    listJobs() {
        this.add('iot:ListJobs');
        return this;
    }
    /**
     * Gets a list of all mitigation actions that match the specified filter criteria.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListMitigationActions.html
     */
    listMitigationActions() {
        this.add('iot:ListMitigationActions');
        return this;
    }
    /**
     * Lists all named shadows for a given thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListNamedShadowsForThing.html
     */
    listNamedShadowsForThing() {
        this.add('iot:ListNamedShadowsForThing');
        return this;
    }
    /**
     * Lists OTA update jobs in the account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListOTAUpdates.html
     */
    listOTAUpdates() {
        this.add('iot:ListOTAUpdates');
        return this;
    }
    /**
     * Lists certificates that are being transfered but not yet accepted.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListOutgoingCertificates.html
     */
    listOutgoingCertificates() {
        this.add('iot:ListOutgoingCertificates');
        return this;
    }
    /**
     * Lists your policies.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicies.html
     */
    listPolicies() {
        this.add('iot:ListPolicies');
        return this;
    }
    /**
     * Lists the principals associated with the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyPrincipals.html
     */
    listPolicyPrincipals() {
        this.add('iot:ListPolicyPrincipals');
        return this;
    }
    /**
     * Lists the versions of the specified policy, and identifies the default version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPolicyVersions.html
     */
    listPolicyVersions() {
        this.add('iot:ListPolicyVersions');
        return this;
    }
    /**
     * Lists the policies attached to the specified principal. If you use an Amazon Cognito identity, the ID needs to be in Amazon Cognito Identity format.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalPolicies.html
     */
    listPrincipalPolicies() {
        this.add('iot:ListPrincipalPolicies');
        return this;
    }
    /**
     * Lists the things associated with the specified principal.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListPrincipalThings.html
     */
    listPrincipalThings() {
        this.add('iot:ListPrincipalThings');
        return this;
    }
    /**
     * A list of fleet provisioning template versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplateVersions.html
     */
    listProvisioningTemplateVersions() {
        this.add('iot:ListProvisioningTemplateVersions');
        return this;
    }
    /**
     * Lists the fleet provisioning templates in your AWS account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListProvisioningTemplates.html
     */
    listProvisioningTemplates() {
        this.add('iot:ListProvisioningTemplates');
        return this;
    }
    /**
     * Lists role aliases.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListRoleAliases.html
     */
    listRoleAliases() {
        this.add('iot:ListRoleAliases');
        return this;
    }
    /**
     * Lists all of your scheduled audits.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListScheduledAudits.html
     */
    listScheduledAudits() {
        this.add('iot:ListScheduledAudits');
        return this;
    }
    /**
     * Lists the Device Defender security profiles you have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfiles.html
     */
    listSecurityProfiles() {
        this.add('iot:ListSecurityProfiles');
        return this;
    }
    /**
     * Lists the Device Defender security profiles attached to a target.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListSecurityProfilesForTarget.html
     */
    listSecurityProfilesForTarget() {
        this.add('iot:ListSecurityProfilesForTarget');
        return this;
    }
    /**
     * Lists the streams in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListStreams.html
     */
    listStreams() {
        this.add('iot:ListStreams');
        return this;
    }
    /**
     * Lists all tags for a given resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('iot:ListTagsForResource');
        return this;
    }
    /**
     * List targets for the specified policy.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForPolicy.html
     */
    listTargetsForPolicy() {
        this.add('iot:ListTargetsForPolicy');
        return this;
    }
    /**
     * Lists the targets associated with a given Device Defender security profile.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTargetsForSecurityProfile.html
     */
    listTargetsForSecurityProfile() {
        this.add('iot:ListTargetsForSecurityProfile');
        return this;
    }
    /**
     * Lists all thing groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroups.html
     */
    listThingGroups() {
        this.add('iot:ListThingGroups');
        return this;
    }
    /**
     * List thing groups to which the specified thing belongs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingGroupsForThing.html
     */
    listThingGroupsForThing() {
        this.add('iot:ListThingGroupsForThing');
        return this;
    }
    /**
     * Lists the principals associated with the specified thing.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingPrincipals.html
     */
    listThingPrincipals() {
        this.add('iot:ListThingPrincipals');
        return this;
    }
    /**
     * Lists information about bulk thing registration tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTaskReports.html
     */
    listThingRegistrationTaskReports() {
        this.add('iot:ListThingRegistrationTaskReports');
        return this;
    }
    /**
     * Lists bulk thing registration tasks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingRegistrationTasks.html
     */
    listThingRegistrationTasks() {
        this.add('iot:ListThingRegistrationTasks');
        return this;
    }
    /**
     * Lists all thing types.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingTypes.html
     */
    listThingTypes() {
        this.add('iot:ListThingTypes');
        return this;
    }
    /**
     * Lists all things.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThings.html
     */
    listThings() {
        this.add('iot:ListThings');
        return this;
    }
    /**
     * Lists all things in the specified billing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInBillingGroup.html
     */
    listThingsInBillingGroup() {
        this.add('iot:ListThingsInBillingGroup');
        return this;
    }
    /**
     * Lists all things in the specified thing group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListThingsInThingGroup.html
     */
    listThingsInThingGroup() {
        this.add('iot:ListThingsInThingGroup');
        return this;
    }
    /**
     * Lists the rules for the specific topic.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTopicRules.html
     */
    listTopicRules() {
        this.add('iot:ListTopicRules');
        return this;
    }
    /**
     * Lists tunnels.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListTunnels.html
     */
    listTunnels() {
        this.add('iot:ListTunnels');
        return this;
    }
    /**
     * Lists the v2 logging levels.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListV2LoggingLevels.html
     */
    listV2LoggingLevels() {
        this.add('iot:ListV2LoggingLevels');
        return this;
    }
    /**
     * Lists the Device Defender security profile violations discovered during the given time period.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ListViolationEvents.html
     */
    listViolationEvents() {
        this.add('iot:ListViolationEvents');
        return this;
    }
    /**
     * Opens a tunnel.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_OpenTunnel.html
     */
    openTunnel() {
        this.add('iot:OpenTunnel');
        return this;
    }
    /**
     * Publish to the specified topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    publish() {
        this.add('iot:Publish');
        return this;
    }
    /**
     * Receive from the specified topic.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    receive() {
        this.add('iot:Receive');
        return this;
    }
    /**
     * Registers a CA certificate with AWS IoT.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCACertificate.html
     */
    registerCACertificate() {
        this.add('iot:RegisterCACertificate');
        return this;
    }
    /**
     * Registers a device certificate with AWS IoT.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificate.html
     */
    registerCertificate() {
        this.add('iot:RegisterCertificate');
        return this;
    }
    /**
     * Registers a device certificate with AWS IoT without a registered CA (certificate authority).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterCertificateWithoutCA.html
     */
    registerCertificateWithoutCA() {
        this.add('iot:RegisterCertificateWithoutCA');
        return this;
    }
    /**
     * Registers your thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RegisterThing.html
     */
    registerThing() {
        this.add('iot:RegisterThing');
        return this;
    }
    /**
     * Rejects a pending certificate transfer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RejectCertificateTransfer.html
     */
    rejectCertificateTransfer() {
        this.add('iot:RejectCertificateTransfer');
        return this;
    }
    /**
     * Removes thing from the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromBillingGroup.html
     */
    removeThingFromBillingGroup() {
        this.add('iot:RemoveThingFromBillingGroup');
        return this;
    }
    /**
     * Removes thing from the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_RemoveThingFromThingGroup.html
     */
    removeThingFromThingGroup() {
        this.add('iot:RemoveThingFromThingGroup');
        return this;
    }
    /**
     * Replaces the specified rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_ReplaceTopicRule.html
     */
    replaceTopicRule() {
        this.add('iot:ReplaceTopicRule');
        return this;
    }
    /**
     * Search IoT fleet index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SearchIndex.html
     */
    searchIndex() {
        this.add('iot:SearchIndex');
        return this;
    }
    /**
     * Sets the default authorizer. This will be used if a websocket connection is made without specifying an authorizer.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultAuthorizer.html
     */
    setDefaultAuthorizer() {
        this.add('iot:SetDefaultAuthorizer');
        return this;
    }
    /**
     * Sets the specified version of the specified policy as the policy's default (operative) version.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetDefaultPolicyVersion.html
     */
    setDefaultPolicyVersion() {
        this.add('iot:SetDefaultPolicyVersion');
        return this;
    }
    /**
     * Sets the logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetLoggingOptions.html
     */
    setLoggingOptions() {
        this.add('iot:SetLoggingOptions');
        return this;
    }
    /**
     * Sets the v2 logging level.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingLevel.html
     */
    setV2LoggingLevel() {
        this.add('iot:SetV2LoggingLevel');
        return this;
    }
    /**
     * Sets the v2 logging options.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_SetV2LoggingOptions.html
     */
    setV2LoggingOptions() {
        this.add('iot:SetV2LoggingOptions');
        return this;
    }
    /**
     * Starts a task that applies a set of mitigation actions to the specified target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartAuditMitigationActionsTask.html
     */
    startAuditMitigationActionsTask() {
        this.add('iot:StartAuditMitigationActionsTask');
        return this;
    }
    /**
     * Gets and starts the next pending job execution for a thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartNextPendingJobExecution.html
     */
    startNextPendingJobExecution() {
        this.add('iot:StartNextPendingJobExecution');
        return this;
    }
    /**
     * Starts an on-demand Device Defender audit.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartOnDemandAuditTask.html
     */
    startOnDemandAuditTask() {
        this.add('iot:StartOnDemandAuditTask');
        return this;
    }
    /**
     * Starts a bulk thing registration task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StartThingRegistrationTask.html
     */
    startThingRegistrationTask() {
        this.add('iot:StartThingRegistrationTask');
        return this;
    }
    /**
     * Stops a bulk thing registration task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_StopThingRegistrationTask.html
     */
    stopThingRegistrationTask() {
        this.add('iot:StopThingRegistrationTask');
        return this;
    }
    /**
     * Subscribe to the specified TopicFilter.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    subscribe() {
        this.add('iot:Subscribe');
        return this;
    }
    /**
     * Tag a specified resource
     *
     * Access Level: Tagging
     */
    tagResource() {
        this.add('iot:TagResource');
        return this;
    }
    /**
     * Test the policies evaluation for group policies
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TestAuthorization.html
     */
    testAuthorization() {
        this.add('iot:TestAuthorization');
        return this;
    }
    /**
     * Invoke the specified custom authorizer for testing purposes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TestInvokeAuthorizer.html
     */
    testInvokeAuthorizer() {
        this.add('iot:TestInvokeAuthorizer');
        return this;
    }
    /**
     * Transfers the specified certificate to the specified AWS account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_TransferCertificate.html
     */
    transferCertificate() {
        this.add('iot:TransferCertificate');
        return this;
    }
    /**
     * Untag a specified resource
     *
     * Access Level: Tagging
     */
    untagResource() {
        this.add('iot:UntagResource');
        return this;
    }
    /**
     * Configures or reconfigures the Device Defender audit settings for this account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateAccountAuditConfiguration.html
     */
    updateAccountAuditConfiguration() {
        this.add('iot:UpdateAccountAuditConfiguration');
        return this;
    }
    /**
     * Updates an authorizer
     *
     * Access Level: Write
     */
    updateAuthorizer() {
        this.add('iot:UpdateAuthorizer');
        return this;
    }
    /**
     * Updates information associated with the specified billing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateBillingGroup.html
     */
    updateBillingGroup() {
        this.add('iot:UpdateBillingGroup');
        return this;
    }
    /**
     * Updates a registered CA certificate.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCACertificate.html
     */
    updateCACertificate() {
        this.add('iot:UpdateCACertificate');
        return this;
    }
    /**
     * Updates the status of the specified certificate. This operation is idempotent.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateCertificate.html
     */
    updateCertificate() {
        this.add('iot:UpdateCertificate');
        return this;
    }
    /**
     * Updates the definition for a dimension.
     *
     * Access Level: Write
     */
    updateDimension() {
        this.add('iot:UpdateDimension');
        return this;
    }
    /**
     * Updates a Dynamic Thing Group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateDynamicThingGroup.html
     */
    updateDynamicThingGroup() {
        this.add('iot:UpdateDynamicThingGroup');
        return this;
    }
    /**
     * Updates event configurations.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateEventConfigurations.html
     */
    updateEventConfigurations() {
        this.add('iot:UpdateEventConfigurations');
        return this;
    }
    /**
     * Updates fleet indexing configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateIndexingConfiguration.html
     */
    updateIndexingConfiguration() {
        this.add('iot:UpdateIndexingConfiguration');
        return this;
    }
    /**
     * Updates a job.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJob.html
     */
    updateJob() {
        this.add('iot:UpdateJob');
        return this;
    }
    /**
     * Updates a job execution.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateJobExecution.html
     */
    updateJobExecution() {
        this.add('iot:UpdateJobExecution');
        return this;
    }
    /**
     * Updates the definition for the specified mitigation action.
     *
     * Access Level: Write
     */
    updateMitigationAction() {
        this.add('iot:UpdateMitigationAction');
        return this;
    }
    /**
     * Updates a fleet provisioning template.
     *
     * Access Level: Write
     */
    updateProvisioningTemplate() {
        this.add('iot:UpdateProvisioningTemplate');
        return this;
    }
    /**
     * Updates the role alias
     *
     * Access Level: Write
     */
    updateRoleAlias() {
        this.add('iot:UpdateRoleAlias');
        return this;
    }
    /**
     * Updates a scheduled audit, including what checks are performed and how often the audit takes place.
     *
     * Access Level: Write
     */
    updateScheduledAudit() {
        this.add('iot:UpdateScheduledAudit');
        return this;
    }
    /**
     * Updates a Device Defender security profile.
     *
     * Access Level: Write
     */
    updateSecurityProfile() {
        this.add('iot:UpdateSecurityProfile');
        return this;
    }
    /**
     * Updates the data for a stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateStream.html
     */
    updateStream() {
        this.add('iot:UpdateStream');
        return this;
    }
    /**
     * Updates information associated with the specified thing.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThing.html
     */
    updateThing() {
        this.add('iot:UpdateThing');
        return this;
    }
    /**
     * Updates information associated with the specified thing group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroup.html
     */
    updateThingGroup() {
        this.add('iot:UpdateThingGroup');
        return this;
    }
    /**
     * Updates the thing groups to which the thing belongs.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/apireference/API_UpdateThingGroupsForThing.html
     */
    updateThingGroupsForThing() {
        this.add('iot:UpdateThingGroupsForThing');
        return this;
    }
    /**
     * Updates the thing shadow.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/policy-actions.html
     */
    updateThingShadow() {
        this.add('iot:UpdateThingShadow');
        return this;
    }
    /**
     * Validates a Device Defender security profile behaviors specification.
     *
     * Access Level: Read
     */
    validateSecurityProfileBehaviors() {
        this.add('iot:ValidateSecurityProfileBehaviors');
        return this;
    }
    /**
     * Adds a resource of type client to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html
     *
     * @param clientId - Identifier for the clientId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onClient(clientId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:client/${ClientId}';
        arn = arn.replace('${ClientId}', clientId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-indexing.html
     *
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIndex(indexName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:index/${IndexName}';
        arn = arn.replace('${IndexName}', indexName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tunnel to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-tunnels.html
     *
     * @param tunnelId - Identifier for the tunnelId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onTunnel(tunnelId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:tunnel/${TunnelId}';
        arn = arn.replace('${TunnelId}', tunnelId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thing to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-registry.html
     *
     * @param thingName - Identifier for the thingName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onThing(thingName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thing/${ThingName}';
        arn = arn.replace('${ThingName}', thingName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-groups.html
     *
     * @param thingGroupName - Identifier for the thingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onThinggroup(thingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}';
        arn = arn.replace('${ThingGroupName}', thingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type billinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/billing-groups.html
     *
     * @param billingGroupName - Identifier for the billingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onBillinggroup(billingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:billinggroup/${BillingGroupName}';
        arn = arn.replace('${BillingGroupName}', billingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dynamicthinggroup to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/dynamic-thing-groups.html
     *
     * @param thingGroupName - Identifier for the thingGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDynamicthinggroup(thingGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thinggroup/${ThingGroupName}';
        arn = arn.replace('${ThingGroupName}', thingGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type thingtype to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/thing-types.html
     *
     * @param thingTypeName - Identifier for the thingTypeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onThingtype(thingTypeName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:thingtype/${ThingTypeName}';
        arn = arn.replace('${ThingTypeName}', thingTypeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-message-broker.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopic(topicName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:topic/${TopicName}';
        arn = arn.replace('${TopicName}', topicName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type topicfilter to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/topics.html
     *
     * @param topicFilter - Identifier for the topicFilter.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTopicfilter(topicFilter, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:topicfilter/${TopicFilter}';
        arn = arn.replace('${TopicFilter}', topicFilter);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rolealias to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/authorizing-direct-aws.html
     *
     * @param roleAlias - Identifier for the roleAlias.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRolealias(roleAlias, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:rolealias/${RoleAlias}';
        arn = arn.replace('${RoleAlias}', roleAlias);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type authorizer to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/custom-authorizer.html
     *
     * @param authorizerName - Identifier for the authorizerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAuthorizer(authorizerName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:authorizer/${AuthorizerName}';
        arn = arn.replace('${AuthorizerName}', authorizerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type policy to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-policies.html
     *
     * @param policyName - Identifier for the policyName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPolicy(policyName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:policy/${PolicyName}';
        arn = arn.replace('${PolicyName}', policyName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cert to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html
     *
     * @param certificate - Identifier for the certificate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCert(certificate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:cert/${Certificate}';
        arn = arn.replace('${Certificate}', certificate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cacert to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/x509-certs.html
     *
     * @param cACertificate - Identifier for the cACertificate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCacert(cACertificate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:cacert/${CACertificate}';
        arn = arn.replace('${CACertificate}', cACertificate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html
     *
     * @param streamId - Identifier for the streamId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onStream(streamId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:stream/${StreamId}';
        arn = arn.replace('${StreamId}', streamId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type otaupdate to the statement
     *
     * https://docs.aws.amazon.com/freertos/latest/userguide/freertos-ota-dev.html
     *
     * @param otaUpdateId - Identifier for the otaUpdateId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onOtaupdate(otaUpdateId, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:otaupdate/${OtaUpdateId}';
        arn = arn.replace('${OtaUpdateId}', otaUpdateId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type scheduledaudit to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-audit.html
     *
     * @param scheduleName - Identifier for the scheduleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onScheduledaudit(scheduleName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:scheduledaudit/${ScheduleName}';
        arn = arn.replace('${ScheduleName}', scheduleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mitigationaction to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-mitigation-actions.html
     *
     * @param mitigationActionName - Identifier for the mitigationActionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onMitigationaction(mitigationActionName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:mitigationaction/${MitigationActionName}';
        arn = arn.replace('${MitigationActionName}', mitigationActionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securityprofile to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html
     *
     * @param securityProfileName - Identifier for the securityProfileName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onSecurityprofile(securityProfileName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:securityprofile/${SecurityProfileName}';
        arn = arn.replace('${SecurityProfileName}', securityProfileName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type dimension to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/device-defender-detect.html
     *
     * @param dimensionName - Identifier for the dimensionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onDimension(dimensionName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:dimension/${DimensionName}';
        arn = arn.replace('${DimensionName}', dimensionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/iot-rules.html
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRule(ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:rule/${RuleName}';
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type provisioningtemplate to the statement
     *
     * https://docs.aws.amazon.com/iot/latest/developerguide/provision-template.html
     *
     * @param provisioningTemplate - Identifier for the provisioningTemplate.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProvisioningtemplate(provisioningTemplate, account, region, partition) {
        var arn = 'arn:${Partition}:iot:${Region}:${Account}:provisioningtemplate/${ProvisioningTemplate}';
        arn = arn.replace('${ProvisioningTemplate}', provisioningTemplate);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The flag indicating whether or not to also delete an IoT Tunnel immediately
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifDelete(value) {
        return this.if(`iot:Delete`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The list of all IoT Thing Group ARNs that the destination IoT Thing belongs to for an IoT Tunnel
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifThingGroupArn(value, operator) {
        return this.if(`iot:ThingGroupArn`, value, operator || 'StringLike');
    }
    /**
     * The list of all destination services for an IoT Tunnel
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTunnelDestinationService(value, operator) {
        return this.if(`iot:TunnelDestinationService`, value, operator || 'StringLike');
    }
}
exports.Iot = Iot;
//# sourceMappingURL=data:application/json;base64,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