"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mgh = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mgh](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmigrationhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mgh extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mgh](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsmigrationhub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mgh';
        this.actionList = {
            "AssociateCreatedArtifact": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateCreatedArtifact.html",
                "description": "Associate a given AWS artifact to a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "AssociateDiscoveredResource": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateDiscoveredResource.html",
                "description": "Associate a given ADS resource to a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "CreateHomeRegionControl": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateHomeRegionControl.html",
                "description": "Create a Migration Hub Home Region Control",
                "accessLevel": "Write"
            },
            "CreateProgressUpdateStream": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateProgressUpdateStream.html",
                "description": "Create a ProgressUpdateStream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "progressUpdateStream": {
                        "required": true
                    }
                }
            },
            "DeleteProgressUpdateStream": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteProgressUpdateStream.html",
                "description": "Delete a ProgressUpdateStream",
                "accessLevel": "Write",
                "resourceTypes": {
                    "progressUpdateStream": {
                        "required": true
                    }
                }
            },
            "DescribeApplicationState": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeApplicationState.html",
                "description": "Get an Application Discovery Service Application's state",
                "accessLevel": "Read"
            },
            "DescribeHomeRegionControls": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeHomeRegionControls.html",
                "description": "List Home Region Controls",
                "accessLevel": "List"
            },
            "DescribeMigrationTask": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeMigrationTask.html",
                "description": "Describe a MigrationTask",
                "accessLevel": "Read",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "DisassociateCreatedArtifact": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateCreatedArtifact.html",
                "description": "Disassociate a given AWS artifact from a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "DisassociateDiscoveredResource": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateDiscoveredResource.html",
                "description": "Disassociate a given ADS resource from a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "GetHomeRegion": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_GetHomeRegion.html",
                "description": "Get the Migration Hub Home Region",
                "accessLevel": "Read"
            },
            "ImportMigrationTask": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ImportMigrationTask.html",
                "description": "Import a MigrationTask",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "ListCreatedArtifacts": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListCreatedArtifacts.html",
                "description": "List associated created artifacts for a MigrationTask",
                "accessLevel": "List",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "ListDiscoveredResources": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListDiscoveredResources.html",
                "description": "List associated ADS resources from MigrationTask",
                "accessLevel": "List",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "ListMigrationTasks": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListMigrationTasks.html",
                "description": "List MigrationTasks",
                "accessLevel": "List"
            },
            "ListProgressUpdateStreams": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListProgressUpdateStreams.html",
                "description": "List ProgressUpdateStreams",
                "accessLevel": "List"
            },
            "NotifyApplicationState": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyApplicationState.html",
                "description": "Update an Application Discovery Service Application's state",
                "accessLevel": "Write"
            },
            "NotifyMigrationTaskState": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyMigrationTaskState.html",
                "description": "Notify latest MigrationTask state",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            },
            "PutResourceAttributes": {
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html",
                "description": "Put ResourceAttributes",
                "accessLevel": "Write",
                "resourceTypes": {
                    "migrationTask": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "progressUpdateStream": {
                "name": "progressUpdateStream",
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_ProgressUpdateStreamSummary.html",
                "arn": "arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}",
                "conditionKeys": []
            },
            "migrationTask": {
                "name": "migrationTask",
                "url": "https://docs.aws.amazon.com/migrationhub/latest/ug/API_MigrationTask.html",
                "arn": "arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}/migrationTask/${Task}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Associate a given AWS artifact to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateCreatedArtifact.html
     */
    associateCreatedArtifact() {
        this.add('mgh:AssociateCreatedArtifact');
        return this;
    }
    /**
     * Associate a given ADS resource to a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_AssociateDiscoveredResource.html
     */
    associateDiscoveredResource() {
        this.add('mgh:AssociateDiscoveredResource');
        return this;
    }
    /**
     * Create a Migration Hub Home Region Control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateHomeRegionControl.html
     */
    createHomeRegionControl() {
        this.add('mgh:CreateHomeRegionControl');
        return this;
    }
    /**
     * Create a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_CreateProgressUpdateStream.html
     */
    createProgressUpdateStream() {
        this.add('mgh:CreateProgressUpdateStream');
        return this;
    }
    /**
     * Delete a ProgressUpdateStream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DeleteProgressUpdateStream.html
     */
    deleteProgressUpdateStream() {
        this.add('mgh:DeleteProgressUpdateStream');
        return this;
    }
    /**
     * Get an Application Discovery Service Application's state
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeApplicationState.html
     */
    describeApplicationState() {
        this.add('mgh:DescribeApplicationState');
        return this;
    }
    /**
     * List Home Region Controls
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeHomeRegionControls.html
     */
    describeHomeRegionControls() {
        this.add('mgh:DescribeHomeRegionControls');
        return this;
    }
    /**
     * Describe a MigrationTask
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DescribeMigrationTask.html
     */
    describeMigrationTask() {
        this.add('mgh:DescribeMigrationTask');
        return this;
    }
    /**
     * Disassociate a given AWS artifact from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateCreatedArtifact.html
     */
    disassociateCreatedArtifact() {
        this.add('mgh:DisassociateCreatedArtifact');
        return this;
    }
    /**
     * Disassociate a given ADS resource from a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_DisassociateDiscoveredResource.html
     */
    disassociateDiscoveredResource() {
        this.add('mgh:DisassociateDiscoveredResource');
        return this;
    }
    /**
     * Get the Migration Hub Home Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_GetHomeRegion.html
     */
    getHomeRegion() {
        this.add('mgh:GetHomeRegion');
        return this;
    }
    /**
     * Import a MigrationTask
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ImportMigrationTask.html
     */
    importMigrationTask() {
        this.add('mgh:ImportMigrationTask');
        return this;
    }
    /**
     * List associated created artifacts for a MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListCreatedArtifacts.html
     */
    listCreatedArtifacts() {
        this.add('mgh:ListCreatedArtifacts');
        return this;
    }
    /**
     * List associated ADS resources from MigrationTask
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListDiscoveredResources.html
     */
    listDiscoveredResources() {
        this.add('mgh:ListDiscoveredResources');
        return this;
    }
    /**
     * List MigrationTasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListMigrationTasks.html
     */
    listMigrationTasks() {
        this.add('mgh:ListMigrationTasks');
        return this;
    }
    /**
     * List ProgressUpdateStreams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ListProgressUpdateStreams.html
     */
    listProgressUpdateStreams() {
        this.add('mgh:ListProgressUpdateStreams');
        return this;
    }
    /**
     * Update an Application Discovery Service Application's state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyApplicationState.html
     */
    notifyApplicationState() {
        this.add('mgh:NotifyApplicationState');
        return this;
    }
    /**
     * Notify latest MigrationTask state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_NotifyMigrationTaskState.html
     */
    notifyMigrationTaskState() {
        this.add('mgh:NotifyMigrationTaskState');
        return this;
    }
    /**
     * Put ResourceAttributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_PutResourceAttributes.html
     */
    putResourceAttributes() {
        this.add('mgh:PutResourceAttributes');
        return this;
    }
    /**
     * Adds a resource of type progressUpdateStream to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_ProgressUpdateStreamSummary.html
     *
     * @param stream - Identifier for the stream.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProgressUpdateStream(stream, account, region, partition) {
        var arn = 'arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}';
        arn = arn.replace('${Stream}', stream);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type migrationTask to the statement
     *
     * https://docs.aws.amazon.com/migrationhub/latest/ug/API_MigrationTask.html
     *
     * @param stream - Identifier for the stream.
     * @param task - Identifier for the task.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onMigrationTask(stream, task, account, region, partition) {
        var arn = 'arn:${Partition}:mgh:${Region}:${Account}:progressUpdateStream/${Stream}/migrationTask/${Task}';
        arn = arn.replace('${Stream}', stream);
        arn = arn.replace('${Task}', task);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mgh = Mgh;
//# sourceMappingURL=data:application/json;base64,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