"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Securityhub = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [securityhub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecurityhub.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Securityhub extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [securityhub](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssecurityhub.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'securityhub';
        this.actionList = {
            "AcceptInvitation": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptInvitation.html",
                "description": "Grants permission to accept Security Hub invitations to become a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "BatchDisableStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchDisableStandards.html",
                "description": "Grants permission to disable standards in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "BatchEnableStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchEnableStandards.html",
                "description": "Grants permission to enable standards in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "BatchImportFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html",
                "description": "Grants permission to import findings into Security Hub from an integrated product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "product": {
                        "required": true
                    }
                },
                "conditions": [
                    "securityhub:TargetAccount"
                ]
            },
            "BatchUpdateFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html",
                "description": "Grants permission to update customer-controlled fields for a selected set of Security Hub findings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "CreateActionTarget": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateActionTarget.html",
                "description": "Grants permission to create custom actions in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "CreateInsight": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateInsight.html",
                "description": "Grants permission to create insights in Security Hub. Insights are collections of related findings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "CreateMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateMembers.html",
                "description": "Grants permission to create member accounts in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeclineInvitations": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeclineInvitations.html",
                "description": "Grants permission to decline Security Hub invitations to become a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteActionTarget": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteActionTarget.html",
                "description": "Grants permission to delete custom actions in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteInsight": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInsight.html",
                "description": "Grants permission to delete insights from Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteInvitations": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInvitations.html",
                "description": "Grants permission to delete Security Hub invitations to become a member account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DeleteMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteMembers.html",
                "description": "Grants permission to delete Security Hub member accounts",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeActionTargets": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeActionTargets.html",
                "description": "Grants permission to retrieve a list of custom actions using the API",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeHub": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeHub.html",
                "description": "Grants permission to retrieve information about the hub resource in your account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeProducts": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProducts.html",
                "description": "Grants permission to retrieve information about the available Security Hub product integrations",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html",
                "description": "Grants permission to retrieve information about Security Hub standards",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DescribeStandardsControls": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandardsControls.html",
                "description": "Grants permission to retrieve information about Security Hub standards controls",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisableImportFindingsForProduct": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableImportFindingsForProduct.html",
                "description": "Grants permission to disable the findings importing for a Security Hub integrated product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisableSecurityHub": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHub.html",
                "description": "Grants permission to disable Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisassociateFromMasterAccount": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromMasterAccount.html",
                "description": "Grants permission to a Security Hub member account to disassociate from the associated master account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "DisassociateMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateMembers.html",
                "description": "Grants permission to disassociate Security Hub member accounts from the associated master account",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "EnableImportFindingsForProduct": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableImportFindingsForProduct.html",
                "description": "Grants permission to enable the findings importing for a Security Hub integrated product",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "EnableSecurityHub": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHub.html",
                "description": "Grants permission to enable Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "GetEnabledStandards": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetEnabledStandards.html",
                "description": "Grants permission to retrieve a list of the standards that are enabled in Security Hub",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindings.html",
                "description": "Grants permission to retrieve a list of findings from Security Hub",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetInsightResults": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightResults.html",
                "description": "Grants permission to retrieve insight results from Security Hub",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetInsights": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsights.html",
                "description": "Grants permission to retrieve Security Hub insights",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetInvitationsCount": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInvitationsCount.html",
                "description": "Grants permission to retrieve the count of Security Hub membership invitations sent to the account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetMasterAccount": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMasterAccount.html",
                "description": "Grants permission to retrieve details about the Security Hub master account",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "GetMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMembers.html",
                "description": "Grants permission to retrieve the details of Security Hub member accounts",
                "accessLevel": "Read",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "InviteMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_InviteMembers.html",
                "description": "Grants permission to invite other AWS accounts to become Security Hub member accounts",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListEnabledProductsForImport": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListEnabledProductsForImport.html",
                "description": "Grants permission to retrieve the Security Hub integrated products that are currently enabled",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListInvitations": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListInvitations.html",
                "description": "Grants permission to retrieve the Security Hub invitations sent to the account",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListMembers": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListMembers.html",
                "description": "Grants permission to retrieve details about Security Hub member accounts associated with the master account",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListTagsForResource.html",
                "description": "Grants permission to list of tags associated with a resource",
                "accessLevel": "List",
                "resourceTypes": {
                    "hub": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_TagResource.html",
                "description": "Grants permission to add tags to a Security Hub resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": true
                    }
                }
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UntagResource.html",
                "description": "Grants permission to remove tags from a Security Hub resource",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": true
                    }
                }
            },
            "UpdateActionTarget": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateActionTarget.html",
                "description": "Grants permission to update custom actions in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "UpdateFindings": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindings.html",
                "description": "Grants permission to update Security Hub findings",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "UpdateInsight": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateInsight.html",
                "description": "Grants permission to update insights in Security Hub",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            },
            "UpdateStandardsControl": {
                "url": "https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html",
                "description": "Grants permission to update Security Hub standards controls",
                "accessLevel": "Write",
                "resourceTypes": {
                    "hub": {
                        "required": false
                    }
                }
            }
        };
        this.resourceTypes = {
            "product": {
                "name": "product",
                "url": "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources",
                "arn": "arn:${Partition}:securityhub:${Region}:${Account}:product/${Company}/${ProductId}",
                "conditionKeys": []
            },
            "hub": {
                "name": "hub",
                "url": "https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources",
                "arn": "arn:${Partition}:securityhub:${Region}:${Account}:hub/default",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Grants permission to accept Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_AcceptInvitation.html
     */
    acceptInvitation() {
        this.add('securityhub:AcceptInvitation');
        return this;
    }
    /**
     * Grants permission to disable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchDisableStandards.html
     */
    batchDisableStandards() {
        this.add('securityhub:BatchDisableStandards');
        return this;
    }
    /**
     * Grants permission to enable standards in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchEnableStandards.html
     */
    batchEnableStandards() {
        this.add('securityhub:BatchEnableStandards');
        return this;
    }
    /**
     * Grants permission to import findings into Security Hub from an integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchImportFindings.html
     */
    batchImportFindings() {
        this.add('securityhub:BatchImportFindings');
        return this;
    }
    /**
     * Grants permission to update customer-controlled fields for a selected set of Security Hub findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_BatchUpdateFindings.html
     */
    batchUpdateFindings() {
        this.add('securityhub:BatchUpdateFindings');
        return this;
    }
    /**
     * Grants permission to create custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateActionTarget.html
     */
    createActionTarget() {
        this.add('securityhub:CreateActionTarget');
        return this;
    }
    /**
     * Grants permission to create insights in Security Hub. Insights are collections of related findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateInsight.html
     */
    createInsight() {
        this.add('securityhub:CreateInsight');
        return this;
    }
    /**
     * Grants permission to create member accounts in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_CreateMembers.html
     */
    createMembers() {
        this.add('securityhub:CreateMembers');
        return this;
    }
    /**
     * Grants permission to decline Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeclineInvitations.html
     */
    declineInvitations() {
        this.add('securityhub:DeclineInvitations');
        return this;
    }
    /**
     * Grants permission to delete custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteActionTarget.html
     */
    deleteActionTarget() {
        this.add('securityhub:DeleteActionTarget');
        return this;
    }
    /**
     * Grants permission to delete insights from Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInsight.html
     */
    deleteInsight() {
        this.add('securityhub:DeleteInsight');
        return this;
    }
    /**
     * Grants permission to delete Security Hub invitations to become a member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteInvitations.html
     */
    deleteInvitations() {
        this.add('securityhub:DeleteInvitations');
        return this;
    }
    /**
     * Grants permission to delete Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DeleteMembers.html
     */
    deleteMembers() {
        this.add('securityhub:DeleteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of custom actions using the API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeActionTargets.html
     */
    describeActionTargets() {
        this.add('securityhub:DescribeActionTargets');
        return this;
    }
    /**
     * Grants permission to retrieve information about the hub resource in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeHub.html
     */
    describeHub() {
        this.add('securityhub:DescribeHub');
        return this;
    }
    /**
     * Grants permission to retrieve information about the available Security Hub product integrations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeProducts.html
     */
    describeProducts() {
        this.add('securityhub:DescribeProducts');
        return this;
    }
    /**
     * Grants permission to retrieve information about Security Hub standards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html
     */
    describeStandards() {
        this.add('securityhub:DescribeStandards');
        return this;
    }
    /**
     * Grants permission to retrieve information about Security Hub standards controls
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandardsControls.html
     */
    describeStandardsControls() {
        this.add('securityhub:DescribeStandardsControls');
        return this;
    }
    /**
     * Grants permission to disable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableImportFindingsForProduct.html
     */
    disableImportFindingsForProduct() {
        this.add('securityhub:DisableImportFindingsForProduct');
        return this;
    }
    /**
     * Grants permission to disable Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisableSecurityHub.html
     */
    disableSecurityHub() {
        this.add('securityhub:DisableSecurityHub');
        return this;
    }
    /**
     * Grants permission to a Security Hub member account to disassociate from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateFromMasterAccount.html
     */
    disassociateFromMasterAccount() {
        this.add('securityhub:DisassociateFromMasterAccount');
        return this;
    }
    /**
     * Grants permission to disassociate Security Hub member accounts from the associated master account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DisassociateMembers.html
     */
    disassociateMembers() {
        this.add('securityhub:DisassociateMembers');
        return this;
    }
    /**
     * Grants permission to enable the findings importing for a Security Hub integrated product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableImportFindingsForProduct.html
     */
    enableImportFindingsForProduct() {
        this.add('securityhub:EnableImportFindingsForProduct');
        return this;
    }
    /**
     * Grants permission to enable Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_EnableSecurityHub.html
     */
    enableSecurityHub() {
        this.add('securityhub:EnableSecurityHub');
        return this;
    }
    /**
     * Grants permission to retrieve a list of the standards that are enabled in Security Hub
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetEnabledStandards.html
     */
    getEnabledStandards() {
        this.add('securityhub:GetEnabledStandards');
        return this;
    }
    /**
     * Grants permission to retrieve a list of findings from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetFindings.html
     */
    getFindings() {
        this.add('securityhub:GetFindings');
        return this;
    }
    /**
     * Grants permission to retrieve insight results from Security Hub
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsightResults.html
     */
    getInsightResults() {
        this.add('securityhub:GetInsightResults');
        return this;
    }
    /**
     * Grants permission to retrieve Security Hub insights
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInsights.html
     */
    getInsights() {
        this.add('securityhub:GetInsights');
        return this;
    }
    /**
     * Grants permission to retrieve the count of Security Hub membership invitations sent to the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetInvitationsCount.html
     */
    getInvitationsCount() {
        this.add('securityhub:GetInvitationsCount');
        return this;
    }
    /**
     * Grants permission to retrieve details about the Security Hub master account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMasterAccount.html
     */
    getMasterAccount() {
        this.add('securityhub:GetMasterAccount');
        return this;
    }
    /**
     * Grants permission to retrieve the details of Security Hub member accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_GetMembers.html
     */
    getMembers() {
        this.add('securityhub:GetMembers');
        return this;
    }
    /**
     * Grants permission to invite other AWS accounts to become Security Hub member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_InviteMembers.html
     */
    inviteMembers() {
        this.add('securityhub:InviteMembers');
        return this;
    }
    /**
     * Grants permission to retrieve the Security Hub integrated products that are currently enabled
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListEnabledProductsForImport.html
     */
    listEnabledProductsForImport() {
        this.add('securityhub:ListEnabledProductsForImport');
        return this;
    }
    /**
     * Grants permission to retrieve the Security Hub invitations sent to the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListInvitations.html
     */
    listInvitations() {
        this.add('securityhub:ListInvitations');
        return this;
    }
    /**
     * Grants permission to retrieve details about Security Hub member accounts associated with the master account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListMembers.html
     */
    listMembers() {
        this.add('securityhub:ListMembers');
        return this;
    }
    /**
     * Grants permission to list of tags associated with a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('securityhub:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to add tags to a Security Hub resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_TagResource.html
     */
    tagResource() {
        this.add('securityhub:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags from a Security Hub resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UntagResource.html
     */
    untagResource() {
        this.add('securityhub:UntagResource');
        return this;
    }
    /**
     * Grants permission to update custom actions in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateActionTarget.html
     */
    updateActionTarget() {
        this.add('securityhub:UpdateActionTarget');
        return this;
    }
    /**
     * Grants permission to update Security Hub findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateFindings.html
     */
    updateFindings() {
        this.add('securityhub:UpdateFindings');
        return this;
    }
    /**
     * Grants permission to update insights in Security Hub
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateInsight.html
     */
    updateInsight() {
        this.add('securityhub:UpdateInsight');
        return this;
    }
    /**
     * Grants permission to update Security Hub standards controls
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_UpdateStandardsControl.html
     */
    updateStandardsControl() {
        this.add('securityhub:UpdateStandardsControl');
        return this;
    }
    /**
     * Adds a resource of type product to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param company - Identifier for the company.
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProduct(company, productId, account, region, partition) {
        var arn = 'arn:${Partition}:securityhub:${Region}:${Account}:product/${Company}/${ProductId}';
        arn = arn.replace('${Company}', company);
        arn = arn.replace('${ProductId}', productId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type hub to the statement
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#resources
     *
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onHub(account, region, partition) {
        var arn = 'arn:${Partition}:securityhub:${Region}:${Account}:hub/default';
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The ID of the AWS account into which you want to import findings. In the AWS Security Finding format, this field is called AwsAccountId
     *
     * https://docs.aws.amazon.com/securityhub/latest/userguide/securityhub-access.html#conditions
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetAccount(value, operator) {
        return this.if(`securityhub:TargetAccount`, value, operator || 'StringLike');
    }
}
exports.Securityhub = Securityhub;
//# sourceMappingURL=data:application/json;base64,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