"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsoDirectory = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sso-directory](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsssodirectory.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SsoDirectory extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sso-directory](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsssodirectory.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sso-directory';
        this.actionList = {
            "AddMemberToGroup": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Adds member to the group in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "CompleteVirtualMfaDeviceRegistration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Completes the creation process of a virtual MFA device",
                "accessLevel": "Write"
            },
            "CreateAlias": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Creates an alias for the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "CreateBearerToken": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Creates a bearer token for a given provisioning tenant.",
                "accessLevel": "Write"
            },
            "CreateExternalIdPConfigurationForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Create an External Identity Provider configuration for the directory",
                "accessLevel": "Write"
            },
            "CreateGroup": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Creates a group in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "CreateProvisioningTenant": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Creates a provisioning tenant for a given directory.",
                "accessLevel": "Write"
            },
            "CreateUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Creates a user in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "DeleteBearerToken": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deletes the bearer token.",
                "accessLevel": "Write"
            },
            "DeleteExternalIdPConfigurationForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Delete an External Identity Provider configuration associated with the directory",
                "accessLevel": "Write"
            },
            "DeleteGroup": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deletes a group from the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "DeleteMfaDeviceForUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deletes a MFA device by device name for a given user",
                "accessLevel": "Write"
            },
            "DeleteProvisioningTenant": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deletes the provisioning tenant.",
                "accessLevel": "Write"
            },
            "DeleteUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deletes a user from the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "DescribeDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve information about the directory that AWS SSO provides by default",
                "accessLevel": "Read"
            },
            "DescribeGroups": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieves information about group from the directory that AWS SSO provides by default",
                "accessLevel": "List"
            },
            "DescribeUsers": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieves information about user from the directory that AWS SSO provides by default",
                "accessLevel": "List"
            },
            "DisableExternalIdPConfigurationForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Disable authentication of end users with an External Identity Provider",
                "accessLevel": "Write"
            },
            "DisableUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Deactivates user in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "EnableExternalIdPConfigurationForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Enable authentication of end users with an External Identity Provider",
                "accessLevel": "Write"
            },
            "EnableUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Activates user in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "GetAWSSPConfigurationForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieve the AWS SSO Service Provider configurations for the directory",
                "accessLevel": "Read"
            },
            "ListBearerTokens": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists bearer tokens for a given provisioning tenant.",
                "accessLevel": "List"
            },
            "ListExternalIdPConfigurationsForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "List all the External Identity Provider configurations created for the directory",
                "accessLevel": "List"
            },
            "ListGroupsForUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists groups for a user from the directory that AWS SSO provides by default",
                "accessLevel": "List"
            },
            "ListMembersInGroup": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Retrieves all members that are part of the group in the directory that AWS SSO provides by default",
                "accessLevel": "List"
            },
            "ListMfaDevicesForUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists all active MFA devices and their MFA device metadata for a user",
                "accessLevel": "List"
            },
            "ListProvisioningTenants": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Lists provisioning tenants for a given directory.",
                "accessLevel": "List"
            },
            "RemoveMemberFromGroup": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Removes member that are part of the group in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "SearchGroups": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Search for groups within the associated directory",
                "accessLevel": "Read"
            },
            "SearchUsers": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Search for users within the associated directory",
                "accessLevel": "Read"
            },
            "StartVirtualMfaDeviceRegistration": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Begins the creation process of virtual mfa device",
                "accessLevel": "Write"
            },
            "UpdateExternalIdPConfigurationForDirectory": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Update an External Identity Provider configuration associated with the directory",
                "accessLevel": "Write"
            },
            "UpdateGroup": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Updates information about group in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "UpdatePassword": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Updates password by sending password reset link via email or generating one time password for a user in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "UpdateUser": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Updates user information in the directory that AWS SSO provides by default",
                "accessLevel": "Write"
            },
            "VerifyEmail": {
                "url": "https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample",
                "description": "Verify email address of an User",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Adds member to the group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    addMemberToGroup() {
        this.add('sso-directory:AddMemberToGroup');
        return this;
    }
    /**
     * Completes the creation process of a virtual MFA device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    completeVirtualMfaDeviceRegistration() {
        this.add('sso-directory:CompleteVirtualMfaDeviceRegistration');
        return this;
    }
    /**
     * Creates an alias for the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    createAlias() {
        this.add('sso-directory:CreateAlias');
        return this;
    }
    /**
     * Creates a bearer token for a given provisioning tenant.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    createBearerToken() {
        this.add('sso-directory:CreateBearerToken');
        return this;
    }
    /**
     * Create an External Identity Provider configuration for the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    createExternalIdPConfigurationForDirectory() {
        this.add('sso-directory:CreateExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Creates a group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    createGroup() {
        this.add('sso-directory:CreateGroup');
        return this;
    }
    /**
     * Creates a provisioning tenant for a given directory.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    createProvisioningTenant() {
        this.add('sso-directory:CreateProvisioningTenant');
        return this;
    }
    /**
     * Creates a user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    createUser() {
        this.add('sso-directory:CreateUser');
        return this;
    }
    /**
     * Deletes the bearer token.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    deleteBearerToken() {
        this.add('sso-directory:DeleteBearerToken');
        return this;
    }
    /**
     * Delete an External Identity Provider configuration associated with the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    deleteExternalIdPConfigurationForDirectory() {
        this.add('sso-directory:DeleteExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Deletes a group from the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    deleteGroup() {
        this.add('sso-directory:DeleteGroup');
        return this;
    }
    /**
     * Deletes a MFA device by device name for a given user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    deleteMfaDeviceForUser() {
        this.add('sso-directory:DeleteMfaDeviceForUser');
        return this;
    }
    /**
     * Deletes the provisioning tenant.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    deleteProvisioningTenant() {
        this.add('sso-directory:DeleteProvisioningTenant');
        return this;
    }
    /**
     * Deletes a user from the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    deleteUser() {
        this.add('sso-directory:DeleteUser');
        return this;
    }
    /**
     * Retrieve information about the directory that AWS SSO provides by default
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    describeDirectory() {
        this.add('sso-directory:DescribeDirectory');
        return this;
    }
    /**
     * Retrieves information about group from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    describeGroups() {
        this.add('sso-directory:DescribeGroups');
        return this;
    }
    /**
     * Retrieves information about user from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    describeUsers() {
        this.add('sso-directory:DescribeUsers');
        return this;
    }
    /**
     * Disable authentication of end users with an External Identity Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    disableExternalIdPConfigurationForDirectory() {
        this.add('sso-directory:DisableExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Deactivates user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    disableUser() {
        this.add('sso-directory:DisableUser');
        return this;
    }
    /**
     * Enable authentication of end users with an External Identity Provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    enableExternalIdPConfigurationForDirectory() {
        this.add('sso-directory:EnableExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Activates user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    enableUser() {
        this.add('sso-directory:EnableUser');
        return this;
    }
    /**
     * Retrieve the AWS SSO Service Provider configurations for the directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    getAWSSPConfigurationForDirectory() {
        this.add('sso-directory:GetAWSSPConfigurationForDirectory');
        return this;
    }
    /**
     * Lists bearer tokens for a given provisioning tenant.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    listBearerTokens() {
        this.add('sso-directory:ListBearerTokens');
        return this;
    }
    /**
     * List all the External Identity Provider configurations created for the directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    listExternalIdPConfigurationsForDirectory() {
        this.add('sso-directory:ListExternalIdPConfigurationsForDirectory');
        return this;
    }
    /**
     * Lists groups for a user from the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    listGroupsForUser() {
        this.add('sso-directory:ListGroupsForUser');
        return this;
    }
    /**
     * Retrieves all members that are part of the group in the directory that AWS SSO provides by default
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    listMembersInGroup() {
        this.add('sso-directory:ListMembersInGroup');
        return this;
    }
    /**
     * Lists all active MFA devices and their MFA device metadata for a user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    listMfaDevicesForUser() {
        this.add('sso-directory:ListMfaDevicesForUser');
        return this;
    }
    /**
     * Lists provisioning tenants for a given directory.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    listProvisioningTenants() {
        this.add('sso-directory:ListProvisioningTenants');
        return this;
    }
    /**
     * Removes member that are part of the group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    removeMemberFromGroup() {
        this.add('sso-directory:RemoveMemberFromGroup');
        return this;
    }
    /**
     * Search for groups within the associated directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    searchGroups() {
        this.add('sso-directory:SearchGroups');
        return this;
    }
    /**
     * Search for users within the associated directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    searchUsers() {
        this.add('sso-directory:SearchUsers');
        return this;
    }
    /**
     * Begins the creation process of virtual mfa device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    startVirtualMfaDeviceRegistration() {
        this.add('sso-directory:StartVirtualMfaDeviceRegistration');
        return this;
    }
    /**
     * Update an External Identity Provider configuration associated with the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    updateExternalIdPConfigurationForDirectory() {
        this.add('sso-directory:UpdateExternalIdPConfigurationForDirectory');
        return this;
    }
    /**
     * Updates information about group in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    updateGroup() {
        this.add('sso-directory:UpdateGroup');
        return this;
    }
    /**
     * Updates password by sending password reset link via email or generating one time password for a user in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    updatePassword() {
        this.add('sso-directory:UpdatePassword');
        return this;
    }
    /**
     * Updates user information in the directory that AWS SSO provides by default
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    updateUser() {
        this.add('sso-directory:UpdateUser');
        return this;
    }
    /**
     * Verify email address of an User
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    verifyEmail() {
        this.add('sso-directory:VerifyEmail');
        return this;
    }
}
exports.SsoDirectory = SsoDirectory;
//# sourceMappingURL=data:application/json;base64,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