"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Trustedadvisor = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Trustedadvisor extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'trustedadvisor';
        this.actionList = {
            "DescribeAccount": {
                "url": "",
                "description": "View support plan and various TA preferences.",
                "accessLevel": "Read"
            },
            "DescribeAccountAccess": {
                "url": "",
                "description": "Resolve whether Account has disabled Trusted Advisor",
                "accessLevel": "Read"
            },
            "DescribeCheckItems": {
                "url": "",
                "description": "View details for the check items",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckRefreshStatuses": {
                "url": "",
                "description": "Describe check refresh statuses",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeCheckSummaries": {
                "url": "",
                "description": "Describes the check's summaries",
                "accessLevel": "Read",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "DescribeChecks": {
                "url": "",
                "description": "List valid Trusted Advisor checks and details.",
                "accessLevel": "Read"
            },
            "DescribeNotificationPreferences": {
                "url": "",
                "description": "Describes the notification preferences for the account",
                "accessLevel": "Read"
            },
            "ExcludeCheckItems": {
                "url": "",
                "description": "Exclude recommendations for checks for a given customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "IncludeCheckItems": {
                "url": "",
                "description": "Include recommendations for checks for a given customer",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "RefreshCheck": {
                "url": "",
                "description": "Enqueue a refresh for the specified check",
                "accessLevel": "Write",
                "resourceTypes": {
                    "checks": {
                        "required": true
                    }
                }
            },
            "SetAccountAccess": {
                "url": "",
                "description": "Toggle whether TrustedAdvisor is enabled/disabled for the account",
                "accessLevel": "Write"
            },
            "UpdateNotificationPreferences": {
                "url": "",
                "description": "Update notification preferences",
                "accessLevel": "Write"
            }
        };
        this.resourceTypes = {
            "checks": {
                "name": "checks",
                "url": "https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html",
                "arn": "arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * View support plan and various TA preferences.
     *
     * Access Level: Read
     */
    describeAccount() {
        this.add('trustedadvisor:DescribeAccount');
        return this;
    }
    /**
     * Resolve whether Account has disabled Trusted Advisor
     *
     * Access Level: Read
     */
    describeAccountAccess() {
        this.add('trustedadvisor:DescribeAccountAccess');
        return this;
    }
    /**
     * View details for the check items
     *
     * Access Level: Read
     */
    describeCheckItems() {
        this.add('trustedadvisor:DescribeCheckItems');
        return this;
    }
    /**
     * Describe check refresh statuses
     *
     * Access Level: Read
     */
    describeCheckRefreshStatuses() {
        this.add('trustedadvisor:DescribeCheckRefreshStatuses');
        return this;
    }
    /**
     * Describes the check's summaries
     *
     * Access Level: Read
     */
    describeCheckSummaries() {
        this.add('trustedadvisor:DescribeCheckSummaries');
        return this;
    }
    /**
     * List valid Trusted Advisor checks and details.
     *
     * Access Level: Read
     */
    describeChecks() {
        this.add('trustedadvisor:DescribeChecks');
        return this;
    }
    /**
     * Describes the notification preferences for the account
     *
     * Access Level: Read
     */
    describeNotificationPreferences() {
        this.add('trustedadvisor:DescribeNotificationPreferences');
        return this;
    }
    /**
     * Exclude recommendations for checks for a given customer
     *
     * Access Level: Write
     */
    excludeCheckItems() {
        this.add('trustedadvisor:ExcludeCheckItems');
        return this;
    }
    /**
     * Include recommendations for checks for a given customer
     *
     * Access Level: Write
     */
    includeCheckItems() {
        this.add('trustedadvisor:IncludeCheckItems');
        return this;
    }
    /**
     * Enqueue a refresh for the specified check
     *
     * Access Level: Write
     */
    refreshCheck() {
        this.add('trustedadvisor:RefreshCheck');
        return this;
    }
    /**
     * Toggle whether TrustedAdvisor is enabled/disabled for the account
     *
     * Access Level: Write
     */
    setAccountAccess() {
        this.add('trustedadvisor:SetAccountAccess');
        return this;
    }
    /**
     * Update notification preferences
     *
     * Access Level: Write
     */
    updateNotificationPreferences() {
        this.add('trustedadvisor:UpdateNotificationPreferences');
        return this;
    }
    /**
     * Adds a resource of type checks to the statement
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html
     *
     * @param categoryCode - Identifier for the categoryCode.
     * @param checkId - Identifier for the checkId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChecks(categoryCode, checkId, account, region, partition) {
        var arn = 'arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}';
        arn = arn.replace('${CategoryCode}', categoryCode);
        arn = arn.replace('${CheckId}', checkId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Trustedadvisor = Trustedadvisor;
//# sourceMappingURL=data:application/json;base64,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