"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Waf = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [waf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswaf.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Waf extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [waf](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswaf.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'waf';
        this.actionList = {
            "CreateByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateByteMatchSet.html",
                "description": "Creates a ByteMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "CreateGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateGeoMatchSet.html",
                "description": "Creates a GeoMatchSet, which you use to specify which web requests you want to allow or block based on the country that the requests originate from.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "CreateIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateIPSet.html",
                "description": "Creates an IPSet, which you use to specify which web requests you want to allow or block based on the IP addresses that the requests originate from.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "CreateRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRateBasedRule.html",
                "description": "Creates a RateBasedRule, which contains a RateLimit specifying the maximum number of requests that AWS WAF allows from a specified IP address in a five-minute period.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexMatchSet.html",
                "description": "Creates a RegexMatchSet, which you use to specify which web requests you want to allow or block based on the regex patterns you specified in a RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "CreateRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexPatternSet.html",
                "description": "Creates a RegexPatternSet, which you use to specify the regular expression (regex) pattern that you want AWS WAF to search for.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "CreateRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRule.html",
                "description": "Creates a Rule, which contains the IPSet objects, ByteMatchSet objects, and other predicates that identify the requests that you want to block.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRuleGroup.html",
                "description": "Creates a RuleGroup. A rule group is a collection of predefined rules that you add to a WebACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSizeConstraintSet.html",
                "description": "Creates a SizeConstraintSet, which you use to identify the part of a web request that you want to check for length.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "CreateSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSqlInjectionMatchSet.html",
                "description": "Creates a SqlInjectionMatchSet, which you use to allow, block, or count requests that contain snippets of SQL code in a specified part of web requests.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "CreateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACL.html",
                "description": "Creates a WebACL, which contains the Rules that identify the CloudFront web requests that you want to allow, block, or count.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "CreateWebACLMigrationStack": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACLMigrationStack.html",
                "description": "Create and store a CloudFormation tempalte that creates an equivalent WAF v2 WebACL from the given WAF Classic WebACL in the given S3 bucket.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "CreateXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateXssMatchSet.html",
                "description": "Creates an XssMatchSet, which you use to allow, block, or count requests that contain cross-site scripting attacks in the specified part of web requests.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            },
            "DeleteByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteByteMatchSet.html",
                "description": "Permanently deletes a ByteMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "DeleteGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteGeoMatchSet.html",
                "description": "Permanently deletes an GeoMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "DeleteIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteIPSet.html",
                "description": "Permanently deletes an IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "DeleteLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteLoggingConfiguration.html",
                "description": "Permanently deletes the LoggingConfiguration from the specified web ACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DeletePermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeletePermissionPolicy.html",
                "description": "Permanently deletes an IAM policy from the specified RuleGroup.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "DeleteRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRateBasedRule.html",
                "description": "Permanently deletes a RateBasedRule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "DeleteRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexMatchSet.html",
                "description": "Permanently deletes an RegexMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "DeleteRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexPatternSet.html",
                "description": "Permanently deletes an RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "DeleteRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRule.html",
                "description": "Permanently deletes a Rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "DeleteRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRuleGroup.html",
                "description": "Permanently deletes a RuleGroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "DeleteSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSizeConstraintSet.html",
                "description": "Permanently deletes a SizeConstraintSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "DeleteSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSqlInjectionMatchSet.html",
                "description": "Permanently deletes a SqlInjectionMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "DeleteWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteWebACL.html",
                "description": "Permanently deletes a WebACL.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "DeleteXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteXssMatchSet.html",
                "description": "Permanently deletes an XssMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            },
            "GetByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetByteMatchSet.html",
                "description": "Returns the ByteMatchSet specified by ByteMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "GetChangeToken": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeToken.html",
                "description": "When you want to create, update, or delete AWS WAF objects, get a change token and include the change token in the create, update, or delete request.",
                "accessLevel": "Read"
            },
            "GetChangeTokenStatus": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeTokenStatus.html",
                "description": "Returns the status of a ChangeToken that you got by calling GetChangeToken.",
                "accessLevel": "Read"
            },
            "GetGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetGeoMatchSet.html",
                "description": "Returns the GeoMatchSet specified by GeoMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "GetIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetIPSet.html",
                "description": "Returns the IPSet that is specified by IPSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "GetLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetLoggingConfiguration.html",
                "description": "Returns the LoggingConfiguration for the specified web ACL.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "GetPermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetPermissionPolicy.html",
                "description": "Returns the IAM policy attached to the RuleGroup.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "GetRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRule.html",
                "description": "Returns the RateBasedRule that is specified by the RuleId that you included in the GetRateBasedRule request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "GetRateBasedRuleManagedKeys": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRuleManagedKeys.html",
                "description": "Returns an array of IP addresses currently being blocked by the RateBasedRule that is specified by the RuleId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "GetRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexMatchSet.html",
                "description": "Returns the RegexMatchSet specified by RegexMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "GetRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexPatternSet.html",
                "description": "Returns the RegexPatternSet specified by RegexPatternSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "GetRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRule.html",
                "description": "Returns the Rule that is specified by the RuleId that you included in the GetRule request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "GetRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRuleGroup.html",
                "description": "Returns the RuleGroup that is specified by the RuleGroupId that you included in the GetRuleGroup request.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "GetSampledRequests": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSampledRequests.html",
                "description": "Gets detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "rule": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                }
            },
            "GetSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSizeConstraintSet.html",
                "description": "Returns the SizeConstraintSet specified by SizeConstraintSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "GetSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSqlInjectionMatchSet.html",
                "description": "Returns the SqlInjectionMatchSet that is specified by SqlInjectionMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "GetWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetWebACL.html",
                "description": "Returns the WebACL that is specified by WebACLId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "GetXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetXssMatchSet.html",
                "description": "Returns the XssMatchSet that is specified by XssMatchSetId.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            },
            "ListActivatedRulesInRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListActivatedRulesInRuleGroup.html",
                "description": "Returns an array of ActivatedRule objects.",
                "accessLevel": "List"
            },
            "ListByteMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListByteMatchSets.html",
                "description": "Returns an array of ByteMatchSetSummary objects.",
                "accessLevel": "List"
            },
            "ListGeoMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListGeoMatchSets.html",
                "description": "Returns an array of GeoMatchSetSummary objects.",
                "accessLevel": "List"
            },
            "ListIPSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListIPSets.html",
                "description": "Returns an array of IPSetSummary objects in the response.",
                "accessLevel": "List"
            },
            "ListLoggingConfigurations": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListLoggingConfigurations.html",
                "description": "Returns an array of LoggingConfiguration objects.",
                "accessLevel": "List"
            },
            "ListRateBasedRules": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRateBasedRules.html",
                "description": "Returns an array of RuleSummary objects.",
                "accessLevel": "List"
            },
            "ListRegexMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexMatchSets.html",
                "description": "Returns an array of RegexMatchSetSummary objects.",
                "accessLevel": "List"
            },
            "ListRegexPatternSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexPatternSets.html",
                "description": "Returns an array of RegexPatternSetSummary objects.",
                "accessLevel": "List"
            },
            "ListRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRuleGroups.html",
                "description": "Returns an array of RuleGroup objects.",
                "accessLevel": "List"
            },
            "ListRules": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRules.html",
                "description": "Returns an array of RuleSummary objects.",
                "accessLevel": "List"
            },
            "ListSizeConstraintSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSizeConstraintSets.html",
                "description": "Returns an array of SizeConstraintSetSummary objects.",
                "accessLevel": "List"
            },
            "ListSqlInjectionMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSqlInjectionMatchSets.html",
                "description": "Returns an array of SqlInjectionMatchSet objects.",
                "accessLevel": "List"
            },
            "ListSubscribedRuleGroups": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSubscribedRuleGroups.html",
                "description": "Returns an array of RuleGroup objects that you are subscribed to.",
                "accessLevel": "List"
            },
            "ListTagsForResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListTagsForResource.html",
                "description": "Lists the Tags for a given resource.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                }
            },
            "ListWebACLs": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListWebACLs.html",
                "description": "Returns an array of WebACLSummary objects in the response.",
                "accessLevel": "List"
            },
            "ListXssMatchSets": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListXssMatchSets.html",
                "description": "Returns an array of XssMatchSet objects.",
                "accessLevel": "List"
            },
            "PutLoggingConfiguration": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutLoggingConfiguration.html",
                "description": "Associates a LoggingConfiguration with a specified web ACL.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "PutPermissionPolicy": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutPermissionPolicy.html",
                "description": "Attaches a IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "TagResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_TagResource.html",
                "description": "Adds a Tag to a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:RequestTag/${TagKey}",
                    "aws:TagKeys"
                ]
            },
            "UntagResource": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UntagResource.html",
                "description": "Removes a Tag from a given resource.",
                "accessLevel": "Tagging",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": false
                    },
                    "rule": {
                        "required": false
                    },
                    "rulegroup": {
                        "required": false
                    },
                    "webacl": {
                        "required": false
                    }
                },
                "conditions": [
                    "aws:TagKeys"
                ]
            },
            "UpdateByteMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateByteMatchSet.html",
                "description": "Inserts or deletes ByteMatchTuple objects (filters) in a ByteMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "bytematchset": {
                        "required": true
                    }
                }
            },
            "UpdateGeoMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateGeoMatchSet.html",
                "description": "Inserts or deletes GeoMatchConstraint objects in a GeoMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "geomatchset": {
                        "required": true
                    }
                }
            },
            "UpdateIPSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateIPSet.html",
                "description": "Inserts or deletes IPSetDescriptor objects in an IPSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ipset": {
                        "required": true
                    }
                }
            },
            "UpdateRateBasedRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRateBasedRule.html",
                "description": "Inserts or deletes Predicate objects in a rule and updates the RateLimit in the rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "ratebasedrule": {
                        "required": true
                    }
                }
            },
            "UpdateRegexMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexMatchSet.html",
                "description": "Inserts or deletes RegexMatchTuple objects (filters) in a RegexMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexmatchset": {
                        "required": true
                    }
                }
            },
            "UpdateRegexPatternSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexPatternSet.html",
                "description": "Inserts or deletes RegexPatternStrings in a RegexPatternSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "regexpatternset": {
                        "required": true
                    }
                }
            },
            "UpdateRule": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRule.html",
                "description": "Inserts or deletes Predicate objects in a Rule.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rule": {
                        "required": true
                    }
                }
            },
            "UpdateRuleGroup": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRuleGroup.html",
                "description": "Inserts or deletes ActivatedRule objects in a RuleGroup.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "rulegroup": {
                        "required": true
                    }
                }
            },
            "UpdateSizeConstraintSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSizeConstraintSet.html",
                "description": "Inserts or deletes SizeConstraint objects (filters) in a SizeConstraintSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sizeconstraintset": {
                        "required": true
                    }
                }
            },
            "UpdateSqlInjectionMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSqlInjectionMatchSet.html",
                "description": "Inserts or deletes SqlInjectionMatchTuple objects (filters) in a SqlInjectionMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "sqlinjectionmatchset": {
                        "required": true
                    }
                }
            },
            "UpdateWebACL": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateWebACL.html",
                "description": "Inserts or deletes ActivatedRule objects in a WebACL.",
                "accessLevel": "Permissions management",
                "resourceTypes": {
                    "webacl": {
                        "required": true
                    }
                }
            },
            "UpdateXssMatchSet": {
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateXssMatchSet.html",
                "description": "Inserts or deletes XssMatchTuple objects (filters) in an XssMatchSet.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "xssmatchset": {
                        "required": true
                    }
                }
            }
        };
        this.resourceTypes = {
            "bytematchset": {
                "name": "bytematchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ByteMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:bytematchset/${Id}",
                "conditionKeys": []
            },
            "ipset": {
                "name": "ipset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_IPSet.html",
                "arn": "arn:${Partition}:waf::${Account}:ipset/${Id}",
                "conditionKeys": []
            },
            "ratebasedrule": {
                "name": "ratebasedrule",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RateBasedRule.html",
                "arn": "arn:${Partition}:waf::${Account}:ratebasedrule/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "rule": {
                "name": "rule",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_Rule.html",
                "arn": "arn:${Partition}:waf::${Account}:rule/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "sizeconstraintset": {
                "name": "sizeconstraintset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SizeConstraintSet.html",
                "arn": "arn:${Partition}:waf::${Account}:sizeconstraintset/${Id}",
                "conditionKeys": []
            },
            "sqlinjectionmatchset": {
                "name": "sqlinjectionmatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SqlInjectionMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:sqlinjectionset/${Id}",
                "conditionKeys": []
            },
            "webacl": {
                "name": "webacl",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_WebACL.html",
                "arn": "arn:${Partition}:waf::${Account}:webacl/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            },
            "xssmatchset": {
                "name": "xssmatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_XssMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:xssmatchset/${Id}",
                "conditionKeys": []
            },
            "regexmatchset": {
                "name": "regexmatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:regexmatch/${Id}",
                "conditionKeys": []
            },
            "regexpatternset": {
                "name": "regexpatternset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexPatternSet.html",
                "arn": "arn:${Partition}:waf::${Account}:regexpatternset/${Id}",
                "conditionKeys": []
            },
            "geomatchset": {
                "name": "geomatchset",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GeoMatchSet.html",
                "arn": "arn:${Partition}:waf::${Account}:geomatchset/${Id}",
                "conditionKeys": []
            },
            "rulegroup": {
                "name": "rulegroup",
                "url": "https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RuleGroup.html",
                "arn": "arn:${Partition}:waf::${Account}:rulegroup/${Id}",
                "conditionKeys": [
                    "aws:ResourceTag/${TagKey}"
                ]
            }
        };
    }
    /**
     * Creates a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateByteMatchSet.html
     */
    createByteMatchSet() {
        this.add('waf:CreateByteMatchSet');
        return this;
    }
    /**
     * Creates a GeoMatchSet, which you use to specify which web requests you want to allow or block based on the country that the requests originate from.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateGeoMatchSet.html
     */
    createGeoMatchSet() {
        this.add('waf:CreateGeoMatchSet');
        return this;
    }
    /**
     * Creates an IPSet, which you use to specify which web requests you want to allow or block based on the IP addresses that the requests originate from.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateIPSet.html
     */
    createIPSet() {
        this.add('waf:CreateIPSet');
        return this;
    }
    /**
     * Creates a RateBasedRule, which contains a RateLimit specifying the maximum number of requests that AWS WAF allows from a specified IP address in a five-minute period.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRateBasedRule.html
     */
    createRateBasedRule() {
        this.add('waf:CreateRateBasedRule');
        return this;
    }
    /**
     * Creates a RegexMatchSet, which you use to specify which web requests you want to allow or block based on the regex patterns you specified in a RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexMatchSet.html
     */
    createRegexMatchSet() {
        this.add('waf:CreateRegexMatchSet');
        return this;
    }
    /**
     * Creates a RegexPatternSet, which you use to specify the regular expression (regex) pattern that you want AWS WAF to search for.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRegexPatternSet.html
     */
    createRegexPatternSet() {
        this.add('waf:CreateRegexPatternSet');
        return this;
    }
    /**
     * Creates a Rule, which contains the IPSet objects, ByteMatchSet objects, and other predicates that identify the requests that you want to block.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRule.html
     */
    createRule() {
        this.add('waf:CreateRule');
        return this;
    }
    /**
     * Creates a RuleGroup. A rule group is a collection of predefined rules that you add to a WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateRuleGroup.html
     */
    createRuleGroup() {
        this.add('waf:CreateRuleGroup');
        return this;
    }
    /**
     * Creates a SizeConstraintSet, which you use to identify the part of a web request that you want to check for length.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSizeConstraintSet.html
     */
    createSizeConstraintSet() {
        this.add('waf:CreateSizeConstraintSet');
        return this;
    }
    /**
     * Creates a SqlInjectionMatchSet, which you use to allow, block, or count requests that contain snippets of SQL code in a specified part of web requests.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateSqlInjectionMatchSet.html
     */
    createSqlInjectionMatchSet() {
        this.add('waf:CreateSqlInjectionMatchSet');
        return this;
    }
    /**
     * Creates a WebACL, which contains the Rules that identify the CloudFront web requests that you want to allow, block, or count.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACL.html
     */
    createWebACL() {
        this.add('waf:CreateWebACL');
        return this;
    }
    /**
     * Create and store a CloudFormation tempalte that creates an equivalent WAF v2 WebACL from the given WAF Classic WebACL in the given S3 bucket.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateWebACLMigrationStack.html
     */
    createWebACLMigrationStack() {
        this.add('waf:CreateWebACLMigrationStack');
        return this;
    }
    /**
     * Creates an XssMatchSet, which you use to allow, block, or count requests that contain cross-site scripting attacks in the specified part of web requests.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_CreateXssMatchSet.html
     */
    createXssMatchSet() {
        this.add('waf:CreateXssMatchSet');
        return this;
    }
    /**
     * Permanently deletes a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteByteMatchSet.html
     */
    deleteByteMatchSet() {
        this.add('waf:DeleteByteMatchSet');
        return this;
    }
    /**
     * Permanently deletes an GeoMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteGeoMatchSet.html
     */
    deleteGeoMatchSet() {
        this.add('waf:DeleteGeoMatchSet');
        return this;
    }
    /**
     * Permanently deletes an IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteIPSet.html
     */
    deleteIPSet() {
        this.add('waf:DeleteIPSet');
        return this;
    }
    /**
     * Permanently deletes the LoggingConfiguration from the specified web ACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteLoggingConfiguration.html
     */
    deleteLoggingConfiguration() {
        this.add('waf:DeleteLoggingConfiguration');
        return this;
    }
    /**
     * Permanently deletes an IAM policy from the specified RuleGroup.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeletePermissionPolicy.html
     */
    deletePermissionPolicy() {
        this.add('waf:DeletePermissionPolicy');
        return this;
    }
    /**
     * Permanently deletes a RateBasedRule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRateBasedRule.html
     */
    deleteRateBasedRule() {
        this.add('waf:DeleteRateBasedRule');
        return this;
    }
    /**
     * Permanently deletes an RegexMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexMatchSet.html
     */
    deleteRegexMatchSet() {
        this.add('waf:DeleteRegexMatchSet');
        return this;
    }
    /**
     * Permanently deletes an RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRegexPatternSet.html
     */
    deleteRegexPatternSet() {
        this.add('waf:DeleteRegexPatternSet');
        return this;
    }
    /**
     * Permanently deletes a Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRule.html
     */
    deleteRule() {
        this.add('waf:DeleteRule');
        return this;
    }
    /**
     * Permanently deletes a RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteRuleGroup.html
     */
    deleteRuleGroup() {
        this.add('waf:DeleteRuleGroup');
        return this;
    }
    /**
     * Permanently deletes a SizeConstraintSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSizeConstraintSet.html
     */
    deleteSizeConstraintSet() {
        this.add('waf:DeleteSizeConstraintSet');
        return this;
    }
    /**
     * Permanently deletes a SqlInjectionMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteSqlInjectionMatchSet.html
     */
    deleteSqlInjectionMatchSet() {
        this.add('waf:DeleteSqlInjectionMatchSet');
        return this;
    }
    /**
     * Permanently deletes a WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteWebACL.html
     */
    deleteWebACL() {
        this.add('waf:DeleteWebACL');
        return this;
    }
    /**
     * Permanently deletes an XssMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_DeleteXssMatchSet.html
     */
    deleteXssMatchSet() {
        this.add('waf:DeleteXssMatchSet');
        return this;
    }
    /**
     * Returns the ByteMatchSet specified by ByteMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetByteMatchSet.html
     */
    getByteMatchSet() {
        this.add('waf:GetByteMatchSet');
        return this;
    }
    /**
     * When you want to create, update, or delete AWS WAF objects, get a change token and include the change token in the create, update, or delete request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeToken.html
     */
    getChangeToken() {
        this.add('waf:GetChangeToken');
        return this;
    }
    /**
     * Returns the status of a ChangeToken that you got by calling GetChangeToken.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetChangeTokenStatus.html
     */
    getChangeTokenStatus() {
        this.add('waf:GetChangeTokenStatus');
        return this;
    }
    /**
     * Returns the GeoMatchSet specified by GeoMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetGeoMatchSet.html
     */
    getGeoMatchSet() {
        this.add('waf:GetGeoMatchSet');
        return this;
    }
    /**
     * Returns the IPSet that is specified by IPSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetIPSet.html
     */
    getIPSet() {
        this.add('waf:GetIPSet');
        return this;
    }
    /**
     * Returns the LoggingConfiguration for the specified web ACL.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetLoggingConfiguration.html
     */
    getLoggingConfiguration() {
        this.add('waf:GetLoggingConfiguration');
        return this;
    }
    /**
     * Returns the IAM policy attached to the RuleGroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetPermissionPolicy.html
     */
    getPermissionPolicy() {
        this.add('waf:GetPermissionPolicy');
        return this;
    }
    /**
     * Returns the RateBasedRule that is specified by the RuleId that you included in the GetRateBasedRule request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRule.html
     */
    getRateBasedRule() {
        this.add('waf:GetRateBasedRule');
        return this;
    }
    /**
     * Returns an array of IP addresses currently being blocked by the RateBasedRule that is specified by the RuleId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRateBasedRuleManagedKeys.html
     */
    getRateBasedRuleManagedKeys() {
        this.add('waf:GetRateBasedRuleManagedKeys');
        return this;
    }
    /**
     * Returns the RegexMatchSet specified by RegexMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexMatchSet.html
     */
    getRegexMatchSet() {
        this.add('waf:GetRegexMatchSet');
        return this;
    }
    /**
     * Returns the RegexPatternSet specified by RegexPatternSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRegexPatternSet.html
     */
    getRegexPatternSet() {
        this.add('waf:GetRegexPatternSet');
        return this;
    }
    /**
     * Returns the Rule that is specified by the RuleId that you included in the GetRule request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRule.html
     */
    getRule() {
        this.add('waf:GetRule');
        return this;
    }
    /**
     * Returns the RuleGroup that is specified by the RuleGroupId that you included in the GetRuleGroup request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetRuleGroup.html
     */
    getRuleGroup() {
        this.add('waf:GetRuleGroup');
        return this;
    }
    /**
     * Gets detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSampledRequests.html
     */
    getSampledRequests() {
        this.add('waf:GetSampledRequests');
        return this;
    }
    /**
     * Returns the SizeConstraintSet specified by SizeConstraintSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSizeConstraintSet.html
     */
    getSizeConstraintSet() {
        this.add('waf:GetSizeConstraintSet');
        return this;
    }
    /**
     * Returns the SqlInjectionMatchSet that is specified by SqlInjectionMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetSqlInjectionMatchSet.html
     */
    getSqlInjectionMatchSet() {
        this.add('waf:GetSqlInjectionMatchSet');
        return this;
    }
    /**
     * Returns the WebACL that is specified by WebACLId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetWebACL.html
     */
    getWebACL() {
        this.add('waf:GetWebACL');
        return this;
    }
    /**
     * Returns the XssMatchSet that is specified by XssMatchSetId.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GetXssMatchSet.html
     */
    getXssMatchSet() {
        this.add('waf:GetXssMatchSet');
        return this;
    }
    /**
     * Returns an array of ActivatedRule objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListActivatedRulesInRuleGroup.html
     */
    listActivatedRulesInRuleGroup() {
        this.add('waf:ListActivatedRulesInRuleGroup');
        return this;
    }
    /**
     * Returns an array of ByteMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListByteMatchSets.html
     */
    listByteMatchSets() {
        this.add('waf:ListByteMatchSets');
        return this;
    }
    /**
     * Returns an array of GeoMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListGeoMatchSets.html
     */
    listGeoMatchSets() {
        this.add('waf:ListGeoMatchSets');
        return this;
    }
    /**
     * Returns an array of IPSetSummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListIPSets.html
     */
    listIPSets() {
        this.add('waf:ListIPSets');
        return this;
    }
    /**
     * Returns an array of LoggingConfiguration objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListLoggingConfigurations.html
     */
    listLoggingConfigurations() {
        this.add('waf:ListLoggingConfigurations');
        return this;
    }
    /**
     * Returns an array of RuleSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRateBasedRules.html
     */
    listRateBasedRules() {
        this.add('waf:ListRateBasedRules');
        return this;
    }
    /**
     * Returns an array of RegexMatchSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexMatchSets.html
     */
    listRegexMatchSets() {
        this.add('waf:ListRegexMatchSets');
        return this;
    }
    /**
     * Returns an array of RegexPatternSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRegexPatternSets.html
     */
    listRegexPatternSets() {
        this.add('waf:ListRegexPatternSets');
        return this;
    }
    /**
     * Returns an array of RuleGroup objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRuleGroups.html
     */
    listRuleGroups() {
        this.add('waf:ListRuleGroups');
        return this;
    }
    /**
     * Returns an array of RuleSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListRules.html
     */
    listRules() {
        this.add('waf:ListRules');
        return this;
    }
    /**
     * Returns an array of SizeConstraintSetSummary objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSizeConstraintSets.html
     */
    listSizeConstraintSets() {
        this.add('waf:ListSizeConstraintSets');
        return this;
    }
    /**
     * Returns an array of SqlInjectionMatchSet objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSqlInjectionMatchSets.html
     */
    listSqlInjectionMatchSets() {
        this.add('waf:ListSqlInjectionMatchSets');
        return this;
    }
    /**
     * Returns an array of RuleGroup objects that you are subscribed to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListSubscribedRuleGroups.html
     */
    listSubscribedRuleGroups() {
        this.add('waf:ListSubscribedRuleGroups');
        return this;
    }
    /**
     * Lists the Tags for a given resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListTagsForResource.html
     */
    listTagsForResource() {
        this.add('waf:ListTagsForResource');
        return this;
    }
    /**
     * Returns an array of WebACLSummary objects in the response.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListWebACLs.html
     */
    listWebACLs() {
        this.add('waf:ListWebACLs');
        return this;
    }
    /**
     * Returns an array of XssMatchSet objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ListXssMatchSets.html
     */
    listXssMatchSets() {
        this.add('waf:ListXssMatchSets');
        return this;
    }
    /**
     * Associates a LoggingConfiguration with a specified web ACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutLoggingConfiguration.html
     */
    putLoggingConfiguration() {
        this.add('waf:PutLoggingConfiguration');
        return this;
    }
    /**
     * Attaches a IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_PutPermissionPolicy.html
     */
    putPermissionPolicy() {
        this.add('waf:PutPermissionPolicy');
        return this;
    }
    /**
     * Adds a Tag to a given resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_TagResource.html
     */
    tagResource() {
        this.add('waf:TagResource');
        return this;
    }
    /**
     * Removes a Tag from a given resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UntagResource.html
     */
    untagResource() {
        this.add('waf:UntagResource');
        return this;
    }
    /**
     * Inserts or deletes ByteMatchTuple objects (filters) in a ByteMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateByteMatchSet.html
     */
    updateByteMatchSet() {
        this.add('waf:UpdateByteMatchSet');
        return this;
    }
    /**
     * Inserts or deletes GeoMatchConstraint objects in a GeoMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateGeoMatchSet.html
     */
    updateGeoMatchSet() {
        this.add('waf:UpdateGeoMatchSet');
        return this;
    }
    /**
     * Inserts or deletes IPSetDescriptor objects in an IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateIPSet.html
     */
    updateIPSet() {
        this.add('waf:UpdateIPSet');
        return this;
    }
    /**
     * Inserts or deletes Predicate objects in a rule and updates the RateLimit in the rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRateBasedRule.html
     */
    updateRateBasedRule() {
        this.add('waf:UpdateRateBasedRule');
        return this;
    }
    /**
     * Inserts or deletes RegexMatchTuple objects (filters) in a RegexMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexMatchSet.html
     */
    updateRegexMatchSet() {
        this.add('waf:UpdateRegexMatchSet');
        return this;
    }
    /**
     * Inserts or deletes RegexPatternStrings in a RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRegexPatternSet.html
     */
    updateRegexPatternSet() {
        this.add('waf:UpdateRegexPatternSet');
        return this;
    }
    /**
     * Inserts or deletes Predicate objects in a Rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRule.html
     */
    updateRule() {
        this.add('waf:UpdateRule');
        return this;
    }
    /**
     * Inserts or deletes ActivatedRule objects in a RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateRuleGroup.html
     */
    updateRuleGroup() {
        this.add('waf:UpdateRuleGroup');
        return this;
    }
    /**
     * Inserts or deletes SizeConstraint objects (filters) in a SizeConstraintSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSizeConstraintSet.html
     */
    updateSizeConstraintSet() {
        this.add('waf:UpdateSizeConstraintSet');
        return this;
    }
    /**
     * Inserts or deletes SqlInjectionMatchTuple objects (filters) in a SqlInjectionMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateSqlInjectionMatchSet.html
     */
    updateSqlInjectionMatchSet() {
        this.add('waf:UpdateSqlInjectionMatchSet');
        return this;
    }
    /**
     * Inserts or deletes ActivatedRule objects in a WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateWebACL.html
     */
    updateWebACL() {
        this.add('waf:UpdateWebACL');
        return this;
    }
    /**
     * Inserts or deletes XssMatchTuple objects (filters) in an XssMatchSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_UpdateXssMatchSet.html
     */
    updateXssMatchSet() {
        this.add('waf:UpdateXssMatchSet');
        return this;
    }
    /**
     * Adds a resource of type bytematchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_ByteMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBytematchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:bytematchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_IPSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onIpset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:ipset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ratebasedrule to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RateBasedRule.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRatebasedrule(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:ratebasedrule/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_Rule.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRule(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:rule/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sizeconstraintset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SizeConstraintSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSizeconstraintset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:sizeconstraintset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type sqlinjectionmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_SqlInjectionMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSqlinjectionmatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:sqlinjectionset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_WebACL.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onWebacl(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:webacl/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type xssmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_XssMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onXssmatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:xssmatchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexmatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRegexmatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:regexmatch/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexpatternset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RegexPatternSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onRegexpatternset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:regexpatternset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type geomatchset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_GeoMatchSet.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGeomatchset(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:geomatchset/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rulegroup to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_waf_RuleGroup.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible condition keys:
     *  - aws:ResourceTag/${TagKey}
     */
    onRulegroup(id, account, partition) {
        var arn = 'arn:${Partition}:waf::${Account}:rulegroup/${Id}';
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Waf = Waf;
//# sourceMappingURL=data:application/json;base64,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