"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wellarchitected = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [wellarchitected](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswell-architectedtool.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wellarchitected extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [wellarchitected](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswell-architectedtool.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'wellarchitected';
        this.actionList = {
            "CreateWorkload": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/define-workload.html",
                "description": "Creates a new workload.",
                "accessLevel": "Write"
            },
            "CreateWorkloadShare": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-sharing.html",
                "description": "Shares a workload with another account.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workload": {
                        "required": true
                    }
                }
            },
            "DeleteWorkload": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-delete.html",
                "description": "Deletes an existing workload.",
                "accessLevel": "Write",
                "resourceTypes": {
                    "workload": {
                        "required": true
                    }
                }
            },
            "GetWorkload": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workload-details.html",
                "description": "Retrieves the specified workload.",
                "accessLevel": "Read",
                "resourceTypes": {
                    "workload": {
                        "required": true
                    }
                }
            },
            "ListWorkloads": {
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-page.html",
                "description": "Lists the workloads in this account.",
                "accessLevel": "List"
            }
        };
        this.resourceTypes = {
            "workload": {
                "name": "workload",
                "url": "https://docs.aws.amazon.com/wellarchitected/latest/userguide/iam-auth-access.html",
                "arn": "arn:${Partition}:wellarchitected:${Region}:${Account}:workload/${ResourceId}",
                "conditionKeys": []
            }
        };
    }
    /**
     * Creates a new workload.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/define-workload.html
     */
    createWorkload() {
        this.add('wellarchitected:CreateWorkload');
        return this;
    }
    /**
     * Shares a workload with another account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-sharing.html
     */
    createWorkloadShare() {
        this.add('wellarchitected:CreateWorkloadShare');
        return this;
    }
    /**
     * Deletes an existing workload.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-delete.html
     */
    deleteWorkload() {
        this.add('wellarchitected:DeleteWorkload');
        return this;
    }
    /**
     * Retrieves the specified workload.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workload-details.html
     */
    getWorkload() {
        this.add('wellarchitected:GetWorkload');
        return this;
    }
    /**
     * Lists the workloads in this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/workloads-page.html
     */
    listWorkloads() {
        this.add('wellarchitected:ListWorkloads');
        return this;
    }
    /**
     * Adds a resource of type workload to the statement
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/iam-auth-access.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWorkload(resourceId, account, region, partition) {
        var arn = 'arn:${Partition}:wellarchitected:${Region}:${Account}:workload/${ResourceId}';
        arn = arn.replace('${ResourceId}', resourceId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Wellarchitected = Wellarchitected;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzd2VsbC1hcmNoaXRlY3RlZHRvb2wuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhd3N3ZWxsLWFyY2hpdGVjdGVkdG9vbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxzQ0FBb0U7QUFFcEU7Ozs7R0FJRztBQUNILE1BQWEsZUFBZ0IsU0FBUSx3QkFBZTtJQXFEbEQ7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUExRE4sa0JBQWEsR0FBRyxpQkFBaUIsQ0FBQztRQUMvQixlQUFVLEdBQVk7WUFDOUIsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxtRkFBbUY7Z0JBQzFGLGFBQWEsRUFBRSx5QkFBeUI7Z0JBQ3hDLGFBQWEsRUFBRSxPQUFPO2FBQ3ZCO1lBQ0QscUJBQXFCLEVBQUU7Z0JBQ3JCLEtBQUssRUFBRSxxRkFBcUY7Z0JBQzVGLGFBQWEsRUFBRSx5Q0FBeUM7Z0JBQ3hELGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZ0JBQWdCLEVBQUU7Z0JBQ2hCLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSwrQkFBK0I7Z0JBQzlDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLEtBQUssRUFBRSxvRkFBb0Y7Z0JBQzNGLGFBQWEsRUFBRSxtQ0FBbUM7Z0JBQ2xELGFBQWEsRUFBRSxNQUFNO2dCQUNyQixlQUFlLEVBQUU7b0JBQ2YsVUFBVSxFQUFFO3dCQUNWLFVBQVUsRUFBRSxJQUFJO3FCQUNqQjtpQkFDRjthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEtBQUssRUFBRSxrRkFBa0Y7Z0JBQ3pGLGFBQWEsRUFBRSxzQ0FBc0M7Z0JBQ3JELGFBQWEsRUFBRSxNQUFNO2FBQ3RCO1NBQ0YsQ0FBQztRQUNLLGtCQUFhLEdBQWtCO1lBQ3BDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsVUFBVTtnQkFDbEIsS0FBSyxFQUFFLG1GQUFtRjtnQkFDMUYsS0FBSyxFQUFFLDhFQUE4RTtnQkFDckYsZUFBZSxFQUFFLEVBQUU7YUFDcEI7U0FDRixDQUFDO0lBU0YsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixJQUFJLENBQUMsR0FBRyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDaEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsR0FBRyxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsR0FBRyxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksVUFBVSxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekYsSUFBSSxHQUFHLEdBQUcsOEVBQThFLENBQUM7UUFDekYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUE1SUQsMENBNElDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWN0aW9ucywgUG9saWN5U3RhdGVtZW50LCBSZXNvdXJjZVR5cGVzIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbd2VsbGFyY2hpdGVjdGVkXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3N3ZWxsLWFyY2hpdGVjdGVkdG9vbC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFdlbGxhcmNoaXRlY3RlZCBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ3dlbGxhcmNoaXRlY3RlZCc7XG4gIHByb3RlY3RlZCBhY3Rpb25MaXN0OiBBY3Rpb25zID0ge1xuICAgIFwiQ3JlYXRlV29ya2xvYWRcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2VsbGFyY2hpdGVjdGVkL2xhdGVzdC91c2VyZ3VpZGUvZGVmaW5lLXdvcmtsb2FkLmh0bWxcIixcbiAgICAgIFwiZGVzY3JpcHRpb25cIjogXCJDcmVhdGVzIGEgbmV3IHdvcmtsb2FkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIldyaXRlXCJcbiAgICB9LFxuICAgIFwiQ3JlYXRlV29ya2xvYWRTaGFyZVwiOiB7XG4gICAgICBcInVybFwiOiBcImh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93ZWxsYXJjaGl0ZWN0ZWQvbGF0ZXN0L3VzZXJndWlkZS93b3JrbG9hZHMtc2hhcmluZy5odG1sXCIsXG4gICAgICBcImRlc2NyaXB0aW9uXCI6IFwiU2hhcmVzIGEgd29ya2xvYWQgd2l0aCBhbm90aGVyIGFjY291bnQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwid29ya2xvYWRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkRlbGV0ZVdvcmtsb2FkXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dlbGxhcmNoaXRlY3RlZC9sYXRlc3QvdXNlcmd1aWRlL3dvcmtsb2Fkcy1kZWxldGUuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkRlbGV0ZXMgYW4gZXhpc3Rpbmcgd29ya2xvYWQuXCIsXG4gICAgICBcImFjY2Vzc0xldmVsXCI6IFwiV3JpdGVcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwid29ya2xvYWRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkdldFdvcmtsb2FkXCI6IHtcbiAgICAgIFwidXJsXCI6IFwiaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dlbGxhcmNoaXRlY3RlZC9sYXRlc3QvdXNlcmd1aWRlL3dvcmtsb2FkLWRldGFpbHMuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIlJldHJpZXZlcyB0aGUgc3BlY2lmaWVkIHdvcmtsb2FkLlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIlJlYWRcIixcbiAgICAgIFwicmVzb3VyY2VUeXBlc1wiOiB7XG4gICAgICAgIFwid29ya2xvYWRcIjoge1xuICAgICAgICAgIFwicmVxdWlyZWRcIjogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gICAgfSxcbiAgICBcIkxpc3RXb3JrbG9hZHNcIjoge1xuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2VsbGFyY2hpdGVjdGVkL2xhdGVzdC91c2VyZ3VpZGUvd29ya2xvYWRzLXBhZ2UuaHRtbFwiLFxuICAgICAgXCJkZXNjcmlwdGlvblwiOiBcIkxpc3RzIHRoZSB3b3JrbG9hZHMgaW4gdGhpcyBhY2NvdW50LlwiLFxuICAgICAgXCJhY2Nlc3NMZXZlbFwiOiBcIkxpc3RcIlxuICAgIH1cbiAgfTtcbiAgcHVibGljIHJlc291cmNlVHlwZXM6IFJlc291cmNlVHlwZXMgPSB7XG4gICAgXCJ3b3JrbG9hZFwiOiB7XG4gICAgICBcIm5hbWVcIjogXCJ3b3JrbG9hZFwiLFxuICAgICAgXCJ1cmxcIjogXCJodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2VsbGFyY2hpdGVjdGVkL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLmh0bWxcIixcbiAgICAgIFwiYXJuXCI6IFwiYXJuOiR7UGFydGl0aW9ufTp3ZWxsYXJjaGl0ZWN0ZWQ6JHtSZWdpb259OiR7QWNjb3VudH06d29ya2xvYWQvJHtSZXNvdXJjZUlkfVwiLFxuICAgICAgXCJjb25kaXRpb25LZXlzXCI6IFtdXG4gICAgfVxuICB9O1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3dlbGxhcmNoaXRlY3RlZF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzd2VsbC1hcmNoaXRlY3RlZHRvb2wuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyB3b3JrbG9hZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2VsbGFyY2hpdGVjdGVkL2xhdGVzdC91c2VyZ3VpZGUvZGVmaW5lLXdvcmtsb2FkLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVXb3JrbG9hZCgpIHtcbiAgICB0aGlzLmFkZCgnd2VsbGFyY2hpdGVjdGVkOkNyZWF0ZVdvcmtsb2FkJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU2hhcmVzIGEgd29ya2xvYWQgd2l0aCBhbm90aGVyIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dlbGxhcmNoaXRlY3RlZC9sYXRlc3QvdXNlcmd1aWRlL3dvcmtsb2Fkcy1zaGFyaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyBjcmVhdGVXb3JrbG9hZFNoYXJlKCkge1xuICAgIHRoaXMuYWRkKCd3ZWxsYXJjaGl0ZWN0ZWQ6Q3JlYXRlV29ya2xvYWRTaGFyZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYW4gZXhpc3Rpbmcgd29ya2xvYWQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dlbGxhcmNoaXRlY3RlZC9sYXRlc3QvdXNlcmd1aWRlL3dvcmtsb2Fkcy1kZWxldGUuaHRtbFxuICAgKi9cbiAgcHVibGljIGRlbGV0ZVdvcmtsb2FkKCkge1xuICAgIHRoaXMuYWRkKCd3ZWxsYXJjaGl0ZWN0ZWQ6RGVsZXRlV29ya2xvYWQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgdGhlIHNwZWNpZmllZCB3b3JrbG9hZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93ZWxsYXJjaGl0ZWN0ZWQvbGF0ZXN0L3VzZXJndWlkZS93b3JrbG9hZC1kZXRhaWxzLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXRXb3JrbG9hZCgpIHtcbiAgICB0aGlzLmFkZCgnd2VsbGFyY2hpdGVjdGVkOkdldFdvcmtsb2FkJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIHdvcmtsb2FkcyBpbiB0aGlzIGFjY291bnQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2VsbGFyY2hpdGVjdGVkL2xhdGVzdC91c2VyZ3VpZGUvd29ya2xvYWRzLXBhZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIGxpc3RXb3JrbG9hZHMoKSB7XG4gICAgdGhpcy5hZGQoJ3dlbGxhcmNoaXRlY3RlZDpMaXN0V29ya2xvYWRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgd29ya2xvYWQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2VsbGFyY2hpdGVjdGVkL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgYWNjb3VudHMuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCByZWdpb25zLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYC5cbiAgICovXG4gIHB1YmxpYyBvbldvcmtsb2FkKHJlc291cmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06d2VsbGFyY2hpdGVjdGVkOiR7UmVnaW9ufToke0FjY291bnR9Ondvcmtsb2FkLyR7UmVzb3VyY2VJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1Jlc291cmNlSWR9JywgcmVzb3VyY2VJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cbn1cbiJdfQ==