"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Datapipeline = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [datapipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_datapipeline.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Datapipeline extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [datapipeline](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_datapipeline.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'datapipeline';
        this.actionList = {
            "ActivatePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ActivatePipeline.html",
                "description": "Validates the specified pipeline and starts processing pipeline tasks. If the pipeline does not pass validation, activation fails.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "AddTags": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_AddTags.html",
                "description": "Adds or modifies tags for the specified pipeline.",
                "accessLevel": "Tagging",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "CreatePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_CreatePipeline.html",
                "description": "Creates a new, empty pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:Tag"
                ]
            },
            "DeactivatePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeactivatePipeline.html",
                "description": "Deactivates the specified running pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "DeletePipeline": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeletePipeline.html",
                "description": "Deletes a pipeline, its pipeline definition, and its run history.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "DescribeObjects": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribeObjects.html",
                "description": "Gets the object definitions for a set of objects associated with the pipeline.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "DescribePipelines": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribePipelines.html",
                "description": "Retrieves metadata about one or more pipelines.",
                "accessLevel": "List",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "EvaluateExpression": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_EvaluateExpression.html",
                "description": "Task runners call EvaluateExpression to evaluate a string in the context of the specified object.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "GetAccountLimits": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetAccountLimits.html",
                "description": "Description for GetAccountLimits",
                "accessLevel": "List"
            },
            "GetPipelineDefinition": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetPipelineDefinition.html",
                "description": "Gets the definition of the specified pipeline.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "ListPipelines": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ListPipelines.html",
                "description": "Lists the pipeline identifiers for all active pipelines that you have permission to access.",
                "accessLevel": "List"
            },
            "PollForTask": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PollForTask.html",
                "description": "Task runners call PollForTask to receive a task to perform from AWS Data Pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:workerGroup"
                ]
            },
            "PutAccountLimits": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutAccountLimits.html",
                "description": "Description for PutAccountLimits",
                "accessLevel": "Write"
            },
            "PutPipelineDefinition": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutPipelineDefinition.html",
                "description": "Adds tasks, schedules, and preconditions to the specified pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            },
            "QueryObjects": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_QueryObjects.html",
                "description": "Queries the specified pipeline for the names of objects that match the specified set of conditions.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "RemoveTags": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_RemoveTags.html",
                "description": "Removes existing tags from the specified pipeline.",
                "accessLevel": "Tagging",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "ReportTaskProgress": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskProgress.html",
                "description": "Task runners call ReportTaskProgress when assigned a task to acknowledge that it has the task.",
                "accessLevel": "Write"
            },
            "ReportTaskRunnerHeartbeat": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskRunnerHeartbeat.html",
                "description": "Task runners call ReportTaskRunnerHeartbeat every 15 minutes to indicate that they are operational.",
                "accessLevel": "Write"
            },
            "SetStatus": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetStatus.html",
                "description": "Requests that the status of the specified physical or logical pipeline objects be updated in the specified pipeline.",
                "accessLevel": "Write",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag"
                ]
            },
            "SetTaskStatus": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetTaskStatus.html",
                "description": "Task runners call SetTaskStatus to notify AWS Data Pipeline that a task is completed and provide information about the final status.",
                "accessLevel": "Write"
            },
            "ValidatePipelineDefinition": {
                "url": "https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ValidatePipelineDefinition.html",
                "description": "Validates the specified pipeline definition to ensure that it is well formed and can be run without error.",
                "accessLevel": "Read",
                "conditions": [
                    "datapipeline:PipelineCreator",
                    "datapipeline:Tag",
                    "datapipeline:workerGroup"
                ]
            }
        };
        this.resourceTypes = {};
    }
    /**
     * Validates the specified pipeline and starts processing pipeline tasks. If the pipeline does not pass validation, activation fails.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ActivatePipeline.html
     */
    activatePipeline() {
        this.add('datapipeline:ActivatePipeline');
        return this;
    }
    /**
     * Adds or modifies tags for the specified pipeline.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_AddTags.html
     */
    addTags() {
        this.add('datapipeline:AddTags');
        return this;
    }
    /**
     * Creates a new, empty pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_CreatePipeline.html
     */
    createPipeline() {
        this.add('datapipeline:CreatePipeline');
        return this;
    }
    /**
     * Deactivates the specified running pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeactivatePipeline.html
     */
    deactivatePipeline() {
        this.add('datapipeline:DeactivatePipeline');
        return this;
    }
    /**
     * Deletes a pipeline, its pipeline definition, and its run history.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DeletePipeline.html
     */
    deletePipeline() {
        this.add('datapipeline:DeletePipeline');
        return this;
    }
    /**
     * Gets the object definitions for a set of objects associated with the pipeline.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribeObjects.html
     */
    describeObjects() {
        this.add('datapipeline:DescribeObjects');
        return this;
    }
    /**
     * Retrieves metadata about one or more pipelines.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_DescribePipelines.html
     */
    describePipelines() {
        this.add('datapipeline:DescribePipelines');
        return this;
    }
    /**
     * Task runners call EvaluateExpression to evaluate a string in the context of the specified object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_EvaluateExpression.html
     */
    evaluateExpression() {
        this.add('datapipeline:EvaluateExpression');
        return this;
    }
    /**
     * Description for GetAccountLimits
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetAccountLimits.html
     */
    getAccountLimits() {
        this.add('datapipeline:GetAccountLimits');
        return this;
    }
    /**
     * Gets the definition of the specified pipeline.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_GetPipelineDefinition.html
     */
    getPipelineDefinition() {
        this.add('datapipeline:GetPipelineDefinition');
        return this;
    }
    /**
     * Lists the pipeline identifiers for all active pipelines that you have permission to access.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ListPipelines.html
     */
    listPipelines() {
        this.add('datapipeline:ListPipelines');
        return this;
    }
    /**
     * Task runners call PollForTask to receive a task to perform from AWS Data Pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PollForTask.html
     */
    pollForTask() {
        this.add('datapipeline:PollForTask');
        return this;
    }
    /**
     * Description for PutAccountLimits
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutAccountLimits.html
     */
    putAccountLimits() {
        this.add('datapipeline:PutAccountLimits');
        return this;
    }
    /**
     * Adds tasks, schedules, and preconditions to the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_PutPipelineDefinition.html
     */
    putPipelineDefinition() {
        this.add('datapipeline:PutPipelineDefinition');
        return this;
    }
    /**
     * Queries the specified pipeline for the names of objects that match the specified set of conditions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_QueryObjects.html
     */
    queryObjects() {
        this.add('datapipeline:QueryObjects');
        return this;
    }
    /**
     * Removes existing tags from the specified pipeline.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_RemoveTags.html
     */
    removeTags() {
        this.add('datapipeline:RemoveTags');
        return this;
    }
    /**
     * Task runners call ReportTaskProgress when assigned a task to acknowledge that it has the task.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskProgress.html
     */
    reportTaskProgress() {
        this.add('datapipeline:ReportTaskProgress');
        return this;
    }
    /**
     * Task runners call ReportTaskRunnerHeartbeat every 15 minutes to indicate that they are operational.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ReportTaskRunnerHeartbeat.html
     */
    reportTaskRunnerHeartbeat() {
        this.add('datapipeline:ReportTaskRunnerHeartbeat');
        return this;
    }
    /**
     * Requests that the status of the specified physical or logical pipeline objects be updated in the specified pipeline.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetStatus.html
     */
    setStatus() {
        this.add('datapipeline:SetStatus');
        return this;
    }
    /**
     * Task runners call SetTaskStatus to notify AWS Data Pipeline that a task is completed and provide information about the final status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_SetTaskStatus.html
     */
    setTaskStatus() {
        this.add('datapipeline:SetTaskStatus');
        return this;
    }
    /**
     * Validates the specified pipeline definition to ensure that it is well formed and can be run without error.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datapipeline/latest/APIReference/API_ValidatePipelineDefinition.html
     */
    validatePipelineDefinition() {
        this.add('datapipeline:ValidatePipelineDefinition');
        return this;
    }
    /**
     * The IAM user that created the pipeline.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-example-tag-policies.html#ex3
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifPipelineCreator(value, operator) {
        return this.if(`datapipeline:PipelineCreator`, value, operator || 'ArnEquals');
    }
    /**
     * A customer-specified key/value pair that can be attached to a resource.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-resourcebased-access.html#dp-control-access-tags
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifTag(value, operator) {
        return this.if(`datapipeline:Tag`, value, operator || 'ArnEquals');
    }
    /**
     * The name of a worker group for which a Task Runner retrieves work.
     *
     * https://docs.aws.amazon.com/datapipeline/latest/DeveloperGuide/dp-iam-resourcebased-access.html#dp-control-access-workergroup
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnEquals`
     */
    ifWorkerGroup(value, operator) {
        return this.if(`datapipeline:workerGroup`, value, operator || 'ArnEquals');
    }
}
exports.Datapipeline = Datapipeline;
//# sourceMappingURL=data:application/json;base64,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