"use strict";
/**
 * Modifiers can be used to extend the functionality of any `Operator`, except the `null` operator
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = exports.OperatorModifier = void 0;
var OperatorModifier;
(function (OperatorModifier) {
    /**
     * You can add `IfExists` to the end of any condition operator name except the `Null` condition. For example, `StringLikeIfExists`. You do this to say "If the policy key is present in the context of the request, process the key as specified in the policy. If the key is not present, evaluate the condition element as true." Other condition elements in the statement can still result in a nonmatch, but not a missing key when checked with `...IfExists`.
     */
    OperatorModifier["IF_EXISTS"] = "IfExists";
    /**
     * Tests whether the value of every member of the request set is a subset of the condition key set. The condition returns true if every key value in the request matches at least one value in the policy. It also returns true if there are no keys in the request, or if the key values resolve to a null data set, such as an empty string.
     */
    OperatorModifier["FOR_ALL_VALUES"] = "ForAllValues:";
    /**
     * Tests whether at least one member of the set of request values matches at least one member of the set of condition key values. The condition returns true if any one of the key values in the request matches any one of the condition values in the policy. For no matching key or a null dataset, the condition returns false.
     */
    OperatorModifier["FOR_ANY_VALUES"] = "ForAnyValues:";
})(OperatorModifier = exports.OperatorModifier || (exports.OperatorModifier = {}));
/**
 * Use condition operators in the `Condition` element to match the condition key and value in the policy against values in the request context. For more information about the `Condition` element, see [IAM JSON Policy Elements: Condition](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
 *
 * The condition operator that you can use in a policy depends on the condition key you choose. You can choose a global condition key or a service-specific condition key. To learn which condition operator you can use for a global condition key, see [AWS Global Condition Context Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html). To learn which condition operator you can use for a service-specific condition key, see [Actions, Resources, and Condition Keys for AWS Services](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_actions-resources-contextkeys.html) and choose the service that you want to view.
 *
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html
 */
class Operator {
    /**
     * Exact match, case sensitive.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringEquals(...modifiers) {
        return applyModifiers('StringEquals', ...modifiers);
    }
    /**
     * Negated exact match, case sensitive.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotEquals(...modifiers) {
        return applyModifiers('StringNotEquals', ...modifiers);
    }
    /**
     * Exact match, ignore case.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringEqualsIgnoreCase(...modifiers) {
        return applyModifiers('StringEqualsIgnoreCase', ...modifiers);
    }
    /**
     * Exact exact match, ignore case.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotEqualsIgnoreCase(...modifiers) {
        return applyModifiers('StringNotEqualsIgnoreCase', ...modifiers);
    }
    /**
     * Case-sensitive match.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringLike(...modifiers) {
        return applyModifiers('StringLike', ...modifiers);
    }
    /**
     * Negated case-sensitive matching.
     *
     * The values can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`) anywhere in the string.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static stringNotLike(...modifiers) {
        return applyModifiers('StringNotLike', ...modifiers);
    }
    /**
     * Exact match.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericEquals(...modifiers) {
        return applyModifiers('NumericEquals', ...modifiers);
    }
    /**
     * Negated exact match.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericNotEquals(...modifiers) {
        return applyModifiers('NumericNotEquals', ...modifiers);
    }
    /**
     * Match numbers lower than value: `<`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericLessThan(...modifiers) {
        return applyModifiers('NumericLessThan', ...modifiers);
    }
    /**
     * Match numbers lower or equal to value: `<=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericLessThanEquals(...modifiers) {
        return applyModifiers('NumericLessThanEquals', ...modifiers);
    }
    /**
     * Match numbers higher than value: `>`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericGreaterThan(...modifiers) {
        return applyModifiers('NumericGreaterThan', ...modifiers);
    }
    /**
     * Match numbers higher or equal value: `>=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static numericGreaterThanEquals(...modifiers) {
        return applyModifiers('NumericGreaterThanEquals', ...modifiers);
    }
    /**
     * Match a specific date: `=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateEquals(...modifiers) {
        return applyModifiers('DateEquals', ...modifiers);
    }
    /**
     * Negated match for a specific date: `!=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateNotEquals(...modifiers) {
        return applyModifiers('DateNotEquals', ...modifiers);
    }
    /**
     * Match before a specific date and time: `<`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateLessThan(...modifiers) {
        return applyModifiers('DateLessThan', ...modifiers);
    }
    /**
     * Match at or before a specific date and time: `<=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateLessThanEquals(...modifiers) {
        return applyModifiers('DateLessThanEquals', ...modifiers);
    }
    /**
     * Match after a specific a date and time: `>`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateGreaterThan(...modifiers) {
        return applyModifiers('DateGreaterThan', ...modifiers);
    }
    /**
     * Match at or after a specific date and time: `>=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static dateGreaterThanEquals(...modifiers) {
        return applyModifiers('DateGreaterThanEquals', ...modifiers);
    }
    /**
     * Boolean match
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static bool(...modifiers) {
        return applyModifiers('Bool', ...modifiers);
    }
    /**
     * The BinaryEquals condition operator lets you construct Condition elements that test key values that are in binary format. It compares the value of the specified key byte for byte against a [base-64](https://en.wikipedia.org/wiki/Base64) encoded representation of the binary value in the policy.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static binaryEquals(...modifiers) {
        return applyModifiers('BinaryEquals', ...modifiers);
    }
    /**
     * Match an IP address or range: `=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static ipAddress(...modifiers) {
        return applyModifiers('IpAddress', ...modifiers);
    }
    /**
     * All IP addresses except the specified IP address or range `!=`
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static notIpAddress(...modifiers) {
        return applyModifiers('NotIpAddress', ...modifiers);
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnEquals` and `ArnLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnEquals(...modifiers) {
        return applyModifiers('ArnEquals', ...modifiers);
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnNotEquals(...modifiers) {
        return applyModifiers('ArnNotEquals', ...modifiers);
    }
    /**
     * Match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnLike(...modifiers) {
        return applyModifiers('ArnLike', ...modifiers);
    }
    /**
     * Negated match of the ARN, case sensitive.
     *
     * Each of the six colon-delimited components of the ARN is checked separately and each can include a multi-character match wildcard (`*`) or a single-character match wildcard (`?`).
     *
     * `ArnNotEquals` and `ArnNotLike` behave identical.
     *
     * @param modifiers - Pass in any number of `OperatorModifier`s.
     */
    static arnNotLike(...modifiers) {
        return applyModifiers('ArnNotLike', ...modifiers);
    }
    /**
     * Check if a key is present at the time of authorization. In the policy statement, use either true (the key doesn't exist — it is null) or false (the key exists and its value is not null).
     */
    static null() {
        return 'Null';
    }
}
exports.Operator = Operator;
/**
 * Applies modifiers to an operator.
 *
 * @param operator - The original operator
 * @param modifiers - Any number of modifiers
 */
function applyModifiers(operator, ...modifiers) {
    if (modifiers.includes(OperatorModifier.IF_EXISTS)) {
        operator += OperatorModifier.IF_EXISTS;
    }
    if (modifiers.includes(OperatorModifier.FOR_ALL_VALUES)) {
        operator = `${OperatorModifier.FOR_ALL_VALUES}${operator}`;
    }
    if (modifiers.includes(OperatorModifier.FOR_ANY_VALUES)) {
        operator = `${OperatorModifier.FOR_ANY_VALUES}${operator}`;
    }
    return operator;
}
//# sourceMappingURL=data:application/json;base64,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