"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatement = void 0;
var Effect;
(function (Effect) {
    Effect["ALLOW"] = "Allow";
    Effect["DENY"] = "Deny";
})(Effect || (Effect = {}));
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(sid) {
        this.actionList = {};
        this.useNotActions = false;
        this.useNotResources = false;
        this.sid = '';
        this.effect = Effect.ALLOW;
        this.actions = [];
        this.resources = [];
        this.conditions = {};
        this.cdkApplied = false; // internally used to check if resources, actions and conditions have already been applied to the policy
        /**
         * Holds the prefix of the service actions, e.g. `ec2`
         */
        this.servicePrefix = '';
        if (typeof sid !== 'undefined') {
            this.sid = sid;
        }
    }
    /**
     * Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotActions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notActions()`.
     *
     * @param actions Actions that will be added to the statement.
     */
    add(action) {
        this.actions.push(action);
        return this;
    }
    /**
     * Adds actions to the statement, matching an `AccessLevel` or regular expression.
     *
     * When no value is passed, all actions of the service will be added.
     */
    allActions(...rules) {
        if (rules.length) {
            rules.forEach((rule) => {
                for (const [name, action] of Object.entries(this.actionList)) {
                    if (typeof rule === 'object') {
                        //assume it's a regex
                        if (rule.test(name)) {
                            this.add(`${this.servicePrefix}:${name}`);
                        }
                    }
                    else {
                        // assume it's an AccessLevel
                        if (rule == action.accessLevel) {
                            this.add(`${this.servicePrefix}:${name}`);
                        }
                    }
                }
            });
        }
        else {
            this.add(`${this.servicePrefix}:*`);
        }
        return this;
    }
    /**
     * Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     */
    notActions() {
        this.useNotActions = true;
        return this;
    }
    /**
     * Switches the statement to use [`NotResource`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notresource.html).
     */
    notResources() {
        this.useNotResources = true;
        return this;
    }
    /**
     * Checks weather actions have been applied to the policy.
     */
    hasActions() {
        return this.actions.length > 0;
    }
    /**
     * Checks weather any resource was applied to the policy.
     */
    hasResources() {
        return this.resources.length > 0;
    }
    /**
     * Checks weather a condition was applied to the policy.
     */
    hasConditions() {
        return Object.keys(this.conditions).length > 0;
    }
    /**
     * Allow the actions in this statement
     */
    allow() {
        this.effect = Effect.ALLOW;
        return this;
    }
    /**
     * Deny the actions in this statement
     */
    deny() {
        this.effect = Effect.DENY;
        return this;
    }
    /**
     * Limit statement to specified resources.
     *
     * To allow all resources, pass `*`
     */
    on(...arns) {
        this.resources.push(...arns);
        return this;
    }
    /**
     * Adds a condition to the statement
     *
     * @param key The key of the condition
     * @param value The value(s) to check for
     * @param operator [Operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html) of the condition. **Default:** `StringLike`
     */
    if(key, value, operator) {
        if (typeof operator === 'undefined') {
            operator = 'StringLike';
        }
        if (!(operator in this.conditions)) {
            this.conditions[operator] = {};
        }
        this.conditions[operator][key] = value;
        return this;
    }
    /**
     * Compare the services with the services that made requests on behalf of the IAM principal (user or role). When a principal makes a request to an AWS service, that service might use the principal's credentials to make subsequent requests to other services.
     *
     * The `aws:CalledVia` key contains an ordered list of each service in the chain that made requests on the principal's behalf.
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringEquals`
     */
    ifCalledVia(value, operator) {
        return this.if('aws:CalledVia', value, operator || 'ForAnyValue:StringEquals');
    }
    /**
     * Compare the services with the first service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledviafirst
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifCalledViaFirst(value, operator) {
        return this.if('aws:CalledViaFirst', value, operator);
    }
    /**
     * Compare the services with the last service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvialast
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifCalledViaLast(value, operator) {
        return this.if('aws:CalledViaLast', value, operator);
    }
    /**
     * Compare the date and time of the request with the date and time that you specify. To view an example policy that uses this condition key, see [AWS: Allows Access Within Specific Dates](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws-dates.html).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-currenttime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateLessThanEquals`
     */
    ifCurrentTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        return this.if('aws:CurrentTime', value, operator || 'DateLessThanEquals');
    }
    /**
     * Compare the date and time of the request in epoch or Unix time with the value that you specify. This key also accepts the number of seconds since January 1, 1970.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-epochtime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date) and [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `DateLessThanEquals`
     */
    ifEpochTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        return this.if('aws:EpochTime', value, operator || 'DateLessThanEquals');
    }
    /**
     * Compare the number of seconds since the requesting principal was authorized using MFA with the number that you specify. For more information about MFA, see [Using Multi-Factor Authentication (MFA) in AWS](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html).
     *
     * **Availability:** This key is included in the request context only if the principal was authenticated using MFA. If MFA was not used, this key is not present.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthage
     *
     * @param value Number of seconds
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericLessThan`
     */
    ifMultiFactorAuthAge(value, operator) {
        return this.if('aws:MultiFactorAuthAge', value, operator || 'NumericLessThan');
    }
    /**
     * Check whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the request.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. The key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using long-term credentials.
     *
     * Temporary credentials are used to authenticate IAM roles, federated users, IAM users with temporary tokens from `sts:GetSessionToken`, and users of the AWS Management Console. IAM users in the AWS Management Console unknowingly use temporary credentials. Users sign into the console using their user name and password, which are long-term credentials. However, in the background, the console generates temporary credentials on behalf of the user. To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * The `aws:MultiFactorAuthPresent` key is not present when an API or CLI command is called with long-term credentials, such as user access key pairs. Therefore we recommend that when you check for this key that you use the [...IfExists](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IfExists) versions of the condition operators.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthpresent
     *
     * @param value Weather the MFA should be present or absent. **Default:** `true`
     */
    ifMultiFactorAuthPresent(value) {
        return this.if('aws:MultiFactorAuthPresent', value, 'Bool');
    }
    /**
     * Compare the account to which the requesting principal belongs with the account identifier that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalaccount
     *
     * @param value Account identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifPrincipalAccount(value, operator) {
        return this.if('aws:PrincipalAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) (ARN) of the principal that made the request with the ARN that you specify.
     *
     * For IAM roles, the request context returns the ARN of the role, not the ARN of the user that assumed the role. To learn which types of principals you can specify in this condition key, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn
     *
     * @param value Principle ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnEquals`
     */
    ifPrincipalArn(value, operator) {
        return this.if('aws:PrincipalArn', value, operator || 'ArnEquals');
    }
    /**
     * Compare the identifier of the organization in AWS Organizations to which the requesting principal belongs with the identifier you specify.
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * This global key provides an alternative to listing all the account IDs for all AWS accounts in an organization. You can use this condition key to simplify specifying the `Principal` element in a resource-based policy. You can specify the organization ID in the condition element. When you add and remove accounts, policies that include the `aws:PrincipalOrgID` key automatically include the correct accounts and don't require manual updating.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgid
     *
     * @param value Organization ID(s) in format `o-xxxxxxxxxxx`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifPrincipalOrgID(value, operator) {
        return this.if('aws:PrincipalOrgID', value, operator);
    }
    /**
     * Compare the AWS Organizations path for the principal who is making the request to the path you provide. That principal can be an IAM user, IAM role, federated user, or AWS account root user.
     *
     * This condition ensures that the requester is an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see Understand the [AWS Organizations Entity Path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * **Note:** Organization IDs are globally unique but OU IDs and root IDs are unique only within an organization. This means that no two organizations share the same organization ID. However, another organization might have an OU or root with the same ID as yours. We recommend that you always include the organization ID when you specify an OU or root.
     *
     * `aws:PrincipalOrgPaths` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. When you use multiple values with the `ForAnyValue:` condition operator, the principal's path must match one of the paths provided. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array (`"Key":["Value1", "Value2"]`). You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Creating a Condition with Multiple Keys or Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgpaths
     *
     * @param value Organization path(s) in the format of `o-xxxxxxxxxxx/r-xxxxxxxxxx/ou-xxxx-xxxxxxxx/ou-xxxx-xxxxxxxx/`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifPrincipalOrgPaths(value, operator) {
        return this.if('aws:PrincipalOrgPaths', value, operator);
    }
    /**
     * Compare the tag attached to the principal making the request with the tag that you specify. If the principal has more than one tag attached, the request context includes one aws:PrincipalTag key for each attached tag key.
     *
     * **Availability:** This key is included in the request context if the principal is using an IAM user with attached tags. It is included for a principal using an IAM role with attached tags or [session tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html).
     *
     * You can add custom attributes to a user or role in the form of a key-value pair. For more information about IAM tags, see [Tagging IAM Users and Roles](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html). You can use `aws:PrincipalTag` to [control access](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html#access_iam-tags_control-resources) for AWS principals.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifPrincipalTag(key, value, operator) {
        return this.if(`aws:PrincipalTag/${key}`, value, operator);
    }
    /**
     * Compare the type of principal making the request with the principal type that you specify. For details about how the information appears in the request context for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltype
     *
     * @param value The principal type(s). Any of `Account`, `User`, `FederatedUser`, `AssumedRole`, `Anonymous`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifPrincipalType(value, operator) {
        return this.if('aws:PrincipalType', value, operator);
    }
    /**
     * Compare who referred the request in the client browser with the referer that you specify. The `aws:referer` request context value is provided by the caller in an HTTP header.
     *
     * **Availability:** This key is included in the request context only if the request was invoked using a URL in the browser.
     *
     * For example, you can call [Amazon S3 API operations directly using a web browser](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html#example-bucket-policies-use-case-4). This means that you can view S3 objects, such as images and documents, directly through a web browser. The `aws:referer` condition allows you to restrict access to specific values in the HTTP or HTTPS request based on the value of the referrer header.
     *
     * **Warning:** This condition should be used carefully. It is dangerous to include a publicly known referer header value. Unauthorized parties can use modified or custom browsers to provide any `aws:referer` value that they choose. As a result, `aws:referer` should not be used to prevent unauthorized parties from making direct AWS requests. It is offered only to allow customers to protect their digital content, such as content stored in Amazon S3, from being referenced on unauthorized third-party sites.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-referer
     *
     * @param value The referer url(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifReferer(value, operator) {
        return this.if('aws:Referer', value, operator);
    }
    /**
     * Compare the AWS Region that was called in the request with the Region that you specify. You can use this global condition key to control which Regions can be requested. To view the AWS Regions for each service, see [Service endpoints and quotas](https://docs.aws.amazon.com/general/latest/gr/aws-service-information.html) in the Amazon Web Services General Reference.
     *
     * **Availability:** This key is always included in the request context.
     *
     * Some global services, such as IAM, have a single endpoint. Because this endpoint is physically located in the US East (N. Virginia) Region, IAM calls are always made to the us-east-1 Region. For example, if you create a policy that denies access to all services if the requested Region is not us-west-2, then IAM calls always fail. To view an example of how to work around this, see [NotAction with Deny](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * **Note:** The `aws:RequestedRegion` condition key allows you to control which endpoint of a service is invoked but does not control the impact of the operation. Some services have cross-Region impacts. For example, Amazon S3 has API operations that control cross-Region replication. You can invoke `s3:PutBucketReplication` in one Region (which is affected by the `aws:RequestedRegion` condition key), but other Regions are affected based on the replications configuration settings.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requestedregion
     *
     * @param value The region(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifRequestedRegion(value, operator) {
        return this.if('aws:RequestedRegion', value, operator);
    }
    /**
     * Compare the tag key-value pair that was passed in the request with the tag pair that you specify. For example, you could check whether the request includes the tag key `Dept` and that it has the value `Accounting`. For more information, see [Controlling Access During AWS Requests](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-requests).
     *
     * **Availability:** This key is included in the request context when tags are passed in the request. When multiple tags are passed in the request, there is one context key for each tag key-value pair.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a multivalued request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see [Using Multiple Keys and Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifRequestTag(key, value, operator) {
        return this.if(`aws:RequestTag/${key}`, value, operator);
    }
    /**
     * Compare the tag key-value pair that you specify with the key-value pair that is attached to the resource. For example, you could require that access to a resource is allowed only if the resource has the attached tag key `Dept` with the value `Marketing`. For more information, see [Controlling Access to AWS Resources](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-resources).
     *
     * **Availability:** This key is included in the request context when the requested resource already has attached tags. This key is returned only for resources that [support authorization based on tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html). There is one context key for each tag key-value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifResourceTag(key, value, operator) {
        return this.if(`aws:ResourceTag/${key}`, value, operator);
    }
    /**
     * Check whether the request was sent using SSL. The request context returns `true` or `false`. In a policy, you can allow specific actions only if the request is sent using SSL.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-securetransport
     *
     * @param value Weather request was sent using SSL. **Default:** `true`
     */
    ifSecureTransport(value) {
        return this.if('aws:SecureTransport', value, 'Bool');
    }
    /**
     * Compare the account ID of the resource making a service-to-service request with the account ID that you specify.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the resource ARN of the source to the called service. This ARN includes the source account ID.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the account ID from the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceaccount
     *
     * @param value The account ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifSourceAccount(value, operator) {
        return this.if('aws:SourceAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name (ARN)](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) of the resource making a service-to-service request with the ARN that you specify.
     *
     * This key does not work with the ARN of the principal making the request. Instead, use [aws:PrincipalArn](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn). The source's ARN includes the account ID, so it is not necessary to use `aws:SourceAccount` with `aws:SourceArn`.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the ARN of the original resource to the called service.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcearn
     *
     * @param value The source ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnEquals`
     */
    ifSourceArn(value, operator) {
        return this.if('aws:SourceArn', value, operator || 'ArnEquals');
    }
    /**
     * Compare the requester's IP address with the IP address that you specify.
     *
     * **Availability:** This key is included in the request context, except when the requester uses a VPC endpoint to make the request.
     *
     * The `aws:SourceIp` condition key can be used in a policy to allow principals to make requests only from within a specified IP range. However, this policy denies access if an AWS service makes calls on the principal's behalf. In this case, you can use `aws:SourceIp` with the [aws:ViaAWSService](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice) key to ensure that the source IP restriction applies only to requests made directly by a principal.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceip
     *
     * @param value The source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifSourceIp(value, operator) {
        return this.if('aws:SourceIp', value, operator || 'IpAddress');
    }
    /**
     * Check whether the request comes from the VPC that you specify. In a policy, you can use this condition to allow access to only a specific VPC. For more information, see [Restricting Access to a Specific VPC](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc
     *
     * @param value The VPS ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifSourceVpc(value, operator) {
        return this.if('aws:SourceVpc', value, operator);
    }
    /**
     * Compare the VPC endpoint identifier of the request with the endpoint ID that you specify. In a policy, you can use this condition to restrict access to a specific VPC endpoint. For more information, see [Restricting Access to a Specific VPC Endpoint](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc-endpoint) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpce
     *
     * @param value The VPC Endpoint ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifSourceVpce(value, operator) {
        return this.if('aws:SourceVpce', value, operator);
    }
    /**
     * Compare the tag keys in a request with the keys that you specify. As a best practice when you use policies to control access using tags, use the `aws:TagKeys` condition key to define what tag keys are allowed. For example policies and more information, see [Controlling Access Based on Tag Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-tag-keys).
     *
     * **Availability:** This key is included in the request context only if the operation supports attaching tags to resources.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a [multivalued](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html) request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see Using Multiple Keys and Values.
     *
     * Some services support tagging with resource operations, such as creating, modifying, or deleting a resource. To allow tagging and operations as a single call, you must create a policy that includes both the tagging action and the resource-modifying action. You can then use the `aws:TagKeys` condition key to enforce using specific tag keys in the request. For example, to limit tags when someone creates an Amazon EC2 snapshot, you must include the `ec2:CreateSnapshot` creation action ***and*** the `ec2:CreateTags` tagging action in the policy. To view a policy for this scenario that uses `aws:TagKeys`, see [Creating a Snapshot with Tags](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ExamplePolicies_EC2.html#iam-creating-snapshot-with-tags) in the *Amazon EC2 User Guide for Linux Instances*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The tag key(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifTagKeys(value, operator) {
        return this.if('aws:TagKeys', value, operator);
    }
    /**
     * Compare the date and time that temporary security credentials were issued with the date and time that you specify.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. They key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using access keys.
     *
     * To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tokenissuetime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateGreaterThanEquals`
     */
    ifTokenIssueTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        return this.if('aws:TokenIssueTime', value, operator || 'DateGreaterThanEquals');
    }
    /**
     * Compare the requester's client application with the application that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * **Warning:** This key should be used carefully. Since the `aws:UserAgent` value is provided by the caller in an HTTP header, unauthorized parties can use modified or custom browsers to provide any `aws:UserAgent` value that they choose. As a result, `aws:UserAgent` should not be used to prevent unauthorized parties from making direct AWS requests. You can use it to allow only specific client applications, and only after testing your policy.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-useragent
     *
     * @param value The User Agent string(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifUserAgent(value, operator) {
        return this.if('aws:UserAgent', value, operator);
    }
    /**
     * Compare the requester's principal identifier with the ID that you specify. For IAM users, the request context value is the user ID. For IAM roles, this value format can vary. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is included in the request context for all signed requests. Anonymous requests do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-userid
     *
     * @param value The principal identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifUserid(value, operator) {
        return this.if('aws:userid', value, operator);
    }
    /**
     * Compare the requester's user name with the user name that you specify. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context for IAM users. Anonymous requests and requests that are made using the AWS account root user or IAM roles do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-username
     *
     * @param value The user name(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifUsername(value, operator) {
        return this.if('aws:username', value, operator);
    }
    /**
     * Check whether an AWS service makes a request to another service on your behalf.
     *
     * The request context key returns `true` when a service uses the credentials of an IAM principal to make a request on behalf of the principal. The context key returns `false` if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. The request context key also returns `false` when the principal makes the call directly.
     *
     * **Availability:** This key is always included in the request context for most services.
     *
     *The following services do not currently support `aws:ViaAWSService`:
     *- Amazon EC2
     *- AWS Glue
     *- AWS Lake Formation
     *- AWS OpsWorks
     *
     * You can use this condition key to allow or deny access based on whether a request was made by a service. To view an example policy, see [AWS: Denies Access to AWS Based on the Source IP](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws_deny-ip.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice
     *
     * @param value Whether a request was made by a service. **Default:** `true`
     */
    ifViaAWSService(value) {
        return this.if('aws:ViaAWSService', value, 'Bool');
    }
    /**
     * Compare the IP address from which a request was made with the IP address that you specify. In a policy, the key matches only if the request originates from the specified IP address and it goes through a VPC endpoint.
     *
     * **Availability:** This key is included in the request context only if the request is made using a VPC endpoint.
     *
     * For more information, see [Controlling Access to Services with VPC Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) in the *Amazon VPC User Guide*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The VPC source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifVpcSourceIp(value, operator) {
        return this.if('aws:VpcSourceIp', value, operator || 'IpAddress');
    }
    /**
     * JSON-ify the policy statement
     *
     * Also adds `*` to the list of resources, if not was manually added
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        if (!this.hasResources()) {
            // a statement requires resources. if none was added, we assume the user wants all resources
            this.resources.push('*');
        }
        const statement = {};
        if (this.sid.length) {
            statement.Sid = this.sid;
        }
        statement.Effect = this.effect;
        if (this.hasActions()) {
            if (this.useNotActions) {
                statement.NotActions = this.actions;
            }
            else {
                statement.Action = this.actions;
            }
        }
        if (this.useNotResources) {
            statement.NotResource = this.resources;
        }
        else {
            statement.Resource = this.resources;
        }
        if (this.hasConditions()) {
            statement.Condition = this.conditions;
        }
        //TODO: principal
        return statement;
    }
}
exports.PolicyStatement = PolicyStatement;
//# sourceMappingURL=data:application/json;base64,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