import * as cdk8s from 'cdk8s';
import { Construct } from 'constructs';
/**
 * Jenkins plugin.
 */
export interface Plugin {
    /**
     * The url from where plugin has to be downloaded.
     * @default - Plugins are downloaded from Jenkins Update Centers.
     * @see https://github.com/jenkinsci/kubernetes-operator/blob/master/pkg/configuration/base/resources/scripts_configmap.go#L121-L124
     */
    readonly downloadUrl?: string;
    /**
     * The name of Jenkins plugin.
     */
    readonly name: string;
    /**
     * The version of Jenkins plugin.
     */
    readonly version: string;
}
/**
 * Jenkins seed job.
 */
export interface SeedJob {
    /**
     * The unique name for the seed job.
     */
    readonly id: string;
    /**
     * The description of the seed job.
     */
    readonly description: string;
    /**
     * The repository branch where seed job definitions are present.
     */
    readonly repositoryBranch: string;
    /**
     * The repository access URL. Supports SSH and HTTPS.
     */
    readonly repositoryUrl: string;
    /**
     * The repository path where seed job definitions are present.
     */
    readonly targets: string;
}
/**
 * Props for `Jenkins`.
 */
export interface JenkinsProps {
    /**
     * Metadata associated with Jenkins resource.
     *
     * @default: Default metadata values:
     *          {
     *              name: An app-unique name generated by the chart,
     *              annotations: No annotations,
     *              labels: { app: 'jenkins' },
     *              namespace: default,
     *              finalizers: No finalizers,
     *              ownerReferences: Automatically set by Kubernetes
     *          }
     */
    readonly metadata?: cdk8s.ApiObjectMetadata;
    /**
     * Toggle for CSRF Protection on Jenkins resource.
     *
     * @default - false
     */
    readonly disableCsrfProtection?: boolean;
    /**
     * List of plugins required by Jenkins operator.
     *
     * @default - Default base plugins:
     *
     *                { name: 'kubernetes', version: '1.31.3' },
     *                { name: 'workflow-job', version: '1145.v7f2433caa07f' },
     *                { name: 'workflow-aggregator', version: '2.6' },
     *                { name: 'git', version: '4.10.3' },
     *                { name: 'job-dsl', version: '1.78.1' },
     *                { name: 'configuration-as-code', version: '1414.v878271fc496f' },
     *                { name: 'kubernetes-credentials-provider', version: '0.20' }
     */
    readonly basePlugins?: Plugin[];
    /**
     * List of custom plugins applied to Jenkins resource.
     *
     * @default - []
     */
    readonly plugins?: Plugin[];
    /**
     * List of seed job configuration for Jenkins resource.
     * For more information about seed jobs, please take a look at { @link https://github.com/jenkinsci/job-dsl-plugin/wiki/Tutorial---Using-the-Jenkins-Job-DSL Jenkins Seed Jobs Documentation }.
     *
     * @default - No seed jobs
     */
    readonly seedJobs?: SeedJob[];
}
/**
 * A jenkins instance.
 */
export declare class Jenkins extends Construct {
    private _plugins;
    private _basePlugins;
    private _seedJobs;
    constructor(scope: Construct, id: string, props?: JenkinsProps);
    /**
     * Add base plugins to jenkins instance.
     * @param basePlugins List of base plugins.
     */
    addBasePlugins(...basePlugins: Plugin[]): void;
    /**
     * Add custom plugins to jenkins instance.
     * @param plugins List of custom plugins.
     */
    addPlugins(...plugins: Plugin[]): void;
    /**
     * Add seed jobs to jenkins instance.
     * @param seedJobs List of seed jobs.
     */
    addSeedJobs(...seedJobs: SeedJob[]): void;
}
